/* 
 * GIF Decoder 
 * (c) 1997 Fabrice BELLARD
 */

#include "global.h"
#include "images.h"

#ifndef VRENGD


typedef struct {
  void *th;		/* texture handle */
  ReadImageFunc read_func;
  DisplayImageFunc display_func;
  int	screenwidth;	/* The dimensions of the screen    */
  int	screenheight;	/*   (not those of the image)      */
  int	global;		/* Is there a global color map?    */
  int	globalbits;	/* Number of bits of global colors */
  int	backclr;	/* Background color                */
  int	colres;		/* Number of bits of color resol.  */
  int	*interleavetable;
  u_int8 globalmap[256][3];
  Image *screen;
/* lzw */
  int sizbuf, bbits;
  u_int32 bbuf;
  u_int8 buf[256], *pbuf;
  int cursize, codesize, clear_code, end_code, newcodes;
  int slot, fc, oc;
  u_int8 *stack, *suffix;
  short *prefix;
  u_int8 *sp;
} GifInfo;


static int offset;
static void GLZWDecodeInit(GifInfo *s, int csize);
static void GLZWDecode(GifInfo *s, u_int8 *buf, int len);
static int GLZWDecodeEnd(GifInfo *s);
static int gifReadSignature(GifInfo *s);
static int gifReadBlocks(GifInfo *s);
static int gifReadScreen(GifInfo *s);


Image *
loadGIF(void *th, ReadImageFunc read_func, DisplayImageFunc display_func)
{
  GifInfo s1, *s = &s1;

  s->display_func = display_func;
  s->read_func = read_func;
  s->th = th;
  s->screen = NULL;
  offset = 0;

  if (gifReadSignature(s))
    return NULL;
  if (gifReadScreen(s))
    return NULL;
  if (gifReadBlocks(s))
    return NULL;
  return s->screen;
}

/* screen data */
static inline
int gifread(GifInfo *s, u_int8 *buf, int len)
{
  return s->read_func(s->th, (char *) buf, len);
}

static
int gifGetByte(GifInfo *s)
{
  u_int8 ch;

  if ((gifread(s, &ch, 1)) != 1)
    return -1;
  offset++;
  return ch;
}

static
int gifReadSignature(GifInfo *s)
{
  u_int8 buf[6];

  if (gifread(s, buf, 6) != 6)
    return -2;
  if (strncmp((const char *) buf, "GIF", 3) ||
     (strncmp((const char *) &buf[3], "87a", 3) && 
      strncmp((const char *) &buf[3], "89a", 3)))
    return -3;
  return 0;
}

static
int gifReadScreen(GifInfo *s)
{
  u_int8 buf[7];
			
  if (gifread(s, buf, 7) != 7)
    return -4;
  if (buf[6])
    return -5;

  s->screenwidth  = buf[0] + (buf[1] << 8);
  s->screenheight = buf[2] + (buf[3] << 8);
  s->global	  = buf[4] & 0x80;
  s->colres	  = ((buf[4] & 0x70) >> 4) +1;
  s->backclr	  = buf[5];
  if (s->global) {
    s->globalbits = (buf[4] & 0x07) + 1;
    gifread(s, (u_int8 *) s->globalmap, 3*(1<<s->globalbits));
  }
  s->screen = newImage(s->screenwidth, s->screenheight, IMAGE_RGB, IMAGE_FIX);
  return (s->screen == NULL);
}

static
int gifReadImage(GifInfo *s)
{
  int i, j, c, line, pass, nb_pass, pass_height, line_start, line_inc;
  int left, top, width, height, local, localbits, ncolors, cbyte, interleaved;
  u_int8 *line_buf;
  u_int8 *pix,*pix1;
  u_int8 buf[10];
  u_int8 localmap[256][3];

  if ((gifread(s, buf, 10)) != 10)
    return -6;
  left		= buf[0] + (buf[1] << 8);
  top		= buf[2] + (buf[3] << 8);
  width		= buf[4] + (buf[5] << 8);
  height	= buf[6] + (buf[7] << 8);
  interleaved	= buf[8] & 0x40;
  local		= buf[8] & 0x80;
  if (left+width>s->screenwidth || top+height>s->screenheight)
    return -7;

  if (local) {
    localbits = (buf[8] & 0x7) + 1;
    ncolors = 1 << localbits;
    gifread(s, (u_int8 *) localmap, 3 * ncolors);
  }
  else if (s->global) {
    ncolors = (1 << s->globalbits);
    memcpy(localmap, s->globalmap, 3*ncolors);
  }
  else
    return -8;

  cbyte = buf[9];
  if (cbyte<2 || cbyte>8)
    return -9;

  GLZWDecodeInit(s, cbyte);

  if (interleaved)
    nb_pass=4;
  else
    nb_pass=1;
  line_buf = (u_int8 *) malloc(width);
  
  for (pass=0; pass < nb_pass; pass++) {
    switch(pass) {
    case 0:
      line_start=0;
      if (interleaved) {
	pass_height=8;
	line_inc=8;
      } else {
	pass_height=1;
	line_inc=1; 
      }
      break;
    case 1:
      pass_height=4;
      line_start=4;
      line_inc=8;
      break;
    case 2:
      pass_height=2;
      line_start=2;
      line_inc=4;
      break;
    default:
      pass_height=1;
      line_start=1;
      line_inc=2;
      break;
    }
    for (line=line_start; line < height; line += line_inc) {
      GLZWDecode(s, line_buf, width);
      pix = (u_int8 *)s->screen->pixmap + 
	   ((left + top * s->screenwidth) + (s->screenwidth * line)) * 3;
      pix1 = pix;
      for (i=0; i < width; i++) {
	c = line_buf[i];
	pix[0] = localmap[c][0];
	pix[1] = localmap[c][1];
	pix[2] = localmap[c][2];
	pix += 3;
      }
      if (pass_height > 1) {
	for (j=1; j < pass_height; j++) {
	  memcpy(pix, pix1, s->screenwidth*3);
	  pix += s->screenwidth*3;
	}
      }
      s->display_func(s->th, s->screen);
    }
  }
  GLZWDecodeEnd(s);
  free(line_buf);
  return 0;
}

static
int gifReadExtension(GifInfo *s)
{
  int code, count;
  u_int8 buf[255];

  code = gifGetByte(s);
  while (1) {
    count = gifGetByte(s);
    if (count == 0)
      break;
    gifread(s, buf, count);
    switch(code) {
    case 0xF9:
      /* graphic control extension */
      break;
    }
  }
  return 0;
}

/* read gif blocks until we find an image block */
static
int gifReadBlocks(GifInfo *s)
{
  int ch, err;

  while (1) {
    ch = gifGetByte(s);
    switch (ch) {
    case 0x2C:
      err = gifReadImage(s);
      if (err != 0) return err;
      break;
    case 0x3B:
      /* trailer */
      return 0;
    case 0x21:
      err = gifReadExtension(s);
      if (err != 0) return err;
      break;
    default:    
      trace(DBG_ZV, "gifReadBlocks: ch=%02x off=%d(0x%x)", ch, offset, offset);
      continue;
      return -10;
    }
  }
}	

/* LZW decompression */

#define MAXBITS		12
#define	SIZTABLE	(1<<MAXBITS)

static
void GLZWDecodeInit(GifInfo *s, int csize)
{
  s->codesize = csize;
  s->cursize = s->codesize + 1;
  s->clear_code = 1 << s->codesize;
  s->end_code = s->clear_code + 1;
  s->slot = s->newcodes = s->clear_code + 2;
  s->oc = 0;
  s->fc = 0;
  s->stack = (u_int8 *) malloc(SIZTABLE);
  s->suffix = (u_int8 *) malloc(SIZTABLE);
  s->prefix = (int16 *) malloc(SIZTABLE*sizeof(short));
  s->sp = s->stack;
  s->sizbuf = 1;
  s->bbuf = 0;
  s->bbits = 0;
}

static
int GLZWDecodeEnd(GifInfo *s)
{
  int err = 0, size;

  /* end block */
  if (s->sizbuf > 0) {
    size = gifGetByte(s);
    if (size != 0) err = -1;
  }
  free(s->stack);
  free(s->prefix);
  free(s->suffix);
  return err;
}


#define GETCODE(c)				\
{						\
  if (bbits>=cursize) {				\
    c=bbuf & curmask;				\
    bbuf>>=cursize;				\
    bbits-=cursize;				\
  } else {					\
    while (bbits<=24) {				\
      sizbuf--;					\
      if (sizbuf<=0) {				\
	if (sizbuf<0) {		\
	  sizbuf=0;				\
	} else {				\
	  sizbuf=gifGetByte(s);			\
	  gifread(s,s->buf,sizbuf);		\
	}					\
	pbuf=s->buf;				\
      }						\
      bbuf|=(int)(*pbuf++) << bbits;		\
      bbits+=8;					\
    }						\
    c=bbuf & curmask;				\
    bbuf>>=cursize;				\
    bbits-=cursize;				\
  }						\
}

/*
 * Gif LZW Decoder
 */
static
void GLZWDecode(GifInfo *s, u_int8 *out_buf, int out_len)
{
  int sizbuf,bbits;
  u_int32 bbuf;
  u_int8 *pbuf,*obuf;
  int olen;
  int c,code,cursize,curmask,clear_code,end_code,newcodes;
  int top_slot,slot,fc,oc;
  u_int8 *stack,*suffix,*sp;
  short *prefix;

/* cache rw */
  cursize = s->cursize;
  slot = s->slot;
  oc = s->oc;
  fc = s->fc;
  sp = s->sp;

  sizbuf=s->sizbuf;
  bbuf=s->bbuf;
  bbits=s->bbits;
  pbuf = s->pbuf;

/* cache r */
  clear_code = s->clear_code;
  end_code = s->end_code;
  newcodes = s->newcodes;
  clear_code = s->clear_code;
  stack=s->stack;
  prefix=s->prefix;
  suffix=s->suffix;

  top_slot = (1 << cursize); 
  curmask =  top_slot - 1;
  obuf=out_buf;
  olen=out_len;

  if (olen==0) goto the_end;

  while (sp > stack) {
    *obuf++ = *(--sp);
    if ( (--olen)==0 ) goto the_end;
  }

  while (1) {
    GETCODE(c);
    if (c == end_code) break;
    if (c == clear_code) {
      cursize=s->codesize+1;
      top_slot= 1 << cursize;
      curmask=top_slot - 1;
      slot=newcodes;
      do {
	GETCODE(c);
      } while (c == clear_code);
      if (c==end_code) break;
      fc=oc=c;
      *obuf++=c;
      if ( (--olen)==0 ) break;
    } else {
      code=c;
      if (code>=slot) {
	*sp++=fc;
	code=oc;
      }
      while (code>=newcodes) {
	*sp++=suffix[code];
	code=prefix[code];
      }
      *sp++=code;
      if (slot < top_slot) {
	suffix[slot] = fc = code;
	prefix[slot++] = oc;
	oc = c;
      }
      if (slot >= top_slot) {
	if (cursize < MAXBITS) {
	  cursize++;
	  top_slot <<= 1;
	  curmask=top_slot - 1;
	}
      }
      while (sp > stack) {
	*obuf++ = *(--sp);
	if ( (--olen)==0 ) goto the_end;
      }
    }
  }
 the_end:

/* cache rw */
  s->cursize = cursize;
  s->slot = slot;
  s->oc = oc;
  s->fc = fc;
  s->sp = sp;

  s->sizbuf = sizbuf;
  s->bbuf = bbuf;
  s->bbits = bbits;
  s->pbuf = pbuf;
}

#endif /* !VRENGD */
