#include "global.h"

#include "net.h"
#include "netdefs.h"
#include "dates.h"	/* diffDates */
#include "rtpsess.h"	/* Session */
#include "stat.h"

#include "world.h"	/* getCurrentChannelName */
#include "gui.h"	/* GuiGetCycles */


/*** network counters ***/
u_int32 pkts_sent = 0;
u_int32 sum_pkts_lost = 0;

static u_int32 pkts_recvd = 0;
static u_int32 bytes_sent = 0;
static u_int32 bytes_recvd = 0;
static u_int32 pkts_rtp_sent = 0;
static u_int32 bytes_rtp_sent = 0;
static u_int32 pkts_rtcp_sent = 0;
static u_int32 bytes_rtcp_sent = 0;
static u_int32 pkt_max = 0;
static u_int32 pkt_min = 512;
static u_int32 sum_pkts_sent = 0;
static u_int32 sum_pkts_recvd = 0;
static u_int32 sum_pkts_rtp_sent = 0;
static u_int32 sum_pkts_rtcp_sent = 0;
static u_int32 sum_bytes_sent = 0;
static u_int32 sum_bytes_rtp_sent = 0;
static u_int32 sum_bytes_rtcp_sent = 0;
static u_int32 sum_bytes_recvd = 0;
static float kbps_sent = 0.0;
static float kbps_recvd = 0.0;

/*** timers ***/
ProfileTime ptime_init;
ProfileTime ptime_net;
ProfileTime ptime_events;
ProfileTime ptime_world;
ProfileTime ptime_render;
ProfileTime ptime_textures;
ProfileTime ptime_solids;
ProfileTime ptime_buffer;


/* clear time */
void clearTime(ProfileTime *pt)
{
  memset((char *) pt, 0, sizeof(ProfileTime));
}

/* start time */
void startTime(ProfileTime *pt)
{
  gettimeofday(&(pt->start), NULL);
}

/* stop time & cumul time, return last diff */
double stopTime(ProfileTime *pt)
{
  double d;

  gettimeofday(&(pt->stop), NULL);
  d = diffDates(pt->start, pt->stop);
  pt->cumul += d;
  return d;
}

/* compute kbps_sent & co */
void statAdjust(void)
{
  static struct timeval then;
  struct timeval now;
  float d;

  gettimeofday(&now, NULL);
  d = diffDates(then, now);
  if (fabs(d) > 0.1) {
    kbps_sent = bytes_sent/d/1000*8;
    kbps_recvd = bytes_recvd/d/1000*8;
    trace(DBG_NET, "kbps sent: %5.2f recvd: %5.2f",
	   bytes_sent/d/1000*8, bytes_recvd/d/1000*8);
    then = now;
    bytes_sent = bytes_recvd = 0;  
  }
}

void statSendPacket(int pkt_len)
{
  bytes_sent += pkt_len;
  sum_bytes_sent += pkt_len;
  pkts_sent++;
  sum_pkts_sent++;
  pkt_max = MAXI((int) pkt_max, pkt_len);
  pkt_min = MINI((int) pkt_min, pkt_len);
}

void statSendRTP(int pkt_len)
{
  bytes_rtp_sent += pkt_len;
  sum_bytes_rtp_sent += pkt_len;
  pkts_rtp_sent++;
  sum_pkts_rtp_sent++;
}

void statSendRTCP(int pkt_len)
{
  bytes_rtcp_sent += pkt_len;
  sum_bytes_rtcp_sent += pkt_len;
  pkts_rtcp_sent++;
  sum_pkts_rtcp_sent++;
}

void statReceivePacket(int pkt_len)
{
  bytes_recvd += pkt_len;
  sum_bytes_recvd += pkt_len;
  pkts_recvd++;
  sum_pkts_recvd++;
  pkt_max = MAXI((int) pkt_max, pkt_len);
  pkt_min = MINI((int) pkt_min, pkt_len);
}

#ifndef VRENGD
static FILE *flog;
#endif

void statRTP(Session *psession)
{
#ifndef VRENGD

  if ((flog = writelog("--- RTP stat ---")) == (FILE *) NULL)
    return;
  writelog("worldname       : %s", getCurrentWorldName());
  writelog("channel         : %s", getCurrentChannelName());
  writelog("source ssrc     : %x", psession->sinfo->ssrc);
  writelog("sources number  : %d", psession->nbsources);
#if 0
  writelog("sdes item number: %d", psession->nbsdes);
  writelog("source max_seq  : %x", psession->sinfo->s.max_seq);
  writelog("source cycles   : %d", psession->sinfo->s.cycles);
  writelog("source bad_seq  : %x", psession->sinfo->s.bad_seq);
  writelog("source probation: %d", psession->sinfo->s.probation);
#endif
  writelog("source received : %d", psession->sinfo->s.received);
#if 0
  writelog("source sr ssrc  : %x", psession->sr.ssrc);
  writelog("source sr ts    : %x", psession->sr.rtp_ts);
#endif
  writelog("source sr psent : %d", psession->sr.psent);
  writelog("source sr osent : %d", psession->sr.osent);
  writelog("source lost     : %d", psession->sinfo->lost);
  writelog("source next     : %x", psession->sinfo->next);
#if 0
  writelog("source rr lost  : %d", psession->sinfo->rr.lost);
  writelog("source rr ssrc  : %x", psession->sinfo->rr.ssrc);
  writelog("source rr last  : %x", psession->sinfo->rr.last_seq);
  writelog("source rr frac  : %d", psession->sinfo->rr.fraction);
  writelog("source rr lsr   : %x", psession->sinfo->rr.lsr);
  writelog("source rr dlsr  : %x", psession->sinfo->rr.dlsr);
#endif
#endif /* !VRENGD */
}

float getRate(void)
{
#ifndef VRENGD
  float d;
  double time_cycles;

  d = stopTime(&ptime_net);
  time_cycles = ptime_events.cumul + ptime_world.cumul + ptime_render.cumul + ptime_buffer.cumul;
  return (float) (GuiGetCycles() / time_cycles);
#endif /* !VRENGD */
}

void statNetwork(void)
{
#ifndef VRENGD

  float d, bw;
  double time_cycles;

  if (flog != (FILE *) NULL) {
    closelog(flog);
    printlog();
  }

  fprintf(stderr, "### NET stat ###\n");
  d = stopTime(&ptime_net);
  bw = (float) ((sum_bytes_sent + sum_bytes_recvd + 2 * (8 + 20)) * 8) / d;
  time_cycles = ptime_events.cumul + ptime_world.cumul + ptime_render.cumul + ptime_buffer.cumul;

  if (sum_pkts_sent) {
    fprintf(stderr, "pkts sent       : %d\n", sum_pkts_sent);
    fprintf(stderr, "pkts_rtp sent   : %d\n", sum_pkts_rtp_sent);
    fprintf(stderr, "pkts_rtcp sent  : %d\n", sum_pkts_rtcp_sent);
    fprintf(stderr, "pkts sent/s     : %.0f/s\n", sum_pkts_sent/d);
  }
  if (sum_pkts_recvd) {
    fprintf(stderr, "pkts received   : %d\n", sum_pkts_recvd);
    fprintf(stderr, "pkts received/s : %.0f/s\n", sum_pkts_recvd/d);
    fprintf(stderr, "pkts lost       : %d\n", sum_pkts_lost);
    fprintf(stderr, "pkts lost %%     : %2.2f%%\n", ((double) 100*sum_pkts_lost)/(double) sum_pkts_recvd);
  }
  fprintf(stderr, "bytes sent      : %d\n", sum_bytes_sent);
  fprintf(stderr, "bytes_rtp sent  : %d\n", sum_bytes_rtp_sent);
  fprintf(stderr, "bytes_rtcp sent : %d\n", sum_bytes_rtcp_sent);
  fprintf(stderr, "bytes sent/s    : %.0f/s\n", sum_bytes_sent/d);
  if (sum_bytes_recvd) {
    fprintf(stderr, "bytes received  : %d\n", sum_bytes_recvd);
    fprintf(stderr, "bytes received/s: %.0f/s\n", sum_bytes_recvd/d);
  }
  fprintf(stderr, "pkt max         : %d\n", pkt_max);
  fprintf(stderr, "pkt min         : %d\n", pkt_min);
  if (sum_pkts_sent)
    fprintf(stderr, "bytes/pkt sent  : %d\n", sum_bytes_sent / sum_pkts_sent);
  if (sum_pkts_recvd)
    fprintf(stderr, "bytes/pkt recvd : %d\n", sum_bytes_recvd / sum_pkts_recvd);
  fprintf(stderr, "bw IP+UDP+RTP+PL: %.0f bps\n", bw);
  fprintf(stderr, "### Timings ###\n");
  fprintf(stderr, "session time    : %5.2f s\n", d);
  fprintf(stderr, "init time       : %5.2f s\n", (float) ptime_init.cumul);
  fprintf(stderr, "events time     : %5.2f s\n", (float) ptime_events.cumul);
  fprintf(stderr, "world time      : %5.2f s\n", (float) ptime_world.cumul);
  fprintf(stderr, "render time     : %5.2f s\n", (float) ptime_render.cumul);
  // fprintf(stderr, "textures time   : %5.2f s\n", (float) ptime_textures.cumul);
  // fprintf(stderr, "solid time      : %5.2f s\n", (float) ptime_solids.cumul);
  fprintf(stderr, "buffer time     : %5.2f s\n", (float) ptime_buffer.cumul);
  fprintf(stderr, "cycles          : %d\n", GuiGetCycles());
  fprintf(stderr, "cycles/s        : %5.2f/s\n", (float) (GuiGetCycles() / time_cycles));

  fprintf(stderr, "### Sources ###\n");
  dumpSources();

#endif /* !VRENGD */
}
