#ifndef NET_H
#define NET_H


#include "rtp.h"
#include "payload.h"

//
// NetObjectId
//
// A NetObjectId identifies a distributed object
//
typedef struct _NetObjectId {
  // private
  u_int32 src_id;
  u_int16 port_id;
  u_int16 obj_id;	// object id -- network format
} NetObjectId;

// 
// NetProperty
//
typedef struct _NetProperty {
  // private
  u_int8 responsible;
  u_int8 reserved;
  int16 version;
  struct timeval last_seen;
  struct timeval assume_at;
} NetProperty;

//
// Object Header
//
typedef struct _NetObject {
  struct _NetObjectId noid;	// Net object id
  u_int8 type;			// object type
  u_int8 permanent;		// Is object permanent ?
  u_int8 nbprop;		// number of properties
  // private
  struct _NetProperty *prop;	// property[countProperties] of Object
  struct _NetObject *next;	// next Object
  struct _NetObject *prev;	// prev Object
} NetObject;


//
// exports to wmgt
//

/* Create a new object name
 * now we can do getNetObjectId and declareDelta 
 * Il est souhaitable (perfs) de faire un declareObjCreation
 * des que toutes les proprietes sont initialisees.
 */
void createNetObject(NetObject *, u_int8);
void setNetObjectId(NetObject *);

/* Build an object name from a string
 * now we can do getNetObjectId and declareObjDelta 
 * a declareObjCreation on such object produces a fatal().
 */
void createNetObjectFromString(NetObject *, const char*, u_int8);

/* Destroy the object name
 * after this call a getNetObjectId or a * declareObjxxxx will fail
 * to do necessarly before the final free()
 */
void deleteNetObject(NetObject *);

/* Devrait (perfs) etre appelee apres le nommage d'un nouvel objet
 * (createNetObject) et l'initialisation des proprietes.  
 */
void declareObjCreation(NetObject *);

/* A appeler apres changement de valeur de la propriete */ 
void declareObjDelta(NetObject *, u_int8 prop_id );

/* A appeler quand on veut detruire l'objet, juste avant de faire
 * un deleteNetObject puis le free() final.
 */
void declareObjDeletion(NetObject *);

void declarePersistInfo(NetObject *pnoh);

struct _Payload;
void incomingPersist(struct sockaddr_in *, struct _Payload *);
void incomingWorld(struct sockaddr_in *, struct _Payload *);
void incomingOther(int fd, struct sockaddr_in *, struct _Payload *, int len);
// Delta
void sendObjDelta(NetObject *, u_int8);

//
// exports to GUI
//

/* Awake the net when a timeout occurs
 * return the delay after which we want to be awaked (ms)
 */
int networkTimeout(void);

/* Awake the net when data arrive on fd "on" */
void incoming(int);

//
// RTP
//

u_int16 getNumberOfMembers(void);

const char * AvatarName(void);
const char * RtcpEmail(void);
const char * RtcpName(void);
char * ToolName(void);
char * getRtcpNameBySsrc(u_int32);
char * getRtcpEmailBySsrc(u_int32);
char * getRtcpToolBySsrc(u_int32);
char * getSdesItemBySsrc(u_int8, u_int32);

void RtpClearSessionsList(void);
void RtpDeleteSession(u_int32);
void dumpSessions(void);
void dumpSources(void);


#endif /* NET_H */
