/* ==================================================== ======== ======= *
 *
 *  ubxwin.cc : X-Window initialization and management for the VREng GUI
 *  NOTE: this file should be common to all X-Window GUI variants
 *
 *  VREng Project [Elc::nov2001]
 *  Author: Eric Lecolinet
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 */

#ifndef VRENGD

#include "ubit.hh"
#include "ugraph.hh"
#include "unat.hh"

#include "global.h"
#include "zv.h"		// RenderInit
#include "net.h"	// netTimeOut
#include "stat.h"	// ptime_*
#include "world.h"	// doWorldCalculation

#include "guiclass.h"
#include "gui.h"
#include "guiImpl.hh"
#include "widgets.hh"

//catches some X undesirable errors
static void catchXErrors();

// calculates the New World Order and renders it
static Boolean renderingWorkProc(XtPointer pgui);

// checks regularly for various updates
static void netTimeOut(XtPointer pgui, XtIntervalId *iid);


/* ==================================================== ======== ======= */

void GUI::createToplevel(int argc, char *argv[], char **fallback_resources) {

  // creates the Ubit Application Context
  appli = new UAppli(&argc, argv);
  if (!appli) fatal("GUI::createToplevel: Toplevel could not be created");

  // UNatAppli: X Lib implementation of the UAppli
  // makes it possible to retreive the X display, screen, etc.
  // that were initialized by Ubit 

  UNatAppli* natapp = appli->getNatAppli(); 
  display    = natapp->getXDisplay();
  screen     = natapp->getXScreen();
  appContext = natapp->getXContext();
  int scr_no = XScreenNumberOfScreen(screen);
  int dummy, glxmajor = 0, glxminor = 0;

  // Check if GLX is supported
  if (!glXQueryExtension(display, &dummy, &dummy))
    fprintf(stderr, "X server has no OpenGL GLX extension\n");

#if !defined(WITH_TINYGL)
  /* Get GLX version */
  glXQueryVersion(display, &glxmajor, &glxminor);
  // pas notice() car les widgets ne sont pas encore crees!
  fprintf(stderr, "glXQueryVersion: GLX%d.%d\n", glxmajor, glxminor);
#endif

  /* TODO: test if GLX1.3 to do glxCreateWindow */
  static int gl_config[] = {
    GLX_RGBA, GLX_DEPTH_SIZE, 16, GLX_DOUBLEBUFFER, None };

  // find a Visual that match requested depth and OpenGL options
  // (note that we use double buffering)
  if (! (glvisual = glXChooseVisual(display, scr_no, gl_config)))
    fprintf(stderr, "could not get appropriate OpenGL visual\n");
  
  trace(DBG_WIN, "glvisual: depth=%d, visualid=%x, class=%d [PseudoColor=%d, TrueColor=%d]", 
	glvisual->depth, glvisual->visualid, getVisualClass(glvisual),
	PseudoColor, TrueColor);
 
  // tells Ubit to use this GL specific visual
  // (note this fct also initializes the Colormap)
  natapp->setVisualProps(glvisual);

  // an explicit call of realize() is needed now in order to create
  // the toplevel now
  appli->realize();

  //var supprimee: toplevel = natapp->getXWidget();
  cmap = natapp->getXColormap();
  sceneInitialized = FALSE;
}

/* ==================================================== ======== ======= */

void GUI::initX() {} //ne sert plus a rien

/* ==================================================== ======== ======= */

void GUI::mainLoop() {
  //catches some X undesirable errors
  catchXErrors();

  // gets the X Window of the GL Zone that was created by Ubit
  // Note: must be done after 'new GuiWidgets" (of course!)
  glwin = guiWidgets->getGLZone()->getNatWin()->getXWindow();

  // Create an OpenGL rendering context
  if (! (glxc = glXCreateContext(display, glvisual, None, True)))
    fatal("could not create OpenGL rendering context");

  // here, we call GL & ZLib initialization
  glXMakeCurrent(display, glwin, glxc);
  RenderInit(resources.quality);
  RenderSetWindow(resources.width3D, resources.height3D);
  trace(DBG_INIT, "GUI X-Window initialized");

  // get the toplevel size (which is stored for managing window resizes)
  //guiWidgets->getFrame()->getSize(appWidth, appHeight);

  // winResized() is called when the GLZone is resized to tell OGL
  // that the size of the RenderBuffer has changed
  guiWidgets->getGLZone()
    ->add(UOn::viewResize / ucall(this, &GUI::resizeGLBuffer));

  // !!A FAIRE: rajouter un callback pour arreter les traitements qunad
  // !! la fenetre est iconifiee

  XtAppAddWorkProc(appContext, renderingWorkProc, (XtPointer)this);

  // netTimeOut is called regularly during the session
  XtAppAddTimeOut(appContext, NET_TIMEOUT, netTimeOut, (XtPointer)this);

  // opens the "Please Wait" alert box
  //guiWidgets->alert("Please wait: loading the World...");

  //!ATT: ne PLUS appeler XtAppMainLoop(appContext) !!!
  appli->mainLoop();
}

/* ==================================================== ======== ======= */
// called once after the mainloop is started for init. World mgt and rendering

void GUI::initScene() {
  // Note: as this init. may be quite long, it is postponed so that
  // we can start the main loop immediately then run this init.
  // (en pratique c'est WmgtInit(), qui DOIT etre lancee APRES initRender())
  if (vrengInitCB) vrengInitCB();

  // closes the "Please Wait" alert box
  guiWidgets->alert(null);
  stopTime(&ptime_init);

  // the scene is initialized and ready for rendering
  sceneInitialized = TRUE;
  // note: this var is supposed to change when the windo is iconified
  readyAndVisible = TRUE;
}

/* ==================================================== ======== ======= */
// calculates the New World Order and renders it

static Boolean renderingWorkProc(XtPointer pgui) {
  ((GUI*)pgui)->renderScene();
  return False;  // call this function again
}

void GUI::renderScene() {
  if (!sceneInitialized) {
    sceneInitialized = TRUE;
    initScene();
  }

  // at least one postponed Key Release event
  if (pendingPostponedKRs()) {
    flushPostponedKRs();
  }
  
  // the GUI is ready for rendering and is visible
  // (no rendering is performed while iconified)
  if (readyAndVisible) {      

    // Performs virtual world calculation (cf. wmgt.c)
    startTime(&ptime_world);
    doWorldCalculation(ptime_world.start.tv_sec, ptime_world.start.tv_usec);
    stopTime(&ptime_world);
    
    // compute the 3D
    startTime(&ptime_render);
    Render();
    stopTime(&ptime_render);
    // displays on the GL rendering window
    startTime(&ptime_buffer);

    // swap the display and drawing buffers (double buffering mode) :
    // the scene we just rendered in the double buffer is now shown on the screen
    glXSwapBuffers(display, glwin);

    // redraw the Ubit menu (when it is open) on top of the scene 
    // that was just drawn

    if (guiWidgets->getOpenedMenu()) {
      guiWidgets->getOpenedMenu()->update(UUpdate::paint);
      // this is necessary to avoid flicker
      XFlush(display);
    }

    stopTime(&ptime_buffer);
    // count cycles
    cycles++;
  }
}

/* ==================================================== ======== ======= */
/* ==================================================== ======== ======= */
// Handlers and TimeOuts

void GUI::resizeGLBuffer(UEvent* e) {
  UView* glview = e->getView();
  if (glview) {
    resources.width3D  = glview->getWidth();
    resources.height3D = glview->getHeight();
    RenderSetWindow(resources.width3D, resources.height3D);
  }
}

/* ==================================================== ======== ======= */

// This timeOut cheks regularly if various updates are needed
static void netTimeOut(XtPointer pgui, XtIntervalId *iid) {
  GUI *gui = (GUI*)pgui;

  // networkTimeout() cheks if various updates are needed
  u_int32 next_timeout = networkTimeout();

  // TimeOuts are only called ONCE by Xt ==> register this callback again
  // with the same call_data
  XtAppAddTimeOut(gui->appContext, next_timeout, netTimeOut, (XtPointer)gui);
}

/* ==================================================== ======== ======= */
// called when something occurs on a file-descriptor

static void FDEvent(XtPointer, int *fd, XtInputId *iid) {
  incoming(*fd);
}

void GUI::addInputTable(int cnt, int *table, int table_no) {
  //printf("addInputTable: table_no=%d\n", table_no);
  inputTable[table_no] = (XtInputId*) malloc(cnt * sizeof(XtInputId));
  for (int k=0; k < cnt; k++)
    inputTable[table_no][k] = XtAppAddInput(appContext, table[k],
					    (XtPointer) XtInputReadMask, 
					    FDEvent, (XtPointer)NULL);
}

void GUI::removeInputTable(int cnt, int table_no)
{
  //printf("removeInputTable: table_no=%d\n", table_no);
  for (int k=0; k < cnt; k++)
    XtRemoveInput(inputTable[table_no][k]);

  free(inputTable[table_no]);
  inputTable[table_no] = NULL;
}

/* ==================================================== ======== ======= */
// for Handling X errors

// this variable will point to the previous (= standard) error handler
static int (*std_x11error_handler)(Display *, XErrorEvent *);

static int MyX11ErrorHandler(Display *d, XErrorEvent *ev) {
  /* errors for shared-mem */
  if (ev->request_code == 129) {
    fprintf(stderr, "Shared memory unavailable\n");
  }
  else {
    QuitVreng(0);
    std_x11error_handler(d, ev);
    exit(1);
  }
  return 0;
}

static void catchXErrors() {
  std_x11error_handler = XSetErrorHandler(MyX11ErrorHandler);
}

/* ==================================================== ======== ======= */
/* ce code n'est plus utilise
   a conserver temporairement pour prise en compte ulterieure
   de l'iconification de la fenetre

  XtAddEventHandler(toplevel, StructureNotifyMask, False, 
		    configureEH, (XtPointer)this);

static void configureEH(Widget, XtPointer pgui, XEvent *ev, Boolean*)
{
  GUI *gui = (GUI*)pgui;

  if (ev->type == MapNotify) {
    // visible again: restart rendering
    gui->readyAndVisible = TRUE;
  }
  else if (ev->type == UnmapNotify) {
    // iconified => stop rendering
    gui->readyAndVisible = FALSE;
    // a quoi ca pourrait bien servir de reactualiser le Monde qd l'appli
    // est iconifiee ?
    //XtAppAddTimeOut(appContext, WMGT_TIMEOUT, TimeOut, (XtPointer)WMGTTO);
  }
  else if (ev->type == DestroyNotify) {
    fprintf(stderr, ">DestroyNotify!!\n");
  }

  // !Caution: we don't know if the window was moved or resized
  // => test if the size has changed
  else if (ev->type == ConfigureNotify
	   && (gui->appWidth != ev->xconfigure.width
	       || gui->appHeight != ev->xconfigure.height)
	   ) {
    resources.width3D  += ev->xconfigure.width - gui->appWidth;
    resources.height3D += ev->xconfigure.height - gui->appHeight;
    gui->appWidth  = ev->xconfigure.width;
    gui->appHeight = ev->xconfigure.height;

    // change sizes of:
    // - the GLZone widget of the GUI
    gui->guiWidgets->resizeGLZone(resources.width3D, resources.height3D);

    // - the cooresponding OpenGL Viewport
    RenderSetWindow(resources.width3D, resources.height3D);
  }
}
*/

#endif /* !VRENGD */

