/* ==================================================== ======== ======= *
 *
 *  amwidgets.cc : Athena and Motif Widgets for the VREng GUI
 *
 *  VREng Project
 *  Author: Eric Lecolinet @ ENST Paris (elc@enst.fr)
 *  Date:   24/11/00
 *
 *  WWW: http://www.enst.fr/~elc  http://www.infres.enst.fr/net/vreng/
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 */

#ifndef VRENGD

#include "global.h"

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#ifndef WANT_MOTIF
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/Paned.h>
#else
#include <Xm/XmAll.h>
#endif

#include "net.h"
#include "zv.h"
#include "wobject.h"
#include "world.h"
#include "keys.h"
#include "move.h"		/* Move */
#include "user.h"		/* FOVY* */
#include "helpers.h"

#include "gui.h"
#include "guiImpl.hh"
#include "widgets.hh"


// MACROS form Motif & Athena Form Attachments

#if WANT_MOTIF
// XXX_F : attach to Form Widget
#define TOP_F	    XmNtopAttachment,   XmATTACH_FORM
#define BOTTOM_F    XmNbottomAttachment,XmATTACH_FORM
#define LEFT_F	    XmNleftAttachment,  XmATTACH_FORM
#define RIGHT_F	    XmNrightAttachment, XmATTACH_FORM
// XXX_W(y) : attach to (y) widget
#define TOP_W(w)    XmNtopAttachment,   XmATTACH_WIDGET, XmNtopWidget,   (w)
#define LEFT_W(w)   XmNbottomAttachment,XmATTACH_WIDGET, XmNleftWidget,  (w)
#define BOTTOM_W(w) XmNleftAttachment,  XmATTACH_WIDGET, XmNbottomWidget,(w)
#define RIGHT_W(w)  XmNrightAttachment, XmATTACH_WIDGET, XmNrightWidget, (w)

#else //ATHENA

#define TOP_F	    XtNtop,   XtChainTop
#define BOTTOM_F    XtNbottom,XtChainBottom
#define LEFT_F	    XtNleft,  XtChainLeft
#define RIGHT_F	    XtNright, XtChainRight
#define TOP_W(w)    XtNfromVert, (w)
#define LEFT_W(w)   XtNfromHoriz,(w)
//NB: BOTTOW_W() and RIGHT_W() are not available with Athena
#endif

// forward defs

static Widget CreateLabel(Widget parent, String name);
static Widget CreateSeparator(Widget parent, String name);
static Widget CreateButton(Widget parent, String name, void (*fun)(int), int farg);
static Widget CreateMButton(Widget parent, String name, void (*fun)(int), int farg);
static Widget CreateToggle(Widget parent, String name, void (*fun)(int));
static Widget CreateRadio(Widget parent, String name, int *no, Widget *radiogroup,
			  void (*fun)(int), int farg);
static Widget CreateButtonAndMenu(Widget menubar, String name);
static Widget CreateButtonAndOptionMenu(Widget menubar, String name);

static void SetLabelValue(Widget ww, const char *value);
static void SetTextFieldValue(Widget ww, const char *s);
static char *GetTextFieldValue(Widget ww);
static void SetTextAreaValue(Widget ww, const char *s, boolean add_at_the_end);
static Boolean GetToggleValue(Widget togglew);
static void SetToggleValue(Widget togglew, Boolean state);
#ifndef WANT_MOTIF
Pixmap TOGGLE_ON_PIXMAP = XtUnspecifiedPixmap;
Pixmap TOGGLE_OFF_PIXMAP = XtUnspecifiedPixmap;
static void initTogglePixmap(Widget toplevel);
#endif


GuiWidgets::GuiWidgets(GUI *pgui) {
  Widget btn;
  gui = pgui;	// points to the containing GUI class

  // Creates the Main Window (also called the "Main Frame")
#if WANT_MOTIF
  frame = XtCreateManagedWidget("frame", xmFormWidgetClass, gui->toplevel, NULL, 0);
#else
  frame = XtCreateManagedWidget("frame", panedWidgetClass, gui->toplevel, NULL, 0);
#endif

  // MenuBar of the Appli
  // (NB: there is no real "menu bar" widget in the Athena toolkit)

#if WANT_MOTIF
  menuBar = XtVaCreateManagedWidget("menuBar", xmRowColumnWidgetClass, frame,
				    XmNrowColumnType, XmMENU_BAR, 
				    TOP_F, LEFT_F, RIGHT_F, NULL);
#else
  menuBar = XtVaCreateManagedWidget("menuBar", boxWidgetClass, frame, 
				    XtNorientation, XtorientHorizontal,
				    XtNshowGrip, False, NULL);
#endif

  CreateButton(menuBar, "Quit", GUI::quit, 0);
#if 0
  CreateButton(menuBar, "Edit", GUI::edit, 0);
#endif
  CreateButton(menuBar, "Back", GUI::back, 0);
  CreateButton(menuBar, "Forward", GUI::forward, 0);
  CreateButton(menuBar, "Home", GUI::home, 0);
#if 0
  CreateButton(menuBar, "Views", GUI::views, 0);
  CreateButton(menuBar, "Bookmarks", GUI::bookmarks, 0);
#endif

  // creates the tool menu
  Widget toolMenu = CreateButtonAndMenu(menuBar, "Tools");
  CreateToggle(toolMenu, "Audio", GUI::setAudio);
  CreateToggle(toolMenu, "Video", GUI::setVideo);
  CreateToggle(toolMenu, "Whiteboard", GUI::setWhiteboard);
  CreateToggle(toolMenu, "Slidecast", GUI::setSlidecast);
  CreateToggle(toolMenu, "Jmrc", GUI::setJmrc);
  CreateToggle(toolMenu, "Modeler", GUI::setModeler);

  // creates the option menu
  CreateButtonAndOptionMenu(menuBar, "Options");

#if 0
  CreateButton(menuBar, "Stats", GUI::stats, 0);
#endif

  // creates the help button
  btn = CreateButton(menuBar, "Help", GUI::help, 0);
#if WANT_MOTIF
  // sets the Help button at the right side of the menubar
  XtVaSetValues(menuBar, XmNmenuHelpWidget, btn, NULL);
#endif

  // toolBar: will contain the objectBar selection when available

  objectBar = NULL; //created when an object is selected
  objectMenu = NULL;

#if WANT_MOTIF
  Widget framed =
    XtVaCreateManagedWidget("framedBox", xmFrameWidgetClass, frame,
			    TOP_W(menuBar), LEFT_F, RIGHT_F, NULL);
  toolBar = 
    XtVaCreateManagedWidget("toolBar", xmRowColumnWidgetClass, framed, 
			    XmNorientation, XmHORIZONTAL,
			    XmNresizeHeight, False,
			    XmNheight, TOOLBAR_HEIGHT,
			    //TOP_W(menuBar), LEFT_F, RIGHT_F, 
			    NULL);
  //just to make it bigger!
  CreateLabel(toolBar, "Selection:");
#else
  toolBar = 
    XtVaCreateManagedWidget("toolBar", formWidgetClass, frame, 
			    XtNorientation, XtorientHorizontal,
			    XtNshowGrip, False, NULL);
  btn = CreateLabel(toolBar, "No selection");
  XtVaSetValues(btn, TOP_F, LEFT_F, BOTTOM_F, XtNright, XtChainLeft, NULL);
#endif


  // WorkArea that contains the GL rendering zone and the 2 lists

#if WANT_MOTIF
  Widget workArea = 
    XtVaCreateManagedWidget("workArea", xmFormWidgetClass, frame,
			    TOP_W(toolBar), LEFT_F, RIGHT_F, NULL);
#else
  // seule cette combinaison (workArea = Box) semble etre OK avec les window resize 
  Widget workArea = 
    XtVaCreateManagedWidget("workArea", boxWidgetClass, frame, 
			    XtNorientation, XtorientHorizontal,
			    XtNshowGrip, False,
			    // faire en sorte que ce soit workArea qui change 
			    // de taille qd on resize la fenetre
			    XtNresizeToPreferred, True,
			    XtNallowResize, True,
			    NULL);
#endif

  // Drawing Area = GL rendering zone

#if WANT_MOTIF
  glzone = 
    XtVaCreateManagedWidget("glzone", xmDrawingAreaWidgetClass, workArea,
			    XmNwidth, (XtArgVal)resources.width3D,
			    XmNheight, (XtArgVal)resources.height3D, 
			    TOP_F, LEFT_F, BOTTOM_F, NULL);
#else
  glzone = 
    XtVaCreateManagedWidget("glzone", labelWidgetClass, workArea,
			    XtNwidth, (XtArgVal)resources.width3D,
			    XtNheight, (XtArgVal)resources.height3D, 
			    XtNlabel, PLEASE_WAIT,
			    NULL);
#endif

  XtAddEventHandler(glzone, KeyPressMask, False,
		    glzoneKeyPressEH, (XtPointer)this);
  XtAddEventHandler(glzone, KeyReleaseMask, False,
		    glzoneKeyReleaseEH, (XtPointer)this);

  // attention: handler sur toplevel pas sur glzone car celle-ci est cachee
  // par la window d'affichage effectif d'OpenGL qui lui est superposee
  // (donc les ButtonPress sur glzone ne seraient jamais generes)
  // Par contre, le KeyPressMask est bien sur glzone car le focus est
  // *explicitement* redirige sur cette zone par glzoneSetFocusEH & co.

  XtAddEventHandler(gui->toplevel, ButtonPressMask, False,
		    glzoneMousePressEH, (XtPointer)this);
  // move avec bouton defonce
  //XtAddEventHandler(gui->toplevel, PointerMotionMask, False,
  //		    glzoneMouseMoveEH, (XtPointer)this);

  // The rightpanel that contains both lists


#if WANT_MOTIF
  rightPanel =  
    XtVaCreateManagedWidget("rightPanel", xmFormWidgetClass, workArea,
			    TOP_F, RIGHT_F, BOTTOM_F, NULL);
  XtVaSetValues(glzone, RIGHT_W(rightPanel), NULL);
#else
  rightPanel = 
    XtVaCreateManagedWidget("rightPanel", panedWidgetClass, workArea,
			    XtNorientation, XtorientVertical, NULL);
#endif

  btn = CreateButton(rightPanel, "VRENG", GUI::help, 0);
  SetLabelValue(btn, "VREng "VERSION);
#if WANT_MOTIF
  XtVaSetValues(btn, TOP_F, LEFT_F, RIGHT_F, NULL);
#else
  XtVaSetValues(btn, XtNshowGrip, False, NULL);
#endif


#if WANT_MOTIF
  Widget listbox =
    XtVaCreateManagedWidget("listBox", xmFormWidgetClass, rightPanel,
			    TOP_W(btn), LEFT_F, RIGHT_F, BOTTOM_F, NULL);

  //NB: no relationship with the Main Frame !
  Widget framedBox1 = 
    XtVaCreateManagedWidget("framedBox", xmFrameWidgetClass, listbox, 
			    XmNbottomAttachment, XmATTACH_POSITION,
			    XmNbottomPosition,  (XtArgVal)50, 
			    TOP_F, LEFT_F, RIGHT_F, NULL);
  // add title to framed box
  Widget label = CreateLabel(framedBox1, "Worlds");
  XtVaSetValues(label, XmNchildType, XmFRAME_TITLE_CHILD, NULL);

  worldsPane = 
    XtVaCreateManagedWidget("worldsPane", xmScrolledWindowWidgetClass, framedBox1,
			    XmNscrollingPolicy, XmAUTOMATIC,
			    // scrollbars always displayed
			    //XmNscrollBarDisplayPolicy, XmSTATIC, 
			    NULL);
  worlds = 
    XtVaCreateManagedWidget("worlds", xmRowColumnWidgetClass, worldsPane,
			    XmNorientation, XmVERTICAL, NULL);

  Widget framedBox2 = 
    XtVaCreateManagedWidget("framedBox", xmFrameWidgetClass, listbox,
			    XmNtopAttachment, XmATTACH_POSITION,
			    XmNtopPosition, (XtArgVal)50,
			    LEFT_F, RIGHT_F, BOTTOM_F, NULL);
  // add title to framed box
  label = CreateLabel(framedBox2, "Avatars");
  XtVaSetValues(label, XmNchildType, XmFRAME_TITLE_CHILD, NULL);

  usersPane = 
    XtVaCreateManagedWidget("usersPane", xmScrolledWindowWidgetClass, framedBox2,
			    XmNscrollingPolicy, XmAUTOMATIC, NULL);
  users =
    XtVaCreateManagedWidget("users", xmRowColumnWidgetClass, usersPane,
			    XmNorientation, XmVERTICAL, NULL);


#else // ATHENA

  btn = CreateLabel(rightPanel, "Worlds");
  XtVaSetValues(btn, XtNshowGrip, False, NULL);
  worldsPane = 
    XtVaCreateManagedWidget("worldsPane", viewportWidgetClass, rightPanel,
			    XtNallowHoriz, FALSE,
			    XtNallowVert, TRUE,
			    //XtNforceBars, TRUE,
			    XtNshowGrip, False,
			    NULL);
  worlds = 
    XtVaCreateManagedWidget("worlds", boxWidgetClass, worldsPane,
			    XtNorientation, XtorientVertical, NULL);

  btn = CreateLabel(rightPanel, "Avatars");
  XtVaSetValues(btn, XtNshowGrip, False, NULL);
  usersPane = 
    XtVaCreateManagedWidget("usersPane", viewportWidgetClass, rightPanel,
			    XtNallowHoriz, FALSE,
			    XtNallowVert, TRUE,
			    //XtNforceBars, TRUE,
			    XtNshowGrip, False,
			    NULL);
  users =
    XtVaCreateManagedWidget("usersPane", boxWidgetClass, usersPane,
			    XtNorientation, XtorientVertical, NULL);
#endif


  // IO Pane: conatisn the message and entry panes

#if WANT_MOTIF
  Widget ioPane = 
    XtVaCreateManagedWidget("ioPane", xmFormWidgetClass, frame,
			    TOP_W(workArea),
			    LEFT_F, RIGHT_F, BOTTOM_F, NULL);
#else
  Widget ioPane = 
    XtVaCreateManagedWidget("ioPane", formWidgetClass, frame, 
			    XtNshowGrip, False,
			    XtNresizeToPreferred, False,
			    NULL);
#endif


#if WANT_MOTIF
  Widget messagesPane =
    XtVaCreateManagedWidget("messagesPane", xmScrolledWindowWidgetClass, ioPane, 
			    TOP_F, LEFT_F, RIGHT_F, NULL);
  messages = 
    XtVaCreateManagedWidget("messages", xmTextWidgetClass, messagesPane,
			    XmNscrollHorizontal, (XtArgVal)False,
			    XmNeditMode, (XtArgVal)XmMULTI_LINE_EDIT,
			    XmNheight, (XtArgVal)MESSAGES_HEIGHT,
			    XmNeditable, (XtArgVal)False,
			    XmNvalue, VRENG_WELCOME"\n",
			    NULL);
#else
  messages = 
    XtVaCreateManagedWidget("messages", asciiTextWidgetClass, ioPane,
			    XtNscrollHorizontal, (XtArgVal)XawtextScrollWhenNeeded,
			    XtNscrollVertical, (XtArgVal)XawtextScrollWhenNeeded,
			    XtNeditType, (XtArgVal)XawtextRead,
			    XtNdisplayCaret, (XtArgVal)False,
			    XtNheight, (XtArgVal)MESSAGES_HEIGHT,
			    XtNstring, VRENG_WELCOME"\n",
			    TOP_F, LEFT_F, RIGHT_F, NULL);
#endif


#if WANT_MOTIF
  entry = 
    XtVaCreateManagedWidget("entry", xmTextFieldWidgetClass, ioPane, 
			    TOP_W(messages), LEFT_F, RIGHT_F, BOTTOM_F, NULL);

  XtAddCallback(entry, XmNactivateCallback, &GuiWidgets::entryActionCB,
		(XtPointer)this);
#else
  entry = 
    XtVaCreateManagedWidget("entry", asciiTextWidgetClass, ioPane, 
			    XtNeditType, (XtArgVal)XawtextEdit, //can edit
			    TOP_W(messages), LEFT_F, RIGHT_F, BOTTOM_F, NULL);

  // see note at fct entryActionEH
  XtAddEventHandler(entry, KeyPressMask, False, &GuiWidgets::entryActionEH, 
		    (XtPointer)this);
#endif

  // the entry widget gets the keyboard focus when the mouse enters this zone
  XtAddEventHandler(entry, EnterWindowMask, False,&GuiWidgets::entrySetFocusEH, 
		    (XtPointer)this);

  // the glzone widget gets the keyboard focus when the mouse leaves the entry
  // widget (so that the glzone will get the focus whenever the mouse it outside 
  // the entry widget)
  // Note: this is not sufficient when the user clicks a (menu/menubar) button 
  // as this button will get the focus and won't give it back to the glzone 
  // [this problems occurs with Motif not with Athena]
  // => a press action in the glzone will also change the focus

  XtAddEventHandler(entry, LeaveWindowMask, False,&GuiWidgets::glzoneSetFocusEH, 
		    (XtPointer)this);

  // marche pas car il y a une autre window (celle ou est effectivement affiche
  // le graphique) juste au dessus!
  //XtAddEventHandler(glzone,EnterWindowMask, False,&GuiWidgets::glzoneSetFocusEH, 
  //		    (XtPointer)this);


  XtRealizeWidget(gui->toplevel);
  Position top_rootx, top_rooty, glzone_rootx, glzone_rooty;

  XtTranslateCoords(gui->toplevel, 0, 0, &top_rootx, &top_rooty);
  XtTranslateCoords(glzone, 0, 0, &glzone_rootx, &glzone_rooty);
  glwin_x = glzone_rootx - top_rootx;
  glwin_y = glzone_rooty - top_rooty;

  XtSetKeyboardFocus(gui->toplevel, glzone);

  // moveBar not available in this version
  moveBar = NULL;

#ifndef WANT_MOTIF
  // necessaire pour Athena
  initTogglePixmap(gui->toplevel);
  XtVaSetValues(rightPanel, XtNheight, (XtArgVal)resources.height3D, NULL);
  XtVaSetValues(worldsPane, XtNmin, (XtArgVal)resources.height3D/3, NULL);
  XtVaSetValues(usersPane,  XtNmin, (XtArgVal)resources.height3D/3, NULL);
#endif
}


void GuiWidgets::resizeGLZone(int width, int height) {

  XtVaSetValues(glzone,
		XtNwidth,  (XtArgVal)width, 
		XtNheight, (XtArgVal)height,
		NULL);
#ifndef WANT_MOTIF
  // necessaire pour Athena
  XtVaSetValues(rightPanel, XtNheight, (XtArgVal)height, NULL);
  XtVaSetValues(worldsPane, XtNmin, (XtArgVal)resources.height3D/3, NULL);
  XtVaSetValues(usersPane,  XtNmin, (XtArgVal)resources.height3D/3, NULL);
#endif
}

// parent must be the menubar
//
static Widget CreateButtonAndMenu(Widget menubar, String name) {
  char *menu_name = (char*)malloc(strlen(name) + 10);
  sprintf(menu_name, "%s_menu", name);
#if WANT_MOTIF
  Widget btn = XtCreateManagedWidget(name, xmCascadeButtonWidgetClass, 
				     menubar, NULL, 0);
  Widget menu = XmCreatePulldownMenu(menubar, menu_name, NULL, 0);
  XtVaSetValues(btn, XmNsubMenuId, menu, NULL);
  return menu;
#else
  Widget btn = XtCreateManagedWidget(name, menuButtonWidgetClass, 
				     menubar, NULL, 0);
  Widget menu = XtCreatePopupShell(menu_name, simpleMenuWidgetClass,
				   menubar, NULL, 0);
  XtVaSetValues(btn, XtNmenuName, menu_name, NULL);
  return menu;
#endif
}


static Widget CreateButtonAndOptionMenu(Widget menubar, String name) {

  Widget menu = CreateButtonAndMenu(menubar, name);

  // each radiogroup contains a list of related Toggle Buttons so that 
  // selecting one button will unselect other buttons
  // !!!radiogroup MUST be NULL terminated and MUST be STATIC because they
  // !!!are given as an argument to callback functions
  //
  static Widget radiogroup1[5], radiogroup2[5], radiogroup3[5], 
    radiogroup4[5], radiogroup5[5], radiogroup6[5];

  int n = 0;
  CreateRadio(menu, "Vat",    &n,radiogroup1, GUI::setPref, VAT_TOOL);
  CreateRadio(menu, "Rat",    &n,radiogroup1, GUI::setPref, RAT_TOOL);
  CreateRadio(menu, "Fphone", &n,radiogroup1, GUI::setPref, FPHONE_TOOL);
  radiogroup1[n] = NULL;

  CreateSeparator(menu, "separ");

  n = 0;
  CreateRadio(menu, "Vic", &n,radiogroup2, GUI::setPref, VIC_TOOL);
  CreateRadio(menu, "Wb",  &n,radiogroup2, GUI::setPref, WB_TOOL);
  CreateRadio(menu, "Wbd", &n,radiogroup2, GUI::setPref, WBD_TOOL);
  CreateRadio(menu, "Nte", &n,radiogroup2, GUI::setPref, NTE_TOOL);
  radiogroup2[n] = NULL;

  CreateSeparator(menu, "separ");

  n = 0 ;
  CreateRadio(menu, "Netscape",&n,radiogroup3, GUI::setPref, NETSCAPE_TOOL);
  CreateRadio(menu, "mMosaic", &n,radiogroup3, GUI::setPref, MMOSAIC_TOOL);
  radiogroup3[n] = NULL;

  CreateSeparator(menu, "separ");

  n = 0 ;
  CreateRadio(menu, "Webspace", &n,radiogroup4, GUI::setPref, WEBSPACE_TOOL);
  CreateRadio(menu, "Vrweb",    &n,radiogroup4, GUI::setPref, VRWEB_TOOL);
  radiogroup4[2] = NULL;

  CreateSeparator(menu, "separ");

  n = 0 ;
  CreateRadio(menu, "Ssh",    &n,radiogroup5, GUI::setPref, SSH_TOOL);
  CreateRadio(menu, "Telnet", &n,radiogroup5, GUI::setPref, TELNET_TOOL);
  radiogroup5[2] = NULL;

  CreateSeparator(menu, "separ");

  n = 0 ;
  CreateRadio(menu, "Vred",    &n,radiogroup6, GUI::setPref, VRED_TOOL);
  CreateRadio(menu, "Vrem",    &n,radiogroup6, GUI::setPref, VREM_TOOL);
  radiogroup6[2] = NULL;

  return menu;
}


// Callbacks

void GuiWidgets::glzoneSetFocusEH(Widget w, XtPointer cld, XEvent *xe, Boolean*){
  GuiWidgets *gw = (GuiWidgets*)cld;
  XtSetKeyboardFocus(gw->gui->toplevel, gw->glzone);
}

void GuiWidgets::entrySetFocusEH(Widget w, XtPointer cld, XEvent *xe, Boolean*){
  GuiWidgets *gw = (GuiWidgets*)cld;
  XtSetKeyboardFocus(gw->gui->toplevel, gw->entry);
}

// There is no Activate callback (ie when pressing Return) in Athena
// for the Text widget ==> we get all Key Press events, test if its a Return
// and call the standard callback function that is used fore Motif

void GuiWidgets::entryActionEH(Widget w, XtPointer cld, XEvent *xe, Boolean*){
  KeySym keysym; 
  XComposeStatus compose;
  char buf[10];
  int count = XLookupString(&(xe->xkey), buf, sizeof(buf),
			    &keysym, &compose);
  if (count > 0  && (buf[0] == '\n' || buf[0] == '\r'))
    entryActionCB(w, cld, NULL);
}

void GuiWidgets::entryActionCB(Widget, XtPointer cld, XtPointer) {
  GuiWidgets *gw = (GuiWidgets*)cld;

  //printf("entryCB \n");

  char *mess = GetTextFieldValue(gw->entry);
  if (!mess) return;

  char *buf = (char*)malloc(strlen(mess));
  char *in, *out;
  // enlever les chars non printables (Merci Athena qui fait n'importe quoi!)
  for (in = mess, out = buf; *in != '\0'; in++) 
    if (isprint(*in)) *out++ = *in;
  *out = '\0';

  // display in message zone
  gw->writeMessage("chat", resources.nick, buf);
  // send to World Management 
  userWriting(buf);

  SetTextFieldValue(gw->entry, "");   // clear input zone 
  XtFree(mess);
  free(buf);
}


static void popupMenu(Widget menu, Widget where, XEvent* ev);

void GuiWidgets::glzoneKeyPressEH(Widget, XtPointer cld, 
				  XEvent *xe, Boolean*) {
  GuiWidgets *gw = (GuiWidgets*)cld;
  // gets the X Keysym
  gw->gui->processKey(XKeycodeToKeysym(xe->xany.display, xe->xkey.keycode, 0),
		  TRUE);
}

void GuiWidgets::glzoneKeyReleaseEH(Widget, XtPointer cld, 
				    XEvent *xe, Boolean*) {
  GuiWidgets *gw = (GuiWidgets*)cld;
  // gets the X Keysym
  gw->gui->processKey(XKeycodeToKeysym(xe->xany.display, xe->xkey.keycode, 0),
		      FALSE);
}


/* not used
void GuiWidgets::glzoneMouseMoveEH(Widget, XtPointer cld, 
				    XEvent *xe, Boolean*) {
  if (followMouseMode) {
    ObjInfo objinfo[BUTTONSNUMBER + 6];
    struct _ZVSolid *solid =
      GUI::getPointedObject(xe->xbutton.x, xe->xbutton.y, objinfo);

    // 2nd arg is false ==> the popup menu is NOT created
    if (solid) updateObjectInfo(objinfo, false);
  }
}
*/

void GuiWidgets::glzoneMousePressEH(Widget, XtPointer cld, 
				    XEvent *xe, Boolean*) {
  GuiWidgets *gw = (GuiWidgets*)cld;
  xe->xbutton.x -= gw->glwin_x;
  xe->xbutton.y -= gw->glwin_y;

  XtSetKeyboardFocus(gw->gui->toplevel, gw->glzone);  //securite

  ObjInfo objinfo[BUTTONSNUMBER + 6];
  struct _ZVSolid *solid =
    GUI::selectPointedObject(xe->xbutton.x, xe->xbutton.y, objinfo);

  if (solid) {
    // 2nd arg is true ==> the popup menu is created
    gw->updateObjectInfo(objinfo, true);
    // button 2 met a jour l'objectBar sans ouvrir le menu
    if (gw->objectMenu && xe->xbutton.button != 2) 
      popupMenu(gw->objectMenu, gw->glzone, xe);
  }
  // clear the object bar
  else gw->updateObjectInfo(NULL, false);
}

// updates the ObjectBar (and the ObjectMenu when available)
// NB: removes and delete both if argument is NULL

void GuiWidgets::updateObjectInfo(ObjInfo *objinfo, boolean createMenu) {
  if (!objinfo) {
    if (objectBar) XtDestroyWidget(objectBar);
    objectBar = NULL;
    if (objectMenu) XtDestroyWidget(objectMenu);
    objectMenu= NULL;
  }

  else {
    char *classname = objinfo[0].name;
    char *instancename = objinfo[1].name;
    char fullname[64];

    if (objectBar) XtDestroyWidget(objectBar);
    objectBar = NULL;
    if (objectMenu) XtDestroyWidget(objectMenu);
    objectMenu= NULL;

    if (!classname) classname = ""; // avoid seg faults!
    sprintf(fullname, "%s: %s", classname, instancename);

    //NOTE: we will manage the widget once completed
    // useful for Motif, not for Athena because ... well, what works properly
    // in Athena anyway...?
#if WANT_MOTIF
    objectBar = XtVaCreateWidget("objectBar", xmRowColumnWidgetClass, toolBar, 
				 XmNorientation, XmHORIZONTAL, 
				 XmNmarginWidth, 0, XmNmarginHeight, 0,
				 XmNborderWidth, 0, NULL);
    if (createMenu) {
      objectMenu = XmCreatePopupMenu(glzone, "objectMenu", NULL, 0);
      SetLabelValue(CreateLabel(objectMenu, "objectLabel"), fullname);
    }
#else
    objectBar = 
      XtVaCreateManagedWidget("objectBar", boxWidgetClass, toolBar, 
			      XtNorientation, XtorientHorizontal, 
			      XtNhSpace, 0, XtNvSpace, 0, XtNborderWidth, 0,
			      TOP_F, LEFT_F, RIGHT_F, BOTTOM_F,
			      // c'est n'importe quoi mais avec Athena je ne m'etonne
			      // de rien (car cet objet ne se retaille pas qd on ajoute
			      // les enfants du fait que son parent soit deja manage!)
			      XtNwidth, 900,	
			      XtNheight, 35,
			      NULL);
    if (createMenu) {
      objectMenu = 
	XtVaCreatePopupShell("objectMenu", simpleMenuWidgetClass, glzone,
			     XtNlabel, fullname, NULL);
      //SetLabelValue(CreateButton(objectMenu,"objectLabel",NULL,0), fullname);
    }
#endif
    if (objectMenu) CreateSeparator(objectMenu, "separ");
    SetLabelValue(CreateLabel(objectBar, "objectLabel"), fullname);

    // add object buttons if any
    for (ObjInfo *poi = objinfo + 2; poi->name != NULL; poi++) {
      if (objectBar) {
	Widget btn = CreateButton(objectBar, "objectButton", poi->fun, poi->farg);
	SetLabelValue(btn, poi->name);
      }
      if (objectMenu) {
	Widget btn = CreateMButton(objectMenu, "objectButton", poi->fun, poi->farg);
	SetLabelValue(btn, poi->name);
      }
    }

    if (objectBar) XtManageChild(objectBar);
    // objectMenu ouvert dans glzoneMousePressEH
  }
}


static void popupMenu(Widget menu, Widget where, XEvent* ev) {
  Position rootx, rooty;
  XtTranslateCoords(where, 0, 0, &rootx, &rooty);
#if WANT_MOTIF
  XtVaSetValues(menu, 
		XtNx, (XtArgVal)rootx + ev->xbutton.x, 
		XtNy, (XtArgVal)rooty + ev->xbutton.y, 
		NULL);

  XtManageChild(menu);
#else
  XtVaSetValues(menu, 
		XtNx, (XtArgVal)rootx + ev->xbutton.x, 
		XtNy, (XtArgVal)rooty + ev->xbutton.y, 
		NULL);
  //XtPopupSpringLoaded(menu);
  XtPopup(menu, XtGrabNone);
  XGrabPointer(XtDisplay(menu),
	       XtWindow(menu),
	       True,	          //??owner_events,
	       ButtonPressMask|ButtonReleaseMask,
	       GrabModeAsync,    // pointer_mode
	       GrabModeAsync,    // keyboard_mode
	       None,	          // confine_to
	       None,             // cursor, 
	       CurrentTime);     // time

#endif
}


// External API


// mode == chat, warning, notice...
//
void GuiWidgets::writeMessage(const char *mode, const char *from, const char *mess) {
  //  if (mode[0]=='w' || mode[0]=='n' || mode[0]=='W' || mode[0]=='N')....
  char *buf = NULL;

  if (from) {
    buf = (char*) malloc(strlen(from) + strlen(mess) + 10);
    sprintf(buf, "%s> %s\n", from, mess);  
  }
  else {
    buf = (char*) malloc(strlen(mode) + strlen(mess) + 10);
    sprintf(buf, "%s> %s\n", mode, mess);  
  }

  SetTextAreaValue(messages, buf, TRUE);
  free(buf);
}

void GuiWidgets::showAlertBox(boolean state) {}


void GuiWidgets::updateUser(GuiUser gu, struct User *puser) {
  SetLabelValue(gu, puser->name.instance_name);
}

GuiUser GuiWidgets::addUser(struct User *puser) {
  GuiUser gu = CreateLabel(users, "userItem");
  SetLabelValue(gu, puser->name.instance_name);
  XtManageChild(gu);
  return gu;
}

void GuiWidgets::removeUser(GuiUser gu) {
  XtDestroyWidget(gu);
}


GuiWorld GuiWidgets::addWorld(struct _World *pworld, boolean isCurrentWorld) {
 GuiWorld gw = CreateLabel(worlds, "worldItem");
 SetLabelValue(gw, pworld->name);
 XtManageChild(gw);
 return gw;
}

void GuiWidgets::updateWorld(struct _World *pworld, boolean isCurrentWorld) {
  GuiWorld gw = (GuiWorld) pworld->ptrgui;
  SetLabelValue(gw, pworld->name);
}

void GuiWidgets::removeWorld(struct _World *pworld) {
  GuiWorld gw = (GuiWorld) pworld->ptrgui;
  XtDestroyWidget(gw);
}


// intermediate callback function and structure for calling standard GUI
// functions defined in gui.cc and xkeys.cc

struct CallData {
  void (*fct)(int);
  int arg;
  CallData(void (*f)(int), int a)  {fct = f; arg = a;}
};

struct RadioCallData {
  void (*fct)(int);
  int arg, wno;
  Widget *wgroup;
  RadioCallData(void (*f)(int), int a, int wnum, Widget *wgrp) {
    fct = f; arg = a; wno = wnum; wgroup = wgrp;
  }
};


static void actionCB(Widget w, XtPointer cld, XtPointer) {
  CallData *cd = (CallData*)cld;
  (cd->fct)(cd->arg);
}

static void toggleCB(Widget togglew, XtPointer cld, XtPointer) {
  CallData *cd = (CallData*)cld;
  Boolean is_set = FALSE;
#if WANT_MOTIF
    // laisser faire la nature ... (NB: value deja changee par Motif)
    XtVaGetValues(togglew, XmNset, &is_set, NULL);
#else
    is_set = ! GetToggleValue(togglew);
    SetToggleValue(togglew, is_set);
#endif
    //printf("toggleCB: new val = %d\n", is_set);
  (cd->fct)(is_set);
}


static void radioCB(Widget togglew, XtPointer cld, XtPointer) {
  RadioCallData *cd = (RadioCallData*)cld;

  // changer l'etat des autres boutons du wgroup
  for (int k = 0; cd->wgroup[k] != NULL; k++) 
    if (k == cd->wno) SetToggleValue(cd->wgroup[k], TRUE);
  else SetToggleValue(cd->wgroup[k], FALSE);

  //printf("radioCB: arg = %d\n", cd->arg);
  (cd->fct)(cd->arg);
}


static void delCallData(Widget w, XtPointer cld, XtPointer) {
  //  printf("Destroy name=%s - ptr=%d \n", XtName(w), w);
  CallData *cd = (CallData*)cld;
  delete cd;
}


// INTERACTEURS

static Widget CreateLabel(Widget parent, String name) {
#if WANT_MOTIF
  return XtCreateManagedWidget(name, xmLabelWidgetClass, parent, NULL,0);
#else
  return XtCreateManagedWidget(name, labelWidgetClass, parent, NULL,0);
#endif
}


static Widget CreateSeparator(Widget parent, String name) {
#if WANT_MOTIF
  return XtCreateManagedWidget(name, xmSeparatorWidgetClass, parent, NULL,0);
#else
  return XtCreateManagedWidget(name, smeLineObjectClass, parent, NULL,0);
#endif
}


static Widget CreateButton(Widget parent, String name, void (*fun)(int), int farg) {
Widget ww;
#if WANT_MOTIF
  signed char container_type = -1;
  XtVaGetValues(parent, XmNrowColumnType, &container_type, NULL);

  if (container_type == XmMENU_BAR)
    ww = XtCreateManagedWidget(name, xmCascadeButtonWidgetClass,parent,NULL,0);
  else
    ww = XtCreateManagedWidget(name, xmPushButtonWidgetClass,parent,NULL,0);
  if (fun) XtAddCallback(ww, XmNactivateCallback, actionCB, 
			 (XtPointer)new CallData(fun, farg));
#else
  ww = XtCreateManagedWidget(name, commandWidgetClass, parent, NULL,0);
  if (fun)
    XtAddCallback(ww, XtNcallback, actionCB, (XtPointer)new CallData(fun,farg));
#endif
  return ww;
}


static Widget CreateMButton(Widget parent, String name, void (*fun)(int), int farg){
  Widget ww;
  CallData *cd = new CallData(fun, farg);
#if WANT_MOTIF
  ww = XtCreateManagedWidget(name, xmPushButtonWidgetClass,parent,NULL,0);
  if (fun) XtAddCallback(ww, XmNactivateCallback, actionCB, (XtPointer)cd);
#else
  ww = XtCreateManagedWidget(name, smeBSBObjectClass, parent,NULL,0);
  XtAddCallback(ww, XtNcallback, actionCB,  (XtPointer)cd);
#endif
  // both cases
  XtAddCallback(ww, XtNdestroyCallback, delCallData, (XtPointer)cd);
  return ww;
}


static Widget CreateToggle(Widget parent, String name, void (*fun)(int)) {
Widget ww;
#if WANT_MOTIF
  ww = XtCreateManagedWidget(name, xmToggleButtonWidgetClass, parent,NULL,0);
  if (fun) XtAddCallback(ww, XmNvalueChangedCallback, toggleCB, 
			 (XtPointer) new CallData(fun, 0));
#else
  ww = XtVaCreateManagedWidget(name, smeBSBObjectClass, parent,
			       XtNleftMargin, 20, NULL);
  if (fun) XtAddCallback(ww, XtNcallback, toggleCB, 
			 (XtPointer) new CallData(fun, 0));
#endif
  return ww;
}


static Widget CreateRadio(Widget parent, String name, int *no, Widget *radiogroup,
		   void (*fun)(int), int farg) {
Widget ww;
#if WANT_MOTIF
  ww = XtVaCreateManagedWidget(name, xmToggleButtonWidgetClass, parent,
			       XmNindicatorType, XmONE_OF_MANY, NULL);
  if (fun) {
    //!!BEWARE: radiogroup must not be freed (must be malloc or static data)
    radiogroup[*no] = ww;
    XtAddCallback(ww, XmNvalueChangedCallback, radioCB, 
		  (XtPointer)new RadioCallData(fun, farg, *no, radiogroup));
  }
#else
  ww = XtVaCreateManagedWidget(name, smeBSBObjectClass, parent, 
			       XtNleftMargin, 20, NULL);
  if (fun) {
    //!!BEWARE: radiogroup must not be freed (must be malloc or static data)
    radiogroup[*no] = ww;
    XtAddCallback(ww, XtNcallback, radioCB, 
		  (XtPointer)new RadioCallData(fun, farg, *no, radiogroup));
  }
#endif
  (*no)++;  //!!!att
  return ww;
}


static void SetLabelValue(Widget ww, const char *value) {
#if WANT_MOTIF
  XmString item = XmStringCreateLocalized((char*)value);
  XtVaSetValues(ww, XmNlabelString, item, NULL); 
  XmStringFree(item);
#else
  XtVaSetValues(ww, XtNlabel, value, NULL); 
#endif
}


#ifndef WANT_MOTIF
static void initTogglePixmap(Widget toplevel) {
  static char data_on[] = {
    0x00,0x80,0x00,0x84,0x1c,0x8e,0x28,0x9b,0xf0,0x9e,0xa0,0x87,0xe0,0x82,0xa0,
    0x83,0x70,0x86,0x38,0x8e,0x18,0x98,0x0c,0x98,0x04,0x90,0x00,0x80,0x00,0x80
  };
  static char data_off[] = {
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  };
  int toplevel_depth;
  XtVaGetValues(toplevel, XtNdepth, &toplevel_depth, NULL);
  TOGGLE_ON_PIXMAP = 
    XCreatePixmapFromBitmapData( XtDisplay(toplevel), XtWindow(toplevel),
				 data_on, 15, 15,
				 BlackPixelOfScreen(XtScreen(toplevel)),
				 WhitePixelOfScreen(XtScreen(toplevel)),
				 1);
  TOGGLE_OFF_PIXMAP = 
    XCreatePixmapFromBitmapData( XtDisplay(toplevel), XtWindow(toplevel),
				 data_off, 15, 15,
				 BlackPixelOfScreen(XtScreen(toplevel)),
				 WhitePixelOfScreen(XtScreen(toplevel)),
				 1);

}
#endif


static Boolean GetToggleValue(Widget togglew) {
  Boolean state;
#if WANT_MOTIF
  XtVaGetValues(togglew, XmNset, &state, NULL);
#else
  Pixmap pxm = XtUnspecifiedPixmap;
  XtVaGetValues(togglew, XtNleftBitmap, &pxm, NULL);
  state = (pxm == TOGGLE_ON_PIXMAP);
#endif
  return state;
}


static void SetToggleValue(Widget togglew, Boolean state) {
#if WANT_MOTIF
  XtVaSetValues(togglew, XmNset, state, NULL);
#else
  if (state)
    XtVaSetValues(togglew, XtNleftBitmap, TOGGLE_ON_PIXMAP, NULL);
  else
    XtVaSetValues(togglew, XtNleftBitmap, TOGGLE_OFF_PIXMAP, NULL);
#endif
}


static void SetTextFieldValue(Widget ww, const char *s) {
#if WANT_MOTIF
  XtVaSetValues(ww, XmNvalue, s, NULL);
#else
  XtVaSetValues(ww, XtNstring, s, NULL);
#endif
}

static char *GetTextFieldValue(Widget ww)  {
  char *s = NULL;
#if WANT_MOTIF
  XtVaGetValues(ww, XmNvalue, &s, NULL);
#else
  XtVaGetValues(ww, XtNstring, &s, NULL);
  //NB awt copie les Strings au SetValue mais PAS au getValue 
  // (ie. getValue renvoie la representation interne !)
  s = strdup(s);
#endif
  return s;
}

static void SetTextAreaValue(Widget ww, const char *s, boolean add_at_the_end) {
#if WANT_MOTIF
  XmTextPosition lastpos = XmTextGetLastPosition(ww);
  if (!add_at_the_end) 
    XtVaSetValues(ww, XmNvalue, s, NULL);
  else {
    XmTextInsert(ww, lastpos, (char*)s);
    lastpos = XmTextGetLastPosition(ww); // new lastpos
  }
  XmTextShowPosition(ww, lastpos);
#else 
 if (!add_at_the_end) XtVaSetValues(ww, XtNstring, s, NULL); 
 else {
   //NB awt copie les Strings au SetValue mais PAS au getValue 
   // (ie. getValue renvoie la representation interne !)
   char *prev_s = NULL;
   XtVaGetValues(ww, XtNstring, &prev_s, NULL);
 
   if (!prev_s || !prev_s[0]) XtVaSetValues(ww, XtNstring, s, NULL); 
   else {
     int ll = strlen(prev_s) + strlen(s);
     char *buf = (char*)malloc(ll + 2);
     sprintf(buf, "%s%s", prev_s, s);
     XtVaSetValues(ww, XtNstring, buf, NULL); 
     free(buf);

     // indemerdable: il faudrait calculer le nombre de \n
     // XtVaSetValues(ww, XtNdisplayPosition, (XtArgVal)ll, NULL); 
   }
 }
#endif
}


char** GuiWidgets::getFallbackResources() {
  static char* fallbacks[] = {
    "*menuBar*fontList: -*-helvetica-bold-r-normal-*-12-*-*-*-*-*-*-1",
    "*toolBar*fontList: -*-helvetica-bold-o-normal-*-12-*-*-*-*-*-*-1",
    "*VRENG*fontList: -*-helvetica-bold-o-normal-*-14-*-*-*-*-*-*-1",
    "*VRENG*background: red",
    "*VRENG*foreground: white",
#if WANT_MOTIF
    "*messages*fontList: -*-helvetica-medium-r-normal-*-12-*-*-*-*-*-*-1",
    "*worlds*fontList: -*-helvetica-bold-o-normal-*-12-*-*-*-*-*-*-1",
    "*users*fontList: -*-helvetica-bold-o-normal-*-12-*-*-*-*-*-*-1",
    "*background: #b8c5d8",
    "*toolBar*background: #a8b5e8",
    //"*messages*background:  #d5e3f0",
    "*worldsPane*background: white",
    "*worlds*background: white",
    "*usersPane*background: white",
    "*users*background: white",
#else
    "*background: gray84",
    "*toolBar*background: grey75",
    "*messages*background: grey94",
    "*worldsPane*background: grey94",
    "*worlds*background: grey94",
    "*usersPane*background: grey94",
    "*users*background: grey94",
#endif
    "*XmTextField.background: White",
    "*XmList.background: White",
    "*XmList.visibleItemCount: 5",
    "*XmFileSelectionBox.autoUnmanage: True",
    "*XmBulletinBoard.marginWidth: 2",
    "*XmBulletinBoard.marginHeight: 2",
    "*XmRowColumn.spacing: 1",
    "*XmPushButton.shadowThickness: 1",
    "*XmPushButtonGadget.shadowThickness: 1",
    "*XmCascadeButton.shadowThickness: 1",
    "*XmCascadeButtonGadget.shadowThickness: 1",
    "*XmTextField.shadowThickness: 1",
    "*XmText.shadowThickness: 1",
    "*XmList.shadowThickness: 1",
    "*XmScrollBar.shadowThickness: 1",

    "*XmTextField.translations: #override\n<Key>BackSpace: delete-previous-character()\n<Key>Delete:	delete-previous-character()\n<Key>osfDelete:	delete-previous-character()\nMeta <Key>BackSpace:	delete-previous-word()\nMeta <Key>Delete:	delete-previous-word()\nMeta <Key>osfDelete:	delete-previous-word()\nShift <Key>BackSpace:	delete-next-character()\nShift <Key>Delete:	delete-next-character()\nShift <Key>osfDelete:	delete-next-character()\nCtrl <Key>D:	delete-next-character()\nMeta <Key>D:	delete-next-word()\nCtrl <Key>K:	delete-to-end-of-line()\nCtrl <Key>E:	end-of-line()\nCtrl <Key>A:	beginning-of-line()\nCtrl <Key>B:	backward-character()\nCtrl <Key>F:	forward-character()\nMeta <Key>B:	backward-word()\nMeta <Key>F:	forward-word()",
    "*XmText.translations: #override\n<Key>BackSpace: delete-previous-character()\n<Key>Delete:	delete-previous-character()\n<Key>osfDelete:	delete-previous-character()\nMeta <Key>BackSpace:	delete-previous-word()\nMeta <Key>Delete:	delete-previous-word()\nMeta <Key>osfDelete:	delete-previous-word()\nShift <Key>BackSpace:	delete-next-character()\nShift <Key>Delete:	delete-next-character()\nShift <Key>osfDelete:	delete-next-character()\nCtrl <Key>D:	delete-next-character()\nMeta <Key>D:	delete-next-word()\nCtrl <Key>K:	delete-to-end-of-line()\nCtrl <Key>E:	end-of-line()\nCtrl <Key>A:	beginning-of-line()\nCtrl <Key>B:	backward-character()\nCtrl <Key>F:	forward-character()\nMeta <Key>B:	backward-word()\nMeta <Key>F:	forward-word()",
    NULL
  };
  return fallbacks;
}


#endif /* !VRENGD */ 
