/*
 * Copyright (C) 2003  Stefan Kleine Stegemann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "ExtendedScrollView.h"

#include <Foundation/NSString.h>
#include <AppKit/NSButton.h>
#include <AppKit/NSImage.h>
#include <AppKit/NSFont.h>
#include <AppKit/NSScroller.h>
#include <AppKit/NSTextField.h>
#include <AppKit/NSView.h>

/*
 * Non-Public methods.
 */
@interface ExtendedScrollView(Private)
- (void) _createControlsView;
- (NSButton*) _makeButtonAt: (float)posX
                      image: (NSString*)imageName
                      width: (int)width;
@end

/*
 * Provides a custom ScrollView with controls to 
 * display / manipulate the current page etc.
 *
 * Note: The additional controls will only be displayed
 *       correctly if the horizontal scroller is visible.
 */
@implementation ExtendedScrollView

/*
 * Designated initializer.
 */
- (id) initWithFrame: (NSRect)frame
{
   if ((self = [super initWithFrame: frame]))
   {
      [self _createControlsView];
      [self setDisplayPageControls: NO];
   }

   return self;
}


- (void) dealloc
{
   NSLog(@"dealloc ExtendedScrollView, retain count is %d", [self retainCount]);

   RELEASE(controlsView);

   [super dealloc];
}


/* Outlets */

- (NSButton*) firstPageButton
{
   return firstPageButton;
}


- (NSButton*) nextPageButton
{
   return nextPageButton;
}


- (NSButton*) prevPageButton
{
   return prevPageButton;
}


- (NSButton*) lastPageButton
{
   return lastPageButton;
}


- (NSTextField*) currentPageField
{
   return currentPageField;
}


- (NSTextField*) pageOfField
{
   return pageOfField;
}


/*
 * Set whether the page control elements should be
 * display at the left of the horizontal scroller.
 */
- (void) setDisplayPageControls: (BOOL)display
{
   displayPageControls = display;
}


/*
 * Determine whether the page control elements
 * are displayed.
 */
- (BOOL) displayPageControls
{
   return displayPageControls;
}


/*
 * Consider additional components when the ScrollView is
 * layoutet.
 */ 
- (void) tile
{
   [super tile];

   if ([self displayPageControls])
   {
      NSRect scrollerRect;
      NSRect controlsRect;

      scrollerRect = [[self horizontalScroller] frame];
      NSDivideRect(scrollerRect,
                   &controlsRect,
                   &scrollerRect,
                   168.0,
                   NSMinXEdge);
   
      [[self horizontalScroller] setFrame: scrollerRect];
      [controlsView setFrame: NSInsetRect(controlsRect, 1, 1)];
   }
}

@end


/* -------------------------------------------------------------------------- */

@implementation ExtendedScrollView (Private)

/*
 * The controlsView takes the additional components to maniplute
 * some settings. It is displayed along with the horizontal scroller
 * on the left side.
 */
- (void) _createControlsView
{
   NSRect     frame;
   id         pageLabel;
   NSString*  labelText = @"Page";
   int        height;

   // the frame doesn't matter much here because it will
   // be re-calcualted when the scrollview is layoutet.
   frame = NSMakeRect(0, 0, 10, 10);
   controlsView = [[NSView alloc] initWithFrame: frame];
   [controlsView setAutoresizesSubviews: YES];
   [self addSubview: controlsView];

   // the height for all components
   height = [NSScroller scrollerWidthForControlSize: NSRegularControlSize];

   // the "Page" label
   frame = NSMakeRect(0, 0, 35, height);
   pageLabel = AUTORELEASE([[NSTextField alloc] initWithFrame: frame]);

   [pageLabel setFont: [NSFont labelFontOfSize: 10.0]];
   [pageLabel setBordered: NO];
   [pageLabel setBezeled: NO];
   [pageLabel setSelectable: NO];
   [pageLabel setDrawsBackground: NO];
   [pageLabel setEditable: NO];
   [pageLabel setAutoresizingMask: NSViewNotSizable];
   [pageLabel setStringValue: labelText];
   [pageLabel setAlignment: NSCenterTextAlignment];

   [controlsView addSubview: pageLabel];

   // the current page field
   frame = NSMakeRect(35, 0, 30, height);
   currentPageField = AUTORELEASE([[NSTextField alloc] initWithFrame: frame]);
   [currentPageField setAutoresizingMask: NSViewNotSizable];
   [currentPageField setFont: [NSFont labelFontOfSize: 10.0]];
   [currentPageField setBordered: NO];
   [currentPageField setStringValue: @""];
   
   [controlsView addSubview: currentPageField];

   // the " of x" label
   frame = NSMakeRect(68, 0, 42, height);
   pageOfField = AUTORELEASE([[NSTextField alloc] initWithFrame: frame]);

   [pageOfField setFont: [NSFont labelFontOfSize: 10.0]];
   [pageOfField setBordered: NO];
   [pageOfField setBezeled: NO];
   [pageOfField setSelectable: NO];
   [pageOfField setDrawsBackground: NO];
   [pageOfField setEditable: NO];
   [pageOfField setAutoresizingMask: NSViewNotSizable];
   [pageOfField setStringValue: labelText];
   [pageOfField setAlignment: NSLeftTextAlignment];
   [pageOfField setStringValue: @"of ?"];

   [controlsView addSubview: pageOfField];

   // Buttons
   firstPageButton = [self _makeButtonAt: 112
                                   image: @"first_page.tiff"
                                   width: -1];
   [controlsView addSubview: firstPageButton];

   prevPageButton = [self _makeButtonAt: [firstPageButton frame].origin.x
                                       + [firstPageButton frame].size.width
                                  image: @"prev_page.tiff"
                                  width: 15];
   [controlsView addSubview: prevPageButton];

   nextPageButton = [self _makeButtonAt: [prevPageButton frame].origin.x
                                       + [prevPageButton frame].size.width
                                  image: @"next_page.tiff"
                                  width: 15];
   [controlsView addSubview: nextPageButton];

   lastPageButton = [self _makeButtonAt: [nextPageButton frame].origin.x
                                       + [prevPageButton frame].size.width
                                 image :@"last_page.tiff"
                                  width: -1];
   [controlsView addSubview: lastPageButton];
}


/*
 * Create a button that is shown on the vertical
 * scroller.
 */
- (NSButton*) _makeButtonAt: (float)posX
                      image: (NSString*)imageName
                      width: (int)width
{
   NSButton* theButton;
   NSRect    frame;

   frame = NSMakeRect(posX, 0, 0, 0);
   theButton = AUTORELEASE([[NSButton alloc] initWithFrame: frame]);
   [theButton setButtonType: NSMomentaryLight];
   [theButton setImagePosition: NSImageOnly];
   [theButton setImage: [NSImage imageNamed:imageName]];
   [theButton setBordered: NO];
   [theButton sizeToFit];
   [theButton setEnabled: YES];

   if (width > 0)
   {
      [theButton setFrameSize: NSMakeSize(width,
                                          [theButton frame].size.height)];
   }

   return theButton;
}

@end
