#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <GL/gl.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "v3dmp.h"
#include "v3dmodel.h"

#include "guiutils.h"
#include "cdialog.h"

#include "editor.h"
#include "editorcb.h"
#include "editorviewcb.h"
#include "editorlist.h"
#include "editortdialog.h"
#include "editortexture.h"
#include "editorhf.h"
#include "editorundo.h"
#include "editorp.h"
#include "editorpcreatecb.h"
#include "editorpcb.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


#include "images/icon_mp_color_20x20.xpm"
#include "images/icon_mp_comment_20x20.xpm"
#include "images/icon_mp_heightfield_load_20x20.xpm"
#include "images/icon_mp_line_20x20.xpm"
#include "images/icon_mp_lineloop_20x20.xpm"
#include "images/icon_mp_linestrip_20x20.xpm"
#include "images/icon_mp_point_20x20.xpm"
#include "images/icon_mp_polygon_20x20.xpm"
#include "images/icon_mp_quad_20x20.xpm"
#include "images/icon_mp_quadstrip_20x20.xpm"
#include "images/icon_mp_rotate_20x20.xpm"
#include "images/icon_mp_texture_orient_20x20.xpm"
#include "images/icon_mp_texture_select_20x20.xpm"
#include "images/icon_mp_texture_off_20x20.xpm"
#include "images/icon_mp_translate_20x20.xpm"
#include "images/icon_mp_triangle_20x20.xpm"
#include "images/icon_mp_trianglefan_20x20.xpm"
#include "images/icon_mp_trianglestrip_20x20.xpm"


int EditorPrimitiveDoCreateSimple(
	ma_editor_struct *editor, int ptype, int create_pos, void *p
);
int EditorPrimitiveDoCreate(
	ma_editor_struct *editor,
	int model_num,
	int create_pos, int insert_mode,
	int select,
	void *p,
	gbool record_undo
);
void EditorPrimitiveDoDelete(
	ma_editor_struct *editor,
	int model_num,
	int *list, int total,
	gbool record_undo
);

gbool EditorPrimitiveIsRealized(ma_editor_struct *editor, void *p);
void EditorPrimitiveRealize(
	ma_editor_struct *editor, void *p,
	gbool rerealize
);
void EditorPrimitiveUnrealize(ma_editor_struct *editor, void *p);
void EditorPrimitiveRealizeAll(
	ma_editor_struct *editor, int model_num,
	gbool rerealize
);
void EditorPrimitiveRealizeAllType(
	ma_editor_struct *editor, int model_num, int ptype,
	gbool rerealize
);
void EditorPrimitiveUnrealizeAll(ma_editor_struct *editor, int model_num);

int EditorPrimitiveVertexAdd(
	ma_editor_struct *editor, int model_num, int pn, int vertex_num,
	mp_vertex_struct *v, mp_vertex_struct *n, mp_vertex_struct *tc,
	gbool report_errors
);
void EditorPrimitiveVertexRemove(
	ma_editor_struct *editor, int model_num, int pn, int vertex_num,
	gbool record_undo, gbool report_errors
);

GtkWidget *EditorPrimitivesCreateCreateMenu(ma_editor_struct *editor);
int EditorPrimitivesCreateToolBar(ma_editor_struct *editor);


#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))

/*
 *      Checks if editor has changes, if it does not then it will
 *      mark it as having changes.
 */
#define EDITOR_DO_UPDATE_HAS_CHANGES    \
{ \
 if(!editor->has_changes) \
  editor->has_changes = TRUE; \
}


/*
 *      Create primitive front end procedure, creates a new primitive
 *      on the given editor's currently selected model.   
 *
 *      ptype specifies the type of the new primitive (which should be
 *      the same type as the given referance primitive p if it is not
 *      NULL).
 *
 *      pos can be any valid insert position, -1 for append, or -2 to
 *      let this function decide on the best position (uses current last
 *      selected primitive on the editor.
 *
 *      The primitive p will be transfered to the selected model's
 *      primitives list (or destroyed on error), so the given primitive p
 *      should not be referanced again after calling this function. If it
 *	is NULL then a new primitive specified by ptype will be created.
 *
 *	This function will translate the given primitive to a new position
 *	specified by the editor's current view cursor position.
 *
 *      Returns the index to the newly created primitive on the currently
 *      selected model or -1 on failure.
 *
 *	Editor will be properly updated and views redrawn.
 */
int EditorPrimitiveDoCreateSimple(
	ma_editor_struct *editor, int ptype, int create_pos, void *p
)
{
	gint new_pn, model_num;


	if(editor == NULL)
	{
	    V3DMPDestroy(p);
	    return(-1);
	}

	/* Get current selected model on editor. */
	model_num = EditorSelectedModelIndex(editor);

	/* If given primitive is NULL, then create a new one. */
	if(p == NULL)
	{
	    gint i, total;
	    gdouble s = 1.0;
	    mp_vertex_struct *v, *n, *tc;

	    p = V3DMPCreate(ptype);
	    if(p == NULL)
		return(-1);

#define DO_INSERT_VERTEXES	\
{ \
 for(i = 0; i < total; i++) \
 { \
  if(V3DMPInsertVertex( \
   p, -1, \
   &v, &n, &tc \
  ) < 0) \
   break; \
 } \
}

	    /* Handle newly created primitive by type. */
	    ptype = (*(int *)p);
	    switch(ptype)
	    {
	      case V3DMP_TYPE_LINE_STRIP:
		total = 2;
		DO_INSERT_VERTEXES
		break;

	      case V3DMP_TYPE_LINE_LOOP:
		total = 3;
		DO_INSERT_VERTEXES
		break;

	      case V3DMP_TYPE_TRIANGLE_STRIP:
		total = 3;
		DO_INSERT_VERTEXES
		break;

	      case V3DMP_TYPE_TRIANGLE_FAN:
		total = 3;
		DO_INSERT_VERTEXES
		break;

	      case V3DMP_TYPE_QUAD_STRIP:
		total = 4;
		DO_INSERT_VERTEXES
		break;

	      case V3DMP_TYPE_POLYGON:
		total = 4;
		DO_INSERT_VERTEXES
		break;
	    }
#undef DO_INSERT_VERTEXES

	    /* Set up new vertices to some initial positions. */
	    i = 0;
	    v = V3DMPGetVertex(p, i);
	    if(v != NULL)
	    {
		v->x = 0.0 * s;
		v->y = 1.0 * s;
		v->z = 0.0 * s;
	    }
	    i = 1;
	    v = V3DMPGetVertex(p, i);
	    if(v != NULL)
	    {
		v->x = 1.0 * s;
		v->y = 1.0 * s;
		v->z = 1.0 * s;
	    }
	    i = 2;
	    v = V3DMPGetVertex(p, i);
	    if(v != NULL)
	    {
		v->x = 1.0 * s;
		v->y = 0.0 * s;
		v->z = 0.0 * s;
	    }
	    i = 3;
	    v = V3DMPGetVertex(p, i);
	    if(v != NULL)
	    {
		v->x = 0.0 * s;
		v->y = 0.0 * s;
		v->z = 0.0 * s;
	    }
	    i = 4;
	    v = V3DMPGetVertex(p, i);
	    if(v != NULL)
	    {
		v->x = -0.5 * s;
		v->y = 0.5 * s;
		v->z = -0.5 * s;
	    }
	}


	/* Offset editor's view cursor position to the primitive's
	 * vertices.
	 */
	if(p != NULL)
	{
	    int i;
	    mp_vertex_struct *v;

	    for(i = 0; (v = V3DMPGetVertex(p, i)) != NULL; i++)
	    {
		v->x += editor->vcursor_x;
		v->y += editor->vcursor_y;
		v->z += editor->vcursor_z;
	    }
	}


	/* Create new primitive. */
	new_pn = EditorPrimitiveDoCreate(
	    editor,
	    model_num,
	    create_pos,		/* Create position. */
	    0,			/* Insert mode. */
	    1,			/* Select. */
	    p,			/* Transfer p, do not referance again. */
	    FALSE		/* Record undo. */
	);
	p = NULL;

	return(new_pn);
}

/*
 *	Procedure to create the specified primitive p on the given editor.
 *
 *	Inputs; editor, model_num, and p must be valid.
 *
 *	The primitive p will be transfered to the specified model's
 *	primitives list (or destroyed on error), so the given primitive p
 *	should not be referanced again after calling this function.
 *	No translation will be performed on the given primitive, so any
 *	translations must be done by the calling function.
 *
 *	Views, lists, and menus will be updated after this call.
 *
 *	If the model_num matches the current selected model on the editor
 *	and if select is 1 then the newly added primitive will be
 *	selected.
 *
 *	If create_pos is -1, then the primitive will be appended.
 *	If create_pos is -2, then this function will choose the best
 *	position to create it.
 *	If insert_mode is 0 then it will be inserted at create_pos
 *	(if valid) or if insert_mode is 1 then it will be inserted
 *	after the create_pos.
 *
 *	Returns non-zero on error.
 */
int EditorPrimitiveDoCreate(
	ma_editor_struct *editor,
	int model_num,
	int create_pos, int insert_mode,
	int select,
	void *p,
	gbool record_undo
)
{
	int n, j;
	v3d_model_struct *model_ptr;


	if((editor == NULL) || (p == NULL))
	{
	    V3DMPDestroy(p);
	    return(-1);
	}

	if(!editor->initialized)
	{
	    V3DMPDestroy(p);
	    return(-1);
	}

	/* Get model pointer. */
	model_ptr = V3DModelListGetPtr(
	    editor->model, editor->total_models, model_num
	);
	if(model_ptr == NULL)
	{
	    V3DMPDestroy(p);
	    return(-2);
	}

	/* Model type not V3D_MODEL_TYPE_STANDARD? */
	if(model_ptr->type != V3D_MODEL_TYPE_STANDARD)
	{
	    /* Cannot create primitives on a model type that is not
	     * V3D_MODEL_TYPE_STANDARD.
	     */
	    V3DMPDestroy(p);
	    return(-2);
	}

	EDITOR_DO_UPDATE_HAS_CHANGES

	if(model_ptr->total_primitives < 0)
	    model_ptr->total_primitives = 0;

	/* Create at best position? */
	if(create_pos == -2)
	{
	    /* Use best position. */
	    /* One or more primitives selected? */
	    if(editor->total_selected_primitives > 0)
	    {
		/* Get last selected primitive index. */
		create_pos = editor->selected_primitive[
		    editor->total_selected_primitives - 1
		];
		/* Last selected primitives index invalid? */
		if((create_pos < 0) || (create_pos >= model_ptr->total_primitives))
		    create_pos = model_ptr->total_primitives;
	    }
	    else
	    {
		/* Nothing selected, so insert as first. */
		create_pos = 0;
	    }
	}
	/* Append? */
	else if(create_pos < 0)
	{
	    /* Appened. */
	    create_pos = model_ptr->total_primitives;
	}

	/* Final clip (include index at model_ptr->total_primitives
	 * since it will be valid when we increase pointer array
	 * allocation).
	 */
	if(create_pos > model_ptr->total_primitives)
	    create_pos = model_ptr->total_primitives;
	if(create_pos < 0)
	    create_pos = 0;

	/* Got new sanitize create position. */
	n = create_pos;

	/* Apply values from related resources if this model if
	 * selected.
	 */
	if(model_num == EditorSelectedModelIndex(editor))
	{
	    GtkCList *clist = (GtkCList *)editor->primitives_list;

	    /* Unselect the row we are inserting at, this will cause
	     * data values to be synced.
	     */
	    if(clist != NULL)
	    {
		if((n >= 0) && (n < clist->rows))
		    gtk_clist_unselect_row(clist, n, 0);
	    }
	}

	/* Increase model's primitives pointer array allocation. */
	model_ptr->total_primitives++;
	model_ptr->primitive = (void **)realloc(
	    model_ptr->primitive,
	    model_ptr->total_primitives * sizeof(void *)
	);
	if(model_ptr->primitive == NULL)
	{
	    model_ptr->total_primitives = 0;
	    V3DMPDestroy(p);
	    return(-1);
	}

	/* Shift pointers. */
	for(j = model_ptr->total_primitives - 1; j > n; j--)
	    model_ptr->primitive[j] = model_ptr->primitive[j - 1];

	/* Transfer primitive p to new primitives list allocation
	 * index n.
	 */
	model_ptr->primitive[n] = p;

/* Never recording undo for create primitive (for now atleast...) */

	/* Need to make sure new primitive is realized. */
	EditorPrimitiveRealize(editor, p, TRUE);

	/* Is given model selected on the editor? */
	if(model_num == EditorSelectedModelIndex(editor))
	{
	    /* Clear and reget primitives list. */
	    EditorListDeletePrimitivesG(editor);
	    EditorListAddPrimitivesRG(editor, model_ptr);

	    /* Clear values list. */
	    EditorListDeleteValuesG(editor);

	    /* Select newly created primitive? */
	    if(select)
	    {
		GtkCList *clist;

		clist = (GtkCList *)editor->primitives_list;
		if(clist != NULL)
		{
		    gtk_clist_select_row(
			clist,
			n,	/* Row. */
			0	/* Column. */
		    );
		}

		/* Select first row on values list. */
/*
Don't do this anymore.
		clist = (GtkCList *)editor->values_list;
		if(clist != NULL)
		{
		    gtk_clist_select_row(
			clist, 0, 0
		    );
		}
 */
	    }	/* Select newly created primitive? */
	}	/* Is given model selected on the editor? */

	/* Update menus and redraw all views on editor. */
	EditorUpdateMenus(editor);
	EditorUpdateAllViewMenus(editor);
	EditorRedrawAllViews(editor);

	return(n);
}

/*
 *	Deletes the specified primitives on the editor with the given
 *	list of primitive numbers and the total number of primitives in
 *	the list to be deleted.
 *
 *	The given pointer to list will be immediatly duplicated in this
 *	function so it can be considered constant.
 *
 *	Menus, views and lists will be updated.
 */
void EditorPrimitiveDoDelete(
	ma_editor_struct *editor,
	int model_num,
	int *list, int total,	/* List of primitive indexes to delete. */
	gbool record_undo
)
{
	gbool yes_to_all = FALSE;
	int i, j, pn, status;
	int first_sel_row = 0;
	v3d_model_struct *model_ptr;
	void *p;
	gbool just_destroy_it;
	const char *model_name;


	if((editor == NULL) ||
	   (list == NULL)
	)
	    return;

	if(!editor->initialized)
	    return;

	model_ptr = V3DModelListGetPtr(
	    editor->model, editor->total_models, model_num
	);
	if(model_ptr == NULL)
	    return;

	model_name = (const char *)((model_ptr->name == NULL) ?
	    "Untitled" : model_ptr->name
	);

	EDITOR_DO_UPDATE_HAS_CHANGES

	/* Duplicate selection buffer, so after this point list
	 * will be a locally allocated buffer which we need to free().
	 */
	if(total > 0)
	{
	    int *tmp_list = (int *)malloc(total * sizeof(int));

	    if(tmp_list != NULL)
		memcpy(tmp_list, list, total * sizeof(int));

	    list = tmp_list;
	}
	if(list == NULL)
	    return;


	EditorSetBusy(editor);

	/* Go through duplicated selections list of primitives. */
	for(i = 0; i < total; i++)
	{
	    pn = list[i];		/* Get selected primitive number. */

	    /* Get pointer to selected primitive on model. */
	    p = V3DMPListGetPtr(
		model_ptr->primitive, model_ptr->total_primitives, pn
	    );
	    if(p == NULL)
		continue;

	    /* Record first selected row on primitives list. */
	    if(i == 0)
		first_sel_row = pn;                                     


	    /* Check if recording undo, if so, then ask for comfermation.
	     * Otherwise we should interprite this as a forced call.
	     */
	    if(record_undo && !yes_to_all)
	    {
		gbool need_continue = FALSE;
		gbool need_break = FALSE;
		char *buf;
		int buf_len;


		/* Confirmation. */
		buf_len = 256 + strlen(model_name);
		buf = (char *)malloc((buf_len + 1) * sizeof(char));
		if(buf != NULL)
		    sprintf(buf,
 "Delete primitive #%i on model \"%s\"?",
			pn, model_name
		    );
		CDialogSetTransientFor(editor->toplevel);
		status = CDialogGetResponse(
"Delete Primitive?",
buf,
"Are you sure you want to delete the specified\n\
primitive from the model?",
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_YES_TO_ALL |
		    CDIALOG_BTNFLAG_NO | CDIALOG_BTNFLAG_CANCEL |
		    CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_NO
		);
		CDialogSetTransientFor(NULL);
		free(buf);
		buf = NULL;
		buf_len = 0;
		switch(status)
		{
		  case CDIALOG_RESPONSE_YES_TO_ALL:
		    yes_to_all = TRUE;
		  case CDIALOG_RESPONSE_YES:
		  case CDIALOG_RESPONSE_OK:
		    break;

		  case CDIALOG_RESPONSE_NO:
		    need_continue = TRUE;
		    break;

		  case CDIALOG_RESPONSE_CANCEL:
		  case CDIALOG_RESPONSE_NOT_AVAILABLE:
		    need_break = TRUE;
		    break;
		}
		if(need_continue)
		    continue;
		if(need_break)
		    break;
	    }

	    /* Certain primitives need their substructures deallocated
	     * explicitly before deallocating or passing to undo.
	     *
	     * Since some of these structures contain GL lists and they
	     * must be destroyed by first putting the view3d into context
	     * and then explicitly destroying the GL list.
	     *
	     * So unrealize this primitive.
	     */
	    EditorPrimitiveUnrealize(editor, p);

	    /* Record undo for deletion of this primitive? */
	    just_destroy_it = TRUE;
	    if(record_undo)
	    {
		/* Passing the primitive to the undo structure will
		 * effectivly destroy it, since its allocated structure
		 * is directly passed to the undo structure.
		 */
		gint insert_at;
		vma_undo_create_primitive_struct *u = (vma_undo_create_primitive_struct *)VMAUndoNew(
		    VMA_UNDO_TYPE_CREATE_PRIMITIVE,
		    "Delete Primitive"
		);

		/* Calculate insert at position for proper reinserting
		 * if this primitive delete operation was to be undone.
		 * Basically any selected item older than this one
		 * (i - 1) which has a lower selection value than
		 * this primitive (list[j] < pn) then this primitive's
		 * reinsert point should be decreased by the number of
		 * older selected primitives with index values less than
		 * this primitive.
		 */
		for(insert_at = pn, j = (i - 1);
		    j >= 0;
		    j--
		)
		{
		    if((list[j] < pn) && (list[j] >= 0))
			insert_at--;
		}
		if(insert_at < 0)
		    insert_at = 0;

		if(u != NULL)
	        {
		    u->editor = editor;
		    u->model_num = model_num;
		    u->insert_at = insert_at;
		    u->p = p;	/* By passing to undo, we consider
				 * it destroyed.
				 */
		    p = NULL;
		    EditorUndoRecord(editor, u);

		    /* We passed the primitive structure to be recorded
		     * on the undo structure, so do not destroy the
		     * the primitive.
		     */
		    just_destroy_it = FALSE;
		}
	    }
	    /* Just destroy the primitive? */
	    if(just_destroy_it)
	    {
		V3DMPDestroy(p);
	    }

	    /* Reset pointer to NULL, but do not shift pointers just
	     * yet. The pointers will be shifted after all primitives
	     * specified to be deleted have actually been deleted.
	     */
	    model_ptr->primitive[pn] = NULL;

	}	/* Go through duplicated selections list of primitives. */

	/* Inputs; list and total should be considered invalid now. */

	/* Remove NULL pointers from primitives list. */
	for(pn = 0; pn < model_ptr->total_primitives; pn++)
	{
	    p = model_ptr->primitive[pn];
	    if(p != NULL)
	        continue;

	    model_ptr->total_primitives--;
	    for(j = pn; j < model_ptr->total_primitives; j++)
		model_ptr->primitive[j] = model_ptr->primitive[j + 1];

	    /* Reallocate primitives list pointers. */
	    if(model_ptr->total_primitives > 0)
	    {
		model_ptr->primitive = (void **)realloc(
		    model_ptr->primitive,
		    model_ptr->total_primitives * sizeof(void *)
		);
		if(model_ptr->primitive == NULL)
		{
		    model_ptr->total_primitives = 0;
		    break;
		}
	    }
	    else
	    {
		free(model_ptr->primitive);
		model_ptr->primitive = NULL;
		model_ptr->total_primitives = 0;
		break;
	    }

	    /* Start at 0 again. */
	    pn = -1;
	}

	/* Is the specified model currently selected on the editor? */
	if(model_num == EditorSelectedModelIndex(editor))
	{
	    /* Need to update editor's primitives list since its
	     * displaying the primitives for the given model.
	     */
	    gint row;
	    GtkCList *clist;


	    /* Clear and reget primitives gui list. */
	    EditorListDeletePrimitivesG(editor);
	    EditorListAddPrimitivesRG(editor, model_ptr);

	    /* Delete values gui list. */
	    EditorListDeleteValuesG(editor);


	    /* Scroll to first selected row in primitives list. */
	    row = first_sel_row;

	    /* Get primitives list. */
	    clist = (GtkCList *)editor->primitives_list;
	    if(clist != NULL)
	    {
		if((row >= 0) && (row < clist->rows))
		{
		    /* Scroll to row if not visable. */
		    if(gtk_clist_row_is_visible(clist, row) !=
			GTK_VISIBILITY_FULL
		    )
			gtk_clist_moveto(
			    clist,
			    row, 0,	/* Row, column. */
			    0.5, 0.0	/* Row, column. */
			);
		}
	    }
	}

	/* Update menus and redraw all views on editor. */
	EditorUpdateMenus(editor);
	EditorUpdateAllViewMenus(editor);
	EditorRedrawAllViews(editor);

	EditorSetReady(editor);

	/* Free local selections list which we allocated. */
	free(list);

	return;
}


/*
 *	Checks if the primitive p with respect to the given editor
 *	appears to be realized or not. Primitives that do not need
 *	realizing will always be returned TRUE.
 *
 *	Returns FALSE on error.
 */
gbool EditorPrimitiveIsRealized(ma_editor_struct *editor, void *p)
{
	mp_texture_select_struct *mp_texture_select;
	mp_heightfield_load_struct *mp_heightfield_load;


	if((editor == NULL) || (p == NULL))
	    return(FALSE);

	switch(*(int *)p)
	{
	  case V3DMP_TYPE_TEXTURE_SELECT:
	    mp_texture_select = (mp_texture_select_struct *)p;
/* we should be checking mp_texture_select->client_data here but
   since it can be NULL and still valid (since it's probably a number)
   there is no good way to check so return it's always being realized
   for now.
 */
	    return(TRUE);
	    break;

	  case V3DMP_TYPE_HEIGHTFIELD_LOAD:
	    mp_heightfield_load = (mp_heightfield_load_struct *)p;
	    if(mp_heightfield_load->gl_list == NULL)
		return(FALSE);
	    else
		return(TRUE);
	    break;

	  default:
	    /* All other types return already realized. */
	    return(TRUE);
	    break;
	}

	return(FALSE);
}

/*
 *	Realizes the primitive with respect to the given editor.
 *	This only has affect on primitives that have resources that
 *	need to be (re)loaded (such as V3DMP_TYPE_TEXTURE_SELECT and
 *	V3DMP_TYPE_HEIGHTFIELD_LOAD).
 *
 *	The given editor should be the editor that contains the model
 *	which the primitive is on. The 3D view on the given editor
 *	will may be put into context after a call to this function.
 *
 *	If rerealize is TRUE and the primitive is already realized then
 *	the primitive will be rerealized (reloading of data).
 */
void EditorPrimitiveRealize(
	ma_editor_struct *editor, void *p,
	gbool rerealize
)
{
	mp_texture_select_struct *mp_texture_select;
	mp_heightfield_load_struct *mp_heightfield_load;


	if((editor == NULL) || (p == NULL))
	    return;

	switch(*(int *)p)
	{
	  case V3DMP_TYPE_TEXTURE_SELECT:
	    mp_texture_select = (mp_texture_select_struct *)p;
	    EditorTexturePrimitiveRealize(
		editor, mp_texture_select, rerealize
	    );
	    break;

	  case V3DMP_TYPE_HEIGHTFIELD_LOAD:
	    mp_heightfield_load = (mp_heightfield_load_struct *)p;
	    EditorHFPrimitiveRealize(
		editor, mp_heightfield_load, rerealize
	    );
	    break;
	}
}

/*
 *	Unrealizes the primitive, effectivly doing the opposite of
 *	EditorPrimitiveRealize().
 *
 *      This only has affect on primitives that have resources that
 *      need to be unloaded (such as V3DMP_TYPE_TEXTURE_SELECT and
 *      V3DMP_TYPE_HEIGHTFIELD_LOAD).
 *
 *      The given editor should be the editor that contains the model
 *      which the primitive is on. The 3D view on the given editor
 *      will may be put into context after a call to this function.
 */
void EditorPrimitiveUnrealize(ma_editor_struct *editor, void *p)
{
	mp_texture_select_struct *mp_texture_select;
	mp_heightfield_load_struct *mp_heightfield_load;
 
 
	if((editor == NULL) || (p == NULL))
	    return;

	switch(*(int *)p)
	{
	  case V3DMP_TYPE_TEXTURE_SELECT:
	    mp_texture_select = (mp_texture_select_struct *)p;
	    EditorTexturePrimitiveUnrealize(
		editor, mp_texture_select
	    );
	    break;

	  case V3DMP_TYPE_HEIGHTFIELD_LOAD:
	    mp_heightfield_load = (mp_heightfield_load_struct *)p;
	    EditorHFPrimitiveUnrealize(
		editor, mp_heightfield_load
	    );
	    break;
	}

	return;
}

/*
 *	Same as EditorPrimitiveRealize(), except it realizes all
 *	primitives on the specified model number on the given editor.
 */
void EditorPrimitiveRealizeAll(
	ma_editor_struct *editor, int model_num, gbool rerealize
)
{
	int i;
	void *p;
	v3d_model_struct *model_ptr;


	if(editor == NULL)
	    return;

	model_ptr = V3DModelListGetPtr(
	    editor->model, editor->total_models, model_num
	);
	if(model_ptr == NULL)
	    return;

	/* Go through all primitives on the model. */
	for(i = 0; i < model_ptr->total_primitives; i++)
	{
	    p = model_ptr->primitive[i];
	    if(p == NULL)
		continue;

	    EditorPrimitiveRealize(editor, p, rerealize);
	}

	return;
}

/*
 *	Same as EditorPrimitiveRealizeAll() except that it will only
 *	realize primitives on the specified model that match the
 *	specified primitive type.
 */
void EditorPrimitiveRealizeAllType(
	ma_editor_struct *editor, int model_num, int ptype, gbool rerealize
)
{
	gint i;
	gpointer p;
	v3d_model_struct *model_ptr;


	if(editor == NULL)
	    return;

	model_ptr = V3DModelListGetPtr(
	    editor->model, editor->total_models, model_num
	);
	if(model_ptr == NULL)
	    return;

	/* Go through all primitives on the model. */
	for(i = 0; i < model_ptr->total_primitives; i++)
	{
	    p = model_ptr->primitive[i];
	    if(p == NULL)
		continue;

	    /* Primitive types must match. */
	    if(ptype != (*(gint *)p))
		continue;  

	    EditorPrimitiveRealize(editor, p, rerealize);
	}
}

/*
 *      Same as EditorPrimitiveUnrealize(), except it unrealizes all
 *      primitives on the specified model number on the given editor.
 */
void EditorPrimitiveUnrealizeAll(ma_editor_struct *editor, int model_num)
{
	int i;
	void *p;
	v3d_model_struct *model_ptr;


	if(editor == NULL)
	    return;

	/* Get pointer to model. */
	model_ptr = V3DModelListGetPtr(
	    editor->model, editor->total_models, model_num
	);
	if(model_ptr == NULL)
	    return;

	/* Go through all primitives on the model. */
	for(i = 0; i < model_ptr->total_primitives; i++)
	{
	    p = model_ptr->primitive[i];
	    if(p == NULL)
		continue;

	    EditorPrimitiveUnrealize(editor, p);
	}

	return;
}


/*
 *	Adds a vertex to the primitive pn on the given model at the
 *	position vertex_num.
 *
 *	If v or tc are NULL then default values for v and tc will be
 *	generated. If they are not then they will be coppied over to
 *	the new vertex and texture coordinate and should not be referanced
 *	again after calling this function.
 *
 *	Returns the new vertex number or -1 on failure.
 */
int EditorPrimitiveVertexAdd(
	ma_editor_struct *editor, int model_num, int pn, int vertex_num,
	mp_vertex_struct *v, mp_vertex_struct *n, mp_vertex_struct *tc,
	gbool report_errors
)
{
	int i, ptype, row, last_values_list_row;
	int new_vertex_num = -1;
	void *p;
	v3d_model_struct *model_ptr;
	GtkCList *values_list;
	mp_line_strip_struct *mp_line_strip;
	mp_line_loop_struct *mp_line_loop;
	mp_triangle_strip_struct *mp_triangle_strip;
	mp_triangle_fan_struct *mp_triangle_fan;
	mp_quad_strip_struct *mp_quad_strip;
	mp_polygon_struct *mp_polygon;


#define DEALLOC_VERTEXES	\
{ \
 if(v != NULL) \
 { \
  free(v); \
  v = NULL; \
 } \
 if(n != NULL) \
 { \
  free(n); \
  n = NULL; \
 } \
 if(tc != NULL) \
 { \
  free(tc); \
  tc = NULL; \
 } \
}

	if((editor == NULL) || (model_num < 0) || (pn < 0))
	{
	    DEALLOC_VERTEXES
	    return(new_vertex_num);
	}

	/* If v, n, or tc are NULL then allocate them. */
	if(v == NULL)
	{
	    v = (mp_vertex_struct *)calloc(1, sizeof(mp_vertex_struct));
	    if(v == NULL)
	    {
		DEALLOC_VERTEXES
		return(new_vertex_num);
	    }
	    else
	    {
		v->x = editor->vcursor_x;
		v->y = editor->vcursor_y;
		v->z = editor->vcursor_z;
	    }
	}
	if(n == NULL)
	{
	    n = (mp_vertex_struct *)calloc(1, sizeof(mp_vertex_struct));
	    if(n == NULL)
	    {
		DEALLOC_VERTEXES
		return(new_vertex_num);
	    }
	    else
	    {   
		n->x = 0.0;
		n->y = 0.0;
		n->z = 0.0;
	    }
	}
	if(tc == NULL)
	{
	    tc = (mp_vertex_struct *)calloc(1, sizeof(mp_vertex_struct));
	    if(tc == NULL)
	    {
		DEALLOC_VERTEXES
		return(new_vertex_num);
	    } 
	    else
	    {
		tc->x = 0.0;
		tc->y = 0.0;
		tc->z = 0.0;
	    }
	}

	/* Record current values list row. */
	last_values_list_row = vertex_num;

	/* Get pointer to values list. */
	values_list = (GtkCList *)editor->values_list;
	if(values_list == NULL)
	{
	    DEALLOC_VERTEXES
	    return(new_vertex_num);
	}

	/* Get pointer to model. */
	model_ptr = V3DModelListGetPtr(
	    editor->model, editor->total_models, model_num
	);
	if(model_ptr == NULL)
	{
	    DEALLOC_VERTEXES
	    return(new_vertex_num);
	}

	/* Get pointer to primitive on model. */
	p = V3DMPListGetPtr(
	    model_ptr->primitive, model_ptr->total_primitives, pn
	);
	if(p == NULL)
	{
	    DEALLOC_VERTEXES
	    return(new_vertex_num);
	}
	else
	{
	    ptype = (*(int *)p);
	}

/* Records a remove vertex for the vertex specified by vertex_num. */
#define DO_RECORD_UNDO	\
{ \
 if(record_undo) \
 { \
  vma_undo_remove_vertex_struct *u = (vma_undo_remove_vertex_struct *)VMAUndoNew( \
    VMA_UNDO_TYPE_REMOVE_VERTEX, "Add Vertex" \
   ); \
  if(u != NULL) \
  { \
   u->editor = (void *)editor; \
   u->model_num = model_num; \
   u->primitive_num = pn; \
   u->delete_at = vertex_num; \
   EditorUndoRecord(editor, u); \
  } \
}

/* Macro to add given vertex v and tc to the primitive specified by
 * whatever MP_PTR is pointing to (must be a variable number vertex
 * primitive). The variables v and tc will be reset to NULL after this
 * call. Can return if allocation failed.
 */
#define DO_ADD_VERTEX	\
{ \
 /* Sanitize then increment total. */ \
 if(MP_PTR->total < 0) \
  MP_PTR->total = 0; \
 MP_PTR->total++; \
 /* Sanitize vertex_num to insert at. */ \
 if(vertex_num >= MP_PTR->total) \
  vertex_num = MP_PTR->total - 1; \
 if(vertex_num < 0) \
  vertex_num = MP_PTR->total - 1; \
 /* Allocate more pointers for v and tc array. */ \
 MP_PTR->v = (mp_vertex_struct **)realloc( \
  MP_PTR->v, \
  MP_PTR->total * sizeof(mp_vertex_struct *) \
 ); \
 MP_PTR->n = (mp_vertex_struct **)realloc( \
  MP_PTR->n, \
  MP_PTR->total * sizeof(mp_vertex_struct *) \
 ); \
 MP_PTR->tc = (mp_vertex_struct **)realloc( \
  MP_PTR->tc, \
  MP_PTR->total * sizeof(mp_vertex_struct *) \
 ); \
 if((MP_PTR->v == NULL) || (MP_PTR->n == NULL) || (MP_PTR->tc == NULL)) \
 { \
  /* Allocation failed, reset total and deallocate given vertices. */ \
  MP_PTR->total = 0; \
  DEALLOC_VERTEXES \
  return(new_vertex_num); \
 } \
 /* Shift pointers. */ \
 for(i = MP_PTR->total - 1; i > vertex_num; i--) \
 { \
  MP_PTR->v[i] = MP_PTR->v[i - 1]; \
  MP_PTR->n[i] = MP_PTR->n[i - 1]; \
  MP_PTR->tc[i] = MP_PTR->tc[i - 1]; \
 } \
 /* Record new vertex on primitive, reset given pointers so they \
  * do not get freed later after being transfered to the primitve. \
  */ \
 new_vertex_num = i = vertex_num; \
 MP_PTR->v[i] = v; \
 v = NULL; \
 MP_PTR->n[i] = n; \
 n = NULL; \
 MP_PTR->tc[i] = tc; \
 tc = NULL; \
}

	/* Handle by primitive type. */
	switch(ptype)
	{
	  case V3DMP_TYPE_LINE_STRIP:
#define MP_PTR  mp_line_strip
	    MP_PTR = (mp_line_strip_struct *)p;
	    DO_ADD_VERTEX
	    break;

	  case V3DMP_TYPE_LINE_LOOP:
#undef MP_PTR
#define MP_PTR  mp_line_loop
	    MP_PTR = (mp_line_loop_struct *)p;
	    DO_ADD_VERTEX
	    break;

	  case V3DMP_TYPE_TRIANGLE_STRIP:
#undef MP_PTR
#define MP_PTR  mp_triangle_strip
	    MP_PTR = (mp_triangle_strip_struct *)p;
	    DO_ADD_VERTEX
	    break;

	  case V3DMP_TYPE_TRIANGLE_FAN:
#undef MP_PTR
#define MP_PTR  mp_triangle_fan
	    MP_PTR = (mp_triangle_fan_struct *)p;
	    DO_ADD_VERTEX
	    break;

	  case V3DMP_TYPE_QUAD_STRIP:
#undef MP_PTR
#define MP_PTR  mp_quad_strip
	    MP_PTR = (mp_quad_strip_struct *)p;
	    DO_ADD_VERTEX
	    break;

	  case V3DMP_TYPE_POLYGON:
#undef MP_PTR
#define MP_PTR  mp_polygon
	    MP_PTR = (mp_polygon_struct *)p;
	    DO_ADD_VERTEX
	    break;

#undef MP_PTR

	  default:
	    if(report_errors)
	    {
		CDialogSetTransientFor(editor->toplevel);
		CDialogGetResponse(
"Cannot add vertex!",
"This primitive has a fixed number of vertices, you\n\
cannot add a new vertex to it.",
"The number of vertices on this primitive must remain\n\
constant due to its geometric characteristic\n\
constraints.",
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	    }
	    DEALLOC_VERTEXES
	    return(new_vertex_num);
	    break;
	}




	EDITOR_DO_UPDATE_HAS_CHANGES

	/* Update primitives list. */
	EditorListPrimitivesSet(
	    editor->primitives_list, pn, p, TRUE
	);

	/* Reget values list. */
	EditorListDeleteValuesG(editor);
	EditorListAddValuesRG(editor, p);

	/* Restore previous values list scroll position and selection.
	 * When we select the row, the select callback will do the
	 * scrolling for us.
	 */ 
	row = last_values_list_row;
	if((row >= 0) && (row < values_list->rows))
	    gtk_clist_select_row(
		values_list, row, 0
	    );

	/* Update menus and redraw all views on editor. */
	EditorUpdateMenus(editor);
	EditorUpdateAllViewMenus(editor);
	EditorRedrawAllViews(editor);

	DEALLOC_VERTEXES

#undef DO_RECORD_UNDO
#undef DEALLOC_VERTEXES

	return(new_vertex_num);
}

/*
 *	Removes vertex specified by vertex_num on the primitive from the
 *	given model.
 */
void EditorPrimitiveVertexRemove(
	ma_editor_struct *editor, int model_num, int pn, int vertex_num,
	gbool record_undo, gbool report_errors
)
{
	int i, ptype, row, last_values_list_row;
	GtkCList *values_list;
	v3d_model_struct *model_ptr;
	void *p;
	mp_line_strip_struct *mp_line_strip;
	mp_line_loop_struct *mp_line_loop;
	mp_triangle_strip_struct *mp_triangle_strip;
	mp_triangle_fan_struct *mp_triangle_fan;
	mp_quad_strip_struct *mp_quad_strip;
	mp_polygon_struct *mp_polygon;
	mp_vertex_struct *v, *n, *tc;


	if((editor == NULL) || (model_num < 0) || (pn < 0) ||
	   (vertex_num < 0)
	)
	    return;

	/* Record current values list row. */
	last_values_list_row = vertex_num;

	/* Get pointer to values list. */
	values_list = (GtkCList *)editor->values_list;
	if(values_list == NULL)
	    return;

	/* Get pointer to model. */
	model_ptr = V3DModelListGetPtr(
	    editor->model, editor->total_models, model_num
	);
	if(model_ptr == NULL)
	    return;

	/* Get pointer to primitive on model. */
	p = V3DMPListGetPtr(
	    model_ptr->primitive, model_ptr->total_primitives, pn
	);
	if(p == NULL)
	    return;
	else
	    ptype = (*(int *)p);

#define DO_WARN_TOO_FEW		\
{ \
 if(report_errors) \
 { \
  CDialogSetTransientFor(editor->toplevel); \
  CDialogGetResponse( \
"Cannot remove vertex!", \
"This primitive requires a minimum amount of vertices,\n\
you cannot delete additional vertices.", \
"The number of vertices on this primitive must at a\n\
minimum amount due to its geometric characteristic\n\
constraints.", \
   CDIALOG_ICON_ERROR, \
   CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP, \
   CDIALOG_BTNFLAG_OK \
  ); \
  CDialogSetTransientFor(NULL); \
 } \
}

/* Macro to record undo as needed or delete the vertex.
 * Variables v and tc should be considered garbage and deallocated after
 * this macro.
 */
#define DO_DELETE_VERTEX	\
{ \
 if(record_undo) \
 { \
  vma_undo_add_vertex_struct *u = (vma_undo_add_vertex_struct *)VMAUndoNew( \
    VMA_UNDO_TYPE_ADD_VERTEX, "Remove Vertex" \
   ); \
  if(u != NULL) \
  { \
   u->editor = (void *)editor; \
   u->model_num = model_num; \
   u->primitive_num = pn; \
   u->insert_at = vertex_num; \
   u->v = v; \
   u->n = n; \
   u->tc = tc; \
   EditorUndoRecord(editor, u); \
  } \
 } \
 else \
 { \
  /* Do not record undo, so just delete. */ \
  free(v); \
  free(n); \
  free(tc); \
 } \
}

	/* Handle by primitive type. */
	switch(ptype)
	{
	  case V3DMP_TYPE_LINE_STRIP:
#define MP_PTR	mp_line_strip
	    MP_PTR = (mp_line_strip_struct *)p;
	    if(MP_PTR->total <= 2)
	    {
		DO_WARN_TOO_FEW
		return;
	    }
	    if((vertex_num >= 0) && (vertex_num < MP_PTR->total))
	    {
		v = MP_PTR->v[vertex_num];
		n = MP_PTR->n[vertex_num];
		tc = MP_PTR->tc[vertex_num];
		DO_DELETE_VERTEX

		/* Reduce total and shift. */
		MP_PTR->total--;
		for(i = vertex_num; i < MP_PTR->total; i++)
		{
		    MP_PTR->v[i] = MP_PTR->v[i + 1];
		    MP_PTR->n[i] = MP_PTR->n[i + 1];
		    MP_PTR->tc[i] = MP_PTR->tc[i + 1]; 
		}
	    }
	    break;

	  case V3DMP_TYPE_LINE_LOOP:
#undef MP_PTR
#define MP_PTR  mp_line_loop
	    MP_PTR = (mp_line_loop_struct *)p;
	    if(MP_PTR->total <= 3)
	    {
		DO_WARN_TOO_FEW
		return;
	    }
	    if((vertex_num >= 0) && (vertex_num < MP_PTR->total))
	    {
		v = MP_PTR->v[vertex_num];
		n = MP_PTR->n[vertex_num];
		tc = MP_PTR->tc[vertex_num];
		DO_DELETE_VERTEX

		/* Reduce total and shift. */
		MP_PTR->total--;
		for(i = vertex_num; i < MP_PTR->total; i++)
		{
		    MP_PTR->v[i] = MP_PTR->v[i + 1];
		    MP_PTR->n[i] = MP_PTR->n[i + 1];
		    MP_PTR->tc[i] = MP_PTR->tc[i + 1];
		}
	    }
	    break;

	  case V3DMP_TYPE_TRIANGLE_STRIP:
#undef MP_PTR
#define MP_PTR  mp_triangle_strip
	    MP_PTR = (mp_triangle_strip_struct *)p;
	    if(MP_PTR->total <= 3)
	    {
		DO_WARN_TOO_FEW
		return;
	    }
	    if((vertex_num >= 0) && (vertex_num < MP_PTR->total))
	    {
		v = MP_PTR->v[vertex_num];
		n = MP_PTR->n[vertex_num];
		tc = MP_PTR->tc[vertex_num];
		DO_DELETE_VERTEX

		/* Reduce total and shift. */
		MP_PTR->total--;
		for(i = vertex_num; i < MP_PTR->total; i++)
		{
		    MP_PTR->v[i] = MP_PTR->v[i + 1];
		    MP_PTR->n[i] = MP_PTR->n[i + 1];
		    MP_PTR->tc[i] = MP_PTR->tc[i + 1];
		}
	    }
	    break;

	  case V3DMP_TYPE_TRIANGLE_FAN:
#undef MP_PTR
#define MP_PTR  mp_triangle_fan
	    MP_PTR = (mp_triangle_fan_struct *)p;
	    if(MP_PTR->total <= 3)
	    {
		DO_WARN_TOO_FEW
		return;
	    }
	    if((vertex_num >= 0) && (vertex_num < MP_PTR->total))
	    {
		v = MP_PTR->v[vertex_num];
		n = MP_PTR->n[vertex_num];
		tc = MP_PTR->tc[vertex_num];
		DO_DELETE_VERTEX

		/* Reduce total and shift. */
		MP_PTR->total--;
		for(i = vertex_num; i < MP_PTR->total; i++)
		{
		    MP_PTR->v[i] = MP_PTR->v[i + 1];
		    MP_PTR->n[i] = MP_PTR->n[i + 1];
		    MP_PTR->tc[i] = MP_PTR->tc[i + 1];
		}
	    }
	    break;

	  case V3DMP_TYPE_QUAD_STRIP:
#undef MP_PTR
#define MP_PTR	mp_quad_strip
	    MP_PTR = (mp_quad_strip_struct *)p;
	    if(MP_PTR->total <= 4)
	    {
		DO_WARN_TOO_FEW
		return;
	    }
	    if((vertex_num >= 0) && (vertex_num < MP_PTR->total))
	    {
		v = MP_PTR->v[vertex_num];
		n = MP_PTR->n[vertex_num];
		tc = MP_PTR->tc[vertex_num];
		DO_DELETE_VERTEX

		/* Reduce total and shift. */
		MP_PTR->total--;
		for(i = vertex_num; i < MP_PTR->total; i++)
		{
		    MP_PTR->v[i] = MP_PTR->v[i + 1];
		    MP_PTR->n[i] = MP_PTR->n[i + 1];
		    MP_PTR->tc[i] = MP_PTR->tc[i + 1];
		}
	    }
	    break;

	  case V3DMP_TYPE_POLYGON:
#undef MP_PTR
#define MP_PTR  mp_polygon
	    MP_PTR = (mp_polygon_struct *)p;
	    if(MP_PTR->total <= 3)
	    {
		DO_WARN_TOO_FEW
		return;
	    }
	    if((vertex_num >= 0) && (vertex_num < MP_PTR->total))
	    {       
		v = MP_PTR->v[vertex_num];
		n = MP_PTR->n[vertex_num];
		tc = MP_PTR->tc[vertex_num];
		DO_DELETE_VERTEX
    
		/* Reduce total and shift. */
		MP_PTR->total--;
		for(i = vertex_num; i < MP_PTR->total; i++)
		{
		    MP_PTR->v[i] = MP_PTR->v[i + 1];
		    MP_PTR->n[i] = MP_PTR->n[i + 1];
		    MP_PTR->tc[i] = MP_PTR->tc[i + 1];
		}
	    }
	    break;

	  default:
	    /* This primitive cannot have a variable number of
	     * vertices.
	     */
	    if(report_errors)
	    {
		CDialogSetTransientFor(editor->toplevel);
		CDialogGetResponse(
"Cannot remove vertex!",
"This primitive has a fixed number of vertices, you\n\
cannot remove any vertices from it.",
"The number of vertices on this primitive must remain\n\
constant due to its geometric characteristic\n\
constraints.",
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	    }
	    return;	/* Return, do not continue. */
	    break;
	}
#undef MP_PTR


	EDITOR_DO_UPDATE_HAS_CHANGES

	/* Update primitives list. */
	EditorListPrimitivesSet(
	    editor->primitives_list, pn, p, TRUE
	);

	/* Reget values list. */
	EditorListDeleteValuesG(editor);
	EditorListAddValuesRG(editor, p);

	/* Restore previous values list scroll position and selection.
	 * When we select the row, the select callback will do the
	 * scrolling for us.
	 */
	row = last_values_list_row;
	if((row >= 0) && (row < values_list->rows))
	    gtk_clist_select_row(
		values_list, row, 0
	    );

	/* Update menus and redraw all views on editor. */
	EditorUpdateMenus(editor);
	EditorUpdateAllViewMenus(editor);
	EditorRedrawAllViews(editor);

#undef DO_DELETE_VERTEX
#undef DO_WARN_TOO_FEW

	return;
}


/*
 *	Creates primitive create submentu for the given editor and
 *	returns the pointer to the parent menu widget.
 */
GtkWidget *EditorPrimitivesCreateCreateMenu(ma_editor_struct *editor)
{
	GtkWidget *menu, *w, *fw;
	int accel_key;
	void *accel_group = NULL;
	unsigned int accel_mods;
	void *client_data = editor;
	u_int8_t **icon;
	char *label = NULL;
	void (*func_cb)(GtkWidget *w, void *);


	if(editor == NULL)
	    return(NULL);

#define DO_ADD_MENU_ITEM_LABEL	\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_LABEL, accel_group, \
  icon, label, accel_key, accel_mods, (void **)&fw, \
  client_data, func_cb \
 ); \
}
#define DO_ADD_MENU_SEP	\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL, \
  NULL, NULL, 0, 0, NULL, \
  NULL, NULL \
 ); \
}

	menu = GUIMenuCreate();
	if(menu != NULL)
	{
	    icon = (u_int8_t **)icon_mp_comment_20x20_xpm;
	    label = "Comment";
	    accel_key = 0; 
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateCommentCB;
	    DO_ADD_MENU_ITEM_LABEL

	    DO_ADD_MENU_SEP

	    icon = (u_int8_t **)icon_mp_translate_20x20_xpm;
	    label = "Translate";
	    accel_key = 0; 
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateTranslateCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_rotate_20x20_xpm;
	    label = "Rotate";
	    accel_key = 0; 
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateRotateCB;
	    DO_ADD_MENU_ITEM_LABEL

	    DO_ADD_MENU_SEP

	    icon = (u_int8_t **)icon_mp_point_20x20_xpm;
	    label = "Point";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreatePointCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_line_20x20_xpm;
	    label = "Line";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateLineCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_linestrip_20x20_xpm;
	    label = "Line Strip";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateLineStripCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_lineloop_20x20_xpm;
	    label = "Line Loop";  
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateLineLoopCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_triangle_20x20_xpm;
	    label = "Triangle";   
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateTriangleCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_trianglestrip_20x20_xpm;
	    label = "Triangle Strip";
	    accel_key = 0; 
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateTriangleStripCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_trianglefan_20x20_xpm;
	    label = "Triangle Fan";
	    accel_key = 0; 
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateTriangleFanCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_quad_20x20_xpm;
	    label = "Quad";
	    accel_key = 0; 
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateQuadCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_quadstrip_20x20_xpm;
	    label = "Quad Strip";
	    accel_key = 0;   
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateQuadStripCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_polygon_20x20_xpm;
	    label = "Polygon";
	    accel_key = 0; 
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreatePolygonCB;
	    DO_ADD_MENU_ITEM_LABEL

	    DO_ADD_MENU_SEP

	    icon = (u_int8_t **)icon_mp_color_20x20_xpm;
	    label = "Color";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateColorCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_texture_select_20x20_xpm;
	    label = "Texture Select";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateTextureSelectCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_texture_orient_20x20_xpm;
	    label = "Texture Orient XY";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateTextureOrientXYCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_texture_orient_20x20_xpm;
	    label = "Texture Orient YZ";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateTextureOrientYZCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_texture_orient_20x20_xpm;
	    label = "Texture Orient XZ";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateTextureOrientXZCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_texture_off_20x20_xpm;
	    label = "Texture Off";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateTextureOffCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = (u_int8_t **)icon_mp_heightfield_load_20x20_xpm;
	    label = "HeightField";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EditorPrimitiveCreateHeightFieldLoadCB;
	    DO_ADD_MENU_ITEM_LABEL

	}

#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_SEP

	return(menu);
}


/*
 *	Creates primitive create toolbar on the given editor,
 *	this function should be called from EditorCreate().
 */
int EditorPrimitivesCreateToolBar(ma_editor_struct *editor)
{




	return(0);
}
