#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#include <sys/types.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>

#include "../include/disk.h"
#include "../include/string.h"

#include "msglist.h"
#include "v3dfio.h"

#include "guiutils.h"
#include "csd.h"

#include "prefwin.h"
#include "prefwincb.h"

#include "vmastyles.h"
#include "vma.h"
#include "vmapixmaps.h"

#include "messages.h"
#include "config.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


#include "images/icon_vma_48x48.xpm"
#include "images/icon_ok_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_plugin_enabled_20x20.xpm"
#include "images/icon_plugin_disabled_20x20.xpm"
#include "images/icon_reload_20x20.xpm"
#include "images/icon_options_20x20.xpm"
#include "images/vertex_banner_400x50.xpm"


GtkWidget *PrefParmGetWidget(vma_pref_struct *pref, int parameter);
GtkWidget *PrefPanelGetWidget(vma_pref_struct *pref, int catagory);
char *PrefPanelGetBranchText(vma_pref_struct *pref, GtkCTreeNode *branch);

int PrefParmRecord(
        vma_pref_struct *pref,
        int parameter,          /* One of VMA_PREF_PARM_*. */
        GtkWidget *w
);
int PrefPanelRecord(
        vma_pref_struct *pref,
        int catagory,           /* One of VMA_PREF_CAT_. */
        GtkWidget *w            /* A vbox. */
);

void *PrefParmGetValueP(
        vma_pref_struct *pref,
        int parameter           /* One of VMA_PREF_PARM_*. */
);
int PrefParmGetValueI(
        vma_pref_struct *pref,
        int parameter           /* One of VMA_PREF_PARM_*. */
);
float PrefParmGetValueF(
        vma_pref_struct *pref,
        int parameter           /* One of VMA_PREF_PARM_*. */
);
double PrefParmGetValueD(
        vma_pref_struct *pref,
        int parameter           /* One of VMA_PREF_PARM_*. */
);
gbool PrefParmGetValueB(  
        vma_pref_struct *pref,
        int parameter           /* One of VMA_PREF_PARM_*. */
);

void PrefBranchSetData(
        GtkCTree *ctree, GtkCTreeNode *branch,
        vma_pref_item_struct *item_ptr, GtkDestroyNotify destroy
);
vma_pref_item_struct *PrefBranchGetData(
	GtkCTree *ctree, GtkCTreeNode *branch
);
void PrefSetBusy(vma_pref_struct *pref);
void PrefSetReady(vma_pref_struct *pref);

GtkCTreeNode *PrefAddCatagoryBranch(
        vma_pref_struct *pref,
        const char *catagory_name,
        GtkCTreeNode *parent, GtkCTreeNode *sibling,
        GdkPixmap *pixmap_closed, GdkBitmap *mask_closed,
        GdkPixmap *pixmap_opened, GdkBitmap *mask_opened,
        gbool is_leaf,   
        GtkWidget *panel_vbox,
        int catagory
);

static void PrefCreateHeading(
	vma_pref_struct *pref,
        GtkWidget *parent,	/* An hbox. */
        gint parent_border,	/* Border when pack into given parent. */
	const gchar *label
);
static void PrefCreateColorButton(
        vma_pref_struct *pref,
        GtkWidget *parent,	/* An hbox. */
        gint parent_border,	/* Border when pack into given parent. */
        const gchar *label,
        gint label_width,
        gint parameter
);
static void PrefCreateFontButton(
        vma_pref_struct *pref,
        GtkWidget *parent,	/* An hbox. */
        gint parent_border,	/* Border when pack into given parent. */
        const gchar *label,
        const gchar *font_name,
        gint label_width,
        gint entry_width,
        gint parameter
);
static void PrefCreateSpinPrompt(
        vma_pref_struct *pref,
        GtkWidget *parent,	/* An hbox. */
        gint parent_border,	/* Border when pack into given parent. */
        const gchar *label,
        GtkAdjustment *adj,	/* Adjustment for spin widget. */
        guint digits,		/* Digits. */
        gint label_width,
        gint spin_width,
        gint parameter,
	const gchar *tt_name
);
static void PrefCreateScalePrompt(
        vma_pref_struct *pref,
        GtkWidget *parent,	/* An hbox. */
        gint parent_border,	/* Border when pack into given parent. */
        const gchar *label,
        GtkAdjustment *adj,	/* Adjustment for spin widget. */
        guint digits,		/* Digits. */
        gint label_width,
        gint scale_width, 
        gint parameter,
	const gchar *tt_name
);

static void PrefCreatePanelMain(vma_pref_struct *pref, GtkWidget *parent);
static void PrefCreatePanelApperance(vma_pref_struct *pref, GtkWidget *parent);
static void PrefCreatePanelFonts(vma_pref_struct *pref, GtkWidget *parent);
static void PrefCreatePanelColors(vma_pref_struct *pref, GtkWidget *parent);
static void PrefCreatePanelPrecision(vma_pref_struct *pref, GtkWidget *parent);
static void PrefCreatePanelRendering(vma_pref_struct *pref, GtkWidget *parent);
static void PrefCreatePanelDevices(vma_pref_struct *pref, GtkWidget *parent);
static void PrefCreatePanelEditor(vma_pref_struct *pref, GtkWidget *parent);
static void PrefCreatePanelLocations(vma_pref_struct *pref, GtkWidget *parent);
static void PrefCreatePanelV3DFormat(vma_pref_struct *pref, GtkWidget *parent);
static void PrefCreatePanelBackup(vma_pref_struct *pref, GtkWidget *parent);
static void PrefCreatePanelPlugins(vma_pref_struct *pref, GtkWidget *parent);
static void PrefCreateCatagories(vma_pref_struct *pref);

vma_pref_struct *PrefNew(void *core_ptr);
void PrefUpdateMenus(vma_pref_struct *pref);
void PrefReset(vma_pref_struct *pref, gbool need_unmap);
void PrefMap(vma_pref_struct *pref);  
void PrefUnmap(vma_pref_struct *pref);
void PrefDelete(vma_pref_struct *pref);


#define PREF_BTN_WIDTH	(100 + (2 * 3))
#define PREF_BTN_HEIGHT	(30 + (2 * 3))


/*
 *	Returns the widget for the given parameter which matches
 *	the index of widgets holding parameter values on the preferences
 *	strcture.
 *
 *	Can return NULL on error.
 */
GtkWidget *PrefParmGetWidget(vma_pref_struct *pref, int parameter)
{
	if(pref == NULL)
	    return(NULL);
	else if(pref->parm_widget == NULL)
	    return(NULL);
	else if((parameter < 0) || (parameter >= pref->total_parm_widgets))
	    return(NULL);

	return(pref->parm_widget[parameter]);
}

/*
 *      Returns the widget for the given category which matches
 *      the index of panel parent widgets on the preferences structure.
 *
 *      Can return NULL on error.
 */
GtkWidget *PrefPanelGetWidget(vma_pref_struct *pref, int catagory)
{
        if(pref == NULL)
            return(NULL);
        else if(pref->panel_parent == NULL)
            return(NULL);
        else if((catagory < 0) || (catagory >= pref->total_panel_parents))
            return(NULL);

        return(pref->panel_parent[catagory]);
}


/*
 *	Convience function to return the text of the given branch.
 *
 *	Return is statically allocated, do not deallocate.
 */
char *PrefPanelGetBranchText(vma_pref_struct *pref, GtkCTreeNode *branch)
{
	gchar *text;
	guint8 spacing;
	GdkPixmap *pixmap;
	GdkBitmap *mask;


	if((pref == NULL) || (branch == NULL))
	    return(NULL);

	if(pref->catagory_ctree == NULL)
	    return(NULL);

	if(!gtk_ctree_node_get_pixtext(
	    (GtkCTree *)pref->catagory_ctree,
	    branch, 0,
	    &text, &spacing, &pixmap, &mask
	))
	    text = NULL;

	return((char *)text);
}

/*
 *	Records the widget on the preferences structure.
 *
 *	Returns parameter on success or -1 on error.
 */
int PrefParmRecord(
        vma_pref_struct *pref,
        int parameter,          /* One of VMA_PREF_PARM_*. */
        GtkWidget *w
)
{
	if(pref == NULL)
	    return(-1);

	if(pref->total_parm_widgets < 0)
	    pref->total_parm_widgets = 0;

	if(parameter >= pref->total_parm_widgets)
	{
	    /* Need to allocate more pointers. */
	    int i;
	    int prev_total = pref->total_parm_widgets;

	    /* Increase total. */
	    pref->total_parm_widgets = parameter + 1;

	    /* Reallocate pointer array. */
	    pref->parm_widget = (GtkWidget **)g_realloc(
		pref->parm_widget,
		pref->total_parm_widgets * sizeof(GtkWidget *)
	    );
	    if(pref->parm_widget == NULL)
	    {
		pref->total_parm_widgets = 0;
		return(-1);
	    }

	    /* Reset new pointers. */
	    for(i = prev_total; i < pref->total_parm_widgets; i++)
		pref->parm_widget[i] = NULL;
	}

	/* Set new pointer. */
	if((parameter >= 0) && (parameter < pref->total_parm_widgets))
	{
	    pref->parm_widget[parameter] = w;
	}

	return(parameter);
}

/*
 *      Records the panel parent vbox widget for the given catagory
 *	on the preferences structure.
 *
 *      Returns catagory on success or -1 on error.
 */
int PrefPanelRecord(
	vma_pref_struct *pref,
        int catagory,           /* One of VMA_PREF_CAT_. */
        GtkWidget *w            /* A vbox. */
)
{
        if(pref == NULL)
            return(-1);

        if(pref->total_panel_parents < 0)
            pref->total_panel_parents = 0;

        if(catagory >= pref->total_panel_parents)
        {
            /* Need to allocate more pointers. */
            int i;
            int prev_total = pref->total_panel_parents;

            /* Increase total. */
            pref->total_panel_parents = catagory + 1;

            /* Reallocate pointer array. */
            pref->panel_parent = (GtkWidget **)g_realloc(
                pref->panel_parent,
                pref->total_panel_parents * sizeof(GtkWidget *)
            );
            if(pref->panel_parent == NULL)
            {
                pref->total_panel_parents = 0;
                return(-1);
            }

            /* Reset new pointers. */
            for(i = prev_total; i < pref->total_panel_parents; i++)
                pref->panel_parent[i] = NULL;
        }

        /* Set new pointer. */
        if((catagory >= 0) && (catagory < pref->total_panel_parents))
        {
            pref->panel_parent[catagory] = w;
        }

	return(catagory);
}


/*
 *	Returns the buffer pointer of the widget for the parameter
 *	on the preferences widget. Can return NULL on error.
 */
void *PrefParmGetValueP(
	vma_pref_struct *pref,
        int parameter           /* One of VMA_PREF_PARM_*. */
)
{
	GtkWidget *w = PrefParmGetWidget(pref, parameter);
	if(w == NULL)
	    return(NULL);

        if(GTK_IS_TOGGLE_BUTTON((GtkObject *)w))
        {
            return(NULL);
	}
        else if(GTK_IS_ENTRY((GtkObject *)w))
        {
            char *strptr = gtk_entry_get_text(GTK_ENTRY(w));
            return((void *)strptr);
        }
        else if(GTK_IS_COMBO((GtkObject *)w))
        {
            char *strptr;
            GtkCombo *combo = GTK_COMBO(w);
            strptr = gtk_entry_get_text(GTK_ENTRY(combo->entry));
            return((void *)strptr);
        }

	return(NULL);
}

/*
 *	Gets the value from the widget matching the parameter as an int.
 */
int PrefParmGetValueI(
        vma_pref_struct *pref,
        int parameter           /* One of VMA_PREF_PARM_*. */
)
{
        GtkWidget *w = PrefParmGetWidget(pref, parameter);
        if(w == NULL)
            return(0);

	if(GTK_IS_RANGE((GtkObject *)w))
	{
	    GtkAdjustment *adj = gtk_range_get_adjustment(GTK_RANGE(w));
	    return((int)adj->value);
	}
	else if(GTK_IS_SPIN_BUTTON((GtkObject *)w))
        {
            return(
		gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w))
	    );
        }
        else if(GTK_IS_TOGGLE_BUTTON((GtkObject *)w))
        {
            return((GTK_TOGGLE_BUTTON(w)->active) ? 1 : 0); 
	}
	else if(GTK_IS_ENTRY((GtkObject *)w))
	{
	    char *strptr = gtk_entry_get_text(GTK_ENTRY(w));
	    return((strptr == NULL) ? 0 : atoi(strptr));
	}
	else if(GTK_IS_COMBO((GtkObject *)w))
	{
	    char *strptr;
	    GtkCombo *combo = GTK_COMBO(w);
	    strptr = gtk_entry_get_text(GTK_ENTRY(combo->entry));
	    return((strptr == NULL) ? 0 : atoi(strptr));
	}

	return(0);
}

/*
 *      Gets the value from the widget matching the parameter as a float.
 */
float PrefParmGetValueF(
        vma_pref_struct *pref,
        int parameter           /* One of VMA_PREF_PARM_*. */
)
{
        GtkWidget *w = PrefParmGetWidget(pref, parameter);
        if(w == NULL)
            return(0.0);

        if(GTK_IS_RANGE((GtkObject *)w))
        {
            GtkAdjustment *adj = gtk_range_get_adjustment(GTK_RANGE(w));
            return((float)adj->value);
        }
        else if(GTK_IS_SPIN_BUTTON((GtkObject *)w))
        {
            return(
                gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(w))
            );
        }
        else if(GTK_IS_TOGGLE_BUTTON((GtkObject *)w))
        {
	    return((GTK_TOGGLE_BUTTON(w)->active) ? 1.0 : 0.0);
        }
        else if(GTK_IS_ENTRY((GtkObject *)w))
        {
            char *strptr = gtk_entry_get_text(GTK_ENTRY(w));
            return((strptr == NULL) ? 0.0 : atof(strptr));
        }
        else if(GTK_IS_COMBO((GtkObject *)w))
        {
            char *strptr;
            GtkCombo *combo = GTK_COMBO(w);
            strptr = gtk_entry_get_text(GTK_ENTRY(combo->entry));
            return((strptr == NULL) ? 0.0 : atof(strptr));
        }

        return(0.0);
}

/*
 *      Gets the value from the widget matching the parameter as a double.
 */
double PrefParmGetValueD(
        vma_pref_struct *pref,
        int parameter           /* One of VMA_PREF_PARM_*. */
)
{
        GtkWidget *w = PrefParmGetWidget(pref, parameter);
        if(w == NULL)
            return(0.0);

        if(GTK_IS_RANGE((GtkObject *)w))
        {
            GtkAdjustment *adj = gtk_range_get_adjustment(GTK_RANGE(w));
            return((double)adj->value);
        }
        else if(GTK_IS_SPIN_BUTTON((GtkObject *)w))
        {
            return(
        (double)gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(w))
            );
        }
        else if(GTK_IS_TOGGLE_BUTTON((GtkObject *)w))
        {
            return((GTK_TOGGLE_BUTTON(w)->active) ? 1.0 : 0.0);
        }
        else if(GTK_IS_ENTRY((GtkObject *)w))
        {
            char *strptr = gtk_entry_get_text(GTK_ENTRY(w));
            return((strptr == NULL) ? 0.0 : atof(strptr));   
        }
        else if(GTK_IS_COMBO((GtkObject *)w))
        {
            char *strptr;
            GtkCombo *combo = GTK_COMBO(w);
            strptr = gtk_entry_get_text(GTK_ENTRY(combo->entry));
            return((strptr == NULL) ? 0.0 : atof(strptr));
        }

        return(0.0);
}

/*
 *      Gets the value from the widget matching the parameter as a boolean.
 */
gbool PrefParmGetValueB(
	vma_pref_struct *pref,
        int parameter           /* One of VMA_PREF_PARM_*. */
)
{
        GtkWidget *w = PrefParmGetWidget(pref, parameter);
        if(w == NULL)
            return(FALSE);

        if(GTK_IS_RADIO_BUTTON((GtkObject *)w))
        {
            return((GTK_TOGGLE_BUTTON(w)->active) ? TRUE : FALSE);
        }
	else if(GTK_IS_CHECK_BUTTON((GtkObject *)w))
	{
	    return((GTK_TOGGLE_BUTTON(w)->active) ? TRUE : FALSE);
	}
        else if(GTK_IS_TOGGLE_BUTTON((GtkObject *)w))
        {
            return((GTK_TOGGLE_BUTTON(w)->active) ? TRUE : FALSE);
	}
        else if(GTK_IS_ENTRY((GtkObject *)w))
        {
            char *strptr = gtk_entry_get_text(GTK_ENTRY(w));
	    if(strptr == NULL)
	    {
		return(FALSE);
	    }
	    else
	    {
		while(ISBLANK(*strptr))
		    strptr++;
		return(toupper(*strptr) == 'Y');
	    }
        }
        else if(GTK_IS_COMBO((GtkObject *)w))
        {
            char *strptr;
            GtkCombo *combo = GTK_COMBO(w);
            strptr = gtk_entry_get_text(GTK_ENTRY(combo->entry));
            if(strptr == NULL)
            {
                return(FALSE);
            }
            else
            {
                while(ISBLANK(*strptr))
                    strptr++;
                return(toupper(*strptr) == 'Y');
            }
        }

        return(FALSE);
}


/*
 *      Sets the data and destroy function for the given GtkCTreeNode
 *      branch.
 */
void PrefBranchSetData(
        GtkCTree *ctree, GtkCTreeNode *branch,
	vma_pref_item_struct *item_ptr, GtkDestroyNotify destroy
)
{
        if((ctree == NULL) || (branch == NULL))
            return;

        gtk_ctree_node_set_row_data_full(
            ctree, branch,
            (gpointer)item_ptr, destroy
        );   
}

/*
 *      Returns the user set data for the given GtkCTreeNode branch.
 */
vma_pref_item_struct *PrefBranchGetData(
        GtkCTree *ctree, GtkCTreeNode *branch
)
{
        if((ctree == NULL) || (branch == NULL))
            return(NULL);

        return(
            (vma_pref_item_struct *)gtk_ctree_node_get_row_data(
                ctree, branch
            )
        );
}

/*
 *      Marks preferences window as busy.
 */
void PrefSetBusy(vma_pref_struct *pref)
{
        GtkWidget *w;
        GdkCursor *cur;

        if(pref == NULL)
            return;

	w = pref->toplevel;
	if(w == NULL)
	    return;

        if(GTK_WIDGET_NO_WINDOW(w))
            return;

	cur = pref->busy_cur;
	if(cur != NULL)
	{
	    gdk_window_set_cursor(w->window, cur);
	    gdk_flush(); 
	}
}

/*
 *      Marks preferences window as ready (opposite affect of
 *	PrefSetBusy()).
 */
void PrefSetReady(vma_pref_struct *pref)
{
        GtkWidget *w;

        if(pref == NULL)
            return;
  
        w = pref->toplevel;
        if(w == NULL)
            return;

        if(GTK_WIDGET_NO_WINDOW(w))
            return;   
        
        gdk_window_set_cursor(w->window, NULL);
        gdk_flush();
}


/*
 *	Adds a new branch and creates and sets a new item data.
 *
 *	The given vbox will be parented to the preferance window's
 *	panel_toplevel.
 *
 *	Returns the pointer to the newly created branch.
 */
GtkCTreeNode *PrefAddCatagoryBranch(
	vma_pref_struct *pref,
	const char *catagory_name,
	GtkCTreeNode *parent, GtkCTreeNode *sibling,
	GdkPixmap *pixmap_closed, GdkBitmap *mask_closed,
	GdkPixmap *pixmap_opened, GdkBitmap *mask_opened,
	gbool is_leaf,
	GtkWidget *panel_vbox,
	int catagory
)
{
	GtkCTreeNode *new_branch;
	GtkCTree *ctree;
	gchar *text[1];
	vma_pref_item_struct *item_ptr;
	GtkWidget *panel_toplevel;


	if(pref == NULL)
	    return(NULL);

	if(catagory < 0)
	    return(NULL);

	ctree = (GtkCTree *)pref->catagory_ctree;
	if(ctree == NULL)
	    return(NULL);

	/* Add panel_vbox to the preferance window's panel_toplevel. */
	panel_toplevel = pref->panel_toplevel;
	if((panel_toplevel != NULL) && (panel_vbox != NULL))
	{
	    gtk_box_pack_start(
		GTK_BOX(panel_toplevel), panel_vbox, TRUE, TRUE, 0
	    );

	    /* Add this panel vbox to the panel_parent list on pref. */
	    PrefPanelRecord(
		pref, catagory, panel_vbox
	    );
	}

	/* Copy catagory name. */
	text[0] = g_strdup((catagory_name == NULL) ?
	    "Untitled" : catagory_name
	);
	/* Create new branch. */
	new_branch = gtk_ctree_insert_node(
	    ctree, parent, sibling,
	    text,
	    VMA_LIST_PIXMAP_TEXT_SPACING,
	    pixmap_closed, mask_closed,
	    pixmap_opened, mask_opened,
	    is_leaf,			/* Is leaf. */
	    FALSE			/* Expanded. */
	);
	/* Free coppied catagory name. */
	g_free(text[0]);

	/* Allocate item data for branch. */
	item_ptr = (vma_pref_item_struct *)calloc(
	    1, sizeof(vma_pref_item_struct)
	);
	if(item_ptr != NULL)
	{
	    item_ptr->catagory = catagory;
	    item_ptr->pref_ptr = (void *)pref;
	    PrefBranchSetData(
		ctree, new_branch,
		item_ptr, PrefDestroyItem
	    );
	}

	return(new_branch);
}


/*
 *	Creates a banner like heading for the headings of pref panels.
 */
static void PrefCreateHeading(  
	vma_pref_struct *pref,
        GtkWidget *parent,      /* An hbox. */
        gint parent_border,	/* Border when pack into given parent. */
        const gchar *label
)
{
	GtkWidget *w;
	GtkStyle *style_ptr;


	style_ptr = styles_list.pref_banner;
	if(style_ptr == NULL)
	    return;

	w = (GtkWidget *)GUIBannerCreate(
	    label, VMA_GTK_FONT_NAME_PREF_BANNER,
	    style_ptr->fg[GTK_STATE_NORMAL],
	    style_ptr->bg[GTK_STATE_NORMAL],
	    GTK_JUSTIFY_LEFT,
	    TRUE
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, parent_border);
        gtk_widget_show(w);
}

/*
 *	Creates a new select color button.
 */
static void PrefCreateColorButton(
	vma_pref_struct *pref,
	GtkWidget *parent,	/* An hbox. */
	gint parent_border,	/* Border when pack into given parent. */
	const gchar *label,
	gint label_width,
	gint parameter
)
{
	GdkColormap *colormap;
	GtkWidget *w, *btn, *da;
	csd_color_struct *color_ptr;
	GdkColor gdk_color;


	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, parent_border);
        gtk_widget_show(w);
	parent = w;

	w = gtk_label_new(label);
	gtk_widget_set_usize(w, label_width, -1);
        gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	btn = w = gtk_button_new();
/*	gtk_widget_set_usize(w, 25, 25); */
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent = w;

	w = gtk_frame_new(NULL);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_container_border_width(GTK_CONTAINER(w), 1);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_OUT);
	gtk_widget_show(w);
        parent = w;

	/* Create drawing area. */
	da = w = gtk_drawing_area_new();
	gtk_drawing_area_size(GTK_DRAWING_AREA(w), 15, 15);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_realize(w);
	gtk_widget_show(w);

	/* Allocate a new csd_color_struct for the drawing area. */
	color_ptr = (csd_color_struct *)calloc(
	    1, sizeof(csd_color_struct)
	);
	if(color_ptr != NULL)
	{
	    color_ptr->a = 1.0;
	    color_ptr->r = 0.0;
	    color_ptr->g = 0.0;
	    color_ptr->b = 0.0;

	    gdk_color.red = (guint16)(color_ptr->r * 65535.0);
	    gdk_color.green = (guint16)(color_ptr->g * 65535.0);
	    gdk_color.blue = (guint16)(color_ptr->b * 65535.0);

	    gtk_object_set_user_data(GTK_OBJECT(da), (gpointer)color_ptr);
	    gtk_signal_connect(
		GTK_OBJECT(da), "destroy",
		GTK_SIGNAL_FUNC(PrefDrawingAreaDestroyCB),
		(gpointer)color_ptr
	    );
	}

	/* Set background color of GtkDrawingArea widget. */
	colormap = gdk_window_get_colormap(da->window);
	if(colormap != NULL)
	{
	    gdk_colormap_alloc_color(colormap, &gdk_color, TRUE, TRUE);
	    gdk_window_set_background(da->window, &gdk_color);
	    gdk_colormap_free_colors(colormap, &gdk_color, 1);
	}

	/* Record drawing area widget. */
	PrefParmRecord(pref, parameter, da);

	/* Attach the "clicked" signal to the GtkButton widget and pass
	 * the GtkDrawingArea widget as the data pointer (not the
	 * pref_struct).
	 */
        gtk_signal_connect(
            GTK_OBJECT(btn), "clicked",
            GTK_SIGNAL_FUNC(PrefSelectColorCB),
            (gpointer)da
        );

	return;
}

/*
 *      Creates a new select font label and button.
 */
static void PrefCreateFontButton(
	vma_pref_struct *pref,
	GtkWidget *parent,	/* An hbox. */
	int parent_border,	/* Border when pack into given parent. */
	const char *label,
	const char *font_name,
	int label_width,
	int entry_width,
	int parameter
)
{
        GtkWidget *w, *btn, *entry;


        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, parent_border);
        gtk_widget_show(w);
        parent = w;

	if(label != NULL)
	{
	    w = gtk_label_new(label);
            gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_widget_set_usize(w, label_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}

        entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_set_usize(w, entry_width, -1);
	gtk_entry_set_editable(GTK_ENTRY(w), FALSE);
	gtk_entry_set_text(GTK_ENTRY(w), font_name);
        gtk_widget_show(w);


        btn = w = gtk_button_new_with_label("Select...");
	gtk_widget_set_usize(w, 60, -1);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 5);
        gtk_widget_show(w);

        /* Record entry widget. */
        PrefParmRecord(pref, parameter, entry);

        /* Attach the "clicked" signal to the GtkButton widget and pass
         * the GtkEntry widget as the data pointer (not the pref_struct).
         */
        gtk_signal_connect(
            GTK_OBJECT(btn), "clicked",
            GTK_SIGNAL_FUNC(PrefSelectFontCB),
            (gpointer)entry
        );

	return;
}

/*
 *	Creates a new spin prompt.
 */
static void PrefCreateSpinPrompt(
        vma_pref_struct *pref,
        GtkWidget *parent,	/* An hbox. */
        gint parent_border,	/* Border when pack into given parent. */
        const gchar *label,
	GtkAdjustment *adj,	/* Adjustment for spin widget. */
	guint digits,		/* Digits. */
        gint label_width,
        gint spin_width,
        gint parameter,
	const char *tt_name
)
{
        GtkWidget *w, *spin;
	const gchar *msglist[] = VMA_MSGLIST_PREFWIN_TOOLTIPS;

	/* Must be 5 or less digits. */
	if(digits > 5)
	    digits = 5;

        w = gtk_hbox_new(FALSE, 2);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, parent_border);
        gtk_widget_show(w);
        parent = w;

        if(label != NULL)
        {
            w = gtk_label_new(label);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_widget_set_usize(w, label_width, -1);
            gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
            gtk_widget_show(w);
        }

	/* Create spin button widget. */
        spin = w = gtk_spin_button_new(
            adj,
            1.0,	/* Climb rate (0.0 to 1.0). */
            digits	/* Digits. */
        );
	if(w != NULL)
	{
            gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
            gtk_widget_set_usize(w, spin_width, -1);
	    if(tt_name != NULL)
		GUISetWidgetTip(
		    w, MsgListMatchCaseMessage(msglist, tt_name)
		);
            gtk_widget_show(w);

            /* Record spin widget. */
            PrefParmRecord(pref, parameter, spin);
	}

	return;
}

/*
 *      Creates a new scale prompt.
 */
static void PrefCreateScalePrompt(
        vma_pref_struct *pref,
        GtkWidget *parent,	/* An hbox. */
        gint parent_border,	/* Border when pack into given parent. */
        const gchar *label,
        GtkAdjustment *adj,	/* Adjustment for spin widget. */
        guint digits,		/* Digits. */
        gint label_width,
        gint scale_width,
        gint parameter,
	const gchar *tt_name
)
{
	const char *msglist[] = VMA_MSGLIST_PREFWIN_TOOLTIPS;
	GtkWidget *w, *scale;


        /* Must be 5 or less digits. */
        if(digits > 5)
            digits = 5;

        w = gtk_hbox_new(FALSE, 2);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, parent_border);
        gtk_widget_show(w);
        parent = w;

        if(label != NULL)
        {
            w = gtk_label_new(label);
            gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
            gtk_widget_set_usize(w, label_width, -1);
            gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
            gtk_widget_show(w);
        }

        /* Create scale widget. */
	scale = w = gtk_hscale_new(adj);
        if(w != NULL)
        {
            gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
            gtk_widget_set_usize(w, scale_width, -1);
            if(tt_name != NULL)
                GUISetWidgetTip(
                    w, MsgListMatchCaseMessage(msglist, tt_name)
                );
            gtk_widget_show(w);

	    gtk_scale_set_draw_value(GTK_SCALE(w), TRUE);
	    gtk_scale_set_value_pos(GTK_SCALE(w), GTK_POS_RIGHT);
	    gtk_scale_set_digits(GTK_SCALE(w), digits);

            /* Record spin widget. */
            PrefParmRecord(pref, parameter, scale);
        }
        
        return;
}

/*
 *	Create main program panel, this is just a place holder displaying
 *	the program preferance and other front page preferences stuff.
 */
static void PrefCreatePanelMain(vma_pref_struct *pref, GtkWidget *parent)
{
        GtkWidget *w, *parent2, *parent3, *parent4;
        gint	border_major = 5,
		border_minor = 2;
	const gchar *font_name;
	const gchar *str_parm, *str_val;
	gint state;
	GdkColor *c;
	GtkRcStyle *rcstyle, *rcstyle_bold;


	/* Set border of main vbox parent. */
	gtk_container_border_width(GTK_CONTAINER(parent), border_major);

	/* Create style for base event box. */
        rcstyle = gtk_rc_style_new();
        state = GTK_STATE_NORMAL;
        rcstyle->color_flags[state] = GTK_RC_BG /* | GTK_RC_BASE */;
        c = &rcstyle->bg[state];
        c->red = 1.0 * (guint16)-1;
        c->green = 1.0 * (guint16)-1;
        c->blue = 1.0 * (guint16)-1;
/*
        c = &rcstyle->base[state];
        c->red = 1.0 * (guint16)-1;
        c->green = 1.0 * (guint16)-1;
        c->blue = 1.0 * (guint16)-1;
 */
	/* Event box so we can have a styled base. */
	w = gtk_event_box_new();
	gtk_widget_add_events(w, GDK_EXPOSURE_MASK);
        gtk_widget_modify_style(w, rcstyle);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
	parent2 = w;

        GUIRCStyleDeallocUnref(rcstyle);

	/* Create vbox for holding all further child widgets. */
	w = gtk_vbox_new(FALSE, border_major);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_widget_show(w);
	parent2 = w;


	/* Hbox to contain heading banner. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;
	/* Alignment. */
        w = gtk_alignment_new(1.0, 0.5, 0.0, 0.0);
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
	parent3 = w;
	/* Heading banner. */
        w = (GtkWidget *)GUICreateMenuItemIcon(
            (u_int8_t **)vertex_banner_400x50_xpm
        );
        gtk_widget_set_usize(w, 400, 50);
	gtk_container_add(GTK_CONTAINER(parent3), w);
        gtk_widget_show(w);


	/* Preferences banner. */
        rcstyle = gtk_rc_style_new();
        font_name = VMA_GTK_FONT_NAME_TEXT_HEADING1;
        if(font_name != NULL)
        {
            g_free(rcstyle->font_name);
            rcstyle->font_name = g_strdup(font_name);
        }

        w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

        w = gtk_label_new("Preferences");
        gtk_widget_modify_style(w, rcstyle);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        GUIRCStyleDeallocUnref(rcstyle);


	/* Create styles for labels. */
        rcstyle = gtk_rc_style_new();
        font_name =
	    "-adobe-helvetica-medium-r-normal-*-14-*-*-*-p-*-iso8859-1";
        if(font_name != NULL)
        {
            g_free(rcstyle->font_name);
            rcstyle->font_name = g_strdup(font_name);
        }

        rcstyle_bold = gtk_rc_style_new();
        font_name = "-adobe-helvetica-bold-r-normal-*-14-*-*-*-p-*-iso8859-1";
        if(font_name != NULL)
        {
            g_free(rcstyle_bold->font_name);
            rcstyle_bold->font_name = g_strdup(font_name);
        }


	/* Hbox to hold version labels. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;
	/* Alignment. */
	w = gtk_alignment_new(0.0, 0.5, 0.0, 0.0);
	gtk_widget_set_usize(w, 100, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
	parent4 = w;
	/* Label. */
	w = gtk_label_new("Version:");
        gtk_widget_modify_style(w, rcstyle_bold);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);

        /* Label. */
        w = gtk_label_new(PROG_VERSION);
        gtk_widget_modify_style(w, rcstyle);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);





	/* Deallocate styles. */
	GUIRCStyleDeallocUnref(rcstyle);
        GUIRCStyleDeallocUnref(rcstyle_bold);

}

/*
 *      Create appearance panel.
 */
static void PrefCreatePanelApperance(vma_pref_struct *pref, GtkWidget *parent)
{
	GtkWidget *w, *parent2, *parent3;
        GSList *gslist;
	gint	border_major = 5,
		border_minor = 2;


	/* Heading. */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	PrefCreateHeading(
	    pref, parent2, border_major,
	    "Appearance"
	);


	/* Window placement frame. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_frame_new("Window Placement");
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Record window positions and sizes check button. */
        w = gtk_check_button_new_with_label("Record Last Position & Size");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_RECORD_WIN_POS_AND_SIZES, w);


	/* Show tool bar as frame. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_frame_new("Show Toolbar As");
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_hbox_new(FALSE, border_major);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;
	/* Left column vbox. */
        w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* `Pictures and text' radio button. */
	gslist = NULL;
        w = gtk_radio_button_new_with_label(gslist, "Pictures & Text");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_set_usize(w, 200, -1);
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_TOOLBAR_PT, w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

        /* `Pictures only' radio button. */
        w = gtk_radio_button_new_with_label(gslist, "Pictures Only");
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_set_usize(w, 200, -1);
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_TOOLBAR_P, w);
        gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

        /* `Text only' radio button. */
        w = gtk_radio_button_new_with_label(gslist, "Text Only");
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_set_usize(w, 200, -1);
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_TOOLBAR_T, w);


        /* Right column vbox. */
        w = gtk_vbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent3 = w;

        /* Tool tips check button. */
        w = gtk_check_button_new_with_label("Tooltips");
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_TOOLTIPS, w);

        /* Show tip of day check button (on right side). */
        w = gtk_check_button_new_with_label("Show Tip Of Day");
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_SHOW_TIP_OF_DAY, w);

        return;
}


/*
 *      Create fonts panel.
 */
static void PrefCreatePanelFonts(vma_pref_struct *pref, GtkWidget *parent)
{
        GtkWidget *w, *parent2, *parent3, *parent4;
        gint	border_major = 5,
		border_minor = 2;
	gint	label_width = 100,
		entry_width = 175;
        const gchar	*label,
			*font_name;
        gint pref_parm;


        /* Heading. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        PrefCreateHeading(
            pref, parent2, border_major,
            "Fonts"
        );


        /* Using a scrolled window to hold all the widgets. */
        w = gtk_scrolled_window_new(NULL, NULL);
        gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, border_major);
        gtk_scrolled_window_set_policy(
            GTK_SCROLLED_WINDOW(w),
            GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC
        );
        gtk_widget_show(w);
        parent2 = w;

        /* Create hbox as the first child in the scrolled window. */
        w = gtk_vbox_new(FALSE, 0);
        gtk_scrolled_window_add_with_viewport(
            GTK_SCROLLED_WINDOW(parent2), w
        );
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;


        /* Editor fonts. */
        w = gtk_frame_new("Editor");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent3 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent3), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent3 = w;


        /* Begin creating view color widgets. */
#define DO_CREATE_FONT_ENTRY	\
{ \
 w = gtk_hbox_new(FALSE, 0); \
 gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0); \
 gtk_widget_show(w); \
 parent4 = w; \
 PrefCreateFontButton( \
  pref, parent4, border_major, label, font_name, \
  label_width, entry_width, \
  pref_parm \
 ); \
}
        pref_parm = VMA_PREF_PARM_FONT_EDITABLE;
        label = "Editable:";
	font_name = "";
        DO_CREATE_FONT_ENTRY

        pref_parm = VMA_PREF_PARM_FONT_TERMINAL;
        label = "Terminal:";
	font_name = "";
        DO_CREATE_FONT_ENTRY




#undef DO_CREATE_FONT_ENTRY


	return;
}



/*
 *	Create colors panel.
 */
static void PrefCreatePanelColors(vma_pref_struct *pref, GtkWidget *parent)
{
        GtkWidget *w, *parent2, *parent3, *parent4;
        gint border_major = 5;
	gint border_minor = 2;
	const char *label;
	int pref_parm;


        /* Heading. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        PrefCreateHeading(
            pref, parent2, border_major,
            "Colors"
        );


	/* Using a scrolled window to hold all the widgets. */
        w = gtk_scrolled_window_new(NULL, NULL);
        gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, border_major);
        gtk_scrolled_window_set_policy(
            GTK_SCROLLED_WINDOW(w),
            GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC
        );
        gtk_widget_show(w);
        parent2 = w;

	/* Create hbox as the first child in the scrolled window. */
        w = gtk_vbox_new(FALSE, 0);
        gtk_scrolled_window_add_with_viewport(
            GTK_SCROLLED_WINDOW(parent2), w
        );
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;


	/* View colors. */
        w = gtk_frame_new("View");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent3 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent3 = w;

        /* Begin creating view color widgets. */
#define DO_CREATE_COLOR_BUTTON	\
{ \
 w = gtk_hbox_new(FALSE, 0); \
 gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0); \
 gtk_widget_show(w); \
 parent4 = w; \
 PrefCreateColorButton( \
  pref, parent4, border_major, label, 200, \
  pref_parm \
 ); \
}
	pref_parm = VMA_PREF_PARM_COLOR_VIEW_BG;
	label = "Background:";
	DO_CREATE_COLOR_BUTTON

	pref_parm = VMA_PREF_PARM_COLOR_VIEW_POINT;
	label = "Point:";
	DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_LINE;
        label = "Line:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_LINE_STRIP;
        label = "Line Strip:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_LINE_LOOP;
        label = "Line Loop:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_TRIANGLE;
        label = "Triangle:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_TRIANGLE_STRIP;
        label = "Triangle Strip:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_TRIANGLE_FAN;
        label = "Triangle Fan:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_QUAD;
        label = "Quad:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_QUAD_STRIP;
        label = "Quad Strip:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_POLYGON;
        label = "Polygon:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_TEXTURE_OUTLINE;
        label = "Texture Outline:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_TEXTURE_SELECTED;
        label = "Texture Selected:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_HEIGHTFIELD;
        label = "Heightfield:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_GRID;
        label = "Grid:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_SELECTED;
        label = "Selected:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_SELECTED_VERTEX;
        label = "Selected Vertex:";
        DO_CREATE_COLOR_BUTTON

        pref_parm = VMA_PREF_PARM_COLOR_VIEW_CURSORY;
        label = "Cursory:";
        DO_CREATE_COLOR_BUTTON

	pref_parm = VMA_PREF_PARM_COLOR_VIEW_NORMAL_VECTOR;
        label = "Normal Vector:";
        DO_CREATE_COLOR_BUTTON



#undef DO_CREATE_COLOR_BUTTON


	return;
}

/*
 *      Create precision panel.
 */
static void PrefCreatePanelPrecision(vma_pref_struct *pref, GtkWidget *parent)
{
        GtkWidget *w, *parent2, *parent3;
        gint border_major = 5;
        gint border_minor = 2;
	GtkAdjustment *adj;


        /* Heading. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        PrefCreateHeading(
            pref, parent2, border_major,
            "Precision"
        );


        /* Using a scrolled window to hold all the widgets. */
        w = gtk_scrolled_window_new(NULL, NULL);
        gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, border_major);
        gtk_scrolled_window_set_policy(
            GTK_SCROLLED_WINDOW(w),
            GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC
        );
        gtk_widget_show(w);
        parent2 = w;

        /* Create hbox as the first child in the scrolled window. */  
        w = gtk_vbox_new(FALSE, 0);
        gtk_scrolled_window_add_with_viewport(
            GTK_SCROLLED_WINDOW(parent2), w
        );
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Data precision. */
        w = gtk_frame_new("Data");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent3 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent3 = w;

        /* Begin creating data precision widgets. */
	/* Coordinate decimals. */
        adj = (GtkAdjustment *)gtk_adjustment_new(
            0.0,		/* Initial. */
	    0.0,		/* Minimum. */
	    64.0,		/* Maximum. */
	    1.0,		/* Step inc. */
	    5.0,		/* Page inc. */
	    5.0			/* Page size. */
        );
	PrefCreateSpinPrompt(
	    pref, parent3, 0,
	    "Coordinate Decimals:", adj,
	    0,		/* Digits. */
	    200,	/* Label width. */
	    50,		/* Spin width. */
	    VMA_PREF_PARM_DATA_DECIMALS_POSITION,
	    VMA_MSGNAME_PREFWIN_DATA_DECIMALS_POSITION
	);

        /* Angle decimals. */
        adj = (GtkAdjustment *)gtk_adjustment_new(
            0.0,                /* Initial. */
            0.0,                /* Minimum. */
            64.0,               /* Maximum. */
            1.0,                /* Step inc. */
            5.0,                /* Page inc. */
            5.0                 /* Page size. */
        );
        PrefCreateSpinPrompt(
            pref, parent3, 0,
            "Angle Decimals:", adj,
            0,		/* Digits. */
            200,	/* Label width. */
            50,		/* Spin width. */
            VMA_PREF_PARM_DATA_DECIMALS_ANGLE,
	    VMA_MSGNAME_PREFWIN_DATA_DECIMALS_ANGLE
        );


        /* View precision. */
        w = gtk_frame_new("Views");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent3 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent3), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent3 = w;

        /* Begin creating view precision widgets. */
        /* Coordinate decimals. */
        adj = (GtkAdjustment *)gtk_adjustment_new(
            0.0,                /* Initial. */
            0.0,                /* Minimum. */
            64.0,               /* Maximum. */
            1.0,                /* Step inc. */
            5.0,                /* Page inc. */   
            5.0                 /* Page size. */
        );
        PrefCreateSpinPrompt(
            pref, parent3, 0,
            "Coordinate Decimals:", adj,
            0,          /* Digits. */
            200,        /* Label width. */
            50,        /* Spin width. */
            VMA_PREF_PARM_VIEW_DECIMALS_POSITION,
	    VMA_MSGNAME_PREFWIN_VIEW_DECIMALS_POSITION
        );

        /* Angle decimals. */
        adj = (GtkAdjustment *)gtk_adjustment_new(
            0.0,                /* Initial. */
            0.0,                /* Minimum. */
            64.0,               /* Maximum. */
            1.0,                /* Step inc. */
            5.0,                /* Page inc. */
            5.0                 /* Page size. */
        );
        PrefCreateSpinPrompt( 
            pref, parent3, 0,
            "Angle Decimals:", adj,
            0,		/* Digits. */
            200,	/* Label width. */
            50,		/* Spin width. */
            VMA_PREF_PARM_VIEW_DECIMALS_ANGLE,
	    VMA_MSGNAME_PREFWIN_VIEW_DECIMALS_ANGLE
        );

}

/*
 *	Create rendering panel.
 */
static void PrefCreatePanelRendering(vma_pref_struct *pref, GtkWidget *parent)
{
	const gchar *msglist[] = VMA_MSGLIST_PREFWIN_TOOLTIPS;
	const gchar *cstrptr;
        GtkWidget *w, *parent2, *parent3, *parent4;
	GtkAdjustment *adj;
        GSList *gslist;
        gint border_major = 5; 
        gint border_minor = 2;


        /* Heading. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        PrefCreateHeading(
            pref, parent2, border_major,
            "Rendering"
        );

        /* Intro. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        cstrptr = (const char *)MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_RENDERING_INTRO
        );
        if(cstrptr == NULL)
            cstrptr = "";
        w = gtk_label_new(cstrptr);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_major);
        gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
        gtk_widget_show(w);


        /* Using a scrolled window to hold all the widgets. */
        w = gtk_scrolled_window_new(NULL, NULL);
        gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, border_major);
        gtk_scrolled_window_set_policy(
            GTK_SCROLLED_WINDOW(w),
            GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC
        );
        gtk_widget_show(w);
        parent2 = w;

        /* Create hbox as the first child in the scrolled window. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_scrolled_window_add_with_viewport(
            GTK_SCROLLED_WINDOW(parent2), w
        );
        gtk_widget_show(w);
	parent2 = w;

        /* Vbox inside hbox for alignment and as parent for all child widgets. */
        w = gtk_vbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, border_major);
        gtk_widget_show(w);
        parent2 = w;


        /* Views. */
        w = gtk_frame_new("Views");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent3 = w;

        w = gtk_vbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent3 = w;

	/* Render state check button. */
        w = gtk_check_button_new_with_label("Enable Rendering");
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, border_minor);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_VIEW_RENDER_ACTUAL
            )
        );
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_VIEW_RENDER, w);

        /* Hbox for cull faces check button. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, border_minor);
        gtk_widget_show(w);
        parent4 = w;
        /* Cull faces check button. */
        w = gtk_check_button_new_with_label("Cull Faces");
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrefCullFaceCheckCB), (gpointer)pref
	);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_VIEW_CULL_FACES
            )
        );
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_VIEW_CULL_FACES, w);

        /* Cull clockwise radio button. */
        gslist = NULL;
        w = gtk_radio_button_new_with_label(gslist, "Clockwise");
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_VIEW_CULL_DIRECTION_CW
            )
        );
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_CULL_DIRECTION_CW, w);
        gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

        /* `Cull counter-clockwise radio button. */
        w = gtk_radio_button_new_with_label(gslist, "Counter-Clockwise");
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_VIEW_CULL_DIRECTION_CCW
            )
        );
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_CULL_DIRECTION_CCW, w);
	gslist = NULL;


        /* Perform rotations and translations check button. */
        w = gtk_check_button_new_with_label("Perform Rotations and Translations");
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, border_minor);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_VIEW_TRANSLATIONS
            )
        );
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_VIEW_TRANSLATIONS_STATE, w);

        /* Enable alpha channel check button. */
        w = gtk_check_button_new_with_label("Enable Alpha Channel");
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, border_minor);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_VIEW_ENABLE_ALPHA_CHANNEL
            )
        );
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_VIEW_ENABLE_ALPHA_CHANNEL, w);


        /* Camera. */
        w = gtk_frame_new("Camera");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent3 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent3), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent3 = w;

        /* Camera clip near spin button. */
        adj = (GtkAdjustment *)gtk_adjustment_new(
            0.0,                /* Initial. */
            0.0,                /* Minimum. */
            10000000.0,         /* Maximum. */
            0.001,              /* Step inc. */
            0.01,               /* Page inc. */
            0.01                /* Page size. */
        );
        PrefCreateSpinPrompt(
            pref, parent3, 0,
            "Clip Near:", adj,
            5,          /* Digits. */
            120,        /* Label width. */
            100,        /* Spin width. */
            VMA_PREF_PARM_VIEW_CAM_CLIP_NEAR,
	    VMA_MSGNAME_PREFWIN_VIEW_CAM_CLIP_NEAR
        );

        /* Camera clip far spin button. */
        adj = (GtkAdjustment *)gtk_adjustment_new(
            0.0,                /* Initial. */
            0.0,                /* Minimum. */
            10000000.0,         /* Maximum. */
            100,		/* Step inc. */
            1000,		/* Page inc. */
            1000		/* Page size. */
        );
        PrefCreateSpinPrompt(
            pref, parent3, 0,
            "Clip Far:", adj,
            0,          /* Digits. */
            120,        /* Label width. */
            100,        /* Spin width. */
            VMA_PREF_PARM_VIEW_CAM_CLIP_FAR,
	    VMA_MSGNAME_PREFWIN_VIEW_CAM_CLIP_FAR
        );

        /* Camera clip field of view (FOV) spin button. */
        adj = (GtkAdjustment *)gtk_adjustment_new(
            0.0,                /* Initial. */
            0.0,                /* Minimum. */
            360.0,		/* Maximum. */
            1.0,		/* Step inc. */
            10.0,		/* Page inc. */
            10.0		/* Page size. */
        );
        PrefCreateSpinPrompt(
            pref, parent3, 0,
            "Field Of View:", adj,
            2,          /* Digits. */
            120,        /* Label width. */
            100,        /* Spin width. */ 
            VMA_PREF_PARM_VIEW_CAM_FOV,
	    VMA_MSGNAME_PREFWIN_VIEW_CAM_FOV
        );
}

/*
 *	Create devices panel.
 */
static void PrefCreatePanelDevices(vma_pref_struct *pref, GtkWidget *parent)
{
	const gchar *msglist[] = VMA_MSGLIST_PREFWIN_TOOLTIPS;
	GtkWidget *w, *parent2;
	gint border_major = 5;
	gint border_minor = 2;


	/* Heading. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        PrefCreateHeading(
            pref, parent2, border_major,
            "Devices"
        );

        /* Pointer frame. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_frame_new("Pointer");
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Pointer emulated as 2 only buttons. */
        w = gtk_check_button_new_with_label("Emulate 2 Buttons");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_DEVICES_POINTER_EMULATE_2BUTTON
            )
        );
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_DEVICES_POINTER_2BUTTON, w);


	return;
}

/*
 *	Create editor panel.
 */
static void PrefCreatePanelEditor(vma_pref_struct *pref, GtkWidget *parent)
{
/*	const gchar *msglist[] = VMA_MSGLIST_PREFWIN_TOOLTIPS; */
        GtkWidget *w, *parent2;
	GtkAdjustment *adj;
        gint border_major = 5;
        gint border_minor = 2;


        /* Heading. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        PrefCreateHeading(
            pref, parent2, border_major,
            "Editor"
        );


        /* Undo frame. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_frame_new("Undo");
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Undo items max spin. */
        adj = (GtkAdjustment *)gtk_adjustment_new(
            0.0,                /* Initial. */
            0.0,                /* Minimum. */
            10000000.0,         /* Maximum. */
            1.0,                /* Step inc. */
            5.0,                /* Page inc. */
            5.0                 /* Page size. */
        );
        PrefCreateSpinPrompt(
            pref, parent2, 0,
            "Maximum Items:", adj,
            0,          /* Digits. */
            -1,         /* Label width. */
            80,         /* Spin width. */
            VMA_PREF_PARM_EDITOR_UNDO_MAX,
	    VMA_MSGNAME_PREFWIN_EDITOR_UNDO_MAX
        );
}

/*
 *      Create locations panel.
 */
static void PrefCreatePanelLocations(vma_pref_struct *pref, GtkWidget *parent)
{
	const gchar *msglist[] = VMA_MSGLIST_PREFWIN_TOOLTIPS;
        GtkWidget *w, *parent2, *parent3;
	gpointer entry_rtn, browse_rtn;
        gint border_major = 5;
	gint border_minor = 2;


        /* Heading. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        PrefCreateHeading(
            pref, parent2, border_major,
            "Locations"
        );


        /* Using a scrolled window to hold all the widgets. */
        w = gtk_scrolled_window_new(NULL, NULL);
        gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, border_major);
        gtk_scrolled_window_set_policy(
            GTK_SCROLLED_WINDOW(w),
            GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC
        );
        gtk_widget_show(w);
        parent2 = w;

        /* Create vbox as the first child in the scrolled window. */
        w = gtk_vbox_new(FALSE, 0);
        gtk_scrolled_window_add_with_viewport(
            GTK_SCROLLED_WINDOW(parent2), w
        );
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;


	/* Directories frame. */
        w = gtk_frame_new("Directories");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent3 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent3), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent3 = w;

        /* Local data directory. */
        w = GUIPromptBarWithBrowse(
            NULL, "Data Local:",
            NULL, &entry_rtn, &browse_rtn,
            NULL, NULL
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_LOCATION_DATA_LOCAL
            )
        );
        gtk_widget_show(w);

        w = (GtkWidget *)browse_rtn;
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefBrowsePathCB),
            (gpointer)entry_rtn         /* Pass data as entry widget. */  
        );
        
        w = (GtkWidget *)entry_rtn; 
        PrefParmRecord(pref, VMA_PREF_PARM_DIRECTORY_LOCAL, w);


        /* Global data directory. */
        w = GUIPromptBarWithBrowse(
            NULL, "Data Global:",
            NULL, &entry_rtn, &browse_rtn,
            NULL, NULL
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_LOCATION_DATA_GLOBAL
            )
        );
        gtk_widget_show(w);

        w = (GtkWidget *)browse_rtn;
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefBrowsePathCB),
            (gpointer)entry_rtn         /* Pass data as entry widget. */
        );

        w = (GtkWidget *)entry_rtn;
        PrefParmRecord(pref, VMA_PREF_PARM_DIRECTORY_GLOBAL, w);


        /* Local preset models directory. */
        w = GUIPromptBarWithBrowse(
            NULL, "Preset Models Local:",
            NULL, &entry_rtn, &browse_rtn,
            NULL, NULL
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
            MsgListMatchCaseMessage(
                msglist,
		VMA_MSGNAME_PREFWIN_LOCATION_PRESET_MODELS_LOCAL
            )
        );
        gtk_widget_show(w);

        w = (GtkWidget *)browse_rtn;
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefBrowsePathCB),
            (gpointer)entry_rtn         /* Pass data as entry widget. */
        );

        w = (GtkWidget *)entry_rtn;
        PrefParmRecord(pref, VMA_PREF_PARM_DIRECTORY_PRESET_MODELS_LOCAL, w);

        /* Global preset models directory. */
        w = GUIPromptBarWithBrowse(
            NULL, "Preset Models Global:",
            NULL, &entry_rtn, &browse_rtn,
            NULL, NULL
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
            MsgListMatchCaseMessage(
                msglist,
                VMA_MSGNAME_PREFWIN_LOCATION_PRESET_MODELS_GLOBAL
            )
        );
        gtk_widget_show(w);

        w = (GtkWidget *)browse_rtn;
        gtk_signal_connect( 
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefBrowsePathCB),
            (gpointer)entry_rtn         /* Pass data as entry widget. */
        );

        w = (GtkWidget *)entry_rtn;
        PrefParmRecord(pref, VMA_PREF_PARM_DIRECTORY_PRESET_MODELS_GLOBAL, w);


        /* Global preset primitives directory. */
        w = GUIPromptBarWithBrowse(
            NULL, "Preset Primitives Local:",
            NULL, &entry_rtn, &browse_rtn,
            NULL, NULL
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
            MsgListMatchCaseMessage(
                msglist,
                VMA_MSGNAME_PREFWIN_LOCATION_PRESET_PRIMITIVES_LOCAL
            )
        );
        gtk_widget_show(w);

        w = (GtkWidget *)browse_rtn;
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefBrowsePathCB),
            (gpointer)entry_rtn         /* Pass data as entry widget. */
        );

        w = (GtkWidget *)entry_rtn;
        PrefParmRecord(pref, VMA_PREF_PARM_DIRECTORY_PRESET_PRIMITIVES_LOCAL, w);

        /* Global preset primitives directory. */
        w = GUIPromptBarWithBrowse( 
            NULL, "Preset Primitives Global:",
            NULL, &entry_rtn, &browse_rtn,
            NULL, NULL
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
            MsgListMatchCaseMessage(
                msglist,
                VMA_MSGNAME_PREFWIN_LOCATION_PRESET_PRIMITIVES_GLOBAL
            )
        );
        gtk_widget_show(w);

        w = (GtkWidget *)browse_rtn;
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefBrowsePathCB),
            (gpointer)entry_rtn         /* Pass data as entry widget. */
        );

        w = (GtkWidget *)entry_rtn;
        PrefParmRecord(pref, VMA_PREF_PARM_DIRECTORY_PRESET_PRIMITIVES_GLOBAL, w);


        /* Tmp directory. */
        w = GUIPromptBarWithBrowse(
            NULL, "Tempory:",
            NULL, &entry_rtn, &browse_rtn,
            NULL, NULL
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_LOCATION_TMP_DIR
            )
        );
        gtk_widget_show(w);

        w = (GtkWidget *)browse_rtn;
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefBrowsePathCB),
            (gpointer)entry_rtn         /* Pass data as entry widget. */
        );

        w = (GtkWidget *)entry_rtn;
        PrefParmRecord(pref, VMA_PREF_PARM_DIRECTORY_TMP, w);


        /* Files frame. */
        w = gtk_frame_new("Files");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent3 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent3), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent3 = w;


        /* Plug-ins configuration file. */
        w = GUIPromptBarWithBrowse(
            NULL, "ScratchPad:",
            NULL, &entry_rtn, &browse_rtn,
            NULL, NULL
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_LOCATION_SCRATCHPAD_FILE
            )
        );
        gtk_widget_show(w);

        w = (GtkWidget *)browse_rtn;
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefBrowsePathCB),
            (gpointer)entry_rtn         /* Pass data as entry widget. */
        );

        w = (GtkWidget *)entry_rtn;
        PrefParmRecord(pref, VMA_PREF_PARM_FILE_SCRATCHPAD, w);


        /* Plug-ins configuration file. */
        w = GUIPromptBarWithBrowse(
            NULL, "Plug-Ins:",
            NULL, &entry_rtn, &browse_rtn,
            NULL, NULL
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_LOCATION_PLUGINS_FILE
            )
        );
        gtk_widget_show(w);

        w = (GtkWidget *)browse_rtn;
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefBrowsePathCB),
            (gpointer)entry_rtn         /* Pass data as entry widget. */
        );

        w = (GtkWidget *)entry_rtn;
        PrefParmRecord(pref, VMA_PREF_PARM_FILE_PLUGINS, w);



        return;
}

/*
 *	Create V3D format options panel.
 */
static void PrefCreatePanelV3DFormat(vma_pref_struct *pref, GtkWidget *parent)
{
	const char *msglist[] = VMA_MSGLIST_PREFWIN_TOOLTIPS;
	const char *cstrptr;
        GtkWidget *w, *parent2;
        GtkAdjustment *adj;
        gint border_major = 5;
        gint border_minor = 2;


        /* Heading. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        PrefCreateHeading(
            pref, parent2, border_major,   
            "V3D Format Options"
        );

        /* Intro. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        cstrptr = (const char *)MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_V3D_INTRO
        );
        if(cstrptr == NULL)
            cstrptr = "";
        w = gtk_label_new(cstrptr);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_major);
        gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
        gtk_widget_show(w);


        /* Query frame. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_frame_new("Queries");
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Prompt on save check button. */
        w = gtk_check_button_new_with_label("Prompt On Save");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_V3D_PROMPT_ON_SAVE
            )
        );
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_V3DFMT_PROMPT_ON_SAVE, w);


        /* Optimization frame. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_frame_new("Optimization");
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Optimization level scale. */
        adj = (GtkAdjustment *)gtk_adjustment_new(
            0.0,					/* Initial. */
            0.0,					/* Minimum. */
            V3D_SAVE_FILE_OPTIMIZATION_MAX + 1.0,	/* Maximum. */
            1.0,		/* Step inc. */
            1.0,		/* Page inc. */
            1.0			/* Page size. */
        );
        PrefCreateScalePrompt(
            pref, parent2, 0,
            "Level:", adj,
            0,		/* Digits. */
            -1,		/* Label width. */
            150,	/* Scale width. */
	    VMA_PREF_PARM_V3DFMT_OPTIMIZATION,
	    VMA_MSGNAME_PREFWIN_V3D_OPTIMIZATION
        );

        /* Strip extraneous data check button. */
        w = gtk_check_button_new_with_label("Strip Extraneous Data");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_V3D_STRIP_EXTRAS
            )
        );
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_V3DFMT_STRIP_EXTRAS, w);



	return;
}

/*
 *      Create backup panel.
 */
static void PrefCreatePanelBackup(vma_pref_struct *pref, GtkWidget *parent)
{
        const char *msglist[] = VMA_MSGLIST_PREFWIN_TOOLTIPS;
	const char *cstrptr;
        GtkWidget *w, *parent2, *parent3;
        GtkAdjustment *adj;
	GSList *gslist;
        gint border_major = 5;
        gint border_minor = 2;

        /* Heading. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        PrefCreateHeading(
            pref, parent2, border_major,
            "Backup"
        );

	/* Intro. */
	w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

	cstrptr = (const char *)MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_BACKUP_INTRO
        );
	if(cstrptr == NULL)
	    cstrptr = "";
	w = gtk_label_new(cstrptr);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_major);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_widget_show(w);


        /* Backup when frame. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_frame_new("Backup When");
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Opening files check button. */
        w = gtk_check_button_new_with_label("Opening File");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_BACKUP_ON_OPEN
            )
        );
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_BACKUP_ON_OPEN, w);

        /* Saving files check button. */
        w = gtk_check_button_new_with_label("Saving File");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_BACKUP_ON_SAVE
            )
        );
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_BACKUP_ON_SAVE, w);


        /* Periodic (automatic) backup frame. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major); 
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_frame_new("Automatic Backup");
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Hbox for the enable periodic (automatic) backups check button. */
        w = gtk_hbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;
        /* Enable periodic (automatic) backups check button. */
        w = gtk_check_button_new_with_label("Enable");
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "toggled",
            GTK_SIGNAL_FUNC(PrefBackupEnablePeriodicCheckCB),
	    (gpointer)pref
        );
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_BACKUP_PERIODIC
            )
        );
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_BACKUP_PERIODIC, w);

        /* Periodic (automatic) backup interval spin button (units
	 * are in minutes).
	 */
        adj = (GtkAdjustment *)gtk_adjustment_new(
            1.0,                /* Initial. */
            1.0,                /* Minimum. */
            10000000.0,         /* Maximum. */
            1.0,		/* Step inc. */
            5.0,		/* Page inc. */
            5.0			/* Page size. */
        );
        PrefCreateSpinPrompt(
            pref, parent3, 0,
            "Interval (Minutes):", adj,
            0,		/* Digits. */
            -1,		/* Label width. */
            60,		/* Spin width. */
            VMA_PREF_PARM_BACKUP_PERIODIC_INT,
            NULL	/* Tooltip message name. */
        );


        /* Limits frame. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major); 
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_frame_new("Limits");
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Maximum backups per file. */
        adj = (GtkAdjustment *)gtk_adjustment_new(
            5.0,		/* Initial. */
            1.0,		/* Minimum. */
            10000000.0,		/* Maximum. */
            1.0,		/* Step inc. */
            5.0,		/* Page inc. */
            5.0			/* Page size. */
        );
        PrefCreateSpinPrompt(
            pref, parent2, 0,
            "Maximum Backups Per File:", adj,
            0,          /* Digits. */
            -1,		/* Label width. */
            60,		/* Spin width. */
            VMA_PREF_PARM_BACKUP_MAX,
	    VMA_MSGNAME_PREFWIN_BACKUP_MAX
        );


	/* Hbox to hold new backup at index position radio buttons. */
        w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

	w = gtk_label_new("New backups order:");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

        /* New backup at highest index radio. */
        gslist = NULL;
        w = gtk_radio_button_new_with_label(gslist, "Highest Index");
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_BACKUP_NEW_HIGHEST_INDEX, w);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage( 
                msglist, VMA_MSGNAME_PREFWIN_BACKUP_NEW_HIGHEST_INDEX
            )
        );
        gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

        /* New backup at lowest index radio. */
        w = gtk_radio_button_new_with_label(gslist, "Lowest Index");
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        PrefParmRecord(pref, VMA_PREF_PARM_BACKUP_NEW_LOWEST_INDEX, w);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_PREFWIN_BACKUP_NEW_LOWEST_INDEX
            )
        );

	return;
}

/*
 *      Create plug-ins panel.
 */
void PrefCreatePanelPlugins(vma_pref_struct *pref, GtkWidget *parent)
{
        const char *msglist[] = VMA_MSGLIST_PREFWIN_TOOLTIPS;
        const char *cstrptr;
        GtkWidget *w, *fw, *menu, *parent2, *parent3, *scroll_parent;
	GtkCList *clist;
        gint border_major = 5;
        gint border_minor = 2;
	gchar *heading[4];

        gpointer accel_group = NULL;
	gint accel_key;
        guint accel_mods;
        const gchar *label;
        u_int8_t **icon;
        gpointer mclient_data = (gpointer)pref;
        void (*func_cb)(GtkWidget *, gpointer);

#define DO_ADD_MENU_ITEM_LABEL		\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_LABEL, accel_group, \
  icon, label, accel_key, accel_mods, (void **)&fw, \
  mclient_data, func_cb \
 ); \
}

#define DO_ADD_MENU_SEP			\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL, \
  NULL, NULL, 0, 0, NULL, \
  NULL, NULL \
 ); \
}


        /* Heading. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        PrefCreateHeading(
            pref, parent2, border_major,
            "Plug-Ins"
        );

        /* Intro. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        cstrptr = (const char *)MsgListMatchCaseMessage(
	    msglist, VMA_MSGNAME_PREFWIN_PLUGINS_INTRO
        );
        if(cstrptr == NULL)
            cstrptr = "";
        w = gtk_label_new(cstrptr);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_major);
        gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
        gtk_widget_show(w);


        /* Plug-ins list frame. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_frame_new("Plug-Ins List");
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, border_major);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Plugins clist. */
        w = gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(
            GTK_SCROLLED_WINDOW(w),
            GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC
        );
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

        heading[0] = g_strdup("File");
        heading[1] = g_strdup("Title");
        heading[2] = g_strdup("");
	heading[3] = g_strdup("");
        w = gtk_clist_new_with_titles(4, heading);
        clist = GTK_CLIST(w);
        g_free(heading[0]);
        g_free(heading[1]);
        g_free(heading[2]);
        g_free(heading[3]);
        gtk_clist_set_row_height(clist, VMA_LIST_ROW_SPACING);
        gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
        gtk_clist_set_selection_mode(clist, GTK_SELECTION_SINGLE);
        gtk_clist_set_column_width(clist, 0, 130);
        gtk_clist_set_column_width(clist, 1, 180);
        gtk_clist_set_column_width(clist, 2, 20);
        gtk_clist_set_column_width(clist, 3, 20);
        gtk_widget_set_usize(w, -1, 160);
        gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(PrefMenuMapCB),
	    (gpointer)pref
	);
        gtk_container_add(GTK_CONTAINER(parent3), w);
        gtk_signal_connect(
            GTK_OBJECT(w), "select_row",
            GTK_SIGNAL_FUNC(PrefPluginsListSelectCB),
            (gpointer)pref
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "unselect_row",
            GTK_SIGNAL_FUNC(PrefPluginsListUnselectCB),
            (gpointer)pref
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "click_column",
            GTK_SIGNAL_FUNC(PrefPluginsListColumnClickCB),
            (gpointer)pref  
        );
        PrefParmRecord(pref, VMA_PREF_PARM_PLUGINS_LIST, w);
        gtk_widget_show(w);


	/* Right-click menu. */
	menu = GUIMenuCreate();
	if(menu != NULL)
	{
	    PrefParmRecord(pref, VMA_PREF_PARM_PLUGINS_LIST_MENU, menu);

	    /* Enable and disable menu items. */
	    /* Enable menu item, this item is labled "Disable" because
	     * its shown when the plug-in is currently enabled. It will
	     * disable the plug-in if you click on it.
	     */
            icon = (u_int8_t **)icon_plugin_disabled_20x20_xpm;
	    label = "Disable";
            accel_key = 0;
            accel_mods = 0;
            func_cb = PrefPluginsEnableCB;
            DO_ADD_MENU_ITEM_LABEL
            PrefParmRecord(pref, VMA_PREF_PARM_PLUGINS_LIST_MENU_ENABLE, w);
            gtk_widget_hide(w);
	    gtk_widget_set_sensitive(w, FALSE);

	    /* Disable menu item, this item is labled "Enable" because
	     * its shown when the plug-in is currently disabled. It will
	     * enable the plug-in if you click on it.
	     */
            icon = (u_int8_t **)icon_plugin_enabled_20x20_xpm;
            label = "Enable";
            accel_key = 0;
            accel_mods = 0;
            func_cb = PrefPluginsEnableCB;
            DO_ADD_MENU_ITEM_LABEL
            PrefParmRecord(pref, VMA_PREF_PARM_PLUGINS_LIST_MENU_DISABLE, w);
	    gtk_widget_set_sensitive(w, FALSE);

	    DO_ADD_MENU_SEP

            icon = (u_int8_t **)icon_options_20x20_xpm;
            label = "Configure...";
            accel_key = 0;
            accel_mods = 0;
            func_cb = PrefPluginsConfigureCB;
            DO_ADD_MENU_ITEM_LABEL
	    PrefParmRecord(pref, VMA_PREF_PARM_PLUGINS_LIST_MENU_CONFIGURE, w);
            gtk_widget_set_sensitive(w, FALSE);
        }


	/* Hbox to hold some buttons. */
	w = gtk_hbox_new(TRUE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

        /* Enable and disable buttons. */
	/* Enable button, this button is labled "Disable" because its shown
	 * when the plug-in is currently enabled. It will disable the plug-in
	 * if you click on it.
	 */
	w = GUIButtonPixmapLabelH(
	    (u_int8_t **)icon_plugin_disabled_20x20_xpm,
	    "Disable", NULL
	);
        gtk_widget_set_usize(w, PREF_BTN_WIDTH, PREF_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefPluginsEnableCB),
            (gpointer)pref
        );
        PrefParmRecord(pref, VMA_PREF_PARM_PLUGINS_ENABLE, w);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, 
		VMA_MSGNAME_PREFWIN_PLUGINS_ENABLE_BUTTON
            )
        );
	gtk_widget_set_sensitive(w, FALSE);

        /* Disable button, this button is labled "Enable" because its shown
         * when the plug-in is disabled. It will enable the plug-in if you
         * click on it.
         */
        w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_plugin_enabled_20x20_xpm,
	    "Enable", NULL
        );
	gtk_widget_set_usize(w, PREF_BTN_WIDTH, PREF_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefPluginsEnableCB),
            (gpointer)pref
        );
        PrefParmRecord(pref, VMA_PREF_PARM_PLUGINS_DISABLE, w);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist,
                VMA_MSGNAME_PREFWIN_PLUGINS_DISABLE_BUTTON
            )
        );
	gtk_widget_show(w);
        gtk_widget_set_sensitive(w, FALSE);


        /* Reload button. */
        w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_reload_20x20_xpm,
            "Reload", NULL
        );
        gtk_widget_set_usize(w, PREF_BTN_WIDTH, PREF_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefPluginsReloadCB),
            (gpointer)pref
        );
        PrefParmRecord(pref, VMA_PREF_PARM_PLUGINS_RELOAD, w);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist,
                VMA_MSGNAME_PREFWIN_PLUGINS_RELOAD_BUTTON
            )
        );
        gtk_widget_show(w);

        /* Configure button. */
        w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_options_20x20_xpm,
            "Configure...", NULL
        );
        gtk_widget_set_usize(w, PREF_BTN_WIDTH, PREF_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefPluginsConfigureCB),
            (gpointer)pref
        );
        PrefParmRecord(pref, VMA_PREF_PARM_PLUGINS_CONFIGURE, w);
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist,
                VMA_MSGNAME_PREFWIN_PLUGINS_CONFIGURE_BUTTON
            )
        );
        gtk_widget_set_sensitive(w, FALSE);
        gtk_widget_show(w);



	/* Descriptions text box. */
        w = gtk_table_new(2, 2, FALSE);
        gtk_table_set_row_spacing(GTK_TABLE(w), 0, 2);
        gtk_table_set_col_spacing(GTK_TABLE(w), 0, 2);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 2);
        gtk_widget_show(w);
        parent3 = w;

        w = gtk_text_new(NULL, NULL);
        gtk_text_set_editable(GTK_TEXT(w), FALSE);
	gtk_widget_set_usize(w, -1, 70);
        gtk_text_set_word_wrap(GTK_TEXT(w), TRUE);
        gtk_table_attach(
            GTK_TABLE(parent3), w,
            0, 1, 0, 1,
            GTK_EXPAND | GTK_SHRINK | GTK_FILL,
            GTK_EXPAND | GTK_SHRINK | GTK_FILL,
            0, 0
        );
        gtk_widget_realize(w);
/*
        if(pref->text_cur != NULL)
            gdk_window_set_cursor(w->window, pref->text_cur);
 */
        PrefParmRecord(pref, VMA_PREF_PARM_PLUGINS_DESCRIPTION, w);
        gtk_widget_show(w);

        scroll_parent = gtk_vscrollbar_new(GTK_TEXT(w)->vadj);
        gtk_table_attach(
            GTK_TABLE(parent3), scroll_parent,
            1, 2, 0, 1,
            GTK_FILL,
            GTK_EXPAND | GTK_SHRINK | GTK_FILL,   
            0, 0
        );
        gtk_widget_show(scroll_parent);

#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_SEP

	return;
}



/*
 *	Creates each catagory into the preferences window.
 */
static void PrefCreateCatagories(vma_pref_struct *pref)
{
	GtkWidget *parent;
	GtkCTreeNode *branch, *parent_branch, *main_branch;
	GtkCTree *ctree;
	GdkPixmap *closed_pixmap, *opened_pixmap;
	GdkBitmap *closed_mask, *opened_mask;


	ctree = (GtkCTree *)pref->catagory_ctree;
	if(ctree == NULL)
	    return;


	VMAPixmapsListGetValues(
	    &vma_pixmaps_list, VMA_PIXMAP_VMA_20x20,
	    &closed_pixmap, &closed_mask,
	    NULL
	);
	opened_pixmap = closed_pixmap;
	opened_mask = closed_mask;

	/* Main branch. */
	parent = gtk_vbox_new(FALSE, 0);
        branch = PrefAddCatagoryBranch(
            pref, PROG_NAME " Preferences",
            NULL, NULL,
	    closed_pixmap, closed_mask, opened_pixmap, opened_mask,
            FALSE,              /* Is leaf. */
            parent,
            VMA_PREF_CAT_MAIN
        );
	main_branch = parent_branch = branch;
        PrefCreatePanelMain(pref, parent);
	gtk_ctree_expand(ctree, branch);

	/* Apperance. */
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, VMA_PIXMAP_APPERANCE_20x20,
            &closed_pixmap, &closed_mask,
            NULL
        );
        opened_pixmap = closed_pixmap;
        opened_mask = closed_mask;
	parent = gtk_vbox_new(FALSE, 0);
	branch = PrefAddCatagoryBranch(
	    pref, "Appearance",
	    main_branch, NULL,
            closed_pixmap, closed_mask, opened_pixmap, opened_mask,
	    FALSE,		/* Is leaf. */
	    parent,
	    VMA_PREF_CAT_APPERANCE
	);
	parent_branch = branch;
	PrefCreatePanelApperance(pref, parent);

	/* Fonts. */
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, VMA_PIXMAP_FONTS_20x20,
            &closed_pixmap, &closed_mask,
            NULL
        );
        opened_pixmap = closed_pixmap;
        opened_mask = closed_mask;
        parent = gtk_vbox_new(FALSE, 0);
        branch = PrefAddCatagoryBranch(
            pref, "Fonts",
            parent_branch, NULL,
            closed_pixmap, closed_mask, opened_pixmap, opened_mask,
            TRUE,               /* Is leaf. */
            parent,
            VMA_PREF_CAT_FONTS
        );
	PrefCreatePanelFonts(pref, parent);

        /* Colors. */
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, VMA_PIXMAP_MP_COLOR_20x20,
            &closed_pixmap, &closed_mask,
            NULL
        );
        opened_pixmap = closed_pixmap; 
        opened_mask = closed_mask;
	parent = gtk_vbox_new(FALSE, 0);
        branch = PrefAddCatagoryBranch(
            pref, "Colors",
            parent_branch, NULL,
            closed_pixmap, closed_mask, opened_pixmap, opened_mask,
            TRUE, 		/* Is leaf. */
            parent,
            VMA_PREF_CAT_COLORS
        );
	PrefCreatePanelColors(pref, parent);

	/* Precision. */
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, VMA_PIXMAP_PRECISION_20x20,
            &closed_pixmap, &closed_mask,
            NULL
        );
        opened_pixmap = closed_pixmap;
        opened_mask = closed_mask;
	parent = gtk_vbox_new(FALSE, 0);
	branch = PrefAddCatagoryBranch(
            pref, "Precision",
            parent_branch, NULL,
            closed_pixmap, closed_mask, opened_pixmap, opened_mask,
            TRUE,               /* Is leaf. */
            parent,
            VMA_PREF_CAT_PRECISION
        );
	PrefCreatePanelPrecision(pref, parent);

	/* Rendering. */
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, VMA_PIXMAP_RENDER_20x20,
            &closed_pixmap, &closed_mask,
            NULL
        );
        opened_pixmap = closed_pixmap;
        opened_mask = closed_mask;
	parent = gtk_vbox_new(FALSE, 0);
        branch = PrefAddCatagoryBranch(
            pref, "Rendering",
            parent_branch, NULL,
            closed_pixmap, closed_mask, opened_pixmap, opened_mask,
            TRUE,               /* Is leaf. */
            parent,
            VMA_PREF_CAT_RENDERING
        );
	PrefCreatePanelRendering(pref, parent);


	/* Devices. */
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, VMA_PIXMAP_DEVICES_20x20,
            &closed_pixmap, &closed_mask,
            NULL
        );
        opened_pixmap = closed_pixmap;
        opened_mask = closed_mask;
        parent = gtk_vbox_new(FALSE, 0);
        branch = PrefAddCatagoryBranch(
            pref, "Devices",
            main_branch, NULL,
            closed_pixmap, closed_mask, opened_pixmap, opened_mask,
            TRUE,               /* Is leaf. */
            parent,
            VMA_PREF_CAT_DEVICES
        );
        PrefCreatePanelDevices(pref, parent);

	/* Editor. */
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, VMA_PIXMAP_VMA_20x20,
            &closed_pixmap, &closed_mask,
            NULL
        );
        opened_pixmap = closed_pixmap;
        opened_mask = closed_mask;
        parent = gtk_vbox_new(FALSE, 0);
        branch = PrefAddCatagoryBranch(
            pref, "Editor",
            main_branch, NULL,
            closed_pixmap, closed_mask, opened_pixmap, opened_mask,
            TRUE,               /* Is leaf. */
            parent,
            VMA_PREF_CAT_EDITOR
        );
	PrefCreatePanelEditor(pref, parent);

        /* Locations. */
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, VMA_PIXMAP_FOLDER_CLOSED_20x20,
            &closed_pixmap, &closed_mask,
            NULL
        );
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, VMA_PIXMAP_FOLDER_OPENED_20x20,
            &opened_pixmap, &opened_mask,
            NULL
        );
	parent = gtk_vbox_new(FALSE, 0);
        branch = PrefAddCatagoryBranch(
            pref, "Locations",
            main_branch, NULL,
            closed_pixmap, closed_mask, opened_pixmap, opened_mask,
	    TRUE,		/* Is leaf. */
            parent,
            VMA_PREF_CAT_LOCATIONS
        );
        PrefCreatePanelLocations(pref, parent);

	/* V3D Format Options. */
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, VMA_PIXMAP_FILE_V3D_20x20,
            &closed_pixmap, &closed_mask,
            NULL   
        );
        opened_pixmap = closed_pixmap;
        opened_mask = closed_mask;
	parent = gtk_vbox_new(FALSE, 0);
        branch = PrefAddCatagoryBranch(
            pref, "V3D Format",
            main_branch, NULL,
            closed_pixmap, closed_mask, opened_pixmap, opened_mask,
            TRUE,               /* Is leaf. */
            parent,
            VMA_PREF_CAT_V3DFORMAT
        );
        PrefCreatePanelV3DFormat(pref, parent);

        /* Backup Options. */
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, VMA_PIXMAP_BACKUP_20x20,
            &closed_pixmap, &closed_mask,
            NULL
        );
        opened_pixmap = closed_pixmap;
        opened_mask = closed_mask;
        parent = gtk_vbox_new(FALSE, 0);
        branch = PrefAddCatagoryBranch(
            pref, "Backup",
            main_branch, NULL,
            closed_pixmap, closed_mask, opened_pixmap, opened_mask,
            TRUE,               /* Is leaf. */
            parent,
            VMA_PREF_CAT_BACKUP
        );
        PrefCreatePanelBackup(pref, parent);

        /* Plugins. */
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, VMA_PIXMAP_PLUGIN_20x20,
            &closed_pixmap, &closed_mask,
            NULL
        );
        opened_pixmap = closed_pixmap;
        opened_mask = closed_mask;
        parent = gtk_vbox_new(FALSE, 0);
        branch = PrefAddCatagoryBranch(
            pref, "Plug-Ins",
            main_branch, NULL,
            closed_pixmap, closed_mask, opened_pixmap, opened_mask,
            TRUE,               /* Is leaf. */
            parent,
            VMA_PREF_CAT_PLUGINS
        );
        PrefCreatePanelPlugins(pref, parent);



	/* Update column width. */
	gtk_clist_set_column_width(
	    (GtkCList *)ctree,
	    0,			/* Column. */
	    gtk_clist_optimal_column_width((GtkCList *)ctree, 0)
	);

	return;
}


/*
 *	Creates a new preferences window.
 */
vma_pref_struct *PrefNew(void *core_ptr)
{
	GdkWindow *window, *root = (GdkWindow *)GDK_ROOT_PARENT();
        GtkWidget *w, *parent, *parent2, *parent3;
        GtkWidget *scroll_parent;
	GtkCList *clist;
	gchar *title[1];

        vma_pref_struct *pref = (vma_pref_struct *)calloc(
            1,
            sizeof(vma_pref_struct)
        );
        if(pref == NULL)
            return(NULL);

        /* Reset values. */
        pref->initialized = TRUE;
        pref->map_state = FALSE;
	pref->has_changes = FALSE;
        pref->processing = FALSE;

        pref->core_ptr = core_ptr;

	pref->panel_parent = NULL;
	pref->total_panel_parents = 0;

	pref->parm_widget = NULL;
	pref->total_parm_widgets = 0;


        /* Load cursors. */
        pref->busy_cur = gdk_cursor_new(GDK_WATCH);


        /* Toplevel. */
        pref->toplevel = w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
        gtk_window_set_policy(
            GTK_WINDOW(w),
            TRUE, TRUE, FALSE
        );
        gtk_widget_set_usize(
            w, VMA_PREF_DEF_WIDTH, VMA_PREF_DEF_HEIGHT
        );
        gtk_widget_realize(w);
	window = w->window;
        if(!GTK_WIDGET_NO_WINDOW(w) && (window != NULL))
        {
            GdkGeometry geometry;

            geometry.min_width = 100;
            geometry.min_height = 70;

            geometry.base_width = 0;
            geometry.base_height = 0;

            geometry.width_inc = 1;
            geometry.height_inc = 1;
/*
            geometry.min_aspect = 1.3;
            geometry.max_aspect = 1.3;
 */
            gdk_window_set_geometry_hints(
                window,
                &geometry,
                GDK_HINT_MIN_SIZE |
                GDK_HINT_BASE_SIZE |
                /* GDK_HINT_ASPECT | */
                GDK_HINT_RESIZE_INC
            );
/*
            gdk_window_set_decorations(
                window,
                GDK_DECOR_TITLE | GDK_DECOR_MENU | GDK_DECOR_MINIMIZE
            );
 */
/*
            gdk_window_set_functions(
                window,
                GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
            );
 */
            GUISetWMIcon(window, (u_int8_t **)icon_vma_48x48_xpm);
            if(root != NULL)
            {
                gint rx, ry, rw, rh, rd;

                gdk_window_get_geometry(
                    root, &rx, &ry, &rw, &rh, &rd
                );
                gtk_widget_set_uposition(
                    w,
                    (rw / 2) - (w->allocation.width / 2),
                    (rh / 2) - (w->allocation.height / 2)
                );
            }
        }
        gtk_signal_connect(
            GTK_OBJECT(w), "destroy",
            GTK_SIGNAL_FUNC(PrefDestroyCB),
            (gpointer)pref
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "delete_event",
            GTK_SIGNAL_FUNC(PrefCloseCB),
            (gpointer)pref
        );
        gtk_window_set_title(GTK_WINDOW(w), PROG_NAME " Preferences");
        parent = w;


	/* Main vbox. */
	w = gtk_vbox_new(FALSE, 0);
	pref->main_vbox = w;
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


        /* Paned. */
        w = gtk_hpaned_new();
        gtk_paned_set_handle_size(GTK_PANED(w), VMA_DEF_PANED_HANDLE_SIZE);
        gtk_paned_set_gutter_size(GTK_PANED(w), VMA_DEF_PANED_GUTTER_SIZE);
        gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
        gtk_paned_set_position(GTK_PANED(w), 175);
        gtk_widget_show(w);
        parent2 = w;


        /* Vbox for layout ctree. */  
        w = gtk_vbox_new(FALSE, 0);
        gtk_paned_add1(GTK_PANED(parent2), w);
        gtk_widget_show(w);
        parent3 = w;

        /* Scrolled window for catagory ctree. */
        w = gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(
            GTK_SCROLLED_WINDOW(w),
            GTK_POLICY_AUTOMATIC,
            GTK_POLICY_AUTOMATIC
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        scroll_parent = w;

        title[0] = "Categories";
        w = gtk_ctree_new_with_titles(
            1, 0, title
        );
        clist = (GtkCList *)w;
        pref->catagory_ctree = w;
/*
        gtk_signal_connect_after(
            GTK_OBJECT(w), "button_press_event",
            GTK_SIGNAL_FUNC(PrefMenuMapCB),
            (gpointer)pref
        );
 */
        gtk_clist_column_titles_passive(clist);
	gtk_clist_column_titles_hide(clist);
	gtk_clist_set_row_height(clist, VMA_LIST_ROW_SPACING);
        gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
        gtk_container_add(GTK_CONTAINER(scroll_parent), w);
        gtk_signal_connect(
            GTK_OBJECT(w), "tree_select_row",
            GTK_SIGNAL_FUNC(PrefCatagoryCTreeSelectCB),
            (gpointer)pref
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "tree_unselect_row",
            GTK_SIGNAL_FUNC(PrefCatagoryCTreeUnselectCB),
            (gpointer)pref
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "tree_expand",
            GTK_SIGNAL_FUNC(PrefCatagoryCTreeExpandCB),
            (gpointer)pref
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "tree_collapse",
            GTK_SIGNAL_FUNC(PrefCatagoryCTreeExpandCB),
            (gpointer)pref
        );
        gtk_widget_show(w);


        /* Layout ctree right click menu. */
/*
        menu = (GtkWidget *)GUIMenuCreate();
        pref->catagory_menu = menu;
        accel_group = NULL;
        mclient_data = (void *)pref;

        if(menu != NULL)
        {
            icon = (u_int8_t **)mp_viewer_20x20_xpm;
            label = "Expand";
            accel_key = 0;
            accel_mods = 0;
            func_cb = NULL;
            DO_ADD_MENU_ITEM_LABEL 
            pref->menu_mi = w;

	}
 */


        /* ********************************************************** */
        /* Right panel side vbox. */
        w = gtk_vbox_new(FALSE, 0);
	pref->panel_toplevel = w;
        gtk_paned_add2(GTK_PANED(parent2), w);
        gtk_widget_show(w);
        parent2 = w;






	/* ********************************************************** */

	/* Get main vbox parent. */
	parent = pref->main_vbox;

	/* Horizontal rule above buttons hbox. */
	w = gtk_hseparator_new();
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

	/* Hbox to hold buttons (needs border since main vbox dosen't). */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_container_border_width(GTK_CONTAINER(w), 5);
	gtk_widget_show(w);
        parent2 = w;

	/* OK button. */
        w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_ok_20x20_xpm, "OK", NULL
        );
        pref->ok_btn = w;
	gtk_widget_set_usize(w, PREF_BTN_WIDTH, PREF_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 5);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefOKCB),
            (gpointer)pref
        );
	gtk_widget_show(w);

        /* Apply button. */
        w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_select_20x20_xpm, "Apply", NULL
        );
        pref->apply_btn = w;
        gtk_widget_set_usize(w, PREF_BTN_WIDTH, PREF_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 5);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefApplyCB),
            (gpointer)pref
        );
        gtk_widget_show(w);

        /* Cancel button. */
        w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_cancel_20x20_xpm, "Cancel", NULL
        );
        pref->cancel_btn = w;
        gtk_widget_set_usize(w, PREF_BTN_WIDTH, PREF_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 5);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(PrefCloseMCB),
            (gpointer)pref
        );
        gtk_widget_show(w);


        /* Reset values (before creating catagories). */
        PrefReset(pref, FALSE);

	/* Create all catagories on the preferences window. */
	PrefCreateCatagories(pref);

	/* Update menus. */
	PrefUpdateMenus(pref);
 
        return(pref);
}

/*
 *	Updates all menus on the preferences window.
 */
void PrefUpdateMenus(vma_pref_struct *pref)
{
        static gbool reenterant = FALSE;


        if(pref == NULL)
            return;

        if(!pref->initialized)
            return;

        if(reenterant)
            return;
        else
            reenterant = TRUE;




	reenterant = FALSE;
	return;
}

/*
 *	Resets the preferences window, deallocating any loaded data on
 *	it and unmapping it if requested.
 */
void PrefReset(vma_pref_struct *pref, gbool need_unmap)
{
        static gbool reenterant = FALSE;
        GtkWidget *w;
	GtkCTree *ctree;
	GtkCTreeNode *branch, *prev_branch;
	GtkCTreeRow *branch_row;


        if(pref == NULL)
            return;

        if(!pref->initialized)
            return;

        /* Still processing? */
        if(pref->processing)
            return;

        if(reenterant)
            return;
        else
            reenterant = TRUE;


        /* Reset values. */
        pref->processing = FALSE;
        pref->has_changes = FALSE;

	pref->selected_branch = NULL;


	/* Delete all branches in catagory ctree. */
	w = pref->catagory_ctree;
	if(w != NULL)
	{
	    ctree = (GtkCTree *)w;
	    branch = gtk_ctree_node_nth(ctree, 0);

	    while(branch != NULL)
	    {
		prev_branch = branch;

		branch_row = GTK_CTREE_ROW(branch);
		branch = ((branch_row == NULL) ?
		    NULL : branch_row->sibling
		);

		gtk_ctree_remove_node(ctree, prev_branch);
	    }
	}


        /* Unmap as needed. */
        if(need_unmap)
        {
            w = pref->toplevel;
            if(w != NULL)
                gtk_widget_hide(w);
            pref->map_state = FALSE;
        }

        reenterant = FALSE;
	return;
}

/*
 *	Maps the preferences window as needed.
 */
void PrefMap(vma_pref_struct *pref)
{
        GtkWidget *w;

        if(pref == NULL)
            return;

        if(!pref->initialized)
            return;

        w = pref->ok_btn;
        if(w != NULL)
	{
	    gtk_widget_grab_focus(w);
	    gtk_widget_grab_default(w);
	}

        if(!pref->map_state)
        {   
	    w = pref->toplevel;
	    if(w != NULL)
		gtk_widget_show(w);

            pref->map_state = TRUE;
        }
}

/*
 *	Unmaps the preferences window as needed.
 */
void PrefUnmap(vma_pref_struct *pref)
{
        GtkWidget *w;

        if(pref == NULL)
            return;

        if(!pref->initialized)
            return;

        w = pref->toplevel;
        if(w == NULL)
            return;

        if(pref->map_state)
        {
            pref->map_state = FALSE;
            gtk_widget_hide(w);
        }

        return;
}

/*
 *	Destroys all resources on the preferences window and deallocates
 *	the structure itself.
 */
void PrefDelete(vma_pref_struct *pref)
{
	gint i;
	GdkCursor **cur;
        GtkWidget **w;


        if(pref == NULL)
            return;

        if(pref->initialized)
        {
            /* Still processing? */
            if(pref->processing)
                return;

            /* Reset values to defaults, this will cause any loaded data
             * to be deallocated.
             */
            PrefReset(pref, pref->map_state);

#define DO_DESTROY_CURSOR	\
{ \
 if((*cur) != NULL) \
 { \
  GdkCursor *tc = *cur; \
  (*cur) = NULL; \
  gdk_cursor_destroy(tc); \
 } \
}

#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

	    /* Destroy all parameter widgets. */
	    for(i = pref->total_parm_widgets - 1; i >= 0; i--)
	    {
		w = &pref->parm_widget[i];
		DO_DESTROY_WIDGET
	    }

	    w = &pref->panel_toplevel;
	    DO_DESTROY_WIDGET

	    w = &pref->catagory_ctree;
	    DO_DESTROY_WIDGET

            w = &pref->main_vbox;
            DO_DESTROY_WIDGET

            w = &pref->toplevel;
            DO_DESTROY_WIDGET

            cur = &pref->busy_cur;
	    DO_DESTROY_CURSOR

#undef DO_DESTROY_CURSOR
#undef DO_DESTROY_WIDGET
	}

	/* Deallocate panel and parameter widget lists. */
	g_free(pref->panel_parent);
	pref->panel_parent = NULL;
	pref->total_panel_parents = 0;

	g_free(pref->parm_widget);
	pref->parm_widget = NULL;
	pref->total_parm_widgets = 0;

	/* Deallocate structure itself. */
	g_free(pref);
}
