#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include <gtk/gtk.h>

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"

#include "v3dmh.h"
#include "v3dmp.h"
#include "v3dmodel.h"

#include "editor.h"
#include "editorlist.h"
#include "editorviewcb.h"
#include "editorshipping.h"

#include "vpi.h"
#include "vpiinternal.h"

#include "vmastatusbar.h"
#include "vmacfg.h"
#include "vmacfglist.h"
#include "vma.h"
#include "vmautils.h"

#include "config.h"


gint EditorDoImport(ma_editor_struct *editor);
gint EditorDoExport(ma_editor_struct *editor);


/*
 *	Procedure to check available plug-ins that have import support
 *	and get their supported file types then map the file browser to
 *	query for file to load.
 *
 *	Then appropriate plug-in with import support will be called if
 *	possible.
 *
 *	Return values:
 *
 *	0	Success
 *	-1	General error or editor not available
 *	-2	No support for selected file
 *	-3	Plug-in error or aborted
 *	-4	User aborted
 */
gint EditorDoImport(ma_editor_struct *editor)
{
	gint i, n, status;
        fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;
	gchar **path_rtn;
	gint path_total_rtns;
	vma_plugin_struct *plugin_ptr;
	vma_core_struct *core_ptr;
	gchar *filename, *extension;


	if(editor == NULL)
	    return(-1);

	if(!editor->initialized || editor->processing)
	    return(-1);

        if(VMAWriteProtectCheck(editor))
            return(-1);

	core_ptr = (vma_core_struct *)editor->core_ptr;
	if(core_ptr == NULL)
	    return(-1);


	/* Add the first file extension type to be "*.*" which means
	 * auto detect.
	 */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "Automatically detect type"
	);

	/* Itterate through all plug-ins on core structure, gathering
	 * file extension types.
	 */
	for(i = 0; i < core_ptr->total_plugins; i++)
	{
	    plugin_ptr = core_ptr->plugin[i];
	    if((plugin_ptr == NULL) ? 1 : (plugin_ptr->handle == NULL))
		continue;

	    /* Skip plug-ins that are busy processing. */
	    if(plugin_ptr->processing)
		continue;

	    /* Does not support import? */
	    if(plugin_ptr->import == NULL)
		continue;

	    /* Plug-in is loaded/enabled, check if it supports
	     * importing.
	     */
	    for(n = 0; n < plugin_ptr->total_import_ftypes; n++)
	    {
		FileBrowserTypeListNew(
		    &ftype, &total_ftypes,
		    plugin_ptr->import_ftype_ext[n],
                    plugin_ptr->import_ftype_name[n]
		);
	    }
	}

	/* Got no additional file types? */
	if(total_ftypes <= 1)
	{
	    /* This means we have no plug-ins with stated importing
	     * support.
	     */
	}

	/* Query user for file. */
	filename = NULL;
	extension = NULL;
	FileBrowserSetTransientFor(editor->toplevel);
	status = FileBrowserGetResponse(
	    "Import Model Data", "Import", "Cancel",
	    dname.fb_last_models,		/* Path. */
	    ftype, total_ftypes,
	    &path_rtn, &path_total_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	if(status)
	{
	    if(path_total_rtns > 0)
	    {
		const gchar *cstrptr = (const gchar *)path_rtn[
		    path_total_rtns - 1
		];
		filename = ((cstrptr == NULL) ? NULL : g_strdup(cstrptr));
	    }
	}

	/* Get extension if any. */
	if(ftype_rtn != NULL)
	{
	    if(ftype_rtn->ext != NULL)
		extension = g_strdup(ftype_rtn->ext);
	}

	/* Deallocate all file types. */
	FileBrowserDeleteTypeList(ftype, total_ftypes);
	ftype = NULL;
	total_ftypes = 0;


	/* If filename or extension is NULL then that implies that user
	 * canceled.
	 */
	if((filename == NULL) || (extension == NULL))
	{
	    g_free(filename);
	    g_free(extension);
	    return(-4);
	}

	/* Handle by extension, plugin_ptr will be updated to point
	 * to the correct plug-in that supports the format or NULL
	 * on failure.
	 */
	plugin_ptr = NULL;

	/* Selected file extension type suggests to auto detect? */
	if((*extension) == '*')
	{
	    /* Get file name extension from selected file. */
	    const char *filename_extension = strrchr(filename, '.');
	    if(filename_extension == NULL)
		filename_extension = filename;

	    for(i = 0; i < core_ptr->total_plugins; i++)
	    {
		plugin_ptr = core_ptr->plugin[i];
                if((plugin_ptr == NULL) ? 1 : (plugin_ptr->handle == NULL))
                    continue;

                /* Skip plug-ins that are busy processing. */
                if(plugin_ptr->processing)
                    continue;

                /* Does not support import file type check? */
                if(plugin_ptr->import_fcheck == NULL)
                    continue;

		plugin_ptr->processing = 1;
		status = plugin_ptr->import_fcheck(
		    (vpi_id *)plugin_ptr,
		    (const char *)filename,
		    filename_extension,
		    plugin_ptr->import_fcheck_data
		);
		plugin_ptr->processing = 0;
		if(status)
		{
		    /* This plug-in supports the format! */
		    break;
		}
	    }
	    if(i >= core_ptr->total_plugins)
		plugin_ptr = NULL;
	}
	else
	{
	    /* Match by extension. */
            for(i = 0; i < core_ptr->total_plugins; i++)
            {
                plugin_ptr = core_ptr->plugin[i];
                if((plugin_ptr == NULL) ? 1 : (plugin_ptr->handle == NULL))
                    continue;

                /* Skip plug-ins that are busy processing. */
                if(plugin_ptr->processing)
                    continue;

                /* Does not support import? */
                if(plugin_ptr->import == NULL)
                    continue;

		/* Supports import, check through all supported file
		 * extension types set on the plug-in for any matches.
		 */
		for(n = 0; n < plugin_ptr->total_import_ftypes; n++)
		{
		    const gchar *cstrptr = (const gchar *)plugin_ptr->import_ftype_ext[n];
		    if(cstrptr == NULL)
			continue;

		    if(!strcasecmp(extension, cstrptr))
			break;
		}
		if(n >= plugin_ptr->total_import_ftypes)
		{
		    /* No matching extensions, this plug-in does not
		     * support this extension.
		     */
		    continue;
		}
		else
		{
		    /* Got matching extension. */
		    break;
		}
	    }
            if(i >= core_ptr->total_plugins)
                plugin_ptr = NULL;
	}

	/* Found plug-in that supports this extension type, supports
	 * importing, and is currently not processing?
	 */
	if((plugin_ptr == NULL) ? 0 :
	    (plugin_ptr->import != NULL) &&
            !plugin_ptr->processing
	)
	{
	    vpi_op_import_struct v;


	    /* Set up import operation value structure. */
	    v.flags = (VPI_OP_IMPORT_EDITOR_ID |
		VPI_OP_IMPORT_FILENAME | VPI_OP_IMPORT_HEADER |
		VPI_OP_IMPORT_MODELS
	    );
	    for(i = 0; i < core_ptr->total_editors; i++)
	    {
		if(core_ptr->editor[i] == editor)
		    break;
	    }
	    v.editor_id = i;
	    v.filename = g_strdup(filename);
	    v.mh_item = editor->mh_item;
            v.total_mh_items = editor->total_mh_items;
            v.model = editor->model;
            v.total_models = editor->total_models;

	    /* Mark editor and plug-in as processing. */
	    EditorSetBusy(editor);
/*	    editor->processing = TRUE; */
	    plugin_ptr->processing = 1;

	    /* Call plug-in's import function. */
	    status = plugin_ptr->import(
		(vpi_id *)plugin_ptr,
		&v,
		plugin_ptr->import_data
	    );

	    /* Mark editor and plug-in as done processing. */
	    plugin_ptr->processing = 0;
	    EditorSetReady(editor);
/*	    editor->processing = FALSE; */

	    /* Deallocate import values structure members. */
	    g_free(v.filename);

            /* Importing counts as change on editor, update has_changes. */
	    if(!editor->has_changes);
		editor->has_changes = TRUE;

	    /* Update editor and redraw. */
	    EditorUpdateMenus(editor);
	    EditorUpdateAllViewMenus(editor);
	    EditorRedrawAllViews(editor);

	    VMAStatusBarMessage(editor->sb, "Import done");
	}
	else
	{
	    /* No plug-in available that supports this format. */
	    gint len = (80 * 8) + strlen(filename);
	    gchar *buf;

            VMAStatusBarMessage(editor->sb, "Import failed");

	    buf = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	    if(buf != NULL)
		sprintf(buf,
"There are no installed or enabled plug-ins that\n\
support the format of the file:\n\
\n\
    %s\n\
\n\
You may want to try selecting an exact file format\n\
instead of automatic detect or try to obtain/install\n\
a plug-in that supports this file's format.",
		    filename
		);

            CDialogSetTransientFor(editor->toplevel);
            CDialogGetResponse(
"Import failed!",
		buf,
"Importing of different file formats are handled by\n\
plug-ins. You need to enable or obtain/install\n\
a plug-in that supports the format in question.",
                CDIALOG_ICON_ERROR,
                CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);

	    g_free(buf);
	    buf = NULL;
	}

	/* Record last path. */
	VMARecordFBPath(
	    filename,
	    dname.fb_last_models,
	    1
	);

	/* Deallocate file name and extension. */
	g_free(filename);
	g_free(extension);

	return(0);
}


/*
 *      Procedure to check available plug-ins that have export support
 *      and get their supported file types then map the file browser to
 *      query for file to load.
 *
 *      Then appropriate plug-in with export support will be called if
 *      possible.
 *
 *      Return values:
 *
 *      0       Success
 *      -1      General error or editor not available
 *      -2      No support for selected file
 *      -3      Plug-in error or aborted
 *      -4      User aborted
 */
gint EditorDoExport(ma_editor_struct *editor)
{
        gint i, n, status;
        fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
        gint total_ftypes = 0;
        gchar **path_rtn;
        gint path_total_rtns;
        vma_plugin_struct *plugin_ptr;
        vma_core_struct *core_ptr;
	const gchar *last_extension;
        gchar *filename, *extension;


        if(editor == NULL)
            return(-1);

        if(!editor->initialized || editor->processing)
            return(-1);

        if(VMAWriteProtectCheck(editor))
            return(-1);

        core_ptr = (vma_core_struct *)editor->core_ptr;
        if(core_ptr == NULL)
            return(-1);


        /* Add the first file extension type to be the previous export
	 * type used.
         */
	last_extension = (const gchar *)VMACFGItemListGetValueS(
	    option, VMA_CFG_PARM_LAST_EXT_EXPORT
	);
	if((last_extension == NULL) ? 0 : ((*last_extension) != '\0'))
	    FileBrowserTypeListNew(
		&ftype, &total_ftypes,
		last_extension, "Previous Type"
	    );

        /* Itterate through all plug-ins on core structure, gathering
         * file extension types.
         */
        for(i = 0; i < core_ptr->total_plugins; i++)
        {
            plugin_ptr = core_ptr->plugin[i];
            if((plugin_ptr == NULL) ? 1 : (plugin_ptr->handle == NULL))
                continue;

            /* Skip plug-ins that are busy processing. */
            if(plugin_ptr->processing)
                continue;

            /* Does not support export? */
            if(plugin_ptr->export == NULL)
                continue;

            /* Plug-in is loaded/enabled, check if it supports
             * exporting.
             */
            for(n = 0; n < plugin_ptr->total_export_ftypes; n++)
            {
                FileBrowserTypeListNew(
                    &ftype, &total_ftypes,
                    plugin_ptr->export_ftype_ext[n],
                    plugin_ptr->export_ftype_name[n]
                );
            }
        }

        /* Got no additional file types? */
        if(total_ftypes <= 1)
        {
            /* This means we have no plug-ins with stated exporting
             * support.
             */
        }

        /* Query user for file. */
        filename = NULL;
        extension = NULL;

	FileBrowserSetTransientFor(editor->toplevel);
        status = FileBrowserGetResponse(
            "Export Model Data", "Export", "Cancel",
            dname.fb_last_models,		/* Path. */
            ftype, total_ftypes,
            &path_rtn, &path_total_rtns,
            &ftype_rtn
        );
	FileBrowserSetTransientFor(NULL);

	if(status)
        {
            if(path_total_rtns > 0)
            {
                const gchar *cstrptr = (const gchar *)path_rtn[
                    path_total_rtns - 1
                ];
                filename = ((cstrptr == NULL) ? NULL : g_strdup(cstrptr));
            }
        }

        /* Get extension if any. */
        if(ftype_rtn != NULL)
        {
            if(ftype_rtn->ext != NULL)
                extension = g_strdup(ftype_rtn->ext);
        }

        /* Deallocate all file types. */
        FileBrowserDeleteTypeList(ftype, total_ftypes);
        ftype = NULL;
        total_ftypes = 0;


        /* If filename or extension is NULL then that implies that user
         * canceled.
         */
        if((filename == NULL) || (extension == NULL))
        {
            g_free(filename);
            g_free(extension);
            return(-4);
        }

        /* Handle by extension, plugin_ptr will be updated to point
         * to the correct plug-in that supports the format or NULL
         * on failure.
         */
	plugin_ptr = NULL;
	for(i = 0; i < core_ptr->total_plugins; i++)
	{
	    plugin_ptr = core_ptr->plugin[i];
	    if((plugin_ptr == NULL) ? 1 : (plugin_ptr->handle == NULL))
		continue;

	    /* Skip plug-ins that are busy processing. */
	    if(plugin_ptr->processing)
		continue;

	    /* Does not support export? */
	    if(plugin_ptr->export == NULL)
		continue;

	    /* Supports export, check through all supported file
	     * extension types set on the plug-in for any matches.
	     */
	    for(n = 0; n < plugin_ptr->total_export_ftypes; n++)
	    {
		const gchar *cstrptr = (const gchar *)plugin_ptr->export_ftype_ext[n];
		if(cstrptr == NULL)
		    continue;

		if(!strcasecmp(extension, cstrptr))
		    break;
	    }
	    if(n >= plugin_ptr->total_export_ftypes)
	    {
		/* No matching extensions, this plug-in does not support
		 * this extension.
		 */
		continue;
	    }
	    else
	    {
		/* Got matching extension. */
		break;
	    }
	}
	if(i >= core_ptr->total_plugins)
	    plugin_ptr = NULL;

        /* Found plug-in that supports this extension type, supports
         * exporting, and is currently not processing?
         */  
        if((plugin_ptr == NULL) ? 0 :
            (plugin_ptr->export != NULL) && 
            !plugin_ptr->processing
        )
        {
            vpi_op_export_struct v;


            /* Set up export operation value structure. */
            v.flags = (VPI_OP_EXPORT_EDITOR_ID |
                VPI_OP_EXPORT_FILENAME | VPI_OP_EXPORT_HEADER |
                VPI_OP_EXPORT_MODELS
            );
            for(i = 0; i < core_ptr->total_editors; i++)
            {
                if(core_ptr->editor[i] == editor)
                    break;
            }
            v.editor_id = i;
            v.filename = g_strdup(filename);
            v.mh_item = editor->mh_item;
            v.total_mh_items = editor->total_mh_items;
            v.model = editor->model;
            v.total_models = editor->total_models;

            /* Mark editor and plug-in as processing. */
            EditorSetBusy(editor);
/*	    editor->processing = TRUE; */
            plugin_ptr->processing = 1;

            /* Call plug-in's export function. */
            status = plugin_ptr->export(
                (vpi_id *)plugin_ptr,
                &v,
                plugin_ptr->export_data
            );

            /* Mark editor and plug-in as done processing. */
            plugin_ptr->processing = 0;
            EditorSetReady(editor);
/*	    editor->processing = FALSE; */

            /* Deallocate export values structure members. */
            g_free(v.filename);

	    /* Do not modify has changes mark on editor. */

            /* Update editor and redraw. */
            EditorUpdateMenus(editor);
            EditorUpdateAllViewMenus(editor);
            EditorRedrawAllViews(editor);

            VMAStatusBarMessage(editor->sb, "Export done");
        }
        else
        {       
            /* No plug-in available that supports this format. */
            gint len = (80 * 8) + strlen(filename);
            gchar *buf;

            VMAStatusBarMessage(editor->sb, "Export failed");

            buf = (gchar *)g_malloc((len + 1) * sizeof(gchar));
            if(buf != NULL)
                sprintf(buf,
"There are no installed or enabled plug-ins that\n\
supports the format of the file:\n\
\n\
    %s\n\
\n\
You may want to try selecting an exact file format\n\
instead of automatic detect or try to obtain/install\n\
a plug-in that supports this file's format.",
                    filename
                );

            CDialogSetTransientFor(editor->toplevel);
            CDialogGetResponse(
"Export failed!",
                buf,
"Exporting of different file formats are handled by\n\
plug-ins. You need to enable or obtain/install\n\
a plug-in that supports the format in question.",
                CDIALOG_ICON_ERROR,
                CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);

            g_free(buf);
            buf = NULL;
        }

        /* Record last path. */
        VMARecordFBPath(
            filename,
            dname.fb_last_models,
            1
        );

        /* Update last export file type extension. */
        VMACFGItemListMatchSetValue(
            option, VMA_CFG_PARM_LAST_EXT_EXPORT,
            extension, TRUE
        );
        /* Update last exported file. */
        VMACFGItemListMatchSetValue(
            option, VMA_CFG_PARM_LAST_FILE_EXPORT,
            filename, TRUE
        );

        /* Deallocate file name and extension. */
        g_free(filename);
        g_free(extension);

        return(0);
}
