#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <gtk/gtk.h>

#include "v3dmh.h"
#include "v3dmodel.h"
#include "v3dfio.h"
#include "v3dmp.h"

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "vmapresetmodels.h"
#include "vma.h"
#include "config.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


vma_preset_models_struct vma_preset_models;


static int VMAPresetModelsLoadFromFile(
	vma_preset_models_struct *list,
	const char *filename
);
int VMAPresetModelsLoad(void *core, vma_preset_models_struct *list);
void VMAPresetModelsUnload(void *core, vma_preset_models_struct *list);


/*
 *	Appends list of models loaded from file into the given list.
 *
 *	This function should be called by VMAPresetModelsLoad().
 *
 *	Returns the number of models loaded.
 */
static int VMAPresetModelsLoadFromFile(
        vma_preset_models_struct *list,
        const char *filename
)
{
	int status, total_models_loaded = 0;
	FILE *fp;
	void **mh_item = NULL;
	int total_mh_items = 0;
	v3d_model_struct **model = NULL;
	int total_models = 0;


	if((list == NULL) || (filename == NULL))
	    return(total_models_loaded);

	if(list->total_models < 0)
	    list->total_models = 0;


	/* Open model file. */
	fp = FOpen(filename, "rb");
	if(fp == NULL)
	    return(total_models_loaded);

	/* Load models from fp. */
	status = V3DLoadModel(
	    NULL, fp,
	    &mh_item, &total_mh_items,
	    &model, &total_models,
	    NULL, NULL
	);

	/* Close file (don't need it anymore. */
        FClose(fp);
        fp = NULL;

	/* Delete the headers items, don't need those. */
	V3DMHListDeleteAll(&mh_item, &total_mh_items);

	/* Transfer the models and delete the model pointer array. */
	if(total_models > 0)
	{
	    int n, k, i = list->total_models;

	    /* Allocate more pointers. */
	    total_models_loaded += total_models;
	    list->total_models += total_models;
	    list->model = (v3d_model_struct **)realloc(
		list->model,
		list->total_models * sizeof(v3d_model_struct *)
	    );
	    if(list->model == NULL)
	    {
		list->total_models = 0;
		V3DModelListDeleteAll(&model, &total_models);
		return(0);
	    }

	    /* Transfer loaded models to our list. */
	    for(n = i, k = 0; n < list->total_models; n++, k++)
	    {
		list->model[n] = model[k];
		model[k] = NULL;
	    }

	    /* Deallocate loaded models pointer array only since
	     * each model was transfered to our list.
	     */
	    free(model);
	    model = NULL;
	    total_models = 0;
	}
	else
	{
	    /* Delete all loaded models (if any). */
	    V3DModelListDeleteAll(&model, &total_models);
	}

	return(total_models_loaded);
}


/*
 *	Loads preset models from file.
 */
int VMAPresetModelsLoad(void *core, vma_preset_models_struct *list)
{
	vma_core_struct *core_ptr = (vma_core_struct *)core;


	if((core_ptr == NULL) || (list == NULL))
            return(-1);

        list->initialized = TRUE;

	/* Load global models. */
	if((*dname.preset_models_global) != '\0')
	{
	    int i, total;
	    const char *path_ptr;
	    struct stat stat_buf;
	    char **strv, *tmp_path;

	    strv = GetDirEntNames(dname.preset_models_global);
	    if(strv != NULL)
	    {
	      for(total = 0; strv[total] != NULL; total++);
	      strv = StringQSort(strv, total);

	      for(i = 0; i < total; i++)
	      {
		path_ptr = (const char *)PrefixPaths(
		    dname.preset_models_global, strv[i]
		);
		tmp_path = ((path_ptr == NULL) ? NULL : strdup(path_ptr));

		/* Valid path to load? */
		if((tmp_path == NULL) ?
		    0 : !stat(tmp_path, &stat_buf)
		)
		{
		    /* Regular file? */
		    if(S_ISREG(stat_buf.st_mode))
		    {
			VMAPresetModelsLoadFromFile(list, tmp_path);
		    }
		}

		/* Deallocate paths. */
		free(tmp_path);
		free(strv[i]);
	      }

	      /* Deallocate path array. */
	      free(strv);
	    }
	}

        /* Load local models. */
        if((*dname.preset_models_local) != '\0')
        {
            int i, total;
            const char *path_ptr;
            struct stat stat_buf;
            char **strv, *tmp_path;

	    /* Create local models directory as needed. */
	    if(stat(dname.preset_models_local, &stat_buf))
	    {
		mode_t mode;

		if(stat(dname.data_local, &stat_buf))
		    mode = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
		else
		    mode = stat_buf.st_mode;

		rmkdir(dname.preset_models_local, mode);
	    }

            strv = GetDirEntNames(dname.preset_models_local);
            if(strv != NULL)
            {
              for(total = 0; strv[total] != NULL; total++);
              strv = StringQSort(strv, total); 

              for(i = 0; i < total; i++)
              {
                path_ptr = (const char *)PrefixPaths(
                    dname.preset_models_local, strv[i]
                );
                tmp_path = ((path_ptr == NULL) ? NULL : strdup(path_ptr));

                /* Valid path to load? */
                if((tmp_path == NULL) ?
                    0 : !stat(tmp_path, &stat_buf)
                )
                {
                    /* Regular file? */
                    if(S_ISREG(stat_buf.st_mode))
                    {
                        VMAPresetModelsLoadFromFile(list, tmp_path);
                    }
                }

                /* Deallocate paths. */
		free(tmp_path);
                free(strv[i]);
              }

              /* Deallocate path array. */
              free(strv);
            }
        }


	/* If no models were loaded, then create default models. */
	if(list->total_models <= 0)
	{
	    v3d_model_struct *model_ptr;

	    /* Create two default models. */
	    list->total_models = 2;
	    list->model = (v3d_model_struct **)realloc(
		list->model,
		list->total_models * sizeof(v3d_model_struct *)
	    );

	    model_ptr = V3DModelCreate(
		V3D_MODEL_TYPE_STANDARD,
		"standard"
	    );
	    list->model[0] = model_ptr;

	    model_ptr = V3DModelCreate(
		V3D_MODEL_TYPE_OTHER_DATA,
		"other"
	    );
	    list->model[1] = model_ptr;
	}

	return(0);
}

/*
 *	Deallocates given preset list of V3D models.
 */
void VMAPresetModelsUnload(void *core, vma_preset_models_struct *list)
{
	int i;


	if(list == NULL)
	    return;

	list->initialized = FALSE;

	/* Delete all models, assume no primitives are realized on the
	 * model.
	 */
	for(i = 0; i < list->total_models; i++)
	    V3DModelDestroy(list->model[i]);

	if(list->model != NULL)
	{
	    free(list->model);
	    list->model = NULL;
	}
	list->total_models = 0;

        list->initialized = FALSE;

	return;
}
