/*
 * ipc.h: A plugin for the Video Disk Recorder
 *
 * See the README file for copyright information and how to reach the author.
 *
 * $Id: ipc.h,v 1.1 2004/10/24 12:57:09 chelli-guest Exp $
 */


#ifndef __CON_IPC_H
#define __CON_IPC_H


#include <stdio.h>
#include <sys/types.h>

#include <vdr/thread.h>




// Interface that can be implemented,
// if one can wait (signal) for the object.


class IWaitable {
public:
  virtual ~IWaitable() {}

  virtual int SignalToWaitFor() const = 0;
};




// A List of objects we can wait for.


class cWaitableList {
private:
  IWaitable** _list;
  int  _listCount;

  fd_set _wait;

public:
  cWaitableList();
  ~cWaitableList();

  void Add( IWaitable* pObj );
  void Remove( IWaitable* pObj );
  bool Wait( int timeoutMs = 0 );

  bool IsSignalled( IWaitable* pObj ) { 
    if ( pObj && pObj->SignalToWaitFor() >= 0 )
      return FD_ISSET( pObj->SignalToWaitFor(), &_wait );
    return false;
  }

  bool IsSignalled( int fd ) {
    return FD_ISSET( fd, &_wait );
  }
};






// This is the real pipe, but that name is taken from the vdr source.
// So I name this a simple pipe.


class cSimplePipe {
private:
  int _pipe[ 2 ];

public:
  cSimplePipe();
  virtual ~cSimplePipe();

  bool Open();
  void Close();

  bool IsOpen() { return _pipe[ 0 ] >= 0 && _pipe[ 1 ] >= 0; }

  int getReader() const { return _pipe[ 0 ]; }
  int getWriter() const { return _pipe[ 1 ]; }
};





// Can be used to signal an other thread.
// This could also be realized by using a semaphore.
// But by using a pipe one can wait for a signal with select.


class cSignal
: private cSimplePipe,
  public IWaitable
{
public:
  cSignal();
  virtual ~cSignal() {}

  void Signal();
  void Reset();

  bool IsSignalled();

//IWaitable
  virtual int SignalToWaitFor() const { return getReader(); }
};




#endif //__CON_IPC_H

