/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "extInfos.h"

#include <GL/gl.h>

#include <visu_rendering.h>
#include <openGLFunctions/objectList.h>
#include <openGLFunctions/text.h>

/**
 * SECTION:extInfos
 * @short_description: give the capability to draw some information
 * near each node.
 *
 * <para>This part is used to draw some information near the
 * nodes. This information can be the one of a #VisuNodeProperty or
 * something else. When read from a #VisuNodeProperty, just giving the
 * name will produce the right output. In other cases a print routine
 * must be given.</para>
 */

/* Interface for a routine to draw the informations into a label. */
typedef void (*DrawInfosFunc)(VisuData *data, VisuElement *element,
			      VisuNode *node, DataNode *dataNode);

/* Local variables. */
static OpenGLExtension *extInfos;
static gboolean extIsBuilt;

/* These values are stored to be able to redraw the list when needed.
   One copy of such routine is set as a property to a given VisuData. The
   property is called "extensionInformations". */
typedef struct ExtInfos_struct
{
  int *nodes;
  DrawInfosFunc draw;
  DataNode *dataNode;
} ExtInfos;


/* Private routines. */
static void freeExtInfos(gpointer extInfos);
static ExtInfos* getExtInfos(VisuData *data);
static void drawNumber(VisuData *data, VisuElement *element,
		       VisuNode *node, DataNode *dataNode);
static void drawElement(VisuData *data, VisuElement *element,
			VisuNode *node, DataNode *dataNode);
static void drawInfos(VisuData *data, VisuElement *element,
		      VisuNode *node, DataNode *dataNode);

/* Callbacks. */
static void onPopulationIncrease(VisuData *data, int *newNodes, gpointer user_data);
static void onPopulationDecrease(VisuData *data, int *oldNodes, gpointer user_data);
static void onPositionUpdate(VisuData *data, gpointer user_data);
static void onNodeRender(VisuData *data, gpointer user_data);
static void onElementRender(VisuData *data, VisuElement *element, gpointer user_data);
static void onCameraMoved(VisuData *data, OpenGLView *view, gpointer user_data);


/********************/
/* Public routines. */
/********************/
OpenGLExtension* extInfosInit()
{
  char *name = "Information";
  char *description = _("Draw informations on nodes.");
  int listInfos;

  DBG_fprintf(stderr,"Initialising the informations OpenGL extension...\n");
  listInfos = openGLObjectList_new(1);
  extInfos = OpenGLExtension_new(name, _(name), description,
				 listInfos, extInfosBuild);
  OpenGLExtensionSet_priority(extInfos, OPENGL_EXTENSION_PRIORITY_HIGH);

  /* Initialisation des valeurs par dfaut. */
  extInfos->used = 0;
  extIsBuilt = FALSE;

  return extInfos;
}
void extInfosSet_number(VisuData *data, int *nodes)
{
  ExtInfos *infos;

  infos = getExtInfos(data);
  if (infos->nodes)
    g_free(infos->nodes);
  infos->nodes    = nodes;
  infos->draw     = drawNumber;
  infos->dataNode = (DataNode*)0;

  if (extInfos->used)
    extInfosBuild(data);
  else
    extIsBuilt = FALSE;
}
void extInfosSet_element(VisuData *data, int *nodes)
{
  ExtInfos *infos;

  infos = getExtInfos(data);
  if (infos->nodes)
    g_free(infos->nodes);
  infos->nodes    = nodes;
  infos->draw     = drawElement;
  infos->dataNode = (DataNode*)0;

  if (extInfos->used)
    extInfosBuild(data);
  else
    extIsBuilt = FALSE;
}
void extInfosSet_data(VisuData *data, DataNode *dataNode, int *nodes)
{
  ExtInfos *infos;

  infos = getExtInfos(data);
  if (infos->nodes)
    g_free(infos->nodes);
  infos->nodes    = nodes;
  infos->draw     = drawInfos;
  infos->dataNode = dataNode;

  if (extInfos->used)
    extInfosBuild(data);
  else
    extIsBuilt = FALSE;
}
void extInfosBuild(VisuData *dataObj)
{
  ExtInfos *infos;
  RenderingMethod *currentRenderingMethod;
  float modelView[16];
  float delta[3], xyz[3], size, rgba[4];
  VisuDataIter iter;
  int i;
  OpenGLView *view;

  DBG_fprintf(stderr, "Extension Infos: create the list of node informations.\n");
  g_return_if_fail(dataObj);
  view = visuDataGet_openGLView(dataObj);
  g_return_if_fail(view);
  
  extIsBuilt = TRUE;

  infos = visuDataGet_property(dataObj, "extensionInformations");
  if (!infos)
    return;

  currentRenderingMethod = getRenderingMethodInUse();
  g_return_if_fail(currentRenderingMethod);

  /* Get the camera orientation. */
  glGetFloatv(GL_MODELVIEW_MATRIX, modelView);

  glNewList(extInfos->objectListId, GL_COMPILE);
  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_LIGHTING);
  glPushMatrix();
  glTranslated(- view->box->dxxs2,
	       - view->box->dyys2,
	       - view->box->dzzs2);

  /* If infos->nodes is NULL, we draw for all nodes. */
  if (!infos->nodes)
    {
      DBG_fprintf(stderr, " | use all the nodes.\n");
      visuDataIter_new(dataObj, &iter);
      for (visuDataIter_start(dataObj, &iter); iter.element;
	   visuDataIter_nextElement(dataObj, &iter))
	{
	  if (iter.element->rendered && iter.element->showNodeInfos)
	    {
	      DBG_fprintf(stderr, "Extension Infos: creating glObjectList of node"
			  " names for '%s'.\n", iter.element->name);
	      rgba[0] = 1.f - iter.element->rgb[0];
	      rgba[1] = 1.f - iter.element->rgb[1];
	      rgba[2] = 1.f - iter.element->rgb[2];
	      rgba[3] = iter.element->rgb[3];
	      glColor4fv(rgba);
	      size = renderingMethodGet_sizeOfElement(currentRenderingMethod,
						      iter.element);
	      delta[0] = size * modelView[2];
	      delta[1] = size * modelView[6];
	      delta[2] = size * modelView[10];
	      for(visuDataIter_restartNode(dataObj, &iter); iter.node;
		  visuDataIter_nextNode(dataObj, &iter))
		{
		  if (iter.node->rendered)
		    {
		      visuDataGet_nodePosition(dataObj, iter.node, xyz);
		      glRasterPos3f(xyz[0] + delta[0],
				    xyz[1] + delta[1],
				    xyz[2] + delta[2]);
		      infos->draw(dataObj, iter.element, iter.node, infos->dataNode);
		    }
		}
	    }
	}
    }
  else
    {
      DBG_fprintf(stderr, " | use a restricted list of nodes.\n");
      /* infos->nodes is not NULL, we draw for the given infos->nodes only. */
      for (i = 0; infos->nodes[i] >= 0; i++)
	{
	  iter.node = visuDataGet_nodeFromNumber(dataObj, infos->nodes[i]);
	  g_return_if_fail(iter.node);
	  iter.element = dataObj->fromIntToVisuElement[iter.node->posElement];
	  if (iter.element->rendered && iter.element->showNodeInfos &&
	      iter.node->rendered)
	    {
	      rgba[0] = 1.f - iter.element->rgb[0];
	      rgba[1] = 1.f - iter.element->rgb[1];
	      rgba[2] = 1.f - iter.element->rgb[2];
	      rgba[3] = iter.element->rgb[3];
	      glColor4fv(rgba);
	      size = renderingMethodGet_sizeOfElement(currentRenderingMethod,
						      iter.element);
	      delta[0] = size * modelView[2];
	      delta[1] = size * modelView[6];
	      delta[2] = size * modelView[10];
	      visuDataGet_nodePosition(dataObj, iter.node, xyz);
	      glRasterPos3f(xyz[0] + delta[0],
			    xyz[1] + delta[1],
			    xyz[2] + delta[2]);
	      infos->draw(dataObj, iter.element, iter.node, infos->dataNode);
	    }
	}
    }
  glPopMatrix();
  glPopAttrib();
  glEndList();
}
gboolean extInfosSet_used(VisuData *data, gboolean status)
{
  if (extInfos->used == (int)status)
    return FALSE;

  extInfos->used = (int)status;
  if (status && !extIsBuilt)
    extInfosBuild(data);

  return TRUE;
}

/*********************/
/* Private routines. */
/*********************/
static void freeExtInfos(gpointer extInfos)
{
  DBG_fprintf(stderr, "Extension Infos: free the structure %p.\n", extInfos);
  if (((ExtInfos*)extInfos)->nodes)
    g_free(((ExtInfos*)extInfos)->nodes);
  g_free(extInfos);
}
static ExtInfos* getExtInfos(VisuData *data)
{
  ExtInfos *infos;

  infos = visuDataGet_property(data, "extensionInformations");
  if (!infos)
    {
      infos = g_malloc(sizeof(ExtInfos));
      DBG_fprintf(stderr, "Extension Infos: allocate the structure %p.\n",
		  (gpointer)infos);
      visuDataSet_propertyWithDestroyFunc(data, "extensionInformations",
					  (gpointer)infos, freeExtInfos);
      infos->nodes = (int*)0;

      /* Attach some signals to the data to modify or redraw when needed. */
      g_signal_connect(G_OBJECT(data), "NodePopulationDecrease",
		       G_CALLBACK(onPopulationDecrease), (gpointer)infos);
      g_signal_connect(G_OBJECT(data), "NodePopulationIncrease",
		       G_CALLBACK(onPopulationIncrease), (gpointer)infos);
      g_signal_connect(G_OBJECT(data), "NodePositionChanged",
		       G_CALLBACK(onPositionUpdate), (gpointer)infos);
      g_signal_connect(G_OBJECT(data), "NodeRenderedChanged",
		       G_CALLBACK(onElementRender), (gpointer)infos);
      g_signal_connect(G_OBJECT(data), "ElementRenderedChanged",
		       G_CALLBACK(onNodeRender), (gpointer)infos);
      g_signal_connect(G_OBJECT(data), "OpenGLThetaPhiOmega",
		       G_CALLBACK(onCameraMoved), (gpointer)infos);
    }
  return infos;
}
static void drawNumber(VisuData *data _U_, VisuElement *element _U_,
		       VisuNode *node, DataNode *dataNode _U_)
{
  gchar str[10];

  sprintf(str, "%d", node->number + 1);
  openGLText_drawChars(str);
}
static void drawElement(VisuData *data _U_, VisuElement *element,
			VisuNode *node _U_, DataNode *dataNode _U_)
{
  openGLText_drawChars(element->name);
}
static void drawInfos(VisuData *data, VisuElement *element _U_,
		      VisuNode *node, DataNode *dataNode)
{
  gchar *label;

  label = nodeDataGet_valueAsString(dataNode, data, node);
  openGLText_drawChars(label);
  g_free(label);
}

/*************/
/* Callbacks */
/*************/
static void onPopulationIncrease(VisuData *data, int *newNodes _U_,
				 gpointer user_data)
{
  ExtInfos *infos;

  if (!extInfos->used)
    return;

  DBG_fprintf(stderr, "Extension Informations: caught the 'NodePopulationIncrease'"
	      " signal, rebuild in all nodes case.\n");
  /* If we draw all nodes, then we must redraw. */
  infos = (ExtInfos*)user_data;
  if (infos->nodes)
    return;

  extInfosBuild(data);
}
static void onPopulationDecrease(VisuData *data, int *oldNodes, gpointer user_data)
{
  ExtInfos *infos;
  gboolean redraw;
  int i, j, size;

  if (!extInfos->used)
    return;

  DBG_fprintf(stderr, "Extension Informations: caught the 'NodePopulationDecrease'"
	      " signal, update and rebuild in list nodes case.\n");
  /* If we draw a list of nodes, then we must remove some and redraw. */
  infos = (ExtInfos*)user_data;

  if (!infos->nodes)
    return;

  /* We count the number of drawn nodes. */
  for (size = 0; infos->nodes[size] >= 0; size++);

  redraw = FALSE;
  /* We remove all old nodes. */
  for (i = 0; oldNodes[i] >= 0; i++)
    {
      /* We look for oldNodes[i] in infos->nodes. */
      for (j = 0; infos->nodes[j] != oldNodes[i] && j < size; j++);
      if (j < size)
	{
	  /* OK, found. */
	  redraw = TRUE;
	  size -= 1;
	  infos->nodes[j] = infos->nodes[size];
	  infos->nodes[size] = -1;
	}
    }
  if (redraw)
    extInfosBuild(data);
}
static void onPositionUpdate(VisuData *data, gpointer user_data _U_)
{
  if (!extInfos->used)
    return;

  DBG_fprintf(stderr, "Extension Informations: caught the 'NodePositionChanged'"
	      " signal, rebuild.\n");
  extInfosBuild(data);
}
static void onNodeRender(VisuData *data, gpointer user_data _U_)
{
  if (!extInfos->used)
    return;

  DBG_fprintf(stderr, "Extension Informations: caught the 'NodeRenderedChanged'"
	      " signal, rebuild.\n");
  extInfosBuild(data);
}
static void onElementRender(VisuData *data, VisuElement *element _U_,
			    gpointer user_data _U_)
{
  if (!extInfos->used)
    return;

  DBG_fprintf(stderr, "Extension Informations: caught the 'ElementRenderedChanged'"
	      " signal, rebuild.\n");
  extInfosBuild(data);
}
static void onCameraMoved(VisuData *data, OpenGLView *view _U_,
			  gpointer user_data _U_)
{
  if (!extInfos->used)
    return;

  DBG_fprintf(stderr, "Extension Informations: caught the 'OpenGLThetaPhiOmega'"
	      " signal, rebuild.\n");
  extInfosBuild(data);
}
