/*
 * GLX Hardware Device Driver for S3 ViRGE DX/GX/GX2
 * Copyright (C) 1999 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on Mach64 driver: mach64dma.h
 *
 *    Jim Duchek <jimduchek@ou.edu>
 */

#ifndef __S3VIRGE_DMA_H__
#define __S3VIRGE_DMA_H__

/* hardware registers and dma buffers need to be in little endien format */
#ifdef __PPC__
#define SWAP(a) _SWAP(a)
#else
#define SWAP(a) a
#endif

/* the main initialization of the entire s3virge hardware driver */
GLboolean s3virgeInitGLX( void );

void s3virgeSaveState(void);
void s3virgeRestoreState(void);

void s3virgeDmaStart( void );

/* a flush command will guarantee that all data added to the dma buffer
is on its way to the card, and will eventually complete with no more
intervention.  If running with pseudo dma, this will be the same as a finish
call, but if async dma is active then the card will be executing the commands
while the cpu is doing other work.  A protected memory region keeps X server
interaction with the hardware registers safe. */
void s3virgeDmaFlush( void );

/* the overflow function is called when a block can't be allocated
in the current dma buffer.  It flushes the current buffer and
records some information */
void s3virgeDmaOverflow( int newDwords );

/* a finish command will guarantee that all dma commands have actually
been consumed by the card, and that the engine is completely idle.  It
is safe to do software rendering after this returns */
void s3virgeDmaFinish( void );

/* for routines that need to add a variable stream of register writes, use
the DMAGETPTR() / DMAOUTREG() / DMAADVANCE() interface.  Call
s3virgeDmaGetPtr() with a length in dwords that is guaranteed to be greater
than what you will be writing.  Using a large value, like 100, does not
waste anything.  Program all the registers you need with DMAOUTREG(),
When you have finished,call DMAADVANCE(), which will add any necessary padding
and commit the data to the dma buffer

DMALOCALS must be included at the top of all functions that use these
macros to declare temporary variables.

*/

typedef struct _dma_buffer
{
    s3virgeUI32 *dmaBuf;
    s3virgeUI32	numDwords;
    s3virgeUI32 lastRead;
    s3virgeUI32 physAddr;
    s3virgeUI32 size;
} s3virgeDma_buffer;

/* cardHeap is the 8 / 16 / 32 megs of memory on the video card */
extern	memHeap_t	*cardHeap;
extern	memHeap_t	*sysmemHeap;

extern	s3virgeDma_buffer	*dma_buffer;

extern int s3virgeWaitForDmaCompletion( void );
extern void s3virgeDmaResetBuffer( void );

/* woohoo */

#define DMAGETPTR( len ) \
	do { \
		int num_free = 0; \
		while (num_free < (len+8)) { \
			dma_buffer->lastRead = INREG(S3VIRGE_CMD_DMA_READP_REG) >> 2; \
			if (dma_buffer->lastRead > dma_buffer->numDwords) { \
				num_free = dma_buffer->lastRead - dma_buffer->numDwords; \
			} else { \
				num_free = dma_buffer->size - dma_buffer->numDwords + dma_buffer->lastRead; \
			} \
		} \
	} while (0)

#define DMAOUTREG( addr, len ) \
	do { \
		dma_buffer->dmaBuf[dma_buffer->numDwords] = (len & 0xffff) | ((addr & 0xfffc) << 14); \
		dma_buffer->numDwords++; \
		if (dma_buffer->numDwords >= (dma_buffer->size)) { \
			s3virgeMsg(10, "resetting dwords.\n"); \
			dma_buffer->numDwords = 0; \
		} \
	} while (0)

#define DMAOUT( dword ) \
	do { \
		dma_buffer->dmaBuf[dma_buffer->numDwords] = dword; \
		dma_buffer->numDwords++; \
		if (dma_buffer->numDwords >= (dma_buffer->size)) { \
			s3virgeMsg(10, "resetting dwords.\n"); \
			dma_buffer->numDwords = 0; \
		} \
	} while (0)

#define DMAFINISH() \
	do { \
		OUTREG( S3VIRGE_CMD_DMA_WRITEP_REG, \
		((dma_buffer->numDwords << 2) & 0xFFFC) | \
		S3VIRGE_CMD_DMA_WRITEP_UPDATE); \
	} while (0)

#define EnsureDMAOn() \
	do { \
		if ((!(INREG(S3VIRGE_CMD_DMA_ENABLE_REG) & S3VIRGE_CMD_DMA_ENABLE)) && (s3virgeglx.dmaDriver > 0)) \
 			OUTREG(S3VIRGE_CMD_DMA_ENABLE_REG, S3VIRGE_CMD_DMA_ENABLE); \
	} while (0)

#endif
