/*
 * GLX Hardware Device Driver for S3 Savage3D and probably Savage/MX and
 * Savage/IX
 * Copyright (C) 2000 Dominik Behr
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on S3 Virge driver by Jim Duchek <jimduchek@ou.edu>
 *
 * Dominik Behr <behr@promail.pl>
 * thanks to Raja Koduri and Tim Roberts   
 */

#include <stdlib.h>
#include <stdio.h>
#include <GL/gl.h>

#include "xsmesaP.h"

#include "s3savglx.h"
#include "glx_symbols.h"

/* address of word in 16 bit texture
   tiles have 2K size
   tiles are 64 pixels wide, 128 bytes wide
   tiles are 16 pixels high
*/
static unsigned int
TextureAddr16(unsigned int uWIT, // width in tiles
              unsigned int uX,
	      unsigned int uY)
{
 unsigned int uRes;
 /* [0] = 0 - lowest bit is 0
    [1..6] = uX[0..5]
    [7,8,9,10] = uY[0..3]
    and then tile number is (uX >> 6) + uWIT * (uY >> 4)
    
    hell, its even more stupid because of subtiles
 */    
 uRes = (uX & 0x3) << 1;
 uRes |= (uY & 0x7) << 3;
 uRes |= ((uX >> 2) & 0xF) << 6;
 uRes |= ((uY >> 3) & 0x1) << 10;
 uRes |= ((uX >> 6) + (uY >> 4) * uWIT) << 11;
 return uRes;
}	      


static unsigned int
TextureAddrSub16(unsigned int uWIST, // width in sub tiles
                 unsigned int uX,
	         unsigned int uY)
{
 unsigned int uRes;
 /*
  how to generate address in this mode
  x[0..1] -> a[1..2]
  y[0..2] -> a[3..5]
  subtile number is x[2..] + y[3..] * width_in_subtiles       
 */	      
      
 uRes = (uX & 3) << 1;
 uRes |= (uY & 7) << 3;
 uRes |= ((uX >> 2) + (uY >> 3) * uWIST) << 6;

 return uRes;
}	      

static unsigned int
TextureSize16(unsigned int nWidth,
              unsigned int nHeight)
{
 /* this is special case for duplicated textures */
 if (nWidth <= 4
     && nHeight <= 4)
    return 0;
 /* also when smaller that 8 it has to be duplicated */    
 if (nWidth < 8)
    nWidth = 8;
 if (nHeight < 8)
    nHeight = 8;
 return nWidth * nHeight * 2;
}	      

static unsigned int 
TotalTextureSize16(unsigned int nWidth,
                   unsigned int nHeight)
{
 unsigned int nRes = 0;
 unsigned int nSize;

 do {
  nSize = TextureSize16(nWidth, nHeight);
  if (nSize)
     nRes += nSize;
  else 
     {
      // 3 last mipmap levels are in two 4x8 subtiles
      nRes += 8*8*2;
      break;
     }     
  
  if (nWidth > 1)
     nWidth >>= 1;
  if (nHeight > 1)
     nHeight >>= 1;

 } while (nWidth != 1 && nHeight != 1);

 // align to 2k
 nRes += 0x7FF;
 nRes &= 0xFFFFF800;
 return nRes;
}


static int 
Log2(unsigned int a) 
{
 #if 0
 int nRes;

 nRes = 0;
 while (a > 1)
       {
        nRes++;
        a >>= 1;
       }
 return nRes;
 #endif

 unsigned i;
	
 for (i = 0; i < 32; i++) 
     {
	  if ((1<<i) >= a) 
         {
		  return i;
		 } 	
	 }
 return 31;
}

#define S3SAVTO565(r,g,b) ((((hwUI16)(r) >> 3) << 11) | (((hwUI16)(g) >> 2) << 5) | ((hwUI16)(b) >> 3))
#define S3SAVTO4444(r,g,b,a) ((((hwUI16)(a) >> 4) << 12) | (((hwUI16)(r) >> 4) << 8) | (((hwUI16)(g) >> 4) << 4) | ((hwUI16)(b) >> 4))

#define CALCP p = (hwUI16 *)(uAddr + TextureAddr16(uWIT, nX, nY))
/******************************************************************************

******************************************************************************/
static void
s3savUploadTiled16_565(hwUI32 uAddr,
                       struct gl_texture_image *pImg,
		       unsigned int nStartX,
		       unsigned int nStartY,
		       unsigned int nEndX,
		       unsigned int nEndY)
{
 hwUI16 *p;
 hwUI8 *pSrc;
 unsigned int nX, nY, uWIT;
 uWIT = pImg->Width >> 6;
 if (pImg->Format == GL_RGB)
    for (nY = nStartY; nY < nEndY; nY++)
        for (nX = nStartX; nX < nEndX; nX++)
            { 
	     CALCP;
	     pSrc = (hwUI8 *)pImg->Data + 3 * (nX + nY * pImg->Width);
	     *p = S3SAVTO565(pSrc[0], pSrc[1], pSrc[2]);
	    }
 else	    
 if (pImg->Format == GL_LUMINANCE
     || pImg->Format == GL_INTENSITY)	    
    for (nY = nStartY; nY < nEndY; nY++)
        for (nX = nStartX; nX < nEndX; nX++)
            { 
             CALCP;
	     pSrc = (hwUI8 *)pImg->Data + 1 * (nX + nY * pImg->Width);
	     *p = S3SAVTO565(pSrc[0], pSrc[0], pSrc[0]);
	    }
}		   

/******************************************************************************

******************************************************************************/
static void
s3savUploadTiled16_4444(hwUI32 uAddr,
                       struct gl_texture_image *pImg,
		       unsigned int nStartX,
		       unsigned int nStartY,
		       unsigned int nEndX,
		       unsigned int nEndY)
{
 hwUI16 *p;
 hwUI8 *pSrc;
 unsigned int nX, nY, uWIT;
 uWIT = pImg->Width >> 6;
 if (pImg->Format == GL_RGBA)
    for (nY = nStartY; nY < nEndY; nY++)
        for (nX = nStartX; nX < nEndX; nX++)
            {
             CALCP;
	     pSrc = ((hwUI8 *)(pImg->Data)) + 4 * (nX + nY * pImg->Width);
	     *p = S3SAVTO4444(pSrc[0], pSrc[1], pSrc[2], pSrc[3]);
	    }
 else	    
 if (pImg->Format == GL_ALPHA)	    
    for (nY = nStartY; nY < nEndY; nY++)
        for (nX = nStartX; nX < nEndX; nX++)
            {
             CALCP;
	     pSrc = ((hwUI8 *)(pImg->Data)) + 1 * (nX + nY * pImg->Width);
	     *p = S3SAVTO4444(255, 255, 255, pSrc[0]);
	    }
 else	    
 if (pImg->Format == GL_LUMINANCE_ALPHA)	    
    for (nY = nStartY; nY < nEndY; nY++)
        for (nX = nStartX; nX < nEndX; nX++)
            {
             CALCP;
	     pSrc = ((hwUI8 *)(pImg->Data)) + 2 * (nX + nY * pImg->Width);
	     *p = S3SAVTO4444(pSrc[0], pSrc[0], pSrc[0], pSrc[1]);
	    }	    
}

#undef CALCP
#define CALCP p = (hwUI16 *)(uAddr + TextureAddrSub16(uWIST, nX, nY))

/******************************************************************************

******************************************************************************/
static void
s3savUploadSubTiled16_565(hwUI32 uAddr,
                          struct gl_texture_image *pImg,
  		          unsigned int nStartX,
		          unsigned int nStartY,
		          unsigned int nEndX,
		          unsigned int nEndY)
{
 hwUI16 *p;
 hwUI8 *pSrc;
 unsigned int nX, nY, uWIST;
 uWIST = pImg->Width >> 2;
 if (uWIST < 1)
    uWIST = 1;
 if (pImg->Format == GL_RGB)
    for (nY = nStartY; nY < nEndY; nY++)
        for (nX = nStartX; nX < nEndX; nX++)
            { 
	     CALCP;
	     pSrc = (hwUI8 *)pImg->Data + 3 * (nX + nY * pImg->Width);
	     *p = S3SAVTO565(pSrc[0], pSrc[1], pSrc[2]);
	    }
 else	    
 if (pImg->Format == GL_LUMINANCE
     || pImg->Format == GL_INTENSITY)	    
    for (nY = nStartY; nY < nEndY; nY++)
        for (nX = nStartX; nX < nEndX; nX++)
            { 
             CALCP;
	     pSrc = (hwUI8 *)pImg->Data + 1 * (nX + nY * pImg->Width);
	     *p = S3SAVTO565(pSrc[0], pSrc[0], pSrc[0]);
	    }	    
}		   

/******************************************************************************

******************************************************************************/
static void
s3savUploadSubTiled16_4444(hwUI32 uAddr,
                           struct gl_texture_image *pImg,
  		           unsigned int nStartX,
		           unsigned int nStartY,
		           unsigned int nEndX,
		           unsigned int nEndY)
{
 hwUI16 *p;
 hwUI8 *pSrc;
 unsigned int nX, nY, uWIST;
 uWIST = pImg->Width >> 2;
 if (uWIST < 1)
    uWIST = 1;
 if (pImg->Format == GL_RGBA)
    for (nY = nStartY; nY < nEndY; nY++)
        for (nX = nStartX; nX < nEndX; nX++)
            { 
             CALCP;
	     pSrc = ((hwUI8 *)(pImg->Data)) + 4 * (nX + nY * pImg->Width);
	     *p = S3SAVTO4444(pSrc[0], pSrc[1], pSrc[2], pSrc[3]);
	    }
 else	    
 if (pImg->Format == GL_ALPHA)	    
    for (nY = nStartY; nY < nEndY; nY++)
        for (nX = nStartX; nX < nEndX; nX++)
            {
             CALCP;
	     pSrc = ((hwUI8 *)(pImg->Data)) + 1 * (nX + nY * pImg->Width);
	     *p = S3SAVTO4444(255, 255, 255, pSrc[0]);
	    }
 else	    
 if (pImg->Format == GL_LUMINANCE_ALPHA)	    
    for (nY = nStartY; nY < nEndY; nY++)
        for (nX = nStartX; nX < nEndX; nX++)
            {
             CALCP;
	     pSrc = ((hwUI8 *)(pImg->Data)) + 2 * (nX + nY * pImg->Width);
	     *p = S3SAVTO4444(pSrc[0], pSrc[0], pSrc[0], pSrc[1]);
	    }	    
}

#undef CALCP
#define CALCP p = (hwUI16 *)(uAddr + TextureAddrSub16(uWIST, (nSX << uFacH) + nX, (nSY << uFacV) + nY))

/******************************************************************************

******************************************************************************/
static void
s3savUploadScaled16_565(hwUI32 uAddr,
                        struct gl_texture_image *pImg)
{
 hwUI16 *p;
 hwUI8 *pSrc;
 unsigned int nX, nY, nSX, nSY, uWIST;
 hwUI32 uScaleH, uScaleV;
 hwUI32 uWidth, uHeight;
 hwUI32 uFacH, uFacV;
 hwUI16 uVal;
 
 if (pImg->Width >= 8)
    uWidth = pImg->Width;
 else
    uWidth = 8;    
 if (pImg->Height >= 8)
    uHeight = pImg->Height;
 else
    uHeight = 8;
 uScaleH = uWidth / pImg->Width;
 uScaleV = uHeight / pImg->Height;    
 uFacH = Log2(uScaleH);
 uFacV = Log2(uScaleV);
 
 uWIST = uWidth >> 2;
 if (pImg->Format == GL_RGB)
    for (nY = 0; nY < pImg->Height; nY++)
        for (nX = 0; nX < pImg->Width; nX++)
            {
	     pSrc = (hwUI8 *)pImg->Data + 3 * (nX + nY * pImg->Width);
	     uVal = S3SAVTO565(pSrc[0], pSrc[1], pSrc[2]);
	     for (nSY = 0; nSY < uScaleV; nSY++)
   	         for (nSX = 0; nSX < uScaleH; nSX++)
		     {
		      CALCP;
		      *p = uVal;
		     }
	    }
 else	    
 if (pImg->Format == GL_LUMINANCE
     || pImg->Format == GL_INTENSITY)	    
    for (nY = 0; nY < pImg->Height; nY++)
        for (nX = 0; nX < pImg->Width; nX++)
            {
	     pSrc = (hwUI8 *)pImg->Data + 1 * (nX + nY * pImg->Width);
	     uVal = S3SAVTO565(pSrc[0], pSrc[0], pSrc[0]);
	     for (nSY = 0; nSY < uScaleV; nSY++)
   	         for (nSX = 0; nSX < uScaleH; nSX++)
		     {
                      CALCP;
		      *p = uVal;
		     }
	    } 
}

/******************************************************************************

******************************************************************************/
static void
s3savUploadScaled16_4444(hwUI32 uAddr,
                         struct gl_texture_image *pImg)
{
 hwUI16 *p;
 hwUI8 *pSrc;
 unsigned int nX, nY, nSX, nSY, uWIST;
 hwUI32 uScaleH, uScaleV;
 hwUI32 uWidth, uHeight;
 hwUI32 uFacH, uFacV;
 hwUI16 uVal;
 
 if (pImg->Width >= 8)
    uWidth = pImg->Width;
 else
    uWidth = 8;    
 if (pImg->Height >= 8)
    uHeight = pImg->Height;
 else
    uHeight = 8;
 uScaleH = uWidth / pImg->Width;
 uScaleV = uHeight / pImg->Height;    
 uFacH = Log2(uScaleH);
 uFacV = Log2(uScaleV);
 
 uWIST = uWidth >> 2;

 if (pImg->Format == GL_RGBA)
    for (nY = 0; nY < pImg->Height; nY++)
        for (nX = 0; nX < pImg->Width; nX++)
            {
	     pSrc = (hwUI8 *)pImg->Data + 4 * (nX + nY * pImg->Width);
	     uVal = S3SAVTO4444(pSrc[0], pSrc[1], pSrc[2], pSrc[3]);
	     for (nSY = 0; nSY < uScaleV; nSY++)
   	         for (nSX = 0; nSX < uScaleH; nSX++)
		     {
                      CALCP;
		      *p = uVal;
		     }
	    }
 else	    
 if (pImg->Format == GL_ALPHA)	    
    for (nY = 0; nY < pImg->Height; nY++)
        for (nX = 0; nX < pImg->Width; nX++)
            {
	     pSrc = (hwUI8 *)pImg->Data + 1 * (nX + nY * pImg->Width);
	     uVal = S3SAVTO4444(255, 255, 255, pSrc[0]);
	     for (nSY = 0; nSY < uScaleV; nSY++)
   	         for (nSX = 0; nSX < uScaleH; nSX++)
		     {
                      CALCP;
		      *p = uVal;
		     }
	    }
 else	    
 if (pImg->Format == GL_LUMINANCE_ALPHA)	    
    for (nY = 0; nY < pImg->Height; nY++)
        for (nX = 0; nX < pImg->Width; nX++)
            {
	     pSrc = (hwUI8 *)pImg->Data + 2 * (nX + nY * pImg->Width);
	     uVal = S3SAVTO4444(pSrc[0], pSrc[0], pSrc[0], pSrc[1]);
	     for (nSY = 0; nSY < uScaleV; nSY++)
   	         for (nSX = 0; nSX < uScaleH; nSX++)
		     {
		      CALCP;
		      *p = uVal;
		     }
	    }
}

#undef CALCP

#define CALCP p = (hwUI16 *)(uAddr + (((nSY << uFacV) + nY) << 3) + (((nSX << uFacH) + nX) << 1))

/******************************************************************************

******************************************************************************/
static void
s3savUploadTiny16_565(hwUI32 uAddr,
                      struct gl_texture_image *pImg)
{
 hwUI16 *p;
 hwUI8 *pSrc;
 unsigned int nX, nY, nSX, nSY;
 hwUI32 uScaleH, uScaleV;
 hwUI32 uWidth, uHeight;
 hwUI32 uFacH, uFacV;
 hwUI16 uVal;
 if (pImg->Width == 4
     || pImg->Height == 4)
    uAddr += 0;
 else     
 if (pImg->Width == 2
     || pImg->Height == 2)
    uAddr += 0x20;
 else
    uAddr += 0x30;         
 
 uWidth = 4;
 uHeight = 4;
 uScaleH = uWidth / pImg->Width;
 uScaleV = uHeight / pImg->Height;    
 uFacH = Log2(pImg->Width);
 uFacV = Log2(pImg->Height);

 if (pImg->Format == GL_RGB)
    for (nY = 0; nY < pImg->Height; nY++)
        for (nX = 0; nX < pImg->Width; nX++)
            {
	     pSrc = (hwUI8 *)pImg->Data + 3 * (nX + nY * pImg->Width);
	     uVal = S3SAVTO565(pSrc[0], pSrc[1], pSrc[2]);
	     for (nSY = 0; nSY < uScaleV; nSY++)
   	         for (nSX = 0; nSX < uScaleH; nSX++)
		     {
		      CALCP;
		      *p = uVal;
		     }
	    }
 else	    
 if (pImg->Format == GL_LUMINANCE
     || pImg->Format == GL_INTENSITY)	    
    for (nY = 0; nY < pImg->Height; nY++)
        for (nX = 0; nX < pImg->Width; nX++)
            {
	     pSrc = (hwUI8 *)pImg->Data + 1 * (nX + nY * pImg->Width);
	     uVal = S3SAVTO565(pSrc[0], pSrc[0], pSrc[0]);
	     for (nSY = 0; nSY < uScaleV; nSY++)
   	         for (nSX = 0; nSX < uScaleH; nSX++)
		     {
                      CALCP;
		      *p = uVal;
		     }
	    } 
}

/******************************************************************************

******************************************************************************/
static void
s3savUploadTiny16_4444(hwUI32 uAddr,
                       struct gl_texture_image *pImg)
{
 hwUI16 *p;
 hwUI8 *pSrc;
 unsigned int nX, nY, nSX, nSY;
 hwUI32 uScaleH, uScaleV;
 hwUI32 uWidth, uHeight;
 hwUI32 uFacH, uFacV;
 hwUI16 uVal;
 if (pImg->Width == 4
     || pImg->Height == 4)
    uAddr += 0;
 else     
 if (pImg->Width == 2
     || pImg->Height == 2)
    uAddr += 0x20;
 else
    uAddr += 0x30;         
 
 uWidth = 4;
 uHeight = 4;
 uScaleH = uWidth / pImg->Width;
 uScaleV = uHeight / pImg->Height;    
 uFacH = Log2(pImg->Width);
 uFacV = Log2(pImg->Height);


 if (pImg->Format == GL_RGBA)
    for (nY = 0; nY < pImg->Height; nY++)
        for (nX = 0; nX < pImg->Width; nX++)
            {
	     pSrc = (hwUI8 *)pImg->Data + 4 * (nX + nY * pImg->Width);
	     uVal = S3SAVTO4444(pSrc[0], pSrc[1], pSrc[2], pSrc[3]);
	     for (nSY = 0; nSY < uScaleV; nSY++)
   	         for (nSX = 0; nSX < uScaleH; nSX++)
		     {
                      CALCP;
		      *p = uVal;
		     }
	    }
 else	    
 if (pImg->Format == GL_ALPHA)	    
    for (nY = 0; nY < pImg->Height; nY++)
        for (nX = 0; nX < pImg->Width; nX++)
            {
	     pSrc = (hwUI8 *)pImg->Data + 1 * (nX + nY * pImg->Width);
	     uVal = S3SAVTO4444(255, 255, 255, pSrc[0]);
	     for (nSY = 0; nSY < uScaleV; nSY++)
   	         for (nSX = 0; nSX < uScaleH; nSX++)
		     {
                      CALCP;
		      *p = uVal;
		     }
	    }
 else	    
 if (pImg->Format == GL_LUMINANCE_ALPHA)	    
    for (nY = 0; nY < pImg->Height; nY++)
        for (nX = 0; nX < pImg->Width; nX++)
            {
	     pSrc = (hwUI8 *)pImg->Data + 2 * (nX + nY * pImg->Width);
	     uVal = S3SAVTO4444(pSrc[0], pSrc[0], pSrc[0], pSrc[1]);
	     for (nSY = 0; nSY < uScaleV; nSY++)
   	         for (nSX = 0; nSX < uScaleH; nSX++)
		     {
                      CALCP;
		      *p = uVal;
		     }
	    }
}
#undef CALCP
		   
/******************************************************************************

******************************************************************************/
static void 
s3savUploadSubImage(PS3SAVTEXTURE pTex, 
                    int nLevel,
		    int nX, 
                    int nY, 
                    int nWidth, 
                    int nHeight) 
{
 struct gl_texture_image *pImg;
 unsigned int nEndX, nEndY;
 pImg = pTex->pMT->Image[nLevel];
 
 if (!pImg) 
    return;
 /* yes, it really does happen in quake 3
    pImg->Data is fuckin 0 !!!
 */    
 if (!pImg->Data)
    return;    
 if (!pTex->pMemBlock)
    return;    

 {
  unsigned int uOfs = stS3Sav.uVideoMemoryPtr + pTex->pMemBlock->ofs;
  unsigned int uL;
  for (uL = 0; uL < nLevel; uL++)
      {
       struct gl_texture_image *pImg = pTex->pMT->Image[uL];
       uOfs += TextureSize16(pImg->Width,
                             pImg->Height);
      }
  nEndX = nX + nWidth;
  if (nEndX > pImg->Width)
     nEndX = pImg->Width;     
  nEndY = nY + nHeight;     
  if (nEndY > pImg->Height)
     nEndY = pImg->Height;          
  if (pImg->Width >= 64 
      && pImg->Height >= 16)
     {
      if (pTex->uFormat == S3SAVTEXFMT_RGB565) 
         s3savUploadTiled16_565(uOfs,
                                pImg,
				nX, 
				nY,
				nEndX,
				nEndY); 
      else
      if (pTex->uFormat == S3SAVTEXFMT_ARGB4444) 
         s3savUploadTiled16_4444(uOfs,
                                 pImg,
				 nX, 
				 nY,
				 nEndX,
				 nEndY); 			       
     }
  else     			
  if (pImg->Width >= 8
      || pImg->Height >= 8)
     {     
      if (pTex->uFormat == S3SAVTEXFMT_RGB565) 
         s3savUploadSubTiled16_565(uOfs,
                                   pImg,
  				   nX, 
				   nY,
				   nEndX,
				   nEndY); 
      else
      if (pTex->uFormat == S3SAVTEXFMT_ARGB4444) 
         s3savUploadSubTiled16_4444(uOfs,
                                    pImg,
				    nX, 
				    nY,
				    nEndX,
				    nEndY);
     }
#if 0
  else     			
  if (pImg->Width >= 8
      || pImg->Height >= 8)
     {     
      if (pTex->uFormat == S3SAVTEXFMT_RGB565) 
         s3savUploadScaled16_565(uOfs,
                                   pImg);
      else
      if (pTex->uFormat == S3SAVTEXFMT_ARGB4444) 
         s3savUploadScaled16_4444(uOfs,
                                    pImg);
     }     
#endif     
#if 1
  else     
     {
      if (pTex->uFormat == S3SAVTEXFMT_RGB565) 
         s3savUploadTiny16_565(uOfs,
                               pImg);
      else
      if (pTex->uFormat == S3SAVTEXFMT_ARGB4444) 
         s3savUploadTiny16_4444(uOfs,
                                pImg);
     }     
#endif     
 }     
}


/******************************************************************************
 create texture object and upload it to video memory
 when DriverData isnt NULL texture is resident
******************************************************************************/
void 
s3savCreateTexObj(PS3SAVCONTEXT ctx, 
                  struct gl_texture_object *tObj) 
{
 PS3SAVTEXTURE pTex;
 struct gl_texture_image *pImg;
 unsigned int nLevel;
 
 if (!tObj->Image[0])
    return;
 
 pImg = tObj->Image[0];   
 
 pTex = (PS3SAVTEXTURE)malloc(sizeof(S3SAVTEXTURE));
 pTex->pMT = tObj;
 // FIXME alloc memory here
 // for now we assume textures will be 16bpp only
 // also we always alloc memory for mipmaps
 pTex->pMemBlock = s3savMakeRoom(TotalTextureSize16(pImg->Width, pImg->Height), 8);
 if (!pTex->pMemBlock)
    {
     fprintf(stderr, "[s3sav] cannot allocate %d KB for texture\n",
             TotalTextureSize16(pImg->Width, pImg->Height));
     free((void *)pTex);
     tObj->DriverData = 0;
     return;
    }
 #if 0
 {
  unsigned short *p = (unsigned short *)(pTex->pMemBlock->ofs + stS3Sav.uVideoMemoryPtr);
  unsigned int n = pImg->Width * pImg->Height;
  while (n--)
        *p++ = 0xFFFF;
 }
 #endif
 pTex->pCtx = ctx;
 pTex->nTextureID = 0xFFFFFFFF;
 pTex->pNext = stS3Sav.pTextures;
 stS3Sav.pTextures = pTex;
 tObj->DriverData = pTex; 
 
 switch (pImg->Format) {
  case GL_COLOR_INDEX:
  case GL_INTENSITY:
  case GL_LUMINANCE:
   pTex->uFormat = S3SAVTEXFMT_RGB565;
   break;
  case GL_ALPHA:
  case GL_LUMINANCE_ALPHA:
   pTex->uFormat = S3SAVTEXFMT_ARGB4444;
   break;
  case GL_RGB:
   pTex->uFormat = S3SAVTEXFMT_RGB565;
   break;
  case GL_RGBA:
   pTex->uFormat = S3SAVTEXFMT_ARGB4444;
   break;
  default:
 }
 if (pImg->Width > pImg->Height)
    pTex->nLevels = Log2(pImg->Width);
 else    
    pTex->nLevels = Log2(pImg->Height);
 pTex->nLevels++;
 for (nLevel = 0; nLevel < pTex->nLevels; nLevel++)
     {
      if (pImg = tObj->Image[nLevel])
         s3savUploadSubImage(pTex, 
                             nLevel,
			     0,
			     0,
			     pImg->Width,
			     pImg->Height);
     }
}


/******************************************************************************

******************************************************************************/
static void 
s3savDestroyTexObj(PS3SAVCONTEXT pCtx, 
                   PS3SAVTEXTURE pTex) 
{
 PS3SAVTEXTURE pPrev, p;
 
 if (!pTex)
    return;
	
 /* free the texture memory */
 mmFreeMem(pTex->pMemBlock);
 pTex->pMemBlock = 0;
 
 /* free mesa's link */   
 pTex->pMT->DriverData = 0;	
    
 /* remove from the driver texobj list */
 pPrev = 0;
 for (p = stS3Sav.pTextures; p; p = p->pNext)
     {
      if (p == pTex)
         {
          if (pPrev)
             pPrev->pNext = p->pNext;
          else
             stS3Sav.pTextures = p->pNext;
          break;
         }
      pPrev = p;
     }
	
 free(pTex);
}


/******************************************************************************

******************************************************************************/
void 
s3savTexImage(GLcontext *ctx, 
              GLenum target,
              struct gl_texture_object *tObj, 
              GLint level,
              GLint internalFormat,
              const struct gl_texture_image *image) 
{
 PS3SAVTEXTURE pTex;

 pTex = (PS3SAVTEXTURE)tObj->DriverData;
 if (pTex && level == 0)
    {
     /* texture data has changed, zip it */
     s3savDestroyTexObj(stS3Sav.pContext, pTex);
    }
 
 s3savCreateTexObj(stS3Sav.pContext, tObj);
}

/******************************************************************************

******************************************************************************/
void 
s3savTexSubImage(GLcontext *ctx, 
                 GLenum target,
		 struct gl_texture_object *tObj, 
                 GLint level,
		 GLint xoffset, 
                 GLint yoffset,
		 GLsizei width, 
                 GLsizei height,
		 GLint internalFormat,
		 const struct gl_texture_image *image) 
{
 PS3SAVTEXTURE pTex;

 pTex = (PS3SAVTEXTURE)tObj->DriverData;
 if (!pTex)
    return;

 s3savUploadSubImage(pTex, 
                     level, 
                     xoffset, 
                     yoffset, 
                     width, 
                     height);
}

/******************************************************************************

******************************************************************************/
void
s3savLoadTexturePalette(hwUI8 *pal)
{
 // we do not support no lousy paletted textures (yet ?)
 return;
}

/******************************************************************************

******************************************************************************/
void
s3savUpdateTexturePalette(GLcontext *ctx, 
                          struct gl_texture_object *tObj) 
{
 return;
}


/******************************************************************************

******************************************************************************/
void 
s3savDeleteTexture(GLcontext *ctx, 
                   struct gl_texture_object *tObj) 
{
 if (tObj->DriverData) 
    {
     s3savDestroyTexObj(stS3Sav.pContext, 
                        (PS3SAVTEXTURE)(tObj->DriverData));
    }
}

/******************************************************************************

******************************************************************************/
GLboolean 
s3savIsTextureResident(GLcontext *ctx,
                       struct gl_texture_object *tObj) 
{
 GLboolean is;
	
 is = (tObj->DriverData != NULL);

 return is;
}
