/*********************************************************************
 *
 * This is based on code created by Peter Harvey,
 * (pharvey@codebydesign.com).
 *
 * Modified and extended by Nick Gorham
 * (nick@easysoft.com).
 *
 * Any bugs or problems should be considered the fault of Nick and not
 * Peter.
 *
 * copyright (c) 1999 Nick Gorham
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **********************************************************************
 *
 * $Id: SQLDriverConnect.c,v 1.2 2001/12/13 13:00:32 lurcher Exp $
 *
 * $Log: SQLDriverConnect.c,v $
 * Revision 1.2  2001/12/13 13:00:32  lurcher
 *
 * Remove most if not all warnings on 64 bit platforms
 * Add support for new MS 3.52 64 bit changes
 * Add override to disable the stopping of tracing
 * Add MAX_ROWS support in postgres driver
 *
 * Revision 1.1.1.1  2001/10/17 16:40:05  lurcher
 *
 * First upload to SourceForge
 *
 * Revision 1.22  2001/10/16 10:37:32  nick
 *
 * Getting ready for 2.0.10
 *
 * Revision 1.21  2001/10/09 13:23:30  nick
 *
 * Add filedsn support to ODBCConfig
 *
 * Revision 1.20  2001/10/08 13:38:35  nick
 *
 * Add support for FILEDSN's
 *
 * Revision 1.19  2001/08/08 17:05:17  nick
 *
 * Add support for attribute setting in the ini files
 *
 * Revision 1.18  2001/07/20 13:20:44  nick
 * *** empty log message ***
 *
 * Revision 1.17  2001/07/20 12:35:09  nick
 *
 * Fix SQLBrowseConnect operation
 *
 * Revision 1.16  2001/05/15 10:57:44  nick
 *
 * Add initial support for VMS
 *
 * Revision 1.15  2001/04/18 15:03:37  nick
 *
 * Fix problem when going to DB2 unicode driver
 *
 * Revision 1.14  2001/04/16 22:35:10  nick
 *
 * More tweeks to the AutoTest code
 *
 * Revision 1.13  2001/04/16 15:41:24  nick
 *
 * Fix some problems calling non existing error funcs
 *
 * Revision 1.12  2001/04/12 17:43:36  nick
 *
 * Change logging and added autotest to odbctest
 *
 * Revision 1.11  2001/04/03 16:34:12  nick
 *
 * Add support for strangly broken unicode drivers
 *
 * Revision 1.10  2001/01/03 12:02:03  nick
 *
 * Add missing __
 *
 * Revision 1.9  2001/01/03 11:57:26  nick
 *
 * Fix some name collisions
 *
 * Revision 1.8  2000/12/31 20:30:54  nick
 *
 * Add UNICODE support
 *
 * Revision 1.7  2000/12/18 13:02:13  nick
 *
 * More buf fixes
 *
 * Revision 1.6  2000/12/18 12:53:29  nick
 *
 * More pooling tweeks
 *
 * Revision 1.5  2000/12/18 12:27:50  nick
 *
 * Fix missing check for SQL_NTS
 *
 * Revision 1.4  2000/12/14 18:10:19  nick
 *
 * Add connection pooling
 *
 * Revision 1.3  2000/10/13 15:18:49  nick
 *
 * Change string length parameter from SQLINTEGER to SQLSMALLINT
 *
 * Revision 1.2  2000/10/06 08:49:38  nick
 *
 * Fix duplicated error messages on connect
 *
 * Revision 1.1.1.1  2000/09/04 16:42:52  nick
 * Imported Sources
 *
 * Revision 1.15  2000/08/11 12:11:27  ngorham
 *
 * Make SQLDriverConnect return all the stacked errors from the driver, not
 * just one
 *
 * Revision 1.14  2000/07/13 13:27:24  ngorham
 *
 * remove _ from odbcinst_system_file_path()
 *
 * Revision 1.13  2000/06/21 08:58:26  ngorham
 *
 * Stop SQLDriverConnect dumping core when passed a null dsn string
 *
 * Revision 1.12  2000/02/20 10:18:47  ngorham
 *
 * Add support for ODBCINI environment override for Applix.
 *
 * Revision 1.11  1999/12/14 19:02:25  ngorham
 *
 * Mask out the password fields in the logging
 *
 * Revision 1.10  1999/11/13 23:40:59  ngorham
 *
 * Alter the way DM logging works
 * Upgrade the Postgres driver to 6.4.6
 *
 * Revision 1.9  1999/10/24 23:54:17  ngorham
 *
 * First part of the changes to the error reporting
 *
 * Revision 1.8  1999/10/14 06:49:24  ngorham
 *
 * Remove @all_includes@ from Drivers/MiniSQL/Makefile.am
 *
 * Revision 1.7  1999/09/21 22:34:24  ngorham
 *
 * Improve performance by removing unneeded logging calls when logging is
 * disabled
 *
 * Revision 1.6  1999/08/17 06:20:00  ngorham
 *
 * Remove posibility of returning without clearing the connection mutex.
 *
 * Revision 1.5  1999/08/03 21:47:39  shandyb
 * Moving to automake: changed files in DriverManager
 *
 * Revision 1.4  1999/07/10 21:10:16  ngorham
 *
 * Adjust error sqlstate from driver manager, depending on requested
 * version (ODBC2/3)
 *
 * Revision 1.3  1999/07/04 21:05:07  ngorham
 *
 * Add LGPL Headers to code
 *
 * Revision 1.2  1999/06/30 23:56:54  ngorham
 *
 * Add initial thread safety code
 *
 * Revision 1.1.1.1  1999/05/29 13:41:05  sShandyb
 * first go at it
 *
 * Revision 1.1.1.1  1999/05/27 18:23:17  pharvey
 * Imported sources
 *
 * Revision 1.4  1999/05/09 23:27:11  nick
 * All the API done now
 *
 * Revision 1.3  1999/04/30 16:22:47  nick
 * Another checkpoint
 *
 * Revision 1.2  1999/04/29 20:47:37  nick
 * Another checkpoint
 *
 * Revision 1.1  1999/04/25 23:06:11  nick
 * Initial revision
 *
 *
 **********************************************************************/

#include <string.h>
#include "drivermanager.h"

static char const rcsid[]= "$RCSfile: SQLDriverConnect.c,v $ $Revision: 1.2 $";

/*
 * connection pooling stuff
 */

extern int pooling_enabled;

void __generate_connection_string( struct con_struct *con_str, char *str, int str_len )
{
struct con_pair *cp;
char tmp[ 1024 ];

    str[ 0 ] = '\0';

    if ( con_str -> count == 0 )
    {
        return;
    }

    cp = con_str -> list;
    while( cp )
    {
        if( strcasecmp( cp -> keyword, "DRIVER" ) == 0 )
        {
            sprintf( tmp, "%s={%s};", cp -> keyword, cp -> attribute );
        }
        else
        {
            sprintf( tmp, "%s=%s;", cp -> keyword, cp -> attribute );
        }

        if ( strlen( str ) + strlen( tmp ) > str_len )
        {
            break;
        }
        else
        {
            strcat( str, tmp );
        }
                                
        cp = cp -> next;
    }
}

void __get_attr( char ** cp, char ** keyword, char ** value )
{
char * ptr;
int len;

    *keyword = *value = NULL;

    ptr = *cp;

    if ( !**cp )
        return;

    while ( **cp && **cp != '=' )
    {
        (*cp)++;
    }

    if ( !**cp )
        return;

    len = *cp - ptr;
    *keyword = malloc( len + 1 );
    memcpy( *keyword, ptr, len );
    (*keyword)[ len ] = '\0';

    (*cp)++;
    ptr = *cp;

    if ( strcmp( *keyword, "DRIVER" ) == 0 )
    {
        if ( **cp && **cp == '{' )
        {
            (*cp)++;
            ptr ++;
            while ( **cp && **cp != '}' )
                (*cp)++;

            len = *cp - ptr;
            *value = malloc( len + 1 );
            memcpy( *value, ptr , len );
            (*value)[ len ] = '\0';
            (*cp)++;
        }
        else
        {
            while ( **cp && **cp != ';' )
                (*cp)++;

            len = *cp - ptr;
            *value = malloc( len + 1 );
            memcpy( *value, ptr, len );
            (*value)[ len ] = '\0';
        }
    }
    else
    {
        while ( **cp && **cp != ';' )
            (*cp)++;

        len = *cp - ptr;
        *value = malloc( len + 1 );
        memcpy( *value, ptr, len );
        (*value)[ len ] = '\0';
    }

    if ( **cp )
        (*cp)++;
}

struct con_pair * __get_pair( char ** cp )
{
char *keyword, *value;
struct con_pair * con_p;

    __get_attr( cp, &keyword, &value );
    if ( keyword )
    {
        con_p = malloc( sizeof( *con_p ));
        con_p -> keyword = keyword;
        con_p -> attribute = value;
        return con_p;
    }
    else
    {
        return NULL;
    }
}

int __append_pair( struct con_struct *con_str, char *kword, char *value )
{
struct con_pair *ptr, *end;

    /* check that the keyword is not already in the list */

    end = NULL;
    if ( con_str -> count > 0 )
    {
        ptr = con_str -> list;
        while( ptr )
        {
            if( strcmp( kword, ptr -> keyword ) == 0 )
            {
                free( ptr -> attribute );
                ptr -> attribute = malloc( strlen( value ) + 1 );
                strcpy( ptr -> attribute, value );
                return 0;
            }
            end = ptr;
            ptr = ptr -> next;
        }
    }

    ptr = malloc( sizeof( *ptr ));

    ptr -> keyword = malloc( strlen( kword ) + 1 );
    strcpy( ptr -> keyword, kword );

    ptr -> attribute = malloc( strlen( value ) + 1 );
    strcpy( ptr -> attribute, value );

    con_str -> count ++;

    if ( con_str -> list )
    {
        end -> next = ptr;
        ptr -> next = NULL;
    }
    else
    {
        ptr -> next = NULL;
        con_str -> list = ptr;
    }

    return 0;
}

int __parse_connection_string_ex( struct con_struct *con_str,
    char *str, int str_len, int exclude )
{
struct con_pair *cp;
char *local_str, *ptr;
int got_dsn = 0;    /* if we have a DSN then ignore any DRIVER or FILEDSN */
int got_driver = 0;    /* if we have a DRIVER or FILEDSN then ignore any DSN */

    con_str -> count = 0;
    con_str -> list = NULL;

    if ( str_len != SQL_NTS )
    {
        local_str = malloc( str_len + 1 );
        memcpy( local_str, str, str_len );
        local_str[ str_len ] = '\0';
    }
    else
    {
        local_str = str;
    }

    if ( !local_str || strlen( local_str ) == 0 ||
        ( strlen( local_str ) == 1 && *local_str == ';' ))
    {
        /* connection-string ::= empty-string [;] */
        if ( str_len != SQL_NTS )
            free( local_str );

        __append_pair( con_str, "DSN", "DEFAULT" );
        return 0;
    }

    ptr = local_str;

    while(( cp = __get_pair( &ptr )) != NULL )
    {
        if ( strcasecmp( cp -> keyword, "DSN" ) == 0 )
        {
            if ( got_driver && exclude )
                continue;

            got_dsn = 1;
        }
        else if ( strcasecmp( cp -> keyword, "DRIVER" ) == 0 ||
            strcmp( cp -> keyword, "FILEDSN" ) == 0 )
        {
            if ( got_dsn && exclude )
                continue;

            got_driver = 1;
        }

        __append_pair( con_str, cp -> keyword, cp -> attribute );
        free( cp -> keyword );
        free( cp -> attribute );
        free( cp );
    }

    /* if no dsn or DRIVER, then set DSN=DEFAULT */
    if ( !got_driver && !got_dsn )
    {
        __append_pair( con_str, "DSN", "DEFAULT" );
    }

    if ( str_len != SQL_NTS )
        free( local_str );

    return 0;
}

int __parse_connection_string( struct con_struct *con_str,
    char *str, int str_len )
{
    return  __parse_connection_string_ex( con_str, str, str_len, 1 );
}

char * __get_attribute_value( struct con_struct * con_str, char * keyword )
{
struct con_pair *cp;

    if ( con_str -> count == 0 )
        return NULL;

    cp = con_str -> list;
    while( cp )
    {
        if( strcasecmp( keyword, cp -> keyword ) == 0 )
        {
            if ( cp -> attribute )
                return cp -> attribute;
            else
                return "";
        }
        cp = cp -> next;
    }
    return NULL;
}

void __release_conn( struct con_struct *con_str )
{
    struct con_pair *cp = con_str -> list;
    struct con_pair *save;

    while( cp )
    {
        free( cp -> attribute );
        free( cp -> keyword );
        save = cp;
        cp = cp -> next;
        free( save );
    }

    con_str -> count = 0;
}

SQLRETURN SQLDriverConnect(
    SQLHDBC            hdbc,
    SQLHWND            hwnd,
    SQLCHAR            *conn_str_in,
    SQLSMALLINT        len_conn_str_in,
    SQLCHAR            *conn_str_out,
    SQLSMALLINT        conn_str_out_max,
    SQLSMALLINT        *ptr_conn_str_out,
    SQLUSMALLINT       driver_completion )
{
    DMHDBC connection = (DMHDBC)hdbc;
    struct con_struct con_struct;
    char *driver, *dsn, *filedsn, *tsavefile, savefile[ 128 ];
    char lib_name[ INI_MAX_PROPERTY_VALUE + 1 ];
    char driver_name[ INI_MAX_PROPERTY_VALUE + 1 ];
    SQLRETURN ret_from_connect;
    SQLCHAR s1[ 2048 ];
    SQLCHAR local_conn_str_in[ 2048 ];
    SQLCHAR local_out_conection[ 2048 ];

    /*
     * check connection
     */

    if ( !__validate_dbc( connection ))
    {
        dm_log_write( __FILE__, 
                __LINE__, 
                    LOG_INFO, 
                    LOG_INFO, 
                    "Error: SQL_INVALID_HANDLE" );

        return SQL_INVALID_HANDLE;
    }

    function_entry( connection );

    /*
     * replace this if not set for use by SAVEFILE
     */

    if ( !conn_str_out )
    {
        conn_str_out = local_out_conection;
        conn_str_out_max = sizeof( local_out_conection );
    }

    if ( log_info.log_flag )
    {
        sprintf( connection -> msg, "\n\t\tEntry:\
            \n\t\t\tConnection = %p\
            \n\t\t\tWindow Hdl = %p\
            \n\t\t\tStr In = %s\
            \n\t\t\tStr Out = %p\
            \n\t\t\tStr Out Max = %d\
            \n\t\t\tStr Out Ptr = %p\
            \n\t\t\tCompletion = %d",
                connection,
                hwnd,
                __string_with_length_hide_pwd( s1, conn_str_in, 
                    len_conn_str_in ), 
                conn_str_out,
                conn_str_out_max,
                ptr_conn_str_out,
                driver_completion );

        dm_log_write( __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                connection -> msg );
    }

    thread_protect( SQL_HANDLE_DBC, connection );

    if ( len_conn_str_in < 0 && len_conn_str_in != SQL_NTS )
    {
        dm_log_write( __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                "Error: HY090" );

        __post_internal_error( &connection -> error,
                ERROR_HY090, NULL,
                connection -> environment -> requested_version );

        thread_release( SQL_HANDLE_DBC, connection );

        return function_return( connection, SQL_ERROR );
    }

    if ( driver_completion == SQL_DRIVER_PROMPT &&
            hwnd == NULL )
    {
        dm_log_write( __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                "Error: HY092" );

        __post_internal_error( &connection -> error,
                ERROR_HY092, NULL,
                connection -> environment -> requested_version );

        thread_release( SQL_HANDLE_DBC, connection );

        return function_return( connection, SQL_ERROR );
    }

    if ( driver_completion != SQL_DRIVER_PROMPT &&
            driver_completion != SQL_DRIVER_COMPLETE &&
            driver_completion != SQL_DRIVER_COMPLETE_REQUIRED &&
            driver_completion != SQL_DRIVER_NOPROMPT )
    {
        dm_log_write( __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                "Error: HY110" );

        __post_internal_error( &connection -> error,
                ERROR_HY110, NULL,
                connection -> environment -> requested_version );

        thread_release( SQL_HANDLE_DBC, connection );

        return function_return( connection, SQL_ERROR );
    }

    /*
     * check the state of the connection
     */

    if ( connection -> state != STATE_C2 )
    {
        dm_log_write( __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                "Error: 08002" );

        __post_internal_error( &connection -> error,
                ERROR_08002, NULL,
                connection -> environment -> requested_version );

        thread_release( SQL_HANDLE_DBC, connection );

        return function_return( connection, SQL_ERROR );
    }

    /*
     * parse the connection string
     */

    if ( !conn_str_in )
    {
        /*
         * not quite by the book, but better than nothing
         */

        conn_str_in = (SQLCHAR*)"DSN=DEFAULT;";
        len_conn_str_in = strlen((char*) conn_str_in );
    }

    /*
     * can we find a pooled connection to use here ?
     */

    connection -> pooled_connection = NULL;

    if ( pooling_enabled && search_for_pool( connection, 
                                                NULL, 0,
                                                NULL, 0,
                                                NULL, 0,
                                                conn_str_in, len_conn_str_in ))
    {
        /*
         * copy the in string to the out string
         */

        ret_from_connect = SQL_SUCCESS;

        if ( conn_str_out )
        {
            if ( len_conn_str_in < 0 )
            {
                len_conn_str_in = strlen((char*) conn_str_in );
            }

            if ( len_conn_str_in >= conn_str_out_max )
            {
                memcpy( conn_str_out, conn_str_in, conn_str_out_max );
                conn_str_out[ conn_str_out_max ] = '\0';
                if ( ptr_conn_str_out )
                {
                    *ptr_conn_str_out = len_conn_str_in;
                }

                __post_internal_error( &connection -> error,
                    ERROR_01004, NULL,
                    connection -> environment -> requested_version );

                ret_from_connect = SQL_SUCCESS_WITH_INFO;
            }
            else
            {
                memcpy( conn_str_out, conn_str_in, len_conn_str_in );
                conn_str_out[ len_conn_str_in ] = '\0';
                if ( ptr_conn_str_out )
                {
                    *ptr_conn_str_out = len_conn_str_in;
                }
            }
        }

        if ( log_info.log_flag )
        {
            sprintf( connection -> msg,
                    "\n\t\tExit:[%s]",
                        __get_return_status( ret_from_connect ));

            dm_log_write( __FILE__,
                        __LINE__,
                    LOG_INFO,
                    LOG_INFO,
                    connection -> msg );
        }

        connection -> state = STATE_C4;

        thread_release( SQL_HANDLE_DBC, connection );

        return function_return( connection, ret_from_connect );
    }

    /*
     * else safe the info for later
     */

    if ( pooling_enabled )
    {
        connection -> dsn_length = 0;

        strcpy( connection -> server, "" );
        connection -> server_length = 0;
        strcpy( connection -> user, "" );
        connection -> user_length = 0;
        strcpy( connection -> password, "" );
        connection -> password_length = 0;

        if ( len_conn_str_in == SQL_NTS )
        {
            strcpy( connection -> driver_connect_string, (char*)conn_str_in );
        }
        else
        {
            memcpy( connection -> driver_connect_string, conn_str_in, len_conn_str_in );
        }
        connection -> dsn_length = len_conn_str_in;
    }


    __parse_connection_string( &con_struct,
            (char*)conn_str_in, len_conn_str_in );

    /*
     * open the file dsn, get each entry from it, if its not in the connection 
     * struct, add it
     */

    filedsn = __get_attribute_value( &con_struct, "FILEDSN" );
    if ( filedsn )
    {
        char str[ 2048 ];

        if ( SQLReadFileDSN( filedsn, "ODBC", NULL, str, sizeof( str ), NULL ))
        {
            struct con_struct con_struct1;

            if ( strlen( str ))
            {
                strcpy((char*)local_conn_str_in, (char*)conn_str_in );
                conn_str_in = local_conn_str_in;

                __parse_connection_string( &con_struct1,
                        str, strlen( str ));

                if ( con_struct1.count )
                {
                    struct con_pair *cp;

                    cp = con_struct1.list;
                    while( cp )
                    {
                        if ( !__get_attribute_value( &con_struct, cp -> keyword ))
                        {
                            char str1[ 256 ];

                            sprintf( str1, ";%s=%s", cp -> keyword, cp -> attribute );
                            strcat((char*) conn_str_in, str1 );
                        }
                        cp = cp -> next;
                    }
                }

                len_conn_str_in = strlen((char*) conn_str_in );

                __release_conn( &con_struct1 );
            }

            /*
             * reparse the string
             */

            __release_conn( &con_struct );

            __parse_connection_string( &con_struct,
                    (char*)conn_str_in, len_conn_str_in );
        }
    }

    /*
     * look for some keywords
     *
     * have we got a DRIVER= attribute
     */

    driver = __get_attribute_value( &con_struct, "DRIVER" );
    if ( driver )
    {
        /*
         * look up the driver in the ini file
         */

        strcpy( driver_name, driver );

        SQLGetPrivateProfileString( driver, "Driver", "",
                lib_name, sizeof( lib_name ), "ODBCINST.INI" );

        if ( lib_name[ 0 ] == '\0' )
        {
            /*
             * at this point a box could pop up to allow the selection of a driver
             *
             * do this later
             */

            dm_log_write( __FILE__, 
                    __LINE__, 
                    LOG_INFO, 
                    LOG_INFO, 
                    "Error: IM002" );

            __post_internal_error( &connection -> error,
                    ERROR_IM002, NULL,
                    connection -> environment -> requested_version );
            __release_conn( &con_struct );

            thread_release( SQL_HANDLE_DBC, connection );

            return function_return( connection, SQL_ERROR );
        }

        strcpy( connection -> dsn, "" );
    }
    else
    {
        dsn = __get_attribute_value( &con_struct, "DSN" );
        if ( !dsn )
        {
            dm_log_write( __FILE__, 
                    __LINE__, 
                    LOG_INFO, 
                    LOG_INFO, 
                    "Error: IM002" );

            __post_internal_error( &connection -> error,
                    ERROR_IM002, NULL,
                    connection -> environment -> requested_version );

            __release_conn( &con_struct );

            thread_release( SQL_HANDLE_DBC, connection );

            return function_return( connection, SQL_ERROR );
        }

        if ( strlen( dsn ) > SQL_MAX_DSN_LENGTH )
        {
            dm_log_write( __FILE__, 
                    __LINE__, 
                    LOG_INFO, 
                    LOG_INFO, 
                    "Error: IM012" );

            __post_internal_error( &connection -> error,
                    ERROR_IM012, NULL,
                    connection -> environment -> requested_version );

            __release_conn( &con_struct );

            thread_release( SQL_HANDLE_DBC, connection );

            return function_return( connection, SQL_ERROR );
        }

        /*
         * look up the dsn in the ini file
         */

        if ( !__find_lib_name( dsn, lib_name, driver_name ))
        {
            dm_log_write( __FILE__, 
                    __LINE__, 
                    LOG_INFO, 
                    LOG_INFO, 
                    "Error: IM002" );

            __post_internal_error( &connection -> error,
                    ERROR_IM002, NULL,
                    connection -> environment -> requested_version );

            __release_conn( &con_struct );

            thread_release( SQL_HANDLE_DBC, connection );

            return function_return( connection, SQL_ERROR );
        }

        strcpy( connection -> dsn, dsn );
        __handle_attr_extensions( connection, dsn );
    }

    /*
     * get for later
     */

    tsavefile = __get_attribute_value( &con_struct, "SAVEFILE" );
    if ( tsavefile )
    {
        strcpy( savefile, tsavefile );
    }
    __release_conn( &con_struct );

    /*
     * we have now got the name of a lib to load
     */
    if ( !__connect_part_one( connection, lib_name, driver_name ))
    {
        thread_release( SQL_HANDLE_DBC, connection );
        return function_return( connection, SQL_ERROR );
    }

    if ( !CHECK_SQLDRIVERCONNECT( connection ) &&
        !CHECK_SQLDRIVERCONNECTW( connection ))
    {
        dm_log_write( __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                "Error: IM001" );

        __disconnect_part_one( connection );
        __post_internal_error( &connection -> error,
                ERROR_IM001, NULL,
                connection -> environment -> requested_version );

        thread_release( SQL_HANDLE_DBC, connection );

        return function_return( connection, SQL_ERROR );
    }

    if ( CHECK_SQLDRIVERCONNECT( connection ))
    {
        ret_from_connect = SQLDRIVERCONNECT( connection,
                connection -> driver_dbc,
                hwnd,
                conn_str_in,
                len_conn_str_in,
                conn_str_out,
                conn_str_out_max,
                ptr_conn_str_out,
                driver_completion );

        if ( !SQL_SUCCEEDED( ret_from_connect ))
        {
            SQLCHAR sqlstate[ 6 ];
            SQLINTEGER native_error;
            SQLSMALLINT ind;
            SQLCHAR message_text[ SQL_MAX_MESSAGE_LENGTH + 1 ];
            SQLRETURN ret;

            /*
             * get the errors from the driver before
             * loseing the connection 
             */

            if ( CHECK_SQLERROR( connection ))
            {
                do
                {
                    ret = SQLERROR( connection,
                            SQL_NULL_HENV,
                            connection -> driver_dbc,
                            SQL_NULL_HSTMT,
                            sqlstate,
                            &native_error,
                            message_text,
                            sizeof( message_text ),
                            &ind );

                    if ( SQL_SUCCEEDED( ret ))
                    {
                        __post_internal_error_ex( &connection -> error,
                                sqlstate,
                                native_error,
                                message_text );

                        sprintf( connection -> msg, "\t\tDIAG [%s] %s",
                            sqlstate, message_text );

                        dm_log_write_diag( connection -> msg );
                    }
                }
                while( SQL_SUCCEEDED( ret ));
            }
            else if ( CHECK_SQLGETDIAGREC( connection ))
            {
                int rec = 1;

                do
                {
                    ret = SQLGETDIAGREC( connection,
                            SQL_HANDLE_DBC,
                            connection -> driver_dbc,
                            rec ++,
                            sqlstate,
                            &native_error,
                            message_text,
                            sizeof( message_text ),
                            &ind );


                    if ( SQL_SUCCEEDED( ret ))
                    {
                        __post_internal_error_ex( &connection -> error,
                                sqlstate,
                                native_error,
                                message_text );

                        sprintf( connection -> msg, "\t\tDIAG [%s] %s",
                            sqlstate, message_text );

                        dm_log_write_diag( connection -> msg );
                    }
                }
                while( SQL_SUCCEEDED( ret ));
            }
        }


        /* 
         * if it was a error then return now
         */

        if ( !SQL_SUCCEEDED( ret_from_connect ))
        {
            __disconnect_part_one( connection );

            thread_release( SQL_HANDLE_DBC, connection );

            sprintf( connection -> msg,
                    "\n\t\tExit:[%s]",
                        __get_return_status( ret_from_connect ));

            dm_log_write( __FILE__,
                    __LINE__,
                    LOG_INFO,
                    LOG_INFO,
                    connection -> msg );

            return function_return( connection, ret_from_connect );
        }
        connection -> unicode_driver = 0;
    }
    else
    {
        SQLWCHAR *uc_conn_str_in;

        uc_conn_str_in = ansi_to_unicode( conn_str_in, len_conn_str_in );

        ret_from_connect = SQLDRIVERCONNECTW( connection,
                connection -> driver_dbc,
                hwnd,
                uc_conn_str_in,
                len_conn_str_in,
                conn_str_out,
                conn_str_out_max / 2,
                ptr_conn_str_out,
                driver_completion );

        if ( uc_conn_str_in )
            free( uc_conn_str_in );

        if ( !SQL_SUCCEEDED( ret_from_connect ))
        {
            SQLWCHAR sqlstate[ 6 ];
            SQLINTEGER native_error;
            SQLSMALLINT ind;
            SQLWCHAR message_text[ SQL_MAX_MESSAGE_LENGTH + 1 ];
            SQLRETURN ret;

            /*
             * get the errors from the driver before
             * loseing the connection 
             */

            if ( CHECK_SQLERRORW( connection ))
            {
                do
                {
                    ret = SQLERRORW( connection,
                            SQL_NULL_HENV,
                            connection -> driver_dbc,
                            SQL_NULL_HSTMT,
                            sqlstate,
                            &native_error,
                            message_text,
                            sizeof( message_text ),
                            &ind );


                    if ( SQL_SUCCEEDED( ret ))
                    {
                        __post_internal_error_ex_w( &connection -> error,
                                sqlstate,
                                native_error,
                                message_text );

                        unicode_to_ansi( sqlstate, SQL_NTS );
                        unicode_to_ansi( message_text, SQL_NTS );

                        sprintf( connection -> msg, "\t\tDIAG [%s] %s",
                                sqlstate, message_text );

                        dm_log_write_diag( connection -> msg );
                    }
                }
                while( SQL_SUCCEEDED( ret ));
            }
            else if ( CHECK_SQLGETDIAGRECW( connection ))
            {
                int rec = 1;

                do
                {
                    ret = SQLGETDIAGRECW( connection,
                            SQL_HANDLE_DBC,
                            connection -> driver_dbc,
                            rec ++,
                            sqlstate,
                            &native_error,
                            message_text,
                            sizeof( message_text ),
                            &ind );


                    if ( SQL_SUCCEEDED( ret ))
                    {
                        __post_internal_error_ex_w( &connection -> error,
                                sqlstate,
                                native_error,
                                message_text );

                        unicode_to_ansi( sqlstate, SQL_NTS );
                        unicode_to_ansi( message_text, SQL_NTS );

                        sprintf( connection -> msg, "\t\tDIAG [%s] %s",
                            sqlstate, message_text );

                        dm_log_write_diag( connection -> msg );
                    }
                }
                while( SQL_SUCCEEDED( ret ));
            }
        }

        /* 
         * if it was a error then return now
         */

        if ( !SQL_SUCCEEDED( ret_from_connect ))
        {
            __disconnect_part_one( connection );

            thread_release( SQL_HANDLE_DBC, connection );

            sprintf( connection -> msg,
                    "\n\t\tExit:[%s]",
                        __get_return_status( ret_from_connect ));

            dm_log_write( __FILE__,
                    __LINE__,
                    LOG_INFO,
                    LOG_INFO,
                    connection -> msg );

            return function_return( connection, ret_from_connect );
        }
        else
        {
            if ( conn_str_out )
            {
                unicode_to_ansi((SQLWCHAR*) conn_str_out, SQL_NTS );
            }
        }
        connection -> unicode_driver = 1;
    }

    /*
     * we should be connected now
     */

    connection -> state = STATE_C4;

    /*
     * did we get the type we wanted
     */

    if ( connection -> driver_version !=
            connection -> environment -> requested_version )
    {
        connection -> driver_version =
            connection -> environment -> requested_version;

        __post_internal_error( &connection -> error,
                ERROR_01000, "Driver does not support the requested version",
                connection -> environment -> requested_version );
        ret_from_connect = SQL_SUCCESS_WITH_INFO;
    }

    if ( !__connect_part_two( connection ))
    {
        __disconnect_part_two( connection );
        __disconnect_part_one( connection );

        thread_release( SQL_HANDLE_DBC, connection );

        return function_return( connection, SQL_ERROR );
    }

    if ( log_info.log_flag )
    {
        if ( conn_str_out && strlen((char*) conn_str_out ) > 64 )
        {
            sprintf( connection -> msg, 
                    "\n\t\tExit:[%s]\
                    \n\t\t\tConnection Out [%.64s...]",
                        __get_return_status( ret_from_connect ),
                        conn_str_out );
        }
        else
        {
            sprintf( connection -> msg, 
                    "\n\t\tExit:[%s]\
                    \n\t\t\tConnection Out [%s]",
                        __get_return_status( ret_from_connect ),
                        __string_with_length_hide_pwd( s1, 
                            conn_str_out ? conn_str_out : (SQLCHAR*)"NULL", SQL_NTS ));
        }

        dm_log_write( __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                connection -> msg );
    }

    thread_release( SQL_HANDLE_DBC, connection );

    /*
     * write the connection string out to a file
     */

    if ( tsavefile )
    {
        if ( SQL_SUCCEEDED( ret_from_connect ))
        {
            __parse_connection_string_ex( &con_struct,
                    (char*)conn_str_out, conn_str_out_max, 0 );

            /*
             * remove them
             */

            SQLWriteFileDSN( savefile, "ODBC", NULL, NULL );

            if ( con_struct.count )
            {
                int has_driver = 0;
                struct con_pair *cp;

                cp = con_struct.list;
                while( cp )
                {
                    if ( strcasecmp( cp -> keyword, "PWD" ) == 0 )
                    {
                        /*
                         * don't save this
                         */
                        continue;
                    }
                    else if ( strcasecmp( cp -> keyword, "DSN" ) == 0 )
                    {
                        /*
                         * don't save this either, there should be enough with the added DRIVER=
                         * to make it work
                         */
                        continue;
                    }
                    else if ( strcasecmp( cp -> keyword, "DRIVER" ) == 0 )
                    {
                        has_driver = 1;
                    }
                    SQLWriteFileDSN( savefile, "ODBC", cp -> keyword, cp -> attribute );
                    cp = cp -> next;
                }

                if ( !has_driver )
                {
                    SQLWriteFileDSN( savefile, "ODBC", "Driver", driver_name );
                }
            }

            __release_conn( &con_struct );
        }
    }

    return function_return( connection, ret_from_connect );
}
