/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <stdlib.h>
#include <string.h>
#include <memory.h>

#include <ug_data_download.h>


// ----------------------------------------------------------------------------
// UgDataCommon
static void ug_data_common_init		(UgDataCommon* common);
static void ug_data_common_finalize	(UgDataCommon* common);
static void ug_data_common_assign	(UgDataCommon* common, UgDataCommon* src);

static UgDataEntry	data_entry_common[] =
{
	{"name",				G_STRUCT_OFFSET (UgDataCommon, name),			UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"url",					G_STRUCT_OFFSET (UgDataCommon, url),			UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"file",				G_STRUCT_OFFSET (UgDataCommon, file),			UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"folder",				G_STRUCT_OFFSET (UgDataCommon, folder),			UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"user",				G_STRUCT_OFFSET (UgDataCommon, user),			UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"password",			G_STRUCT_OFFSET (UgDataCommon, password),		UG_DATA_TYPE_STRING,	NULL,	NULL},

	{"ConnectTimeout",		G_STRUCT_OFFSET (UgDataCommon, connect_timeout),	UG_DATA_TYPE_INT,	NULL,	NULL},
	{"TransmitTimeout",		G_STRUCT_OFFSET (UgDataCommon, transmit_timeout),	UG_DATA_TYPE_INT,	NULL,	NULL},
	{"RetryDelay",			G_STRUCT_OFFSET (UgDataCommon, retry_delay),		UG_DATA_TYPE_INT,	NULL,	NULL},
	{"RetryLimit",			G_STRUCT_OFFSET (UgDataCommon, retry_limit),		UG_DATA_TYPE_INT,	NULL,	NULL},
	{"RetryCount",			G_STRUCT_OFFSET (UgDataCommon, retry_count),		UG_DATA_TYPE_INT,	NULL,	NULL},
	{NULL}
};

static UgDataClass	data_class_common =
{
	"common",				// name
	NULL,					// reserve
	sizeof (UgDataCommon),	// instance_size
	data_entry_common,		// entry

	(UgInitFunc)     ug_data_common_init,
	(UgFinalizeFunc) ug_data_common_finalize,
	(UgAssignFunc)   ug_data_common_assign,
};

// extern
const	UgDataClass*		UgDataCommonClass = &data_class_common;


static void ug_data_common_init (UgDataCommon* common)
{
	common->connect_timeout  = 30;
	common->transmit_timeout = 30;
	common->retry_delay = 6;
	common->retry_limit = 99;
}

static void ug_data_common_finalize (UgDataCommon* common)
{
	g_free (common->name);
	g_free (common->url);
	g_free (common->file);
	g_free (common->folder);
	g_free (common->user);
	g_free (common->password);
}

static void ug_data_common_assign (UgDataCommon* common, UgDataCommon* src)
{
	if (src->name)
		common->name = g_strdup (src->name);
	else
		common->name = NULL;

	if (src->url)
		common->url = g_strdup (src->url);
	else
		common->url = NULL;

	if (src->file)
		common->file = g_strdup (src->file);
	else
		common->file = NULL;

	if (src->folder)
		common->folder = g_strdup (src->folder);
	else
		common->folder = NULL;

	if (src->user)
		common->user = g_strdup (src->user);
	else
		common->user = NULL;

	if (src->password)
		common->password = g_strdup (src->password);
	else
		common->password = NULL;

	common->connect_timeout  = src->connect_timeout;
	common->transmit_timeout = src->transmit_timeout;
	common->retry_delay = src->retry_delay;
	common->retry_limit = src->retry_limit;
	common->debug_level = src->debug_level;
}


// ----------------------------------------------------------------------------
// UgDataProxy

static void	ug_data_proxy_finalize	(UgDataProxy* proxy);
static void	ug_data_proxy_assign	(UgDataProxy* proxy, UgDataProxy* src);

static UgDataEntry	data_entry_proxy[] =
{
	{"host",		G_STRUCT_OFFSET (UgDataProxy, host),		UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"port",		G_STRUCT_OFFSET (UgDataProxy, port),		UG_DATA_TYPE_INT,		NULL,	NULL},
	{"type",		G_STRUCT_OFFSET (UgDataProxy, type),		UG_DATA_TYPE_INT,		NULL,	NULL},
	{"user",		G_STRUCT_OFFSET (UgDataProxy, user),		UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"password",	G_STRUCT_OFFSET (UgDataProxy, password),	UG_DATA_TYPE_STRING,	NULL,	NULL},
	{NULL},
};

static UgDataClass	data_class_proxy =
{
	"proxy",				// name
	NULL,					// reserve
	sizeof (UgDataProxy),	// instance_size
	data_entry_proxy,		// entry

	(UgInitFunc)		NULL,
	(UgFinalizeFunc)	ug_data_proxy_finalize,
	(UgAssignFunc)		ug_data_proxy_assign,
};

// extern
const	UgDataClass*	UgDataProxyClass = &data_class_proxy;


static void	ug_data_proxy_finalize	(UgDataProxy* proxy)
{
	g_free (proxy->host);
	g_free (proxy->user);
	g_free (proxy->password);
}

static void	ug_data_proxy_assign (UgDataProxy* proxy, UgDataProxy* src)
{
	if (src->host)
		proxy->host = g_strdup (src->host);
	else
		proxy->host = NULL;

	proxy->port = src->port;
	proxy->type = src->type;

	if (src->user)
		proxy->user = g_strdup (src->user);
	else
		proxy->user = NULL;

	if (src->password)
		proxy->password = g_strdup (src->password);
	else
		proxy->password = NULL;
}


// ----------------------------------------------------------------------------
// UgProgress

static void ug_progress_finalize	(UgProgress* progress);
static void ug_progress_assign		(UgProgress* progress, UgProgress* src);

static UgDataEntry	data_entry_progress[] =
{
	{"complete",	G_STRUCT_OFFSET (UgProgress, complete),		UG_DATA_TYPE_INT64,		NULL,	NULL},
	{"total",		G_STRUCT_OFFSET (UgProgress, total),		UG_DATA_TYPE_INT64,		NULL,	NULL},
	{"percent",		G_STRUCT_OFFSET (UgProgress, percent),		UG_DATA_TYPE_DOUBLE,	NULL,	NULL},
	{"elapsed",		G_STRUCT_OFFSET (UgProgress, consume_time),	UG_DATA_TYPE_DOUBLE,	NULL,	NULL},
	{NULL},
};

static UgDataClass	data_class_progress =
{
	"progress",				// name
	NULL,					// reserve
	sizeof (UgProgress),	// instance_size
	data_entry_progress,	// entry

	(UgInitFunc)		NULL,
	(UgFinalizeFunc)	ug_progress_finalize,
	(UgAssignFunc)		ug_progress_assign,
};

// extern
const UgDataClass*	UgProgressClass = &data_class_progress;


static void	ug_progress_finalize (UgProgress* progress)
{
	g_free (progress->file);
}

static void ug_progress_assign (UgProgress* progress, UgProgress* src)
{
	// copy without private member
	memcpy ( ((guint8*) progress) + sizeof (UgData),
	         ((guint8*) src)      + sizeof (UgData),
	         sizeof (UgProgress)  - sizeof (UgData) );

	// copy others that need allocate memory
	if (src->file)
		progress->file = g_strdup (src->file);
	else
		progress->file = NULL;
}


// ---------------------------------------------------------------------------
// UgDataHttp

static void	ug_data_http_init		(UgDataHttp* http);
static void	ug_data_http_finalize	(UgDataHttp* http);
static void	ug_data_http_assign		(UgDataHttp* http, UgDataHttp* src);

static UgDataEntry	data_entry_http[] =
{
	{"user",				G_STRUCT_OFFSET (UgDataHttp, user),					UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"password",			G_STRUCT_OFFSET (UgDataHttp, password),				UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"referer",				G_STRUCT_OFFSET (UgDataHttp, referer),				UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"UserAgent",			G_STRUCT_OFFSET (UgDataHttp, user_agent),			UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"PostData",			G_STRUCT_OFFSET (UgDataHttp, post_data),			UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"PostFile",			G_STRUCT_OFFSET (UgDataHttp, post_file),			UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"CookieData",			G_STRUCT_OFFSET (UgDataHttp, cookie_data),			UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"CookieFile",			G_STRUCT_OFFSET (UgDataHttp, cookie_file),			UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"RedirectionLimit",	G_STRUCT_OFFSET (UgDataHttp, redirection_limit),	UG_DATA_TYPE_INT,		NULL,	NULL},
	{NULL},
};

static UgDataClass	data_class_http =
{
	"http",					// name
	NULL,					// reserve
	sizeof (UgDataHttp),	// instance_size
	data_entry_http,		// entry

	(UgInitFunc)		ug_data_http_init,
	(UgFinalizeFunc)	ug_data_http_finalize,
	(UgAssignFunc)		ug_data_http_assign,
};

// extern
const	UgDataClass*	UgDataHttpClass = &data_class_http;

static void	ug_data_http_init	(UgDataHttp* http)
{
	http->redirection_limit = 30;
}

static void	ug_data_http_finalize (UgDataHttp* http)
{
	g_free (http->user);
	g_free (http->password);
	g_free (http->referer);
	g_free (http->user_agent);
	g_free (http->post_data);
	g_free (http->post_file);
	g_free (http->cookie_data);
	g_free (http->cookie_file);
}

static void	ug_data_http_assign (UgDataHttp* http, UgDataHttp* src)
{
	if (src->user)
		http->user = g_strdup (src->user);
	else
		http->user = NULL;

	if (src->password)
		http->password = g_strdup (src->password);
	else
		http->password = NULL;

	if (src->referer)
		http->referer = g_strdup (src->referer);
	else
		http->referer = NULL;

	if (src->user_agent)
		http->user_agent = g_strdup (src->user_agent);
	else
		http->user_agent = NULL;

	if (src->post_data)
		http->post_data = g_strdup (src->post_data);
	else
		http->post_data = NULL;

	if (src->post_file)
		http->post_file = g_strdup (src->post_file);
	else
		http->post_file = NULL;

	if (src->cookie_data)
		http->cookie_data = g_strdup (src->cookie_data);
	else
		http->cookie_data = NULL;

	if (src->cookie_file)
		http->cookie_file = g_strdup (src->cookie_file);
	else
		http->cookie_file = NULL;

	http->redirection_limit = src->redirection_limit;
}


// ---------------------------------------------------------------------------
// UgDataFtp

static void	ug_data_ftp_finalize	(UgDataFtp* ftp);
static void	ug_data_ftp_assign		(UgDataFtp* ftp, UgDataFtp* src);

static UgDataEntry	data_entry_ftp[] =
{
	{"user",		G_STRUCT_OFFSET (UgDataFtp, user),		UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"password",	G_STRUCT_OFFSET (UgDataFtp, password),	UG_DATA_TYPE_STRING,	NULL,	NULL},
	{"ExtendedPassiveMode",	G_STRUCT_OFFSET (UgDataFtp, extended_passive_mode),	UG_DATA_TYPE_INT,	NULL,	NULL},
	{NULL},
};

static UgDataClass	data_class_ftp =
{
	"ftp",				// name
	NULL,				// reserve
	sizeof (UgDataFtp),	// instance_size
	data_entry_ftp,		// entry

	(UgInitFunc)		NULL,
	(UgFinalizeFunc)	ug_data_ftp_finalize,
	(UgAssignFunc)		ug_data_ftp_assign,
};

// extern
const	UgDataClass*	UgDataFtpClass = &data_class_ftp;


static void	ug_data_ftp_finalize	(UgDataFtp* ftp)
{
	g_free (ftp->user);
	g_free (ftp->password);
}

static void	ug_data_ftp_assign	(UgDataFtp* ftp, UgDataFtp* src)
{
	if (src->user)
		ftp->user = g_strdup (src->user);
	else
		ftp->user = NULL;

	if (src->password)
		ftp->password = g_strdup (src->password);
	else
		ftp->password = NULL;

	ftp->extended_passive_mode = src->extended_passive_mode;
}

