/* ==================================================== ======== ======= *
 *
 *  uuxml.cpp
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uuxml.cpp	ubit:03.06.00"
#include <iostream>
#include <string.h>
#include <stdio.h>
#include <ctype.h>          // for isspace
#include <ubit/ubit.hpp>
#include <ubit/xml/uxmldoc.hpp>
#include <ubit/xml/uxmlparser.hpp>
using namespace std;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UXmlParser* UXmlDoc::parser;

void UXmlDoc::setXmlError(UDoc::Errors& err, int _stat, 
			  const char* mesg, const XMLCh* arg) {
  err.stat = err.stat | _stat;
  if (arg) {
    char* s_arg = XMLString::transcode(arg);
    err.messages &= mesg;
    err.messages &= s_arg;
    err.messages &= "\n";
    delete[] s_arg;
    //v2.3.0 XMLString::release(&s_arg);
  }
  else err.messages &= mesg + '\n';
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UXmlDoc::UXmlDoc(const UStr& _pathname, UDoc::Callbacks* _call) :
  UXmlElem(null),
  UDoc(_pathname),
  dom_document(null),
  callbacks(_call) {
  if (!parser) parser = new UXmlParser();
}

UXmlDoc::~UXmlDoc() {
  //cerr << "before release doc : " << dom_document << endl;
  if (dom_document) dom_document->release();
  //cerr << "after release doc : " << dom_document << endl;
  dom_document = null;

  /* ne plus faire: les objects sont detruits implicitement
  for (list<UStr*>::iterator k = objects.begin();
       k != objects.end();
       k++)
    delete *k;

  for (list<UStr*>::iterator k = links.begin();
       k != links.end();
       k++)
    delete *k;
   */
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UStr* UXmlDoc::makePath(const XMLCh* ch_path) {
  char* s = null;
  if (ch_path) s = XMLString::transcode(ch_path);

  UStr* path = null;
  if (s) {
    path = new UStr(s);
    path->trim();

    // NB: NE MARCHERA PAS POUR LES HTTP !!!
    int ix = path->find("file:");
    if (ix == 0) path->remove(0, 5); // remove "file:"
        
    if (path->charAt(0) != '/')  path->insert(0, dirname);

    delete[] s;
    //v2.3.0 XMLString::release(&s);
  }
  return path;
}

UCall* UXmlDoc::makeCall(UStr* ref_path) {
  if (!callbacks || !ref_path) return null;
  else return &ucall(callbacks,
                     ref_path, static_cast<UDoc*>(this),
                     &UDoc::Callbacks::notify);
}

void UXmlDoc::storeLink(UXmlLinkElem* e) {
  if (e) links.push_back(e);
}

void UXmlDoc::storeObject(UXmlObjectElem* e) {
  if (e) objects.push_back(e);
}

int UXmlDoc::loadMissingObjects() {
  int count = 0;
  for (UDoc::Objects::const_iterator k = objects.begin();
       k != objects.end();
       k++)
    if (*k && !(*k)->isLoaded()) {
      if ((*k)->load() < UFilestat::Opened) count++;
    }
  return count;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UXmlDoc::createTree(const UStr& pathname, DOMDocument* dd,
			 UDoc::Errors& err) 
{
  initModels();

  dirname  = pathname.getFileDir(true);   // with trailing /
  basename = pathname.getFileName(true);  // with suffix

  if (dom_document) dom_document->release();
  dom_document = dd;
  dom_elem = dd ? dd->getDocumentElement() : null;

  removeAll(true);

  if (!dom_elem)
    setXmlError(err, EmptyDocument,"null or empty DOMDocument");
  else
    createChildren(this, err);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UBrick* UXmlDoc::createAttr(DOMAttr* attr, UDoc::Errors& err) {
  const XMLCh* attr_name = attr->getName();

  for (const UXmlAttrModel* paa = getAttrModels(); 
       paa->cname != null; 
       paa++) {
    if (XMLString::compareString(attr_name, paa->dom_name) == 0) {
      return (paa->create)(attr, err);
    }
  }
  return null;
}

// ========================================================================

UBrick* UXmlDoc::createElem(DOMElement* elem, UDoc::Errors& err) {
  const XMLCh* elem_name = elem->getTagName();
 
  for (const UXmlElemModel* pea = getElemModels(); 
       pea->cname != null; 
       pea++) {
    if (XMLString::compareString(elem_name, pea->dom_name) == 0) { 
      return (pea->create)(elem);
    }
  }
  return null;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

// A REVOIR !!!!!!

void UXmlDoc::processWordText(UGroup* obj, class DOMNode* n) {

  const XMLCh* val = n->getNodeValue();
  char* s = XMLString::transcode(val);
  if (!s) return;
  
  int len = strlen(s);
  int i;
  
  // hack pourri car cette saloperie de Xerces ne renvoie
  // que de l'UTF et non de l'iso-8859 (independemment de ce
  // qui indique dans le document html: c'est sans rapport)
  for (i=0; i<len; i++) {
    //cerr << " " << int((unsigned char)s[i]) ;
    if ((unsigned char)s[i] < ' ') {
      s[i] = ' '; // virer chars speciaux et \n \t etc
    }
    else if ((unsigned char)s[i] > 128) {
      switch((unsigned char)s[i]) {
        case 136: s[i] = 224 /*''*/; break;
        case 137: s[i] = 226 /*''*/; break;
        case 141: s[i] = 231 /*''*/; break;
        case 142: s[i] = 233 /*''*/; break;
        case 143: s[i] = 232 /*''*/; break;
        case 144: s[i] = 234 /*''*/; break;
        case 145: s[i] = 235 /*''*/; break;
        case 148: s[i] = 238 /*''*/; break;
        case 149: s[i] = 239 /*''*/; break;
        case 157: s[i] = 249 /*''*/; break;
        case 158: s[i] = 251 /*''*/; break;
        case 159: s[i] = 252 /*''*/; break;
        default: s[i] = ' '; break;
      }
      // sinon: inchange
    }
    //cerr << "(" << int((unsigned char)s[i]) <<")";
  }
  //cerr << endl << endl;

  UStr* text = new UStr(s);
  text->trim();
  if (text->empty()) delete text;
  else obj->add(text);
  
  delete[] s; //v2.3.0 XMLString::release(&s);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UXmlElem::createChildren(UXmlDoc* doc, UDoc::Errors& err) {
  // this object can't have elem nor attr children if it's not a UGroup 
  // (or a subclass)
  UGroup* obj = dynamic_cast<UGroup*>(this);
  if (!obj) return;
  
  // the corresponding DOM Element is undefined
  if (dom_elem == null) return;

  // attribute children
  DOMNamedNodeMap* attr_map = dom_elem->getAttributes();

  if (attr_map != null) {
    for (unsigned int i = 0; i < attr_map->getLength(); i++) {

      DOMNode* attr_node = attr_map->item(i);
      DOMAttr* attr = dynamic_cast<DOMAttr*>(attr_node);
      UBrick*  child = doc->createAttr(attr, err);

      if (child) obj->add(child);
      else UXmlDoc::setXmlError(err, UXmlDoc::InvalidAttribute, 
				"invalid attribute: ", attr->getName());
    }
  }

  // element children
  for (DOMNode* chnode = dom_elem->getFirstChild(); 
       chnode != null; 
       chnode = chnode->getNextSibling()) 

    switch (chnode->getNodeType()) {

    case DOMNode::ELEMENT_NODE: {
      DOMElement* elem = dynamic_cast<DOMElement*>(chnode);
      UBrick* child = doc->createElem(elem, err);

      if (!child) {
	UXmlDoc::setXmlError(err, UXmlDoc::InvalidElement, 
			     "invalid element: ", elem->getTagName());
      }
      else {
	// this child should logically be an UXmlElem (which is also an Ubrick
	// or a subclass) so we can call createChildren recursively

	UXmlElem *xml_child = dynamic_cast<UXmlElem*>(child);
	if (xml_child) xml_child->createChildren(doc, err);
	
	//if (obj == doc->getDocElem()) doc->add(child);
	//else
	obj->add(child);
      }
    } break;

    case DOMNode::TEXT_NODE: {
      //DOMText& t = static_cast<DOMText&>(chnode);
      doc->processWordText(obj, chnode);
      //..else
      //  if (!t.isIgnorableWhitespace()) {
      //     cerr << "unallowed text: " << t.getNodeValue().transcode() << endl;
      //}
    } break;
    }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UXmlParser::xerces_initialized = false;

UXmlParser::UXmlParser() {
  // Initialize the Xerces-C system 
  // MUST be called before any other client call (and only once!).

  if (!xerces_initialized) {
    xerces_initialized = true;
 
    try {
      XMLPlatformUtils::Initialize();
      //const char* locale = "fr_FR";
      //XMLPlatformUtils::Initialize(locale);
    }
    catch(const XMLException& toCatch) {
      char* message = XMLString::transcode(toCatch.getMessage());
      cerr << "Error during Xerces-c Initialization:\n" << message << endl;
      delete [] message;
      //v2.3.0 XMLString::release(&message);
      return;
    }
  }

  xerces_parser = new XercesDOMParser();

  // xerces_parser->setValidationScheme(XercesDOMParser::Val_Always);
  // xerces_parser->setDoNamespaces(true);
  // xerces_parser->setExternalNoNamespaceSchemaLocation(const char*,
  //              const noNamespaceSchemaLocation);

  xerces_error_handler = new HandlerBase();
  //xerces_error_handler = new ErrorReporter();
  xerces_parser->setErrorHandler(xerces_error_handler);
}


UXmlParser::~UXmlParser() {
  //cerr << "before delete parser : " << xerces_parser << endl;
  delete xerces_parser;
  delete xerces_error_handler;
  //cerr << "after delete parser : " << xerces_parser << endl;
}

void UXmlParser::terminate() {
  XMLPlatformUtils::Terminate();
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// !NB le DOMDocument retounre appartient au client

DOMDocument* UXmlParser::parse(const UStr& _pathname, UDoc::Errors& err) { 

  UStr s = "*** Parsing file: "; s &= _pathname & "\n";
  err.set(0, s);

  if (!_pathname.isFileFound()) {
    err.set(UXmlDoc::InvalidFile, "File not found");
    return null;
  }

  try {
    xerces_parser->parse(_pathname.chars());
  }

  catch (const SAXParseException& toCatch) {
    char ms[50];
    sprintf(ms, "SAXException: at line %ld column %ld : ", 
	    long(toCatch.getLineNumber()), long(toCatch.getColumnNumber()));
    UXmlDoc::setXmlError(err, UXmlDoc::SAXError, ms, toCatch.getMessage());
  }
  
  catch (const XMLException& toCatch) {
    UXmlDoc::setXmlError(err, UXmlDoc::XMLError, "XMLException: ", toCatch.getMessage());
    return null;
  }

  catch (const DOMException& toCatch) {
    UXmlDoc::setXmlError(err, UXmlDoc::DOMError, "DOMException: ", toCatch.msg);
    return null;
  }

  catch (...) {
    //cerr << "OtherError" << endl;
    UXmlDoc::setXmlError(err, UXmlDoc::OtherError, "Unexpected exception");
    return null;
  }

  //if (xerces_parser->getErrorCount() != 0) return ;

  // tells the parser that it should not delete this document
  // once the parser gets geleted
  xerces_parser->adoptDocument();
  return xerces_parser->getDocument();
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/*
class ErrorReporter : public ErrorHandler {
public:
  ErrorReporter() {}
  void warning(const SAXParseException& toCatch);
  void error(const SAXParseException& toCatch);
  void fatalError(const SAXParseException& toCatch);
  void resetErrors() {}
};

void ErrorReporter::warning(const SAXParseException& toCatch) {
  char* mesg = XMLString::transcode(toCatch.getMessage());
  cerr << "WARNING at line " << toCatch.getLineNumber()
       << " column " << toCatch.getColumnNumber()
       << " : " << mesg << endl;
  delete[] mesg;
}

void ErrorReporter::error(const SAXParseException& toCatch) {
  char* mesg = XMLString::transcode(toCatch.getMessage());
  cerr << "ERROR  at line " << toCatch.getLineNumber()
       << " column " << toCatch.getColumnNumber()
       << " : " << mesg << endl;
  delete[] mesg;
}

void ErrorReporter::fatalError(const SAXParseException& toCatch) {
  char* mesg = XMLString::transcode(toCatch.getMessage());
  cerr << "FATAL ERROR at line " << toCatch.getLineNumber()
       << " column " << toCatch.getColumnNumber()
       << " : " << mesg << endl;
  delete[] mesg;
}
*/
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
