/* ==================================================== ======== ======= *
 *
 *  ubox.hh
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

#ifndef _ubox_hh
#define	_ubox_hh
//pragma ident	"@(#)ubox.hh	ubit:b1.11.6"
#include <uctrl.hh>
#include <uboxImpl.hh>


// NOTES on syntax:
//    ubox( UFont::b + ubutton(...) + "a string" + ... )
//
// References:
//    UBox &r = ubox( UFont::b + ubutton(...) +  "a string" + ... )
//
// Pointers
//    UBox *p = &ubox( UFont::b + ubutton(...) +  "a string" + ... )
// OR:
//    UBox *p1 = null, *p2 = null;
//    ...
//    p1 = &ubox( UFont::b + (p2 = &ubutton(...)) +  "a string" + ... )

// WARNINGS
// -- objects can't be copied, only pointers and references can, so:
//        UBox o = ubox(...)  is WRONG and WON'T COMPILE!
//
// --> instead you should write:
//        UBox *p = &ubox(...)          // CORRECT (p is an object pointer)
//        UBox &r = ubox(...)           // CORRECT (r is an object reference)

// -- the TWO FIRST elements of a list can't both be pointers (C++ limitation)
//        ubox( p1 + p2 )               // WRONG
//        ubox( "aaa" + "bbb" )         // WRONG
//
// --> instead you should write:
//        ubox( *p1 + p2 )              // CORRECT
//        ubox( ustr("aaa") + "bbb" )   // CORRECT

class UGroup& ugroup(UArgs = UArgs::none);
class UBox&   ubox(UArgs = UArgs::none);
class UVbox&  uvbox(UArgs = UArgs::none);
class UHbox&  uhbox(UArgs = UArgs::none);
class UBar&   ubar(UArgs = UArgs::none);

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UGroup: public UCtrl {
  friend class UBrick;
  friend class UWin;
protected:
  static UChain firingList; //objects whose callbacks are now being fired
  UChain children;   // child list
public:
  static const UClass  uclass;
  static UStyle *style;

  // !IMPORTANT NOTE: 
  // the destructor ALSO destroys the object's CHILDREN recursively:
  // - if they are in the AUTODEL mode (= the default; see the 'autodel()'
  //   method of the UBrick class in file ubrick.hh) 
  // - and if they do NOT have other (undeleted) parents

  UGroup(UArgs = UArgs::none);
  virtual ~UGroup() {clean();}

  virtual void clean();
  virtual const UClass* getClass()  const {return &uclass;}
  // returns the Contextual Style (depends on parents)
  virtual const UStyle* getStyle(const UBox *parent);
  virtual class UGroup* groupCast() {return this;}

  // INHERITED FUNCTIONS (see UGroup, UCtrl & UBrick classes for details)

  // virtual int getParentCount();
  // virtual UGroup** getParents();
  // virtual UGroup** getParents(int &count);
  // virtual UGroup* getParent(int pos);
  // virtual void removeFromParents();
  // virtual u_bool isChildOf(UGroup *parent, u_bool indirect);

  // virtual void enable(u_bool state);
  // u_bool isEnabled() const;  
  // virtual void select(u_bool state);
  // u_bool isSelected();
  // virtual u_bool isShown() const;     // is the object currently shown ?

  // forces the object to appear or disappear
  virtual void show(u_bool state);

  // prints the text that is enclosed in this object into an UStr
  // (given as the first arg. of the function)
  // - if 'recursive' is true, also retrieves text in object's descendants
  // - returns the text length
  virtual int copyText(UStr &result, u_bool recursive = true);
  //obsolete synonym:
  virtual int sprintText(UStr &result, u_bool recursive = true);

  // Adds a LIST of children at the end of the object's child list
  // NOTE:  this function returns the object and is NOT virtual
  UGroup& addlist(UArgs);

  // Adds 'child' in the object's child list at this position
  // - 'pos' =  0  means "first child"
  // - 'pos' = -1  (the default) means "AFTER THE last child"
  // - 'pos' is out of range: prints an error message and does nothing

  //!IMPORTANT NOTE:
  // - call the 'update()' method after adding or removing objects

  virtual void add(UBrick *child, int pos = -1);
  virtual void add(UBrick &child, int pos = -1);
  // adds a char string
  virtual void add(const char*, int pos = -1);
  // adds a link
  virtual void add(ULink*, int pos = -1);
  virtual void add(ULink&, int pos = -1);


  // Removes AND/OR deletes 'child' and its descendants
  // Step 1: 
  //       the FIRST occurence of 'child' is removed from the object's
  //       child list (other occurences are NOT removed if 'child' appears 
  //       several times in this list)
  // Step 2:
  //   -a- if arg. 'auto_delete' is 'false':
  //       'child' is NOT deleted and is returned by the function
  //   -b- if arg. 'auto_delete' is 'true':
  //       'child' AND its descendants are recursively deleted:
  //       -- if they are in the AUTODEL mode (= the default; see the 
  //          autodel() method of the UBrick class in file ubrick.hh) 
  //       -- and if they do NOT have other (undeleted) parents
  // Returned value:
  //       the fct. returns null if 'child' was deleted and 'child' otherwise
  // Remarks:
  //   -a- as a consequence of previous rules, 'child' is never deleted
  //       if it appears several times in the object's child list
  //   -b- arg. 'auto_delete' should be set to 'true' when removing
  //       objects that are not referenced (as memory would be lost otherwise)

  //!IMPORTANT NOTE:
  // - call the 'update()' method after adding or removing objects

  virtual UBrick* remove(UBrick &child, u_bool auto_delete);
  virtual UBrick* remove(UBrick *child, u_bool auto_delete);

  // Same as before but 'pos' specifies the position in the list:
  // - 'pos' =  0  means "first child" 
  // - 'pos' = -1  means "last child"
  // - 'pos' is out of range: the fct. prints an error message and does nothing

  virtual UBrick* remove(int pos, u_bool auto_delete);

  // Removes AND/OR deletes ALL children and descendants
  // Details: same as remove()

  virtual void removeAll(u_bool auto_delete);

  // Returns a copy of the object's child list
  // This copy:
  // -- is a NULL terminated list 
  // -- must be DESTROYED after use by calling: 
  //        delete[] this_list
  // -- is always created even if the object has no children
  //    (in which case it just contains a null terminating pointer)
  // !Warning:
  //    as children can be multiply added to the same parent object, 
  //    the same element can appear several times in the list.
  //    ==> be careful when you delete children not to delete them twice (or more)

  virtual UBrick** getChildren();
  virtual UBrick** getChildren(int &child_count);

  // returns the number of children
  virtual int getChildCount();

  // returns the number of children whose class is CHILD_CLASS (or a subclass)
  template <class CHILD_CLASS> 
  int getTypedChildCount() {
    int cnt = 0;
    for (ULink *l = children.first(); l!=null; l = l->next(), cnt++)
      if (dynamic_cast<CHILD_CLASS*>(l->brick())) cnt++;
    return cnt;
  }
  
  // Returns the nth element in the object's child list
  // -- 'pos' = 0 means "first child" and 'pos' = -1 means "last child"
  // -- just returns null if 'pos' is out of range (no error message!)
  // Note:
  // -- use fct. getChildren() for scanning the child list (or for accessing
  //    many children in this list). This will be much more efficient than
  //    multiple calls to fct. getChild()

  virtual UBrick* getChild(int pos);

  // returns (first) position of this child in the list. 
  // -- returns -1 if not found.
  virtual int getChildPos(const UBrick *child);

  // returns the 1st child of class UClass
  // -- returns null if not found.
  virtual UBrick* getChild(const UClass*);

   // returns position of the 1st child of class UClass
  // -- returns -1 if not found.
  virtual int getChildPos(const UClass*);

  // returns the nth child whose class is CHILD_CLASS (or a subclass)
  template <class CHILD_CLASS> 
  CHILD_CLASS* getTypedChild(int nth = 1) {
    int cnt = 0;
    for (ULink *l = children.first(); l!=null; l = l->next()) 
      if (dynamic_cast<CHILD_CLASS*>(l->brick())) {
	cnt++;
	if (cnt == nth)	return dynamic_cast<CHILD_CLASS*>(l->brick());
      }
    return null; // not found
  }

  // closes the first UWin (UDialog, UMenu, etc.) that contains this object
  // - the 'class' arg. specifies which type of UWin objects should be closed 
  //   (e.g. &UMenu::uclass, &UDialog::uclass). null means "any" UWin object.
  // NOTE:
  // -> see: ucloseWin(), ucloseDialog(), ucloseMenu() in ucall.hh
  //
  void closeWin(const UClass* = null);
  static void closeWin(UEvent*, const UClass* = null);

  // Updates object and chidren layout then repaints
  // !NOTE: must be called after add() or remove()
  virtual void update();

  // updates object according to upmode (see UUpdate class below)
  virtual void update(UUpdate upmode); 

  //package_private: ====[Ubit intrinsics]=========================

  // fires callbacks
  virtual void fire(class UEvent&, const class UOn&); 
  virtual void highlight(u_bool state);

  // this function makes this object an implicit browsing group
  // if it returns the object default/ NOT a browsing group)
  virtual UGroup* getBrowseGroup() {return null;}
  virtual void init(ULink *selflink, ULink *parlink, UView *parview);
  virtual void updateView(UEvent&, UView*, const UUpdate&);
  ULink* getChildLinks() const;

  // NOTE: the following methods are called by add(), remove(), getChild()
  // and 'delete'. They can can be freely redefined by subclasses. 
  // Note however its is generally necessary to redefine the four of them 
  // simultaneouly as these methods call each other.

  virtual void addImpl(UBrick *child, ULink *childlink, int pos);
  virtual UBrick* removeImpl(UBrick*child, ULink*prevlink, u_bool del_singles);
  virtual u_bool getChildImpl(int pos, ULink **prevlink);
  virtual u_bool getChildImpl(const UBrick *child, ULink **prevlink);

  // intermediate functions for sprintText
  virtual int   getTextLength(u_bool recursive);
  virtual char* getTextData(char *ptr, u_bool recursive);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// UBox : basic box object (use subclasses UHbox, UVbox, UBar etc.)

class UBox: public UGroup {
  friend class UWin;
public:
  static const UClass  uclass;
  static UStyle *style;

  // !IMPORTANT!: 
  // the destructor ALSO destroys the object's CHILDREN recursively:
  // - if they are in the AUTODEL mode (= the default; see the 'autodel()'
  //   method of the UBrick class in file ubrick.hh) 
  // - and if they do NOT have other (undeleted) parents

  UBox(UArgs = UArgs::none);
  virtual ~UBox() {clean();}

  virtual const UClass* getClass() const {return &uclass;}
  // returns the Contextual Style (depends on the Box parent)
  virtual const UStyle* getStyle(const UBox *parent);
  virtual class UBox* boxCast() {return this;}

  // INHERITED FUNCTIONS (see UGroup, UCtrl & UBrick classes for details)
  // - enable, isEnabled
  // - select, isSelected
  // - add
  // - remove, removeAll
  // - getParent, getParentCount, getParents
  // - getChild, getChildCount, getChildren
  // - copyText
  // - closeWin
  // - virtual u_bool isShown() const;   // is the object currently shown ?
  // etc.

  // forces the object to appear or disappear
  virtual void show(u_bool state) {UGroup::show(state);}

  // Adds a list of children at the end of the object's child list
  // !Note: this function returns the object and is NOT virtual
  UBox& addlist(UArgs a) {UGroup::addlist(a); return *this;}

  // Updates object and chidren layout then repaints
  // !NOTE: must be called after add() or remove()
  virtual void update();

  // updates object according to upmode (see UUpdate class below)
  virtual void update(UUpdate upmode);

  // onChildrenEvents catch events that are sent to the children (and the
  // descendants) of this object
  // Details:
  // - events are first sent to children (in the usual way), then they
  //   are sent to this object
  // - all UEvent methods (such as getSource(), getX(), getY() ...)
  //   are relative to the CHILD which received this event first
  // - NOTE that events occuring on this object are also caught
  // Example:
  //   box->onChildrenEvents( ucall(foo1, arg1, UOn::mmove) 
  //                          + ucall(foo2, args, UOn::mpress)
  //                         )
  //   'box' will catch 'mmove' and 'mpress' events occuring on itself 
  //   and on its children and descendants.

  virtual UBox& onChildrenEvents(UArgs);

  // ==== VIEWS ============================================================

  // !!! THIS API is not standardized and will be changed soon !!!
  // This API retrieves the views of the Box
  // Exemple:
  //   UBox *b = ...;
  //   UView *v = null;
  //   for (UViewIndex ix = b->getViewIndex(); (v = b->getView(ix)); ix++) {
  //       ..whatever..
  //   }
  virtual UViewIndex getViewIndex();
  virtual UView* getView(UViewIndex&);

  // Returns the Box's View that contains the Event location
  // (if any and applicable, and null otherwise)

  UView* getViewContaining(UEvent*);

  // Returns the Box's View that contains this child (or descendant) view 
  // (if any, and null otherwise)
  // -- 'child_view' may be a direct or indirect child (= a descendant) 

  UView* getViewContaining(UView *child_view);

  // Returns the nth Box's View that is contained in this (direct) parent view
  // -- 'pos' starts from 0
  // -- 'parent_view' must be a DIRECT parent
  // -- note that several Box's views are contained in 'parent_view'
  //    if the Box has beed multiply added to its parent

  UView* getViewContainedInParent(UView* parent_view, int pos);

  //package_private: ====[Ubit Intrinsics]==============================

  virtual ULink* makeLink() {return new UBoxLink(this);}
  virtual void init(ULink *selflink, ULink *parlink, UView *parview);
  virtual void updateView(UEvent&, UView*, const UUpdate&);
  static void updateLayout(UView* winview, UView* fromview, 
			   u_bool impose_size, u_dim w, u_dim h);
  // number of horizontal OR vertical flexible children (depending on
  // the Box orientation)
  unsigned short flexCount;
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// UHbox = horizontal box

class UHbox: public UBox {
public:
  static const UClass uclass;

  UHbox(UArgs a = UArgs::none) : UBox(a) {}
  virtual const UClass* getClass() const {return &uclass;}
};

/* ==================================================== ======== ======= */
// UHbox = vertical box

class UVbox: public UBox {
public:
  static const UClass uclass;
  static UStyle *style;

  virtual const UClass* getClass() const {return &uclass;}
  virtual const UStyle* getStyle(const UBox *parent);

  UVbox(UArgs a = UArgs::none) : UBox(a) {}
};

/* ==================================================== ======== ======= */
// UBar = horizontal bar (for toolbars)
// NOTE
// -> see: UMenubar in umenu.hh for creating menu bars

class UBar: public UBox {
public:
  static const UClass uclass;
  static UStyle *style;

  virtual const UClass* getClass() const {return &uclass;}
  virtual const UStyle* getStyle(const UBox *parent);

  UBar(UArgs = UArgs::none);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// arguments for the UGroup/UBox/UWin update() methods
// that specifies how to update these objects
// Examples: 
//    box.update(UUpdate::paint)   //just repaints
//    box.update(UUpdate::layout)  //updates layout then repaints
//    box.update()                 //same as: box.update(UUpdate::layout)
// (see alternate usage below)

class UUpdate {
  friend class UBox;
  friend class UWin;
public: //protected:
  int ix;
  u_bool always, doublebuf;
  u_pos delta_x, delta_y;
  const URegion *region;
  const UItem   *item;
public:
  enum Mode {PAINT, MOVE, SCROLL, LAYOUT, SHOW, HIDE, TITLE};
  static const UUpdate paint;    //same as: UUpdate(UUpdate::PAINT)
  static const UUpdate layout;   //same as: UUpdate(UUpdate::LAYOUT)

  UUpdate(Mode);

  // will paint in double buffering mode
  // Example:
  //   UUpdate upmode = UUpdate::paint; //or: UUpdate upmode(UUpdate::PAINT);
  //   upmode.doubleBuffering();
  //   box.update(upmode);
  //
  void doubleBuffering(u_bool = true);

  // will update even if the object is hidden (this is NOT the default!)
  void evenIfHidden(u_bool = true);

  // will only paint this item (if found) and whatever is located
  // beneath or above it
  void paintItem(const UItem*);

  // will only paint an enclosing subpart of this string
  void paintStr(const UStr*, int strpos1, int strpos2); 

  // will only paint this region
  void paintRegion(const URegion*);

  // will paint the title
  void paintTitle(const UStr*); 

  // used with the MOVE and SCROLL modes
  void setOffset(u_pos delta_x, u_pos delta_y);
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */
