"""
Functions for use with Kid template
"""
from kid.pull import START, TEXT, END

import turbogears
# use plain_gettext because Kid template's strings always evaluated immediately
from turbogears.i18n.tg_gettext import plain_gettext as gettext
from turbogears.i18n.utils import google_translate

def translate(item, attr=None):
    """Translates the text of element plus the text of all child elements. If attr is present 
    this is used to provide the locale name; if not then value provided by get_locale is used.
    For example:
    
    <div lang="">
        this is a test
        <a href="de">testing</a>
    </div>
    
    The string 'this is a test' is rendered by the locale provided by get_locale, the 
    string 'testing' by the German locale.
    
    Possible use in Kid template:
    <?python
    from turbogears.i18n import translate
    ?>
    <translate xmlns:py="http://purl.org/kid/ns#" py:match="item.attrib.has_key('lang')" 
    py:replace="translate(item)"/>
    <h1 lang="">Welcome!</h1>
    @param item: element to be translated
    @type item: ElementTree element
    @param attr: attribute name used to store locale, if any
    """
    if attr is None:attr = turbogears.config.get("i18n.templateLocaleAttribute", "lang")    
    translate_all(item, item.get(attr), attr)
    return item
    
def translate_all(tree, lang, attr):
    """Recursive function to translate all text in child elements
    @param tree: parent ElementTree element
    @param lang: language setting
    @param attr: attribute name used to store locale
    """

    if tree.text:
        tree.text = gettext(tree.text.strip(), lang)
    for element in tree:
        # check overriding lang attribute
        if element.get(attr):lang = element.get(attr)
        translate_all(element, lang, attr)

def i18n_filter(stream, template, locale=None):
    """Kid template filter which calls translates all elements matching language 
    attribute(set in configuration as i18n.templateLocaleAttribute, default 'lang')
    """
    
    lang_attr = turbogears.config.get("i18n.templateLocaleAttribute", "lang")    
    locales=[locale]
    
    for ev, item in stream:
                
        if ev==START:        
            l = item.get(lang_attr)
            if l:    
                locale = l
                locales.append(l)                
        elif ev==TEXT:
            prefix = ''
            postfix = ''
            if len(item) > 0 and item[0] == ' ': prefix =' '
            if len(item) > 1 and item[-1] == ' ': postfix =' '
            
            text = item.strip()
            if text:
                item = gettext(text, locale)     
                item = prefix + item + postfix
        elif ev==END:
            if item.get(lang_attr):
                locales.pop()
                locale = locales[-1]
            
        yield (ev, item)
