//-*-c++-*-
/**
 Authors: David Auber, Patrick Mary, Morgan Mathiaut
 from the LaBRI Visualization Team
 Email : auber@tulip-software.org
 Last modification : 13/03/2009 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifndef Tulip_GLTEXTUREMANAGER_H
#define Tulip_GLTEXTUREMANAGER_H

#include <map>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <tulip/tulipconf.h>

#include <cassert>
#if defined(__APPLE__)
#include <OpenGL/glu.h>
#else
#include <GL/glu.h>
#endif
#include <string>

namespace tlp {

  struct GlTexture {
    GLuint id;
    int height;
    int width;
  };

  /** \brief Class to manage textures
   * Singleton class to load/store textures need by OpenGL rendering
   */
  class TLP_GL_SCOPE GlTextureManager {

    typedef std::map<std::string,GlTexture> TextureUnit;
    typedef std::map<unsigned long,TextureUnit> ContextAndTextureMap;

  public:

    /**
     * Create the texture manager singleton
     */
    static void createInst();
    /**
     * Return the texture manager singleton, il singleton doesn't exist this function create it
     */
    static GlTextureManager &getInst() {
      if(!inst)
        inst=new GlTextureManager();
      return *inst;
    }

    /**
     * Change the current OpenGl context (each OpenGl window have a different OpenGl context)
     */
    void changeContext(unsigned long context);
    /**
     * Remove all textures of an OpenGl context and remove this context
     */
    void removeContext(unsigned long context);

    /**
     * Return texture info (id, width and height) for the given name
     */
    GlTexture getTextureInfo(const std::string&);
    /**
     * Load texture with textureName from Raw data
     */
    bool loadTextureFromRawData(const std::string &textureName, int width, int height, bool hasAlpha, unsigned char *data);
    /**
     * Load texture with given name
     */
    bool loadTexture(const std::string&);
    /**
     * Remove texture with given name
     */
    void deleteTexture(const std::string &);
    /**
     * Begin a new texture with given name
     */
    void beginNewTexture(const std::string&);
    /**
     * Activate a texture with given name
     */
    bool activateTexture(const std::string&);
    /**
     * Disable texture with given name
     */
    void desactivateTexture();

  private:

    /**
     * empty private constructor for singleton
     */
    GlTextureManager() {}

    static GlTextureManager* inst;

    unsigned long currentContext;

    ContextAndTextureMap texturesMap;

  };

}

#endif // Tulip_GLTEXTUREMANAGER_H
