/*
 * @(#)MidiMapper.cpp 3.00 24 May 1999
 *
 * Copyright (c) 2000 Pete Goodliffe (pete.goodliffe@pace.co.uk)
 *
 * This file is part of TSE3 - the Trax Sequencer Engine version 3.00.
 *
 * This library is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "tse3/MidiMapper.h"

#include "tse3/MidiScheduler.h"
#include "tse3/Midi.h"

using namespace TSE3;

namespace
{
    /**
     * Helper method to convert channel/port into array index.
     */
    int mapIndex(int channel, int port)
    {
        return (16*port)+channel;
    }
}


/******************************************************************************
 * MidiMapperImpl class
 *****************************************************************************/

class TSE3::MidiMapperImpl
{
    public:

        MidiScheduler *scheduler;

        int            noPorts;
        int           *channelMap;
        int           *portMap;
};


/******************************************************************************
 * MidiMapper class
 *****************************************************************************/

MidiMapper::MidiMapper(MidiScheduler *s)
: pimpl(new MidiMapperImpl)
{
    pimpl->scheduler = s;
    pimpl->noPorts   = s->ports();

    if (pimpl->noPorts < 1) pimpl->noPorts = 1;
    pimpl->channelMap = new int[pimpl->noPorts*16];
    pimpl->portMap    = new int[pimpl->noPorts*16];
    reset();

    attachTo(s);
}


MidiMapper::~MidiMapper()
{
    delete pimpl->channelMap;
    delete pimpl->portMap;

    delete pimpl;
}


MidiScheduler *MidiMapper::scheduler() const
{
    return pimpl->scheduler;
}


const std::pair <int, int> MidiMapper::map(int fromChannel, int fromPort) const
{
    return std::pair<int, int>
        (pimpl->channelMap[mapIndex(fromChannel, fromPort)],
         pimpl->portMap[mapIndex(fromChannel, fromPort)]);
}


void MidiMapper::setMap(int fromChannel, int fromPort,
                        int toChannel,   int toPort)
{
    pimpl->channelMap[mapIndex(fromChannel, fromPort)] = toChannel;
    pimpl->portMap[mapIndex(fromChannel, fromPort)]    = toPort;
    notify(&MidiMapperListener::MidiMapper_Altered, fromChannel, fromPort);
}


const MidiEvent MidiMapper::filter(const MidiEvent &m) const
{
    MidiEvent me = m;
    int       c  = me.data.channel;
    int       p  = me.data.port;
    me.data.channel = pimpl->channelMap[mapIndex(c, p)];
    me.data.port    = pimpl->portMap[mapIndex(c, p)];
    if (me.data.status == MidiCommand_NoteOn)
    {
        int c = me.offData.channel, p = me.offData.port;
        me.offData.channel = pimpl->channelMap[mapIndex(c, p)];
        me.offData.port    = pimpl->portMap[mapIndex(c, p)];
    }
    return me;
}


void MidiMapper::reset()
{
    for(int p = 0; p < pimpl->noPorts; ++p)
    {
        for (int c = 0; c < 16; ++c)
        {
            pimpl->channelMap[mapIndex(c, p)] = c;
            pimpl->portMap[mapIndex(c, p)]    = p;
        }
    }
    notify(&MidiMapperListener::MidiMapper_Altered, -1, -1);
}


void MidiMapper::MidiScheduler_Ports(MidiScheduler *)
{
}
