// m6502.h - Motorola 6502 Microprocessor Definitions
//
// Copyright (c) 2002, Timothy M. Stark
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// TIMOTHY M STARK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Timothy M Stark shall not
// be used in advertising or otherwise to promote the sale, use or other 
// dealings in this Software without prior written authorization from
// Timothy M Stark.

// 6502 Memory Map
//
// 0000 +------------------------------+
//      |       Page Zero Area         |
// 0100 +------------------------------+
//      |         Stack Area           |
// 0200 +------------------------------+
//      |                              |
//      |                              |
//      |                              |
//      |                              |
//      |                              |
//      |                              |
// FFFA +------------------------------+
//      | Non-Maskable Interrupt (NMI) |
// FFFC +------------------------------+
//      |      Reset Vector (RES)      |
// FFFE +------------------------------+
//      |    Interrupt Vector (IRQ)    |
//      |     Break Command Vector     |
// FFFF +------------------------------+

#define PGZ_ADR 0x0000 // Page Zero Addressing
#define STK_ADR 0x0100 // Stack Area
#define NMI_ADR 0xFFFA // Non-Maskiable Interrupt Address
#define RST_ADR 0xFFFC // Reset Address
#define IRQ_ADR 0xFFFE // Interrupt Request Address
#define BRK_ADR 0xFFFE // Break Command Address

// Status Register
//
// +-+-+-+-+-+-+-+-+
// |N|V|T|B|D|I|Z|C|
// +-+-+-+-+-+-+-+-+
//  7 6 5 4 3 2 1 0

#define SR_N  0x80 // Negative Flag
#define SR_V  0x40 // Overflow Flag
#define SR_T  0x20 //
#define SR_B  0x10 // Break Command
#define SR_D  0x08 // Decimal Mode
#define SR_I  0x04 // Interrupt Disable
#define SR_Z  0x02 // Zero Flag
#define SR_C  0x01 // Carry Flag

#define SET_NZ(ac) \
	if ((ac) == 0) P = (P & ~SR_N) | SR_Z; \
	else           P = (P & ~(SR_N|SR_Z)) | ((ac) & SR_N);

typedef struct m65_InstTable M65_INST;
typedef struct m65_Processor M65_CPU;

struct m65_InstTable {
	char   *Name;     // Instruction Name
	char   *Desc;     // Description
	uint32 Flags;     // Flags
	int    nOperands; // Number of Operands
	struct {
		uint8 Type;    // Addressing Mode Type
		uint8 opCode;  // Opcode
		uint8 nBytes;  // Number of bytes
		uint8 Cycles;  // CPU Cycles
	} Operands[8];
};

struct m65_Processor {
	UNIT   Unit;    // Unit Header Information

	// 6502 Registers
	pair16 pc;  // Program Counter
	pair16 ea;  // Effective Address
	pair16 zp;  // Zero Page Address ($0000 to $00FF)
	pair16 sp;  // Stack Pointer     ($0100 to $01FF)
	uint8  a;   // Accumulator
	uint8  x;   // X Index Register
	uint8  y;   // Y Index Register
	uint8  p;   // Status Register
	uint16 irq; // Pending IRQ interrupts

	uint32 ips;    // Instruction Counter for MIPS Meter
	uint32 cycles; // Cycle Counter for CPU throttle

	// External function calls
	uint8 (*ReadB)(uint32 addr);
	void  (*WriteB)(uint32 addr, uint8 data);
};

#define M65_REGS(x) (m65->x)

#define PC  M65_REGS(pc.w)   // Program Counter - Word
#define PCL M65_REGS(pc.b.l) // Program Counter - Low byte
#define PCH M65_REGS(pc.b.h) // Program Counter - High byte

#define EA  M65_REGS(ea.w)   // Effective Address - Word
#define EAL M65_REGS(ea.b.l) // Effective Address - Low byte
#define EAH M65_REGS(ea.b.h) // Effective Address - High byte

#define ZP  M65_REGS(zp.w)   // Zero Page Address - Word
#define ZPL M65_REGS(zp.b.l) // Zero Page Address - Low byte
#define ZPH M65_REGS(zp.b.h) // Zero Page Address - High byte

#define SP  M65_REGS(sp.w)   // Stack Pointer - Word
#define SPL M65_REGS(sp.b.l) // Stack Pointer - Low byte
#define SPH M65_REGS(sp.b.h) // Stack Pointer - High byte

#define A   M65_REGS(a)      // Accumulator      (A)
#define X   M65_REGS(x)      // X Index Register (X)
#define Y   M65_REGS(y)      // Y Index Register (Y)
#define P   M65_REGS(p)      // Status Register  (P)
#define S   M65_REGS(sp.b.l) // Stack Pointer    (S)

#define IPS M65_REGS(ips)    // Instruction Count
#define IRQ M65_REGS(irq)    // Pending IRQ Interrupts


// Instruction Definition
#define INST_DEF(cpu, inst)  void cpu##_op##inst(register M65_CPU *m65)

