#!/usr/bin/env python

"""
****************************************************************************
 treeline.py, the main program file

 TreeLine, an information storage program
 Copyright (C) 2006, Douglas W. Bell

 This is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License, Version 2.  This program is
 distributed in the hope that it will be useful, but WITTHOUT ANY WARRANTY.
*****************************************************************************
"""

__progname__ = 'TreeLine'
__version__ = '1.0.0'
__author__ = 'Doug Bell'

helpFilePath = None    # modified by install script if required
iconPath = None        # modified by install script if required
translationPath = 'translations'


from qt import qVersion, QApplication, QMessageBox, QTranslator
import sys, signal, getopt, os.path, locale, __builtin__
import globalref


def loadTranslator(fileName, app):
    """Load and install qt translator, return True if sucessful"""
    translator = QTranslator(app)
    path = os.path.join(os.path.abspath(sys.path[0]), translationPath)
    result = translator.load(fileName, path)
    if not result:
        path = os.path.join(os.path.abspath(sys.path[0]), '..', translationPath)
        result = translator.load(fileName, path)
    if result:
        app.installTranslator(translator)
        return True
    else:
        print 'Warning: translation file "%s" could not be loaded' % fileName
        return False

def setupTranslator(app):
    """Set language, load translators and setup translator function"""
    try:
        locale.setlocale(locale.LC_ALL, '')
    except locale.Error:
        pass
    globalref.lang = os.environ.get('LANG', '')
    if not globalref.lang:
        try:
            globalref.lang = locale.getdefaultlocale()[0]
        except ValueError:
            pass
    numTranslators = 0
    if globalref.lang and globalref.lang[:2] not in ['C', 'en']:
        numTranslators += loadTranslator('qt_%s' % globalref.lang, app)
        numTranslators += loadTranslator('treeline_%s' % globalref.lang, app)

    def translate(text, comment=''):
        """Translation function that sets context to calling module's
           filename"""
        try:
            frame = sys._getframe(1)
            fileName = frame.f_code.co_filename
        finally:
            del frame
        context = os.path.basename(os.path.splitext(fileName)[0])
        result = unicode(app.translate(context, text, comment))
        if qVersion()[0] < '3' and result == text:
            # ugly hack to fix bug in Qt2.3 that can't find with a comment
            result = unicode(app.translate(context, text))
        return result

    def markNoTranslate(text, comment=''):
        return text

    if numTranslators:
        __builtin__._ = translate
    else:
        __builtin__._ = markNoTranslate
    __builtin__.N_ = markNoTranslate


def main():
    try:
        split = sys.argv.index('--')
        qtOpts, args = sys.argv[:split], sys.argv[split+1:]
    except ValueError:
        qtOpts, args = sys.argv[:1], sys.argv[1:]

    app = QApplication(qtOpts)
    setupTranslator(app)  # must be before importing any treeline modules

    import treedoc
    from cmdline import CmdLine
    from treemainwin import TreeMainWin

    if not treedoc.testXmlParser():
        QMessageBox.critical(None, _('Error'), \
                             _('Error loading XML Parser\n'\
                               'See TreeLine ReadMe file'), 1, 0)
        sys.exit(3)
    treedoc.setLocalEncoding()
    try:
        opts, args = getopt.gnu_getopt(args, CmdLine.options, \
                                       CmdLine.longOptions)
    except getopt.GetoptError:
        import cmdline
        print cmdline.usage
        sys.exit(2)

    if opts:
        CmdLine(opts, args)
    else:
        win = TreeMainWin()
        app.setMainWidget(win)
        if args:
            win.openFile(args[0])
        else:
            win.autoOpen()
        win.show()
        win.lateInit()
        signal.signal(signal.SIGINT, signal.SIG_IGN)
        app.exec_loop()


if __name__ == '__main__':
    main()
