#ifndef _TRAFSTATS_H_
#include "trafstats.h"
#endif

extern pthread_cond_t timer_ok;

extern pthread_mutex_t datalock;
extern pthread_mutex_t timelock;
extern pthread_mutex_t timeroklock;
// static list to be filled.
extern CTrafStatsList theList;
extern int verbosity;
extern char* timestamp;
extern bool run;
extern bool castrate;
// Have to declare this at file scope so that the signal handler will know
// what to do with it.
// 

pcap_t *descr;
char errbuf[PCAP_ERRBUF_SIZE];

// Externally declared functions; 
void *storageThread(void *d);

int try_open(char *dev) {
	// open device for reading.
	// FIXME: This is the ONLY line in the program requiring rootage. 
	descr = pcap_open_live(dev,BUFSIZ,1,-1,errbuf);
	
	if(descr == NULL) { 
		if(verbosity >= LOG_ERR) {
			syslog(LOG_ERR,"[sniffer] pcap_open_live failed: %s",
				errbuf);
		}
		return false;
	} // else
	return true;
}
// Signal handler: handle SIGINT and SIGTERM for now by simply calling
// pcap_close on the descriptor
// 
void snifferSigHandler(int sig) {
	switch(sig) { 
	case SIGINT: // Someone pressed ^C
		// Fall through
	case SIGTERM:
		if(verbosity >= LOG_NOTICE) {
			syslog(LOG_NOTICE,"[sniffer] Received signal %d, exiting",sig);
		}
		if(descr!=NULL) { // Shouldn't happen but you never know.
			if(verbosity >= LOG_INFO) {
				syslog(LOG_INFO,"[sniffer] Closing descriptor.");
			}
			pthread_exit(NULL);
		}
		break;
	default:
		if(verbosity >= LOG_WARNING) {
			syslog(LOG_WARNING,"[sniffer] Don't know how to handle signal %d!",sig);
		}
		break;
	}
}

/* Strip the ethernet header off the packet; code
 * again shamelessly pilfered from aforementioned tutorial.
 */
struct my_ip *strip_ether (const u_char* packet)
{
    struct ether_header *eptr;  /* as seen in net/ethernet.h */

    /* Splice off the ethernet header. */
    eptr = (struct ether_header *) packet;

    /* check to see if we have an ip packet */
    if (ntohs (eptr->ether_type) == ETHERTYPE_IP)
    {
        return (struct my_ip *) (packet + sizeof(struct ether_header));
    } // We're not interested in anything else.
    return NULL;
}


void handle_packet(const u_char* etherpacket)
{
	struct my_ip* ip_packet= strip_ether(etherpacket);
	int iplength;
	struct in_addr source;
	struct in_addr destination; // If I declare them on the same line, 
				    // they stay identical.
	int source_port=0,destination_port=0;
	if(ip_packet==NULL) { // Not an IP packet, so we're not interested.
		return;
	} // else

	// Start analysing the IP packet headers for useful info
	iplength=ntohs(ip_packet->ip_len);
	destination=ip_packet->ip_dst;
	source=ip_packet->ip_src;

	if(!castrate) {
		// Almost there, but if it's TCP or UDP we still need to know
		// the port numbers. So... 
		switch(ip_packet->ip_p) {
			case IPPROTO_UDP:
				// Evil hack, but since the only parts I'm
				// interested in are identical, this works.
				// 
			case IPPROTO_TCP:
				source_port=ntohs(((struct tcphdr *) (ip_packet + 1))->source);
				destination_port=ntohs(((struct tcphdr *) (ip_packet + 1))->dest);
				break;
				
			default:
				// Other protocols don't use ports, so we simply 
				// sure they're they're zeroed
				source_port=destination_port=0;
		}
	}

	// We've got all the relevant info, let's store it.
	CTrafStatsEntry	line(	source,
				source_port,
				destination,
				destination_port,
				timestamp,
				iplength);
	if(verbosity >= LOG_DEBUG+6) {
		syslog(LOG_DEBUG,"Packet: %s",line.asString().c_str());
	}
	// Careful, watch those threads...
	pthread_mutex_lock(&datalock);
	if(verbosity >= LOG_DEBUG+6) {
		syslog(LOG_DEBUG,"[sniffer] Got the mutex.");
	}
	theList.insert(line);
	pthread_mutex_unlock(&datalock);

	return; // Redundant, I know.
}


void *snifferThread(void *d) {

	char *dev=(char *)d;
	descr=NULL;
	bpf_u_int32 maskp;          /* subnet mask               */
	bpf_u_int32 netp;           /* ip                        */
	struct in_addr addr;
	char *net;
	char *mask;
	struct pcap_pkthdr useless;
	u_char* packet=NULL;

	// ask pcap for the network address and mask of the device 
	pcap_lookupnet(dev,&netp,&maskp,errbuf);
	if(verbosity >= LOG_DEBUG) {
		addr.s_addr = netp;
		mask=inet_ntoa(addr);
		net=(char*) malloc(16);
		strcpy(net,mask);
		addr.s_addr = maskp;
		mask=inet_ntoa(addr);
		syslog(LOG_DEBUG,"[sniffer] Device: <%s>: %s/%s",dev,net,mask);
	}

	if(!try_open(dev)) {
		if(verbosity >= LOG_CRIT) {
			syslog(LOG_CRIT,"[sniffer] Failed to open device %s, exiting.", dev);
		}

		run=0;
		pthread_exit(NULL);
	}

	// else 
	// Set the handler to knock me out of it when needed.
	// 
	if(verbosity >= LOG_INFO) {
		syslog(LOG_INFO,"[sniffer] Setting up signal handlers.");
	}
	
	signal(SIGINT,snifferSigHandler);
	signal(SIGTERM,snifferSigHandler);
	if(verbosity >= LOG_INFO) {
		syslog(LOG_INFO,"[sniffer] Starting loop.");
	}


	/* 
	 * No longer used.
	 * int count=pcap_loop(descr,-1,do_sniff,args);
	 *
	 * Changed to:
	 */
	int count=0;
	while(run) {
		// Get the next packet
		packet= (u_char *) pcap_next(descr, &useless);

		if(packet==NULL) { // Whoops.
			if(verbosity >= LOG_DEBUG) {
				syslog(LOG_DEBUG,"[sniffer] Failed to read a packet.");
			}
			if(descr==NULL) {
				if(verbosity >= LOG_ERR) {
					syslog(LOG_ERR,"[sniffer] Failed read from %s: %s",dev,pcap_geterr(descr));
					if(verbosity >= LOG_NOTICE) {
						syslog(LOG_NOTICE,"[sniffer] Trying to re-open interface...");
					}
				}
				// Try to re-open the handle or die.
				if(try_open(dev) == false) {
					
					if(verbosity >= LOG_CRIT) {
						syslog(LOG_CRIT,"[sniffer] Re-open failed. Exiting.");
					}
					run=0;
					pthread_exit(NULL); // We die now.
				}
			}
		}
		// Feed it to handle_packet()
		handle_packet(packet);
		count++;
	}

	if(verbosity >= LOG_DEBUG) {
		syslog(LOG_DEBUG,"[sniffer] Closing file descriptor.");
	}
	if(descr != NULL) {
		pcap_close(descr);
	}
	pthread_exit(NULL);
}

