/*
    Small tester application that runs against CT-API library.
    Handles up to four card-terminal interfaces simultaneously.
    It issues commands for both synchronous and asynchronous cards.
 
    This file is part of the Unix driver for Towitoko smartcard readers
    Copyright (C) 2000 Carlos Prados <cprados@yahoo.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include "ctapi.h"
#include "ctbcs.h"
#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif
#ifdef OS_FREEBSD
#include <sys/time.h>
#endif

/* Class byte for Cryptoflex */
/* #define CLASS 0xC0 */

/* Class byte for Cyberflex */
#define CLASS 0x00

/* Class byte for IBM MFC 3.5 smartcards */
/* #define CLASS 0xA4 */

/* Status of each card-terminal */
struct
{
  unsigned short pn;            /* Serial port (1..4), 0 if not assigned */
  int status;                   /* -1 void, 0 sync, 1 async */
  unsigned char atr[33];        /* ATR bytes */
  unsigned short atr_size;      /* ATR size */
#ifdef HAVE_PTHREAD_H
  pthread_t thread;             /* Monitoring thread for this ctn */
  pthread_mutex_t mutex;        /* Mutex for accessin this element */
#endif
}
ct_list[4];

/* Menu loop */
void menu_loop (void);

/* Event handlers */
void Initialize (void);
unsigned short Change (unsigned short);
void Close (unsigned short);
void SelectFile (unsigned short);
void GetResponse (unsigned short);
void UpdateBinary (unsigned short);
void ReadBinary (unsigned short);
void VerifyKey (unsigned short);
void SendPPS (unsigned short);
void EnterPin (unsigned short);
void ChangePin (unsigned short);
void ReadData (unsigned short);
void WriteData (unsigned short);

#ifdef HAVE_PTHREAD_H
/* Asynchronous monitoring thread function */
void *Monitor (void *);
#else
/* Synchronous polling function */
unsigned short GetSmartcard (unsigned short);
#endif

/* Print status */
void PrintReport (void);

int
main (int argc, char *argv[])
{
  unsigned short ctn;

  /* Initialize status of ct's */
  for (ctn = 0; ctn < 4; ctn++)
    {
      ct_list[ctn].pn = 0;
      ct_list[ctn].status = -1;
      memset (ct_list[ctn].atr, 0, 33);
      ct_list[ctn].atr_size = 0;
#ifdef HAVE_PTHREAD_H
      pthread_mutex_init (&(ct_list[ctn].mutex), NULL);
#endif
    }

  /* Start menu loop */
  menu_loop ();

  /* Close all open ct's */
  for (ctn = 0; ctn < 4; ctn++)
    {
      if (ct_list[ctn].pn != 0)
        {
          Close (ctn);
#ifdef HAVE_PTHREAD_H
          pthread_mutex_destroy (&(ct_list[ctn].mutex));
#endif
        }
    }
    
  return 0;
}

void
menu_loop (void)
{
  char option[32];
  unsigned short ctn = 0, i;
  int dummy;

  while (1)
    {
      PrintReport ();
      printf ("Select option:\n");

      /* Common menu */
      printf ("in: Initialize new terminal\n");
      printf ("ch: Change current terminal\n");
      printf ("cl: Close current terminal\n");

      /* Port is open */
      if (ct_list[ctn].pn != 0)
        {
          /* Processor cards menu */
          if (ct_list[ctn].status == 1)
            {
              printf ("sf: Select File\n");
              printf ("gr: Get Response\n");
              printf ("vk: Verify Key (Cryptoflex)\n");
              printf ("rb: Read Binary\n");
              printf ("ub: Update Binary\n");
	      printf ("pps: Send PPS\n");
            }

          /* Memory card menu */
          else if (ct_list[ctn].status == 0)
            {
              printf ("ep: Enter PIN\n");
              printf ("cp: Change PIN\n");
              printf ("rd: Read Data\n");
              printf ("wd: Write Data\n");
            }
        }

      printf ("*: Refresh Menu\n");
      printf ("q: Quit\n");

      if (ct_list[ctn].pn != 0)
        printf ("COM%d$ ", ct_list[ctn].pn);
      else
        printf ("$ ");

      /* Get  menu option */
      scanf ("%s", option);
      dummy = getchar ();

      /* Convert input to lowercase */
      for (i = 0; option[i]; i++)
        option[i] = tolower (option[i]);

      /* Common options */
      if (strcmp (option, "q") == 0)
        break;

      else if (strcmp (option, "in") == 0)
        Initialize ();

      else if (strcmp (option, "ch") == 0)
        ctn = Change (ctn);

      else if (strcmp (option, "cl") == 0)
        Close (ctn);

      /* Processor card options */
      if (ct_list[ctn].status == 1)
        {
          if (strcmp (option, "sf") == 0)
            SelectFile (ctn);

          else if (strcmp (option, "gr") == 0)
            GetResponse (ctn);

          else if (strcmp (option, "vk") == 0)
            VerifyKey (ctn);

          else if (strcmp (option, "rb") == 0)
            ReadBinary (ctn);

          else if (strcmp (option, "ub") == 0)
            UpdateBinary (ctn);

	  else if (strcmp (option, "pps") == 0)
	    SendPPS (ctn);
        }

      /* Memory card options */
      else if (ct_list[ctn].status == 0)
        {
          if (strcmp (option, "ep") == 0)
            EnterPin (ctn);

          else if (strcmp (option, "cp") == 0)
            ChangePin (ctn);

          else if (strcmp (option, "rd") == 0)
            ReadData (ctn);

          else if (strcmp (option, "wd") == 0)
            WriteData (ctn);
        }
    }
}

void
PrintReport (void)
{
  unsigned short ctn, num, i;

  num = 0;
  for (ctn = 0; ctn < 4; ctn++)
    if (ct_list[ctn].pn != 0)
      num++;

  printf
   
 ("**********************************************************************\n");
  printf ("Towitoko CT-API tester utility\n");
  printf ("Copyright (C) 2000 2001 Carlos Prados <cprados@yahoo.com>\n");
  printf ("Initilized CardTerminals: %d\n", num);

  for (ctn = 0; ctn < 4; ctn++)
    {
      if (ct_list[ctn].pn != 0)
        {
#ifdef HAVE_PTHREAD_H
          pthread_mutex_lock (&(ct_list[ctn].mutex));
#endif
          printf
           
 ("**********************************************************************\n");
          printf ("COM%d\n", ct_list[ctn].pn);
          printf ("Status: %s\n",
                  ct_list[ctn].status == 0 ? "Memory smartcard present" :
                  ct_list[ctn].status == 1 ? "Processor smartcard present" :
                  "No smartcard present (type * to refresh)");
          if (ct_list[ctn].status != -1)
            {
              printf ("ATR: ");
              for (i = 0; i < ct_list[ctn].atr_size; i++)
                printf ("%02X ", ct_list[ctn].atr[i]);
              printf ("\n");
            }
#ifdef HAVE_PTHREAD_H
          pthread_mutex_unlock (&(ct_list[ctn].mutex));
#endif
        }
    }

  printf
   
 ("**********************************************************************\n");
}

#ifndef HAVE_PTHREAD_H
unsigned short
GetSmartcard (unsigned short ctn)
{
  unsigned char cmd[11], res[256], sad, dad;
  unsigned short lr, count;
  char ret;
#ifdef HAVE_NANOSLEEP
  struct timespec req_ts;

  req_ts.tv_sec = 1;
  req_ts.tv_nsec = 0;
#endif

  for (count = 0; count < 60; count ++)
    {
      /* Check if card is inserted */
      cmd[0] = CTBCS_CLA;
      cmd[1] = CTBCS_INS_STATUS;
      cmd[2] = CTBCS_P1_CT_KERNEL;
      cmd[3] = CTBCS_P2_STATUS_ICC;
      cmd[4] = 0x00;

      dad = 1;
      sad = 2;
      lr = 256;

      ret = CT_data (ctn, &dad, &sad, 11, cmd, &lr, res);

      if ((ret != OK) || (res[lr-2] != 0x90))
        {
          fprintf (stderr, "\nError getting status of terminal: %d\n", ret);
          return 0;
        }
 
      if (res[0] == CTBCS_DATA_STATUS_CARD_CONNECT)
        {            
          if (count > 1)                
            printf("\n");
            
          printf ("Activating card...\n");

          /* Activate card */
          cmd[0] = CTBCS_CLA;
          cmd[1] = CTBCS_INS_REQUEST;
          cmd[2] = CTBCS_P1_INTERFACE1;
          cmd[3] = CTBCS_P2_REQUEST_GET_ATR;
          cmd[4] = 0x00;

          dad = 0x01;
          sad = 0x02;
          lr = 256;

          ret = CT_data (ctn, &dad, &sad, 5, cmd, &lr, res);

          if ((ret != OK) || (res[lr - 2] != 0x90))
            {
              fprintf (stderr, "Error activating card: %d\n", ret);
              return 0;
            }

          /* Store the type of card */
          ct_list[ctn].status = res[lr - 1];

          /* Store ATR */
          memcpy (ct_list[ctn].atr, res, lr - 2);
          ct_list[ctn].atr_size = lr - 2;
          return 1;
        }
        
      else
        {
          if (count > 0)
            {
              printf (".");
              fflush (stdout);
            }
          else
            printf ("Please insert a smartcard in the terminal\n");
        }
#ifdef HAVE_NANOSLEEP
          nanosleep (&req_ts, NULL);
#else
          usleep (999999);
#endif                                                                          
 
    }

  printf ("\nTimeout waiting for smartcard insertion\n");
  return 0;
}

#else

void *
Monitor (void *arg)
{
  unsigned char cmd[5], res[256], sad, dad;
  unsigned short lr, ctn;
  char ret;
#ifdef HAVE_NANOSLEEP
  struct timespec req_ts;

  req_ts.tv_sec = 1;
  req_ts.tv_nsec = 0;
#endif

  ctn = *(unsigned short *) arg;
  free (arg);

  while (ct_list[ctn].pn != 0)
    {
      /* Check if card is inserted */
      cmd[0] = CTBCS_CLA;
      cmd[1] = CTBCS_INS_STATUS;
      cmd[2] = CTBCS_P1_CT_KERNEL;
      cmd[3] = CTBCS_P2_STATUS_ICC;
      cmd[4] = 0x00;

      dad = 1;
      sad = 2;
      lr = 256;

      ret = CT_data (ctn, &dad, &sad, 5, cmd, &lr, res);

      if ((ret != OK) || (res[lr-2] != 0x90))
        continue;
      
      pthread_mutex_lock (&(ct_list[ctn].mutex));
      
      if (res[0] == CTBCS_DATA_STATUS_CARD_CONNECT)
        {            
          if (ct_list[ctn].status == -1)
            {
              /* Activate card */
              cmd[0] = CTBCS_CLA;
              cmd[1] = CTBCS_INS_REQUEST;
              cmd[2] = CTBCS_P1_INTERFACE1;
              cmd[3] = CTBCS_P2_REQUEST_GET_ATR;
              cmd[4] = 0x00;

              dad = 0x01;
              sad = 0x02;
              lr = 256;

              ret = CT_data (ctn, &dad, &sad, 5, cmd, &lr, res);

              if ((ret != OK) || (res[lr - 2] != 0x90))
                {
                  pthread_mutex_unlock (&(ct_list[ctn].mutex));
                  continue;
                }   
              
              /* Store the type of card */
              ct_list[ctn].status = res[lr - 1];

              /* Store ATR */
              memcpy (ct_list[ctn].atr, res, lr - 2);
              ct_list[ctn].atr_size = lr - 2;
              
              pthread_mutex_unlock (&(ct_list[ctn].mutex));
            }
        }

      else
        {
          if (ct_list[ctn].status != -1)
            {
              ct_list[ctn].status = -1;
              memset (ct_list[ctn].atr, 0, 33);
              ct_list[ctn].atr_size = 0;
            }
        }
      
      pthread_mutex_unlock (&(ct_list[ctn].mutex));
#ifdef HAVE_NANOSLEEP
      nanosleep (&req_ts, NULL);
#else
      usleep(999999);
#endif
    }

  return NULL;
}
#endif

void
Initialize (void)
{
  unsigned short int pn, ctn;
  char ret;
  int dummy;

  printf ("Port number [COM>=1, USB>=32769]: ");
  scanf ("%hu", &pn);
  dummy = getchar ();

  if (pn < 1)
    {
      printf ("Invalid port number\n");
      return;
    }

  for (ctn = 0; ctn < 4; ctn++)
    {
      if ((ct_list[ctn].pn == pn))
        {
          printf ("Port already open\n");
          return;
        }
      else if (ct_list[ctn].pn == 0)
        break;
    }

  if (ctn > 4)
    {
      printf ("Maximum number of ports open\n");
      return;
    }

  printf ("Initializing terminal at COM%d...\n", pn);

#ifndef CTAPI_WIN32_COM
  ret = CT_init (ctn, pn - 1);
#else
  ret = CT_init (ctn, pn);
#endif

  if (ret != OK)
    {
      fprintf (stderr, "Error on port allocation: %d\n", ret);
      return;
    }

  ct_list[ctn].pn = pn;
  
#ifdef HAVE_PTHREAD_H
    {
      unsigned short *arg;

      arg = (unsigned short *) malloc (sizeof (unsigned short));
      (*arg) = ctn;

      printf ("Starting terminal monitoring job...\n");

      pthread_create (&(ct_list[ctn].thread), NULL, Monitor, (void *) arg);

      {
        /* Wait 0.1 seconds before returning to the menu loop
          to give time the monitor thread to adquire the mutex */

#ifdef HAVE_NANOSLEEP
        struct timespec req_ts;

        req_ts.tv_sec = 0;
        req_ts.tv_nsec = 100000000;

        nanosleep (&req_ts, NULL);
#else
        usleep(100000);
#endif
      }
    }
#else
  if (!GetSmartcard (ctn))
    Close (ctn);
#endif
}

unsigned short
Change (unsigned short ctn)
{
  unsigned short pn, i;
  int dummy;

  /* Show the list of open ports */
  printf ("Port number (");

  for (i = 0; i < 4; i++)
    {
      if (ct_list[i].pn != 0)
        printf ("%d", ct_list[i].pn);
      if (i + 1 < 4)
        {
          if (ct_list[i + 1].pn != 0)
            printf (", ");
        }
      else
        printf ("): ");
    }

  scanf ("%hu", &pn);
  dummy = getchar ();

  /* Search the ctn */
  for (i = 0; i < 4; i++)
    {
      if (ct_list[i].pn == pn)
        break;
    }

  if (i >= 4)
    {
      printf ("Invalid port number\n");
      return ctn;
    }

  return i;
}

void
Close (unsigned short ctn)
{
  unsigned short pn;
  char ret;
 
  pn = ct_list[ctn].pn;

#ifdef HAVE_PTHREAD_H
  pthread_mutex_lock (&(ct_list[ctn].mutex));
#endif
  ct_list[ctn].pn = 0;
  ct_list[ctn].status = -1;
  memset (ct_list[ctn].atr, 0, 33);
  ct_list[ctn].atr_size = 0;
#ifdef HAVE_PTHREAD_H
  pthread_mutex_unlock (&(ct_list[ctn].mutex));
  
  printf ("Waiting for terminal monitoring job to stop...\n");
  pthread_join(ct_list[ctn].thread ,NULL);
#endif

  printf ("Closing terminal at COM%d\n", pn);

  ret = CT_close (ctn);
  if (ret != OK)
    printf ("Error closing terminal at COM%d\n", pn);
}

void
SelectFile (unsigned short ctn)
{
  unsigned char select_file[8] =
    { CLASS, 0xA4, 0x00, 0x00, 0x02, 0x3f, 0x00, 0x00 };
  unsigned char buffer[32];
  unsigned char res[256];
  unsigned char dad;
  unsigned char sad;
  unsigned short lr;
  int i, dummy;
  char ret;

  printf ("File ID: ");
  scanf ("%X %X", buffer, buffer + 1);
  dummy = getchar ();

  select_file[5] = buffer[0];
  select_file[6] = buffer[1];

  dad = 0;
  sad = 2;
  lr = 256;
  printf ("Command: ");

  for (i = 0; i < 8; i++)
    printf ("%02X ", select_file[i]);
  printf ("\n");

  ret = CT_data (ctn, &dad, &sad, 8, select_file, &lr, res);

  if (ret != OK)
    {
      fprintf (stderr, "Error in SELECT FILE: %d\n", ret);
      return;
    }

  printf ("Response: ");

  for (i = 0; i < lr; i++)
    printf ("%02X ", res[i]);
  printf ("\n");
}

void
GetResponse (unsigned short ctn)
{
  unsigned char get_response[5] = { CLASS, 0xC0, 0x00, 0x00, 0x00 };
  char buffer[32];
  unsigned char res[256];
  unsigned char dad;
  unsigned char sad;
  unsigned short lr;
  int i, dummy;
  char ret;

  printf ("Response size (hexadecimal): ");
  scanf ("%X", buffer);
  dummy = getchar ();
  get_response[4] = buffer[0];

  dad = 0;
  sad = 2;
  lr = 256;

  printf ("Command: ");
  for (i = 0; i < 5; i++)
    printf ("%02X ", get_response[i]);
  printf ("\n");

  ret = CT_data (ctn, &dad, &sad, 5, get_response, &lr, res);

  if (ret != OK)
    {
      fprintf (stderr, "Error on GET RESPONSE: %d\n", ret);
      return;
    }

  printf ("Response: ");
  for (i = 0; i < lr; i++)
    printf ("%02X ", res[i]);
  printf ("\n");
}

void
UpdateBinary (unsigned short ctn)
{
  unsigned char update_binary[260] = { CLASS, 0xD6, 0x00, 0x00, 0x00 };
  char buffer[32];
  unsigned char res[256];
  unsigned char dad;
  unsigned char sad;
  unsigned short lr;
  int i, size, dummy;
  char ret;

  printf ("File size (0..255): ");
  scanf ("%d", &size);
  dummy = getchar ();
  update_binary[4] = (unsigned char) (size % 256);

  printf ("Data: ");
  scanf ("%02X", buffer);
  dummy = getchar ();

  for (i = 5; i < (int) update_binary[4] + 5; i++)
    update_binary[i] = buffer[0];

  dad = 0;
  sad = 2;
  lr = 256;

  printf ("Command: ");
  for (i = 0; i < (int) update_binary[4] + 5; i++)
    printf ("%02X ", update_binary[i]);
  printf ("\n");

  ret =
    CT_data (ctn, &dad, &sad, (int) update_binary[4] + 5, update_binary, &lr,
             res);

  if (ret != OK)
    {
      fprintf (stderr, "Error on UPDATE BYNARY: %d\n", ret);
      return;
    }

  printf ("Response: ");
  for (i = 0; i < lr; i++)
    printf ("%02X ", res[i]);
  printf ("\n");
}

void
ReadBinary (unsigned short ctn)
{
  unsigned char read_binary[5] = { CLASS, 0xB0, 0x00, 0x00, 0x00 };
  char buffer[32];
  unsigned char res[258];
  unsigned char dad;
  unsigned char sad;
  unsigned short lr;
  int i, dummy;
  char ret;

  /* Read binary */
  printf ("File size: ");
  scanf ("%X", buffer);
  dummy = getchar ();
  read_binary[4] = buffer[0];

  dad = 0;
  sad = 2;
  lr = 258;

  printf ("Command: ");
  for (i = 0; i < 5; i++)
    printf ("%02X ", read_binary[i]);
  printf ("\n");

  ret = CT_data (ctn, &dad, &sad, 5, read_binary, &lr, res);

  if (ret != OK)
    {
      fprintf (stderr, "Error on READ BYNARY: %d\n", ret);
      return;
    }

  printf ("Response: ");
  for (i = 0; i < lr; i++)
    printf ("%02X ", res[i]);
  printf ("\n");
}

void
VerifyKey (unsigned short ctn)
{
  unsigned char verify_key[13] =
    { 0xF0, 0x2A, 0x00, 0x01, 0x08, 0x47, 0x46, 0x58, 0x49, 0x32, 0x56, 0x78,
    0x40
  };
  unsigned char res[256];
  unsigned char dad;
  unsigned char sad;
  unsigned short lr;
  int i;
  char ret;

  dad = 0;
  sad = 2;
  lr = 256;
  printf ("Command: ");
  for (i = 0; i < 13; i++)
    printf ("%02X ", verify_key[i]);
  printf ("\n");

  ret = CT_data (ctn, &dad, &sad, 13, verify_key, &lr, res);

  if (ret != OK)
    {
      fprintf (stderr, "Error on VERIFY KEY: %d\n", ret);
      return;
    }

  printf ("Response: ");

  for (i = 0; i < lr; i++)
    printf ("%02X ", res[i]);
  printf ("\n");
}

void
SendPPS (unsigned short ctn)
{
  unsigned char reset[9] = {0x20, 0x11, 0x01, 0x01, 0x03, 0xFF, 0x00, 0x00, 0x00};
  unsigned char buffer[32], res[256];
  unsigned char dad, sad;
  unsigned short lr;
  int dummy, i;
  char ret;

  printf ("PPS request (PPSS PPS0 PPS1): ");
  scanf ("%X %X %X", buffer, buffer + 1, buffer + 2);
  dummy = getchar ();

  reset[5] = buffer[0];
  reset[6] = buffer[1];
  reset[7] = buffer[2];

  dad = 0x01;
  sad = 0x02;
  lr = 256;

  printf ("Command: ");
  for (i = 0; i < 8; i++)
    printf ("%02X ", reset[i]);
  printf ("\n");

  ret = CT_data (ctn, &dad, &sad, 9, reset, &lr, res);

  if (ret != OK)
    {
      fprintf (stderr, "Error on Reset CT (Send  PPS): %d\n", ret);
      return;
    }

  printf ("Response: ");

  for (i = 0; i < lr; i++)
    printf ("%02X ", res[i]);
  printf ("\n");
}

void
EnterPin (unsigned short ctn)
{
  unsigned char verify[8] =
    { 0x00, 0x20, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00 };
  unsigned char buffer[32];
  unsigned char res[256];
  unsigned char dad;
  unsigned char sad;
  unsigned short lr;
  int i, dummy;
  char ret;

  printf ("PIN (3 bytes): ");
  scanf ("%X %X %X", buffer, buffer + 1, buffer + 2);
  dummy = getchar ();

  verify[5] = buffer[0];
  verify[6] = buffer[1];
  verify[7] = buffer[2];

  dad = 0;
  sad = 2;
  lr = 256;

  printf ("Command: ");
  for (i = 0; i < 8; i++)
    printf ("%02X ", verify[i]);
  printf ("\n");

  ret = CT_data (ctn, &dad, &sad, 8, verify, &lr, res);

  if (ret != OK)
    {
      fprintf (stderr, "Error on VERIFY: %d\n", ret);
      return;
    }

  printf ("Response: ");

  for (i = 0; i < lr; i++)
    printf ("%02X ", res[i]);
  printf ("\n");
}

void
ChangePin (unsigned short ctn)
{
  unsigned char change[11] =
    { 0x00, 0x24, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
  unsigned char buffer[32];
  unsigned char res[256];
  unsigned char dad;
  unsigned char sad;
  unsigned short lr;
  int i, dummy;
  char ret;

  printf ("PIN (3 bytes): ");
  scanf ("%X %X %X", buffer, buffer + 1, buffer + 2);
  dummy = getchar ();

  change[5] = buffer[0];
  change[6] = buffer[1];
  change[7] = buffer[2];

  printf ("New PIN (3 bytes): ");
  scanf ("%X %X %X", buffer, buffer + 1, buffer + 2);
  dummy = getchar ();

  change[8] = buffer[0];
  change[9] = buffer[1];
  change[10] = buffer[2];

  dad = 0;
  sad = 2;
  lr = 256;

  printf ("Command: ");
  for (i = 0; i < 11; i++)
    printf ("%02X ", change[i]);
  printf ("\n");

  ret = CT_data (ctn, &dad, &sad, 11, change, &lr, res);

  if (ret != OK)
    {
      fprintf (stderr, "Error on CHANGE VERIFICATION DATA: %d\n", ret);
      return;
    }

  printf ("Response: ");

  for (i = 0; i < lr; i++)
    printf ("%02X ", res[i]);
  printf ("\n");
}

void
ReadData (unsigned short ctn)
{
  unsigned char select_file[7] = { 0x00, 0xA4, 0x00, 0x00, 0x02, 0x3F, 0x00 };
  unsigned char read_binary[5] = { 0x00, 0xB0, 0x00, 0x00, 0x00 };
  int address;
  int size;
  unsigned char res[258];
  unsigned char dad;
  unsigned char sad;
  unsigned short lr;
  int i, dummy;
  char ret;

  printf ("Address: ");
  scanf ("%d", &address);
  dummy = getchar ();

  read_binary[2] = (unsigned char) (address >> 8);
  read_binary[3] = (unsigned char) (address & 0x00FF);

  printf ("Size (0..255): ");
  scanf ("%d", &size);
  dummy = getchar ();

  read_binary[4] = (unsigned char) size;

  printf ("Command: ");
  for (i = 0; i < 7; i++)
    printf ("%02X ", select_file[i]);
  printf ("\n");

  dad = 0;
  sad = 2;
  lr = 258;

  /* Select MF */
  ret = CT_data (ctn, &dad, &sad, 7, select_file, &lr, res);

  if (ret != OK)
    {
      fprintf (stderr, "Error on SELECT FILE: %d\n", ret);
      return;
    }

  printf ("Response: ");
  for (i = 0; i < lr; i++)
    printf ("%02X ", res[i]);
  printf ("\n");

  printf ("Command: ");
  for (i = 0; i < 5; i++)
    printf ("%02X ", read_binary[i]);
  printf ("\n");

  dad = 0;
  sad = 2;
  lr = 258;

  /* Read binary */
  ret = CT_data (ctn, &dad, &sad, 5, read_binary, &lr, res);

  if (ret != OK)
    {
      fprintf (stderr, "Error on READ BINARY: %d\n", ret);
      return;
    }

  printf ("Response: ");
  for (i = 0; i < lr; i++)
    printf ("%02X ", res[i]);
  printf ("\n");
}

void
WriteData (unsigned short ctn)
{
  unsigned char select_file[7] = { 0x00, 0xA4, 0x00, 0x00, 0x02, 0x3F, 0x00 };
  unsigned char update_binary[260] = { 0x00, 0xD6, 0x00, 0x00, 0x00 };
  int address;
  int size;
  unsigned char buffer[32];
  unsigned char res[256];
  unsigned char dad;
  unsigned char sad;
  unsigned short lr;
  int i, dummy;
  char ret;

  printf ("Address: ");
  scanf ("%d", &address);
  dummy = getchar ();

  update_binary[2] = (unsigned char) (address >> 8);
  update_binary[3] = (unsigned char) (address & 0x00FF);

  printf ("Size (0..255): ");
  scanf ("%d", &size);
  dummy = getchar ();

  update_binary[4] = (unsigned char) (size % 256);

  printf ("Data: ");
  scanf ("%X", buffer);
  dummy = getchar ();

  for (i = 5; i < update_binary[4] + 5; i++)
    update_binary[i] = buffer[0];

  printf ("Command: ");
  for (i = 0; i < 7; i++)
    printf ("%02X ", select_file[i]);
  printf ("\n");

  dad = 0;
  sad = 2;
  lr = 256;

  /* Select MF */
  ret = CT_data (ctn, &dad, &sad, 7, select_file, &lr, res);

  if (ret != OK)
    {
      fprintf (stderr, "Error on SELECT FILE: %d\n", ret);
      return;
    }

  printf ("Response: ");
  for (i = 0; i < lr; i++)
    printf ("%02X ", res[i]);
  printf ("\n");

  printf ("Command: ");
  for (i = 0; i < update_binary[4] + 5; i++)
    printf ("%02X ", update_binary[i]);
  printf ("\n");

  dad = 0;
  sad = 2;
  lr = 256;

  /* Update binary */
  ret =
    CT_data (ctn, &dad, &sad, update_binary[4] + 5, update_binary, &lr, res);

  if (ret != OK)
    {
      fprintf (stderr, "Error on UPDATE BINARY: %d\n", ret);
      return;
    }

  printf ("Response: ");
  for (i = 0; i < lr; i++)
    printf ("%02X ", res[i]);
  printf ("\n");
}

