/* alarms.c -- draw the alarm settings dialog of TuxTime Configuration Manager
 *
 * Copyright (c) 1996-99  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log: alarms.c,v $
 * Revision 1.2  1999/12/04 13:44:04  jab
 * changed for the new tuxtime-conf program
 *
 * Revision 1.1  1999/05/25 08:06:37  jab
 * Initial revision
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

static const char rcsid[]="$Id: alarms.c,v 1.2 1999/12/04 13:44:04 jab Exp jab $";


#include<stdio.h>
#include<stdlib.h>
#include<gtk/gtk.h>

#include"sci.h"
#include"misc.h"
#include"config.h"
#include"alarms.h"

#include"plus.xpm"
#include"minus.xpm"

static BatteryAlarm low,critical;


GtkWidget *lowtime,*lowpercent;
GtkWidget *criticaltime,*criticalpercent;
GtkWidget *lowtriggertime,*lowtriggerpercent;
GtkWidget *criticaltriggertime, *criticaltriggerpercent;

GdkPixmap *pluspixmap,*minuspixmap;
GdkBitmap *plusmask,*minusmask;


#define MINUTES(h,m) ((h*60)+m)

#define LOW_GT_CRITICAL_PERCENT "\
The Low Battery Percentage threshold\n\
must be greater than the\n\
Critical Battery Percentage threshold"

#define LOW_GT_CRITICAL_TIME "\
The Low Battery Time Remaining value\n\
must be greater than the\n\
Critical Battery Time Remaining value"

#define CRITICAL_LT_LOW_PERCENT "\
The Critical Battery Percentage threshold\n\
must be less than the\n\
Low Battery Percentage threshold"

#define CRITICAL_LT_LOW_TIME "\
The Critical Battery Time Remaining value\n\
must be greater than the\n\
Low Battery Time Remaining value"



/*
 * Update the percentage label and issue any necessary warnings
 */
void PercentCallback(GtkWidget *widget, gpointer data)
{
	char buffer[8];

	switch ((int) data) {
		case LOW_ALARM:
			low.percent++;
			if (low.percent==101)
				low.percent = 100;
			sprintf(buffer, "%d%%", low.percent);
			gtk_label_set(GTK_LABEL(lowpercent), buffer);
			break;
		case (-1*LOW_ALARM):
			low.percent--;
			if (low.percent==5)
				low.percent = 6;
			else if (low.percent<=critical.percent) {
				low.percent++;
				GuiError(ERROR_MESSAGE, LOW_GT_CRITICAL_PERCENT);
			}
			else {
				sprintf(buffer, "%d%%", low.percent);
				gtk_label_set(GTK_LABEL(lowpercent), buffer);
			}
			break;
		case CRITICAL_ALARM:
			critical.percent++;
			if (critical.percent==101)
				critical.percent = 100;
			else if (critical.percent>low.percent) {
				critical.percent--;
				GuiError(ERROR_MESSAGE, CRITICAL_LT_LOW_PERCENT);				
			}
			else {
				sprintf(buffer, "%d%%", critical.percent);
				gtk_label_set(GTK_LABEL(criticalpercent), buffer);
			}
			break;
		case (-1*CRITICAL_ALARM):
			critical.percent--;
			if (critical.percent==5)
				critical.percent = 6;
			else {
				sprintf(buffer, "%d%%", critical.percent);
				gtk_label_set(GTK_LABEL(criticalpercent), buffer);
			}
			break;
	}

	return;
}


/*
 * Update the alarm time label for hour changes and issue any necessary warnings
 */
void BatteryHourCallback(GtkWidget *widget, gpointer data)
{
	char buffer[8];

	switch ((int) data) {
		case LOW_ALARM:
			low.hour++;
			if (low.hour==10)
				low.hour = 9;
			sprintf(buffer, "%02d:%02d", low.hour, low.minute);
			gtk_label_set(GTK_LABEL(lowtime), buffer);
			break;
		case (-1*LOW_ALARM):
			low.hour--;
			if (low.hour==-1)
				low.hour = 0;
			else if (MINUTES(low.hour, low.minute)<=
				MINUTES(critical.hour, critical.minute)) {
				low.hour++;
				GuiError(ERROR_MESSAGE, LOW_GT_CRITICAL_TIME);				
			}
			else {
				sprintf(buffer, "%02d:%02d", low.hour, low.minute);
				gtk_label_set(GTK_LABEL(lowtime), buffer);
			}
			break;
		case CRITICAL_ALARM:
			critical.hour++;
			if (critical.hour==10)
				critical.hour = 9;
			else if (MINUTES(critical.hour, critical.minute)>=
				MINUTES(low.hour, low.minute)) {
				critical.hour--;
				GuiError(ERROR_MESSAGE, CRITICAL_LT_LOW_TIME);				
			}
			else {
				sprintf(buffer, "%02d:%02d", critical.hour, critical.minute);
				gtk_label_set(GTK_LABEL(criticaltime), buffer);
			}
			break;
		case (-1*CRITICAL_ALARM):
			critical.hour--;
			if (critical.hour==-1)
				critical.hour = 0;
			sprintf(buffer, "%02d:%02d", critical.hour, critical.minute);
			gtk_label_set(GTK_LABEL(criticaltime), buffer);
			break;
	}

	return;
}


/*
 * Update the alarm time label for minutes and issue any necessary warnings
 */
void BatteryMinuteCallback(GtkWidget *widget, gpointer data)
{
	char buffer[8];

	switch ((int) data) {
		case LOW_ALARM:
			low.minute++;
			if (low.minute==60)
				low.minute = 59;
			sprintf(buffer, "%02d:%02d", low.hour, low.minute);
			gtk_label_set(GTK_LABEL(lowtime), buffer);
			break;
		case (-1*LOW_ALARM):
			low.minute--;
			if (low.minute==5)
				low.minute = 6;
			else if (MINUTES(low.hour, low.minute)<=
				MINUTES(critical.hour, critical.minute)) {
				low.minute++;
				GuiError(ERROR_MESSAGE, LOW_GT_CRITICAL_TIME);				
			}
			else {
				sprintf(buffer, "%02d:%02d", low.hour, low.minute);
				gtk_label_set(GTK_LABEL(lowtime), buffer);
			}
			break;
		case CRITICAL_ALARM:
			critical.minute++;
			if (critical.minute==60)
				critical.minute = 59;
			else if (MINUTES(critical.hour, critical.minute)>=
				MINUTES(low.hour, low.minute)) {
				critical.minute--;
				GuiError(ERROR_MESSAGE, CRITICAL_LT_LOW_TIME);				
			}
			else {
				sprintf(buffer, "%02d:%02d", critical.hour, critical.minute);
				gtk_label_set(GTK_LABEL(criticaltime), buffer);
			}
			break;
		case (-1*CRITICAL_ALARM):
			critical.minute--;
			if (critical.minute==5)
				critical.minute = 6;
			sprintf(buffer, "%02d:%02d", critical.hour, critical.minute);
			gtk_label_set(GTK_LABEL(criticaltime), buffer);
			break;
	}

	return;
}


/*
 * Update the stored setting of the trigger status
 */
void TriggerCallback(GtkWidget *widget, gpointer data)
{

	switch (((int) data) & 0xf000) {
		case LOW_ALARM:
			low.trigger ^= ((int) data) & 0x0fff;
			break;
		case CRITICAL_ALARM:
			critical.trigger ^= ((int) data) & 0x0fff;
			break;
	}

	return;
}


/*
 * Set the active radio button for the alarm trigger
 */
void SetActiveTriggerButton(int toggle)
{
	int time,percent;

	switch (toggle & 0x0fff) {
		case (TIME_ALARM):
			time = TRUE;
			percent = FALSE;
			break;
		case (PERCENT_ALARM):
			time = FALSE;
			percent = TRUE;
			break;
		case (BOTH_ALARM):
			time = percent = TRUE;
			break;
		case (NEVER_ALARM):
			time = percent = FALSE;
			break;
		default:
			time = percent = FALSE;
			break;
	}

	switch (toggle & 0xf000) {
		case LOW_ALARM:
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(lowtriggertime),
				time);
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(lowtriggerpercent),
				percent);
			break;
		case CRITICAL_ALARM:
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(criticaltriggertime),
				time);
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(criticaltriggerpercent),
				percent);
			break;
	}

	return;
}


/*
 * Save the alarm settings to the configuration file and close dialog
 */
void SaveAlarmSettingsCallback(GtkWidget *widget, gpointer *data)
{
	char buffer[16];

	sprintf(buffer, "%d", low.percent);
	WriteConfigString("AlarmLow", "Percent", buffer, config);
	sprintf(buffer, "%02d:%02d", low.hour, low.minute);
	WriteConfigString("AlarmLow", "Time", buffer, config);
	sprintf(buffer, "%d", low.trigger);
	WriteConfigString("AlarmLow", "Trigger", buffer, config);

	sprintf(buffer, "%d", critical.percent);
	WriteConfigString("AlarmCritical", "Percent", buffer, config);
	sprintf(buffer, "%02d:%02d", critical.hour, critical.minute);
	WriteConfigString("AlarmCritical", "Time", buffer, config);
	sprintf(buffer, "%d", critical.trigger);
	WriteConfigString("AlarmCritical", "Trigger", buffer, config);

	gtk_widget_destroy(GTK_WIDGET(data));

	ExitTuxTimeConfig(0x10);

	return;
}


/*
 * Return the settings to their default values
 */
void SetAlarmDefaultsCallback(GtkWidget *widget, gpointer *data)
{
	char buffer[8];

	/* return the low battery alarm settings to their defaults */

	low.hour = 0;
	low.minute = 20;
	sprintf(buffer, "%02d:%02d", low.hour, low.minute);
	gtk_label_set(GTK_LABEL(lowtime), buffer);

	low.percent = 20;
	sprintf(buffer, "%d%%", low.percent);
	gtk_label_set(GTK_LABEL(lowpercent), buffer);

	low.trigger = BOTH_ALARM;
	SetActiveTriggerButton(BOTH_ALARM | LOW_ALARM);

	/* return the critical battery alarm settings to their defaults */

	critical.hour = 0;
	critical.minute = 10;
	sprintf(buffer, "%02d:%02d", critical.hour, critical.minute);
	gtk_label_set(GTK_LABEL(criticaltime), buffer);

	critical.percent = 10;
	sprintf(buffer, "%d%%", critical.percent);
	gtk_label_set(GTK_LABEL(criticalpercent), buffer);

	critical.trigger = BOTH_ALARM;
	SetActiveTriggerButton(BOTH_ALARM | CRITICAL_ALARM);

	return;
}


/*
 * Quit the alarms dialog without changing the settings

 */
void AlarmCancelCallback(GtkWidget *widget, GtkWidget *dialog)
{
	gtk_widget_destroy(dialog);
	dialog = NULL;

	gdk_pixmap_unref(pluspixmap);
	gdk_bitmap_unref(plusmask);
	gdk_pixmap_unref(minuspixmap);
	gdk_bitmap_unref(minusmask);

	ExitTuxTimeConfig(0);
	
	return;
}


/*
 * Build the Alarm Setting pages of the notebook
 */
void AddAlarmPage(GtkNotebook *notebook, GtkWidget *window, int type)
{
	GtkWidget *box,*frame,*row,*column;
	GtkWidget *format,*trigger,*label,*button;
	GtkWidget *buttons,*alarm,*layout;
	GtkWidget *time,*percent;
	GtkWidget *icon;
	char buffer[8];


	box = gtk_hbox_new(FALSE, 0);

	if (type==LOW_ALARM) {
		frame = gtk_frame_new("Low Battery Alarm");
		low.percent = GetConfigInt("AlarmLow", "Percent", 20, config);
		low.trigger = GetConfigInt("AlarmLow", "Trigger", 3, config);
		GetConfigString("AlarmLow", "Time", "00:20", buffer, 8, config);
		ConvertTime(buffer, &(low.hour), &(low.minute));
	} else {
		frame = gtk_frame_new("Critical Battery Alarm");
		critical.percent = GetConfigInt("AlarmCritical", "Percent", 10,
			config);
		critical.trigger = GetConfigInt("AlarmCritical", "Trigger", 3,
			config);
		GetConfigString("AlarmCritical", "Time", "00:10", buffer, 8,
			config);
		ConvertTime(buffer, &(critical.hour), &(critical.minute));
	}

	gtk_container_border_width(GTK_CONTAINER(frame), 4);
	gtk_box_pack_start(GTK_BOX(box), frame, TRUE, TRUE, 0);

	buttons = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(buttons), 4);
	gtk_box_pack_start(GTK_BOX(box), buttons, FALSE, TRUE, 4);

	AddButtons(buttons, window, BUTTONS_ALARM);

	/* add main frame for the page */

	alarm = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(alarm), 4);
	gtk_container_add(GTK_CONTAINER(frame), alarm);

	/* add a horizontal box to hold the time and percent adjustments */

	row = gtk_hbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(row), 4);
	gtk_box_pack_start(GTK_BOX(alarm), row, TRUE, TRUE, 0);

	/* vertical box for the time adjustment */

	column = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(column), 0);
	gtk_box_pack_start(GTK_BOX(row), column, TRUE, TRUE, 0);

	/* blank titled frame for time adjustment */

	frame = gtk_frame_new(NULL);
	gtk_container_border_width(GTK_CONTAINER(frame), 4);
	gtk_box_pack_start(GTK_BOX(column), frame, TRUE, TRUE, 0);

	/* fill out the time frame */

	format = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(format), 4);
	gtk_container_add(GTK_CONTAINER(frame), format);

	/* set the battery alarm time label */

	if (type==LOW_ALARM) {
		sprintf(buffer, "%02d:%02d", low.hour, low.minute);
		lowtime = gtk_label_new(buffer);
		gtk_box_pack_start(GTK_BOX(format), lowtime, FALSE, TRUE, 0);
	}
	else if (type==CRITICAL_ALARM) {
		sprintf(buffer, "%02d:%02d", critical.hour, critical.minute);
		criticaltime = gtk_label_new(buffer);
		gtk_box_pack_start(GTK_BOX(format), criticaltime, FALSE, TRUE, 0);
	}

	layout = gtk_hbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(layout), 4);
	gtk_box_pack_start(GTK_BOX(format), layout, FALSE, TRUE, 0);


	/* buttons for adjusting the battery alarm time hour */

	button = gtk_button_new();
	icon = gtk_pixmap_new(pluspixmap, plusmask);
	gtk_widget_set_usize(button, 22, 22);
	gtk_container_add(GTK_CONTAINER(button), icon);
	gtk_box_pack_start(GTK_BOX(layout), button, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(BatteryHourCallback), (gpointer) type);

	button = gtk_button_new();
	icon = gtk_pixmap_new(minuspixmap, minusmask);
	gtk_widget_set_usize(button, 22, 22);
	gtk_container_add(GTK_CONTAINER(button), icon);
	gtk_box_pack_start(GTK_BOX(layout), button, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(BatteryHourCallback), (gpointer) (-1*type));	

	/* buttons for adjusting the battery alarm time minutes */

	button = gtk_button_new();
	icon = gtk_pixmap_new(minuspixmap, minusmask);
	gtk_widget_set_usize(button, 22, 22);
	gtk_container_add(GTK_CONTAINER(button), icon);
	gtk_box_pack_end(GTK_BOX(layout), button, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(BatteryMinuteCallback), (gpointer) (-1*type));

	button = gtk_button_new();
	icon = gtk_pixmap_new(pluspixmap, plusmask);
	gtk_widget_set_usize(button, 22, 22);
	gtk_container_add(GTK_CONTAINER(button), icon);
	gtk_box_pack_end(GTK_BOX(layout), button, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(BatteryMinuteCallback), (gpointer) (1*type));

	layout = gtk_hbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(layout), 4);
	gtk_box_pack_start(GTK_BOX(format), layout, FALSE, TRUE, 0);

	label = gtk_label_new("Hours");
	gtk_box_pack_start(GTK_BOX(layout), label, TRUE, TRUE, 0);
	label = gtk_label_new("Minutes");
	gtk_box_pack_start(GTK_BOX(layout), label, TRUE, TRUE, 0);

	/* add the threshold label */

	label = gtk_label_new("Time Remaining\nThreshold");
	gtk_box_pack_start(GTK_BOX(column), label, TRUE, TRUE, 0);

	/* vertical box for the percent adjustment */

	column = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(column), 0);
	gtk_box_pack_start(GTK_BOX(row), column, TRUE, TRUE, 0);

	/* blank titled frame for percent adjustment */

	frame = gtk_frame_new(NULL);
	gtk_container_border_width(GTK_CONTAINER(frame), 4);
	gtk_box_pack_start(GTK_BOX(column), frame, TRUE, TRUE, 0);

	/* fill out the percent frame */

	format = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(format), 4);
	gtk_container_add(GTK_CONTAINER(frame), format);

	/* set the battery alarm percent label */

	if (type==LOW_ALARM) {
		sprintf(buffer, "%d%%", low.percent);
		lowpercent = gtk_label_new(buffer);
		gtk_box_pack_start(GTK_BOX(format), lowpercent, FALSE, TRUE, 0);
	}
	else if (type==CRITICAL_ALARM) {
		sprintf(buffer, "%d%%", critical.percent);
		criticalpercent = gtk_label_new(buffer);
		gtk_box_pack_start(GTK_BOX(format), criticalpercent, FALSE, TRUE, 0);
	}

	layout = gtk_hbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(layout), 4);
	gtk_box_pack_start(GTK_BOX(format), layout, FALSE, TRUE, 0);

	/* buttons for adjusting the battery alarm percent */

	button = gtk_button_new();
	icon = gtk_pixmap_new(pluspixmap, plusmask);
	gtk_widget_set_usize(button, 22, 22);
	gtk_container_add(GTK_CONTAINER(button), icon);
	gtk_box_pack_start(GTK_BOX(layout), button, TRUE, FALSE, 4);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(PercentCallback), (gpointer) type);

	button = gtk_button_new();
	icon = gtk_pixmap_new(minuspixmap, minusmask);
	gtk_widget_set_usize(button, 22, 22);
	gtk_container_add(GTK_CONTAINER(button), icon);
	gtk_box_pack_start(GTK_BOX(layout), button, TRUE, FALSE, 4);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(PercentCallback), (gpointer) (-1*type));

	layout = gtk_hbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(layout), 4);
	gtk_box_pack_start(GTK_BOX(format), layout, FALSE, TRUE, 0);

	label = gtk_label_new("Percent");
	gtk_box_pack_start(GTK_BOX(layout), label, TRUE, TRUE, 0);

	/* add the threshold label */

	label = gtk_label_new("Percent Remaining\nThreshold");
	gtk_box_pack_start(GTK_BOX(column), label, TRUE, TRUE, 0);

	/* add the alarm trigger on selection box */

	frame = gtk_frame_new("Trigger Alarm");
	gtk_container_border_width(GTK_CONTAINER(frame), 4);
	gtk_box_pack_start(GTK_BOX(alarm), frame, TRUE, TRUE, 0);

	/* now pack the trigger alarm frame with radio buttons */

	trigger = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(trigger), 4);
	gtk_container_add(GTK_CONTAINER(frame), trigger);

	time = gtk_check_button_new_with_label("Based on Time Remaining Theshold");
	gtk_box_pack_start(GTK_BOX(trigger), time, FALSE, FALSE, 2);
	gtk_signal_connect(GTK_OBJECT(time), "pressed",
		GTK_SIGNAL_FUNC(TriggerCallback), (gpointer) (TIME_ALARM+type));

	percent = gtk_check_button_new_with_label("Based on Percent Remaining Threshold");
	gtk_box_pack_start(GTK_BOX(trigger), percent, FALSE, FALSE, 2);
	gtk_signal_connect(GTK_OBJECT(percent), "pressed",
		GTK_SIGNAL_FUNC(TriggerCallback), (gpointer) (PERCENT_ALARM+type));


	/* set the active radio button for the alarm trigger */

	if (type==LOW_ALARM) {
		lowtriggertime = time;
		lowtriggerpercent = percent;
		SetActiveTriggerButton(low.trigger | LOW_ALARM);
		label = gtk_label_new("Low Battery Alarm");
	}
	else {
		criticaltriggertime = time;
		criticaltriggerpercent = percent;
		SetActiveTriggerButton(critical.trigger | CRITICAL_ALARM);
		label = gtk_label_new("Critical Battery Alarm");
	}

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), box, label);

	return;
}


/*
 * Build the alarms setting dialog
 */
void AlarmDialog(GtkWidget *window)
{
	GtkWidget *notebook;
	GdkColor *transparent = NULL;


	/* create a new notebook, place the position of the tabs */

	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_container_add(GTK_CONTAINER(window), notebook);

	/* create the pixmaps used by the dialog */

	pluspixmap = gdk_pixmap_create_from_xpm_d(window->window,
		&plusmask, transparent, plus_xpm);
	minuspixmap = gdk_pixmap_create_from_xpm_d(window->window,
		&minusmask, transparent, minus_xpm);


	/* add the two pages to the notebook */

	AddAlarmPage(GTK_NOTEBOOK(notebook), window, LOW_ALARM);
	AddAlarmPage(GTK_NOTEBOOK(notebook), window, CRITICAL_ALARM);

	/* set to start at first page */

	gtk_notebook_set_page(GTK_NOTEBOOK(notebook), 0);

	return;
}
