/* svpw.c -- enter/delete the supervisor password on a Toshiba laptop.
 *
 * Copyright (c) 1998  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log: svpw.c,v $
 * Revision 1.3  1999/07/25 14:41:53  jab
 * changed to use System Configuration Interface routines
 *
 * Revision 1.2  1998/04/19 16:14:45  jab
 * added check to make sure we are running at the console
 *
 * Revision 1.1  1998/04/13 20:22:11  jab
 * Initial revision
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

static const char rcsid[]="$Id: svpw.c,v 1.3 1999/07/25 14:41:53 jab Exp jab $";

#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<unistd.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif
#include<signal.h>
#include<termios.h>
#include<fcntl.h>
#include<paths.h>
#include<sys/ioctl.h>
#include<sys/vt.h>
#include<sys/kd.h>

#include"sci.h"

static int interface = 0;       /* is there an open interface to the laptop */
static int keyboard = -1;
static int terminal = 0;
static int fd;
struct termio saved;


#define PID_FILE _PATH_VARRUN "svpw.pid"

#define USAGE \
"This utility can be used to register the supervisor password and change the\n\
user password mode (allowed or not allowed to run SETUP).\n\n\
Usage: svpw [-u|h]\n\
    without parameters : Display if the supervisor password is registered or\n\
                         not, and allow to register/delete it.\n\
    -u : Display the current user password mode and allow it to be changed.\n"


/*
 * Exit the program closing any open interface to the BIOS if necessary
 */
void leave(int close)
{
	if (interface==1)
		SciCloseInterface();
	unlink(PID_FILE);

	exit(close);
	return;
}


/*
 * Catch any signals and exit the program in nicely.
 */
void catch(int x)
{
	fprintf(stderr, "svpw: caught signal %d, cleaning up...\n", x);
	if (keyboard!=-1)
		ioctl(fd, KDSKBMODE, keyboard);

	if (terminal==1)
		close(fd);

	leave(1);

	return;
}


/*
 * Print Incompatiable machine type and exit
 */
void Incompatible()
{
	fprintf(stderr, "svpw: Incompatiable machine type.\n");
	leave(1);
}


/*
 * Put terminal into raw mode and then get a single keypress
 */
int GetScanCode()
{
	struct termio new;
	unsigned char key[1];

	/* exit if unable to open the console */

  	if ((fd = open("/dev/tty", O_RDWR))<0) {
		fprintf(stderr, "svpw: unable to open console\n");
		leave(1);
	}
	terminal = 1;

	/* get the current terminal state */

	ioctl(fd, KDGKBMODE, &keyboard);
	ioctl(fd, TCGETA, &saved);
	ioctl(fd, TCGETA, &new);

	/* set appropriate terminal mode */

	new.c_lflag &= ~ (ICANON | ECHO | ISIG); 
	new.c_iflag = 0;
	new.c_cc[VMIN] = 1;
	new.c_cc[VTIME] = 1;

	ioctl(fd, TCSETAW, &new);
	ioctl(fd, KDSKBMODE, K_RAW);

	key[0] = 0;
	read(fd, key, 1);

	ioctl(fd, TCSETAW, &saved);
	ioctl(fd, KDSKBMODE, keyboard);
	keyboard = -1;

	/* close the connection to the terminal  */

	close(fd);
	terminal = 0;

	return (int) key[0];
}


/*
 * Get a yes/no reply from the user as a keypress
 */
int YesNo(void)
{
	int reply;
	struct termio new;

	/* exit if unable to open the console */

	if ((fd = open("/dev/tty", O_RDWR))<0) {
		fprintf(stderr, "svpw: unable to open console\n");
		leave(1);
	}
	terminal = 1;

	/* get the current terminal state */

	ioctl(fd, TCGETA, &saved);
	ioctl(fd, TCGETA, &new);

	/* set appropriate terminal mode */

	new.c_iflag = (IXON|IXOFF);
	new.c_oflag = 0;
	new.c_lflag = 0;
	new.c_cc[VMIN] = 1;
	new.c_cc[VTIME] = 0;

	ioctl(fd, TCSETAW, &new);

	for(reply=-1;reply==-1;) {
		switch(getchar()) {
			case 'y': case'Y':
				reply = 1;
				break;
			case 'n': case 'N':
				reply = 0;
				break;
			default:
				printf("\a");
				break;
		}
	}

	/* restore the terminal mode and start a newline*/

	ioctl(fd, TCSETAW, &saved);
	close(fd);
	terminal = 0;
	printf("\n");

	return (reply);
}


/*
 * Get a password from the user echoing astrix's
 */
void GetPassword(char *password)
{
	int loop,scan;

	/* blank the password */

	for(loop=0;loop<10;loop++)
		*(password+loop) = 0;

	fflush(stdout);

	/* now get the password from the user*/

	for (loop=0;;) {
		scan = GetScanCode();
		if ((scan<0x02) || (scan>0x39)) continue;
		if ((scan==0x1d) || (scan==0x2a) || (scan==0x36)
		      || (scan==0x37) || (scan==0x38)) continue;
		if (scan==0x0f) continue;
		if (scan==0x1c) {
			printf("\n");
			fflush(stdout);
			break;
		}
		if (scan==0x0e) {
			if (loop>0) {
				printf("\b \b");
				fflush(stdout);
				loop--;
				*(password+loop) = 0;
			}
			continue;
		}
		if (loop<10) {
			*(password+loop) = scan;
			printf("*");
			fflush(stdout);
			loop++;
		}
	}

	return;
}


/*
 * Set the supervisor for the laptop.
 */
int SetPassword(char *password, int *trys)
{
	SMMRegisters reg;

	reg.ebx = SCI_PASSWORD;
	reg.ecx = SCI_SUPER_PASSWORD;
	reg.edx = *(password+3) + *(password+2)*0x100 + *(password+1)*0x10000 +
	        *(password)*0x1000000;
	reg.esi = *(password+7) + *(password+6)*0x100 + *(password+5)*0x10000 +
	        *(password+4)*0x1000000;
	reg.edi = *(password+9) + *(password+8)*0x100;

	SciSet(&reg);

	*trys = (int) (reg.edx & 0xffff);

	return (int) (reg.eax & 0xff00)>>8;
}


/*
 * Change the user password mode; able/unable to run tsetup
 */
void UserMode()
{
	SMMRegisters reg;
	int error,reply;

	reg.ebx = SCI_PASSWORD_LEVEL;
	error = SciGet(&reg);

	if ((reg.ecx & 0x0001)==1)
		printf("USER PASSWORD MODE = Unable to run SETUP\n");
	else
		printf("USER PASSWORD MODE = Able to run SETUP\n");

	printf("Do you want to change the setting <Y/N>?");
	reply = YesNo();

	/* does the user want the setting changed? */
	
	if (reply==1) {
		reg.ebx = SCI_PASSWORD_LEVEL;
		reg.ecx ^= 0x0001;
		SciSet(&reg);
		if ((reg.ecx & 0x0001)==1)
			printf("USER PASSWORD MODE = Unable to run SETUP\n");
		else
			printf("USER PASSWORD MODE = Able to run SETUP\n");
	}

	return;
}


/*
 * Set/delete the Supervisor password
 */
void SuperPassword(int registered, int trys)
{
	int reply;
	char enter[10],verify[10];

	if (registered==0) {
		printf("SUPERVISOR PASSWORD = Not Registered\n");
		printf("Do you want to register the supervisor password <Y/N>?");
		reply = YesNo();
		while (reply==1) {
			printf("Enter password ---> ");
			GetPassword(enter);
			printf("Verify password --> ");
			GetPassword(verify);
			if (memcmp(enter, verify, 10)!=0) {
				printf("Password verify error !\nDo you want to retry <Y/N>?");
				reply = YesNo();
			}
			else {
				reply = 0;
				SetPassword(enter, &trys);
				printf("SUPERVISOR PASSWORD = Registered\n");
			}
		}
	}
	else {
		printf("SUPERVISOR PASSWORD = Registered\n");
		printf("Do you want to delete the supervisor password <Y/N>?");
		reply = YesNo();
		while (reply==1) {
			if (trys==0) {
				printf("Password access denied !\n");
				leave(3);
			}
			printf("Enter password ---> ");
			GetPassword(enter);
			reply = 0;
			if (SetPassword(enter, &trys)!=SCI_SUCCESS) {
				printf("Password verify error !\nDo you want to retry <Y/N>?");
				reply = YesNo();
			}
			else {
				printf("SUPERVISOR PASSWORD = Not Registered\n");
				reply = 0;
			}
		}
	}

	return;
}


int main(int argc, char *argv[])
{
	SMMRegisters reg;
	int pid,help,user,version;
	FILE *str;
	struct vt_mode vtm;


	/* this program *must* be run as root */

	if (getuid()) {
		fprintf(stderr, "svpw: must be run as root.\n" );
		return 1;
	}

	/* check that we are running at the console */

  	if ((fd = open("/dev/tty", O_RDWR))<0) {
		fprintf(stderr, "svpw: unable to open console.\n");
		return 1;
	}
	if (ioctl(fd, VT_GETMODE, &vtm)!=0) {
		fprintf(stderr, "svpw: not running at the console.\n");
		close(fd);
		return 1;
	}
	close(fd);

	/* process command line arguments */

 	help = user = 0;
	if ((--argc>0) && ((*++argv)[0]=='-'))
		switch (*++argv[0]) {
			case 'h':
				help = 1;
				break;
			case 'u':
				user = 1;
				break;
			default:
				printf("svpw: illegal option %s\n", argv[0]);
				return 1;
		}

	/* if we recieve a signal, exit nicely closing any interface to the BIOS,
	   deleting the run lock and putting the keyboard into a usable state */

	signal(SIGHUP, catch);
	signal(SIGINT, catch);
	signal(SIGQUIT, catch);
	signal(SIGILL, catch);
	signal(SIGTRAP, catch);
	signal(SIGABRT, catch);
	signal(SIGIOT, catch);
	signal(SIGFPE, catch);
	signal(SIGKILL, catch);
	signal(SIGSEGV, catch);
	signal(SIGPIPE, catch);
 	signal(SIGTERM, catch);
	signal(SIGCHLD, catch);
	signal(SIGCONT, catch);
	signal(SIGSTOP, catch);
	signal(SIGTSTP, catch);
	signal(SIGTTIN, catch);
	signal(SIGTTOU, catch);

	/* check to see if a copy of svpw is already running */

	if (!access(PID_FILE, R_OK)) {
		if ((str = fopen(PID_FILE, "r" ))) {
			fscanf(str, "%d", &pid);
			fclose(str);
			fprintf(stderr, "svpw: Already running as process %d.\n", pid);
			return 1;
		}
	}

	/* create the pid file */

	pid = getpid();
	if ((str = fopen(PID_FILE, "w"))) {
		fprintf(str, "%d\n", pid);
		fclose(str);
	}
	
	/* is this a supported machine? */

	if (SciSupportCheck(&version)==SCI_FAILURE)
		Incompatible();

	/* did the user request help? */

	if (help==1) {
		printf(USAGE);
		leave(0);
	}

	/* Try opening an interface to the BIOS */

	if (SciOpenInterface()==SCI_SUCCESS)
		interface = 1;
	else
		Incompatible();

	/* is the supervisor password registered? */

	reg.ebx = SCI_PASSWORD;
	reg.ecx =  SCI_SUPER_PASSWORD;
	if (SciGet(&reg)==SCI_FAILURE)
		Incompatible();

	if ((user==1) && (reg.ecx==SCI_NOT_REGISTERED)) {
		printf("Unable to change user password mode because supervisor"
			" password is not\nregstered.\n");
		leave(2);
	}

	if (user==1)
		UserMode();
	else
		SuperPassword((int) reg.ecx, (int) reg.edx);

	leave(0);

	return 0;
}
