// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef GENERAL_INC
#define GENERAL_INC

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <limits.h>
#include <stdarg.h>
#include <time.h>
#include <sys/times.h>
#include <float.h>

// Old systems need that to define FLT_MAX and DBL_MAX
#ifndef DBL_MAX
#include <values.h>
#endif

namespace Torch {

#ifdef USEDOUBLE
#define INF DBL_MAX
#define REAL_EPSILON DBL_EPSILON
#define real double
#else
#define INF FLT_MAX
#define REAL_EPSILON FLT_EPSILON
#define real float
#endif

/** Sparse definition.
    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
struct sreal
{
    /// Index which is not zero.
    int index;
    /// Value of this index.
    real value;
};

/** Return the size of a sparse vector #line#.

    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
int sparseVectorLength(sreal *line);
//-----------------------------------

//- IO ------------------------------

/** Several functions for Inputs/Ouputs on disk.

    @author Ronan Collobert (collober@iro.umontreal.ca)
 */
//@{

/// Returns #true# if the computer use the Little Endian coding.
bool isLittleEndian();
/// Returns #true# if we'll load/save using the native mode
bool isNativeMode();
/// We'll load/save in native mode.
void setNativeMode();
/// We'll load/save in Little Endian mode.
void setLittleEndianMode();
/// We'll load/save in Big Endian mode.
void setBigEndianMode();
/** Reverse #n_blocks# of memory starting from #ptr_#.
    In each block of size #block_size#, reverse
    the memory byte per byte.
*/
void reverseMemory(void *ptr_, int block_size, int n_blocks);

/** As #fread()#, but reverse the memory if it has been specified
    with #setNativeMode()#, #setLittleEndianMode()# or
    #setBigEndianMode()#.
*/
int xfread(void *ptr, int block_size, int n_blocks, FILE *f);
/** As #fwrite()#, but reverse the memory if it has been specified
    with #setNativeMode()#, #setLittleEndianMode()# or
    #setBigEndianMode()#.
*/
int xfwrite(void *ptr, int block_size, int n_blocks, FILE *f);
//@}
//-----------------------------------

//- Divers --------------------------

/// Memory functions
//@{

/** As #malloc()#, but send an error message if there
    is no more memory.

    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
void *xalloc(int size);

/** As #calloc()#, but send an error message if there
    is no more memory.

    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
void *xcalloc(int n_blocks, int size);

/** Almost as #realloc()#, but send an error message if there
    is no more memory.
    Use #free()# is #size# is #0#.

    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
void *xrealloc(void *ptr, int size);
//@}


/** Text outputs functions.

    These functions are like #printf()#, but you should
    use them instead. Note that you should never try to
    print a message larger than 10000 characters.

    @author Ronan Collobert (collober@iro.umontreal.ca)
 */
//@{

/// Print an error message. The program will exit.
void error(const char* msg, ...);
/// Print a warning message.
void warning(const char* msg, ...);
/// Print a message.
void message(const char* msg, ...);
/// Like printf.
void print(const char* msg, ...);

/** Print a control bar [\#\#\#\#\#\#\#\#\#\#].
    
    First time, you can it with #level=-1#.
    It'll print the control bar at each time
    you will call that.

    #max_level# is the value of the last #level#
    you'll call this function.

    @author Ronan Collobert (collober@iro.umontreal.ca)
 */
void controlBar(int level, int max_level);
//@}

/** Return the time in CLOCKS_PER_SEC

    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
long getRuntime(void);

#ifndef min
/// The min function
#define	min(a,b) ((a) > (b) ? (b) : (a))
#endif

#ifndef max
/// The max function
#define	max(a,b) ((a) > (b) ? (a) : (b))
#endif


}

#endif
