// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "SVMRegression.h"

namespace Torch {

SVMRegression::SVMRegression(Kernel *kernel_, real *C_) : SVM(kernel_)
{
  addROption("eps regression", &eps_regression, 0.7, "size of the error tube");

  l = 2*data->n_real_examples;

  Cup = (real *)xalloc(sizeof(real)*l);
  Cdown = (real *)xalloc(sizeof(real)*l);
  Cuser = C_;

  alpha = (real *)xalloc(sizeof(real)*l);
  grad = (real *)xalloc(sizeof(real)*l);
  y = (real *)xalloc(sizeof(real)*l);

  l = 2*data->n_examples;
}

void SVMRegression::reset()
{
  l = 2*data->n_examples;
  int lm = data->n_examples;

  for(int i = 0; i < l; i++)
    alpha[i] = 0;

  b = 0;

  if(Cuser)
  {
    for(int i = 0; i < lm; i++)
    {
      Cup[i] = 0;
      Cdown[i] = -Cuser[data->selected_examples[i]];
    }

    for(int i = lm; i < l; i++)
    {
      Cup[i] = Cuser[data->selected_examples[i-lm]+lm];
      Cdown[i] = 0;
    }
  }
  else
  {
    for(int i = 0; i < lm; i++)
    {
      Cup[i] = 0;
      Cdown[i] = -C_cst;
    }

    for(int i = lm; i < l; i++)
    {
      Cup[i] = C_cst;
      Cdown[i] = 0;
    }
  }

  for(int i = 0; i < lm; i++)
  {
    data->setExample(i);
    real z = ((real *)data->targets)[0];
    
    grad[i] =  -z - eps_regression;
    y[i] =  1;
    
    grad[i+lm] = -z + eps_regression;
    y[i+lm] = 1;
  }

  n_support_vectors = 0; 
  n_support_vectors_bound = 0; 

  kernel->reset();
}

void SVMRegression::checkSupportVectors()
{
  // La je prie pour que l'utilisateur normal utilise
  // un processeur deterministe ///
  n_support_vectors = 0;
  n_support_vectors_bound = 0;
  for(int i = 0; i < data->n_examples; i++)
  {
    if(alpha[i] < -eps_bornes)
    {
      if(alpha[i] < Cdown[i] + eps_bornes)
        n_support_vectors_bound++;
      
      n_support_vectors++;
    }
  }
  for(int i = data->n_examples; i < 2*data->n_examples; i++)
  {
    if(alpha[i] > eps_bornes)
    {
      if(alpha[i] > Cup[i] - eps_bornes)
        n_support_vectors_bound++;
      
      n_support_vectors++;
    }
  }
  support_vectors = (int *)xrealloc(support_vectors, sizeof(int)*n_support_vectors);
  real_index = (int *)xrealloc(real_index, sizeof(int)*n_support_vectors);

  n_support_vectors = 0;
  for(int i = 0; i < data->n_examples; i++)
  {
    if(alpha[i] < -eps_bornes)
    {
      support_vectors[n_support_vectors  ] = i;
           real_index[n_support_vectors++] = data->selected_examples[i];
    }
  }
  for(int i = data->n_examples; i < 2*data->n_examples; i++)
  {
    if(alpha[i] > eps_bornes)
    {
      support_vectors[n_support_vectors  ] = i;
           real_index[n_support_vectors++] = data->selected_examples[i-data->n_examples];
    }
  }

  if(!bCompute())
  {
    warning("SVMRegression: b is not unique. It's probably wrong");
    warning("SVMRegression: I think you are using silly parameters");
  }
}

SVMRegression::~SVMRegression()
{
  free(Cup);
  free(Cdown);
  free(alpha);
  free(grad);
  free(support_vectors);
  free(real_index);
  free(y);
}

}

