/**
 * Copyright (C) 2007-2012 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 363 $
 * $Date: 2012-01-22 18:45:43 +0800 (Sun, 22 Jan 2012) $
 */
#ifndef INDII_TINT_MODEL_CLUSTERMODEL_HPP
#define INDII_TINT_MODEL_CLUSTERMODEL_HPP

#include "ClusterModelObservable.hpp"
#include "../../model/Model.hpp"
#include "../../image/ImageResource.hpp"
#include "../../image/ImageManipulation.hpp"
#include "../../cluster/DataSet.hpp"
#include "../../cluster/KMeansClusterer.hpp"

#include "wx/colour.h"

#include <vector>

namespace indii {

  class ClusterModelObserver;
  
/**
 * Cluster model.
 */
class ClusterModel : public Model, public ClusterModelObservable {
public:
  /**
   * Constructor.
   *
   * @param res Image resource.
   */
  ClusterModel(ImageResource* res);

  /**
   * Destructor.
   */
  virtual ~ClusterModel();

  /**
   * Get underlying image resource.
   */
  ImageResource* getImageResource();
  
  /**
   * Get number of clusters.
   *
   * @return Number of clusters.
   */
  unsigned getNumClusters();

  /**
   * Set number of clusters.
   *
   * @param k Number of clusters.
   */
  void setNumClusters(const unsigned k);

  /**
   * Is cluster assignment hard?
   *
   * @return True if clustering is hard, false otherwise.
   */
  bool isHard();

  /**
   * Get cluster hardness.
   *
   * @return Cluster hardness.
   */
  float getHard();

  /**
   * Set cluster hardness.
   *
   * @param hard Cluster hardness, between 0 and 1.
   */
  void setHard(const float hard = 1.0f);

  /**
   * Get number of repetitions.
   *
   * @return Number of repetitions.
   */
  unsigned getNumRepetitions();

  /**
   * Set number of clusters.
   *
   * @param reps Number of repetitions.
   */
  void setNumRepetitions(const unsigned reps);

  /**
   * Get saturation threshold.
   *
   * @return Saturation threshold.
   */
  unsigned char getSaturationThreshold();

  /**
   * Set saturation threshold.
   *
   * @param x Saturation threshold.
   */
  void setSaturationThreshold(const unsigned char x);

  /**
   * Get maximum pixels for clustering.
   *
   * @return Maximum pixels for clustering.
   */
  unsigned getMaxPixels();
  
  /**
   * Set maximum pixels for clustering.
   *
   * @param n Maximum pixels for clustering.
   */
  void setMaxPixels(const unsigned n);

  /**
   * Get saturation decay.
   *
   * @return Saturation decay.
   */
  float getSaturationDecay();

  /**
   * Set saturation decay.
   *
   * @param x Saturation decay.
   */
  void setSaturationDecay(const float x);

  /**
   * Get centroid distance decay.
   *
   * @return Centroid distance decay.
   */
  float getCentroidDecay();

  /**
   * Set centroid distance decay.
   *
   * @param x Centroid distance decay.
   */
  void setCentroidDecay(float x);

  /**
   * Get saturation softness.
   *
   * @return Saturation softness.
   */
  float getSaturationSoftness();

  /**
   * Set saturation softness.
   *
   * @param x Saturation softness.
   */
  void setSaturationSoftness(const float x);

  /**
   * Get centroid distance softness.
   *
   * @return Centroid distance softness.
   */
  float getCentroidSoftness();

  /**
   * Set centroid distance softness.
   *
   * @param x Centroid distance softness.
   */
  void setCentroidSoftness(const float x);

  /**
   * Is cluster shown in colour?
   *
   * @param i Cluster number.
   */
  bool isShown(const unsigned i);

  /**
   * Show (or hide) cluster.
   *
   * @param i Cluster number.
   * @param on True to show cluster in colour, false for b&w.
   */
  void show(const unsigned i, const bool on = true);

  /**
   * Show (or hide) all clusters.
   *
   * @param on True to show all clusters in colour, false for b&w.
   */
  void showAll(const bool on = true);

  /**
   * Get cluster colour.
   *
   * @param i Cluster number.
   *
   * @return Colour representing given cluster.
   */
  const wxColour& getColour(const unsigned i);
  
  /**
   * Set cluster colour.
   *
   * @param i Cluster number.
   * @param col Cluster colour.
   */
  void setColour(const unsigned i, const wxColour& col);

  /**
   * Get cluster hue rotation.
   *
   * @param i Cluster number.
   *
   * @return Hue rotation of given cluster.
   */
  float getHue(const unsigned i);
  
  /**
   * Set cluster hue rotation.
   *
   * @param i Cluster number.
   * @param x Hue rotation to set.
   */
  void setHue(const unsigned i, const float x);
  
  /**
   * Get cluster saturation adjustment.
   *
   * @param i Cluster number.
   *
   * @return Saturation adjustment of given cluster.
   */
  float getSat(const unsigned i);
  
  /**
   * Set cluster saturation adjustment.
   *
   * @param i Cluster number.
   * @param x Saturation adjustment to set.
   */
  void setSat(const unsigned i, const float x);

  /**
   * Does cluster have any saturation?
   *
   * @param i Cluster number.
   * 
   * @return True if saturation of cluster @p i is significant (i.e. does not
   * round to grey), false otherwise.
   */
  bool hasSat(const unsigned i);
  
  /**
   * Get cluster lightness adjustment.
   *
   * @param i Cluster number.
   *
   * @return Lightness adjustment of given cluster.
   */
  float getLight(const unsigned i);
  
  /**
   * Set cluster lightness adjustment.
   *
   * @param i Cluster number.
   * @param x Lightness adjustment to set.
   */
  void setLight(const unsigned i, const float x);

  /**
   * Does cluster have any light?
   *
   * @param i Cluster number.
   * 
   * @return True if lightness of cluster @p i is significant (i.e. does not
   * round to black), false otherwise.
   */
  bool hasLight(const unsigned i);

  /**
   * Calculate alpha channel of subimage for particular cluster.
   *
   * @param i Cluster number.
   * @param rect Rectangular region of interest, relative to scaled image.
   * @param width Scaled width of image.
   * @param height Scaled height of image.
   * @param[out] c Output channel.
   */
  void calcAlpha(const unsigned i, const wxRect& rect,
      const unsigned width, const unsigned height, channel& c);

  /**
   * Calculate mask for particular cluster.
   *
   * @param i Cluster number.
   * @param rect Rectangular region of interest, relative to scaled image.
   * @param width Scaled width of image.
   * @param height Scaled height of image.
   *
   * @return Mask for the given cluster and region.
   */
  sparse_mask calcMask(const unsigned i, const wxRect& rect,
      const unsigned width = 0, const unsigned height = 0);

  /**
   * Prepare data set for clustering.
   */
  void prepare();

  /**
   * Cluster image.
   */
  void cluster();

  /**
   * Lock model. Ignore any calls to prepare() or cluster() until unlocked.
   * This is basically a hack to stop them being overcalled, lock(), change
   * a bunch of settings, unlock(), then prepare() and cluster() to keep the
   * model consistent.
   */
  void lock();

  /**
   * Unlock model.
   */
  void unlock();

  virtual void calcFg(const wxRect& rect, const unsigned width,
      const unsigned height, wxImage& o);
  virtual void calcFg(const unsigned width, const unsigned height,
      wxImage& o);
  virtual void setDefaults();
  virtual void setForDialog();

private:
  /**
   * Image resources.
   */
  ImageResource* res;

  /**
   * Number of clusters.
   */
  unsigned k;

  /**
   * Cluster hardness.
   */
  float hard;
  
  /**
   * Number of repetitions.
   */
  unsigned reps;
  
  /**
   * Saturation threshold.
   */
  unsigned char saturationThreshold;
  
  /**
   * Maximum number of pixels for clustering.
   */
  unsigned maxPixels;
  
  /**
   * Saturation decay.
   */
  float saturationDecay;

  /**
   * Centroid decay.
   */
  float centroidDecay;

  /**
   * Saturation softness.
   */
  float saturationSoftness;

  /**
   * Centroid softness.
   */
  float centroidSoftness;

  /**
   * Cluster hue rotation.
   */
  std::vector<float> hues;

  /**
   * Cluster saturation adjustment.
   */
  std::vector<float> sats;

  /**
   * Cluster light adjustment.
   */
  std::vector<float> lights;

  /**
   * Cluster colours.
   */
  std::vector<wxColour> colours;

  /**
   * Data set for clustering.
   */
  indii::DataSet<float,unsigned> data;

  /**
   * Clusterer used for last clustering.
   */
  indii::KMeansClusterer<>* minClusterer;

  /**
   * Is model locked?
   */
  bool locked;
};
}

inline indii::ImageResource*
    indii::ClusterModel::getImageResource() {
  return res;
}

inline unsigned indii::ClusterModel::getNumClusters() {
  return k;
}

inline bool indii::ClusterModel::isHard() {
  return hard >= 1.0f;
}

inline float indii::ClusterModel::getHard() {
  return hard;
}

inline unsigned indii::ClusterModel::getNumRepetitions() {
  return reps;
}

inline unsigned char indii::ClusterModel::getSaturationThreshold() {
  return saturationThreshold;
}

inline unsigned indii::ClusterModel::getMaxPixels() {
  return maxPixels;
}

inline float indii::ClusterModel::getSaturationDecay() {
  return saturationDecay;
}

inline float indii::ClusterModel::getCentroidDecay() {
  return centroidDecay;
}

inline float indii::ClusterModel::getSaturationSoftness() {
  return saturationSoftness;
}

inline float indii::ClusterModel::getCentroidSoftness() {
  return centroidSoftness;
}

inline bool indii::ClusterModel::isShown(const unsigned i) {
  return hasSat(i);
}

inline const wxColour& indii::ClusterModel::getColour(
    const unsigned i) {
  /* pre-condition */
  assert (i < k);

  return colours[i];
}

inline float indii::ClusterModel::getHue(const unsigned i) {
  /* pre-condition */
  assert (i < k);

  return hues[i];
}

inline float indii::ClusterModel::getSat(const unsigned i) {
  /* pre-condition */
  assert (i < k);

  return sats[i];
}

inline float indii::ClusterModel::getLight(const unsigned i) {
  /* pre-condition */
  assert (i < k);

  return lights[i];
}

inline bool indii::ClusterModel::hasLight(const unsigned i) {
  /* pre-condition */
  assert (i < k);
  
  return lights[i] > -255.0f;
}

inline bool indii::ClusterModel::hasSat(const unsigned i) {
  /* pre-condition */
  assert (i < k);
  
  return sats[i] > -1.0f;
}

inline void indii::ClusterModel::lock() {
  locked = true;
}

inline void indii::ClusterModel::unlock() {
  locked = false;
}

#endif

