#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <float.h>
#include "DLTmath.h"
#include "pdbUtils.h"
#include "pdbStats.h"
#include "CovMat.h"
#include "lapack_dlt.h"
#include "FragCoords.h"
#include "ProcJacobiSVD.h"

#if defined(__APPLE__)
    #include <vecLib/clapack.h>
    #include <vecLib/cblas.h>
#endif

static double
CalcE0(const Coords *coords1, const Coords *coords2,
          const double *weights, const double *axesw);

static double
CalcE0Cov(const Coords *coords1, const Coords *coords2,
             const double *axesw);

static void
CalcR(const Coords *coords1, const Coords *coords2, double **Rmat,
         const double *weights, const double *axesw);

static void
CalcRCov(const Coords *coords1, const Coords *coords2, double **Rmat,
            const double **covmat, const double *axesw);

static int
CalcJacobiSVD(double **a, double **U, double *z, double **V, double tol);

static int
CalcRotMat(double **rotmat, double **Umat, double **VTmat);

static double
CalcE0(const Coords *coords1, const Coords *coords2,
         const double *weights, const double *axesw)
{   
    int             i;
    double          sum;
    const double   *x1 = (const double *) coords1->x,
                   *y1 = (const double *) coords1->y,
                   *z1 = (const double *) coords1->z;
    const double   *x2 = (const double *) coords2->x,
                   *y2 = (const double *) coords2->y,
                   *z2 = (const double *) coords2->z;
    double          x1i, y1i, z1i, x2i, y2i, z2i;
    const double    axeswx = axesw[0], axeswy = axesw[1], axeswz = axesw[2];
/* #include "pdbIO.h" */
/* PrintCoords((Coords *)coords1); */
/* PrintCoords((Coords *)coords2); */

    sum = 0.0;
    i = coords1->vlen;
    while(i-- > 0)
    {
        x1i = *x1++;
        y1i = *y1++;
        z1i = *z1++;
        x2i = *x2++;
        y2i = *y2++;
        z2i = *z2++;
        sum +=  *weights++ *
               (axeswx * (x1i * x1i + x2i * x2i) +
                axeswy * (y1i * y1i + y2i * y2i) +
                axeswz * (z1i * z1i + z2i * z2i));
/* printf("\nsum = %d %f %f", i, weight, sum); */
    }
/* exit(0); */
    return(sum);
}


static double
CalcE0Cov(const Coords *coords1, const Coords *coords2,
             const double *axesw)
{   
    int             i;
    double          sum;
    const double   *x2 = (const double *) coords2->x,
                   *y2 = (const double *) coords2->y,
                   *z2 = (const double *) coords2->z;
    const double   *x1 = (const double *) coords1->x,
                   *y1 = (const double *) coords1->y,
                   *z1 = (const double *) coords1->z;
    const double   *cx2 = (const double *) coords2->covx,
                   *cy2 = (const double *) coords2->covy,
                   *cz2 = (const double *) coords2->covz;
    const double   *cx1 = (const double *) coords1->covx,
                   *cy1 = (const double *) coords1->covy,
                   *cz1 = (const double *) coords1->covz;
    double          x1i, y1i, z1i, x2i, y2i, z2i,
                    cx1i, cy1i, cz1i, cx2i, cy2i, cz2i;
    const double    axeswx = axesw[0], axeswy = axesw[1], axeswz = axesw[2];

    sum = 0.0;
    i = coords1->vlen;
    while(i-- > 0)
    {
        x1i = *x1++;
        y1i = *y1++;
        z1i = *z1++;
        x2i = *x2++;
        y2i = *y2++;
        z2i = *z2++;

        cx1i = *cx1++;
        cy1i = *cy1++;
        cz1i = *cz1++;
        cx2i = *cx2++;
        cy2i = *cy2++;
        cz2i = *cz2++;

        sum += (axeswx * (cx1i * x1i + cx2i * x2i) +
                axeswy * (cy1i * y1i + cy2i * y2i) +
                axeswz * (cz1i * z1i + cz2i * z2i));
    }

    return(sum);
}


/* This function assumes that the coordinates have been centered previously
   Use CenMass() and ApplyCenter() */
static void
CalcR(const Coords *coords1, const Coords *coords2, double **Rmat,
         const double *weights, const double *axesw)
{
    int             i;
    double          weight;
    const double   *x2 = (const double *) coords2->x,
                   *y2 = (const double *) coords2->y,
                   *z2 = (const double *) coords2->z;
    const double   *x1 = (const double *) coords1->x,
                   *y1 = (const double *) coords1->y,
                   *z1 = (const double *) coords1->z;
    const double    axeswx = sqrt(axesw[0]),
                    axeswy = sqrt(axesw[1]),
                    axeswz = sqrt(axesw[2]);
    double          x2i, y2i, z2i, x1i, y1i, z1i;
    double          Rmat00, Rmat01, Rmat02,
                    Rmat10, Rmat11, Rmat12,
                    Rmat20, Rmat21, Rmat22;

    Rmat00 = Rmat01 = Rmat02 = Rmat10 = Rmat11 = Rmat12 =
    Rmat20 = Rmat21 = Rmat22 = 0.0;

   /*  printf("\n axes weights: %f %f %f ", axeswx, axeswy, axeswz); */

    i = coords1->vlen;
    while(i-- > 0)
    {
        weight = *weights++;

        x2i = axeswx * *x2++;
        y2i = axeswy * *y2++;
        z2i = axeswz * *z2++;

        x1i = axeswx * *x1++;
        y1i = axeswy * *y1++;
        z1i = axeswz * *z1++;

        Rmat00 += weight * x2i * x1i;
        Rmat01 += weight * x2i * y1i;
        Rmat02 += weight * x2i * z1i;
        
        Rmat10 += weight * y2i * x1i;
        Rmat11 += weight * y2i * y1i;
        Rmat12 += weight * y2i * z1i;
        
        Rmat20 += weight * z2i * x1i;
        Rmat21 += weight * z2i * y1i;
        Rmat22 += weight * z2i * z1i;
    }

    Rmat[0][0] = Rmat00;
    Rmat[0][1] = Rmat01;
    Rmat[0][2] = Rmat02;
    Rmat[1][0] = Rmat10;
    Rmat[1][1] = Rmat11;
    Rmat[1][2] = Rmat12;
    Rmat[2][0] = Rmat20;
    Rmat[2][1] = Rmat21;
    Rmat[2][2] = Rmat22;
}


static void
CalcRvan(const Coords *coords1, const Coords *coords2, double **Rmat)
{
    int             i;
    const double   *x1 = (const double *) coords1->x,
                   *y1 = (const double *) coords1->y,
                   *z1 = (const double *) coords1->z;
    const double   *x2 = (const double *) coords2->x,
                   *y2 = (const double *) coords2->y,
                   *z2 = (const double *) coords2->z;
    double          x2i, y2i, z2i, x1i, y1i, z1i;
    double          Rmat00, Rmat01, Rmat02,
                    Rmat10, Rmat11, Rmat12,
                    Rmat20, Rmat21, Rmat22;

   /*  printf("\n axes weights: %f %f %f ", axeswx, axeswy, axeswz); */

    Rmat00 = Rmat01 = Rmat02 = Rmat10 = Rmat11 = Rmat12 =
    Rmat20 = Rmat21 = Rmat22 = 0.0;

    i = coords1->vlen;
    while(i-- > 0)
    {
        x1i = *x1++;
        y1i = *y1++;
        z1i = *z1++;

        x2i = *x2++;
        y2i = *y2++;
        z2i = *z2++;

        Rmat00 += x2i * x1i;
        Rmat01 += x2i * y1i;
        Rmat02 += x2i * z1i;
        
        Rmat10 += y2i * x1i;
        Rmat11 += y2i * y1i;
        Rmat12 += y2i * z1i;
        
        Rmat20 += z2i * x1i;
        Rmat21 += z2i * y1i;
        Rmat22 += z2i * z1i;
    }

    Rmat[0][0] = Rmat00;
    Rmat[0][1] = Rmat01;
    Rmat[0][2] = Rmat02;
    Rmat[1][0] = Rmat10;
    Rmat[1][1] = Rmat11;
    Rmat[1][2] = Rmat12;
    Rmat[2][0] = Rmat20;
    Rmat[2][1] = Rmat21;
    Rmat[2][2] = Rmat22;
}


static void
CalcRFrag(const FragCoords *coords1, const FragCoords *coords2, double **Rmat)
{
    int             i;
    const double   *x1 = (const double *) coords1->x,
                   *y1 = (const double *) coords1->y,
                   *z1 = (const double *) coords1->z;
    const double   *x2 = (const double *) coords2->x,
                   *y2 = (const double *) coords2->y,
                   *z2 = (const double *) coords2->z;
    double          x2i, y2i, z2i, x1i, y1i, z1i;
    double          Rmat00, Rmat01, Rmat02,
                    Rmat10, Rmat11, Rmat12,
                    Rmat20, Rmat21, Rmat22;

   /*  printf("\n axes weights: %f %f %f ", axeswx, axeswy, axeswz); */

    Rmat00 = Rmat01 = Rmat02 = Rmat10 = Rmat11 = Rmat12 =
    Rmat20 = Rmat21 = Rmat22 = 0.0;

    i = coords1->fraglen;
    while(i-- > 0)
    {
        x1i = *x1++;
        y1i = *y1++;
        z1i = *z1++;

        x2i = *x2++;
        y2i = *y2++;
        z2i = *z2++;

        Rmat00 += x2i * x1i;
        Rmat01 += x2i * y1i;
        Rmat02 += x2i * z1i;
        
        Rmat10 += y2i * x1i;
        Rmat11 += y2i * y1i;
        Rmat12 += y2i * z1i;
        
        Rmat20 += z2i * x1i;
        Rmat21 += z2i * y1i;
        Rmat22 += z2i * z1i;
    }

    Rmat[0][0] = Rmat00;
    Rmat[0][1] = Rmat01;
    Rmat[0][2] = Rmat02;
    Rmat[1][0] = Rmat10;
    Rmat[1][1] = Rmat11;
    Rmat[1][2] = Rmat12;
    Rmat[2][0] = Rmat20;
    Rmat[2][1] = Rmat21;
    Rmat[2][2] = Rmat22;
}


static void
CalcRCov(const Coords *coords1, const Coords *coords2, double **Rmat,
            const double **covmat, const double *axesw)
{
    int             i;
    const double   *x2 = (const double *) coords2->covx,
                   *y2 = (const double *) coords2->covy,
                   *z2 = (const double *) coords2->covz;
    const double   *x1 = (const double *) coords1->x,
                   *y1 = (const double *) coords1->y,
                   *z1 = (const double *) coords1->z;
    const double    axeswx = sqrt(axesw[0]),
                    axeswy = sqrt(axesw[1]),
                    axeswz = sqrt(axesw[2]);
    double          x2i, y2i, z2i, x1i, y1i, z1i;
    double          Rmat00, Rmat01, Rmat02,
                    Rmat10, Rmat11, Rmat12,
                    Rmat20, Rmat21, Rmat22;

    Rmat00 = Rmat01 = Rmat02 = Rmat10 = Rmat11 = Rmat12 =
    Rmat20 = Rmat21 = Rmat22 = 0.0;

   /*  printf("\n axes weights: %f %f %f ", axeswx, axeswy, axeswz); */

    i = coords1->vlen;
    while(i-- > 0)
    {
        x2i = axeswx * *x2++;
        y2i = axeswy * *y2++;
        z2i = axeswz * *z2++;

        x1i = axeswx * *x1++;
        y1i = axeswy * *y1++;
        z1i = axeswz * *z1++;

        Rmat00 += x2i * x1i;
        Rmat01 += x2i * y1i;
        Rmat02 += x2i * z1i;
        
        Rmat10 += y2i * x1i;
        Rmat11 += y2i * y1i;
        Rmat12 += y2i * z1i;
        
        Rmat20 += z2i * x1i;
        Rmat21 += z2i * y1i;
        Rmat22 += z2i * z1i;
    }

    Rmat[0][0] = Rmat00;
    Rmat[0][1] = Rmat01;
    Rmat[0][2] = Rmat02;
    Rmat[1][0] = Rmat10;
    Rmat[1][1] = Rmat11;
    Rmat[1][2] = Rmat12;
    Rmat[2][0] = Rmat20;
    Rmat[2][1] = Rmat21;
    Rmat[2][2] = Rmat22;
}


/* Classic one-sided Jacobi SVD algorithm for 3x3 matrices.
   Calculates the SVD of a, a = UzV^t .

   This is pulled almost verbatim from
   John Nash, _Compact Numerical Methods for Computers_, 1979, Wiley, NY 
   Chapter 3, algorithm 1, pp 30-31.

   It is based on the method of Chartres:

   Chartres, B.A. (1962) "Adaptation of the Jacobi method for a computer with
   magnetic tape backing store." Comput. J. 5:51-60.

   Nash, J.C. (1975) "A one-sided transformation method for the Singular Value
   Decomposition and algebraic eigenproblem."  Comput. J. 18:74-76.

   I have added a convergence criterion based on the sum of cross-products
   after a sweep -- it should be 0 if the matrix is orthogonal. 
   I also made some adjustments for C-style indexing.
   Unlike Nash's implementation, this function returns the
   transpose of V, same behavior as LAPACK DGESVD.
   Singular values are ordered largest to smallest and stored in z. 
   The matrix A is preserved (unlike in Nash 1979 where it is overwritten
   with U) */
static int
CalcJacobiSVD(double **A, double **U, double *z, double **Vt, double tol)
{
    int             i, j, k, count, sweeps;
    double          eps, orth, orthsum;
    double          p, q, r, v, g, h, pp, qr, c = 0.0, s = 1.0;

    /* U = V = I_3 */
    memset(&U[0][0], 0, 9 * sizeof(double));
    for (i = 0; i < 3; i++)
        U[i][i] = 1.0;

    MatCpySym(Vt, (const double **) A, 3);

    eps = tol * tol * fabs(Vt[0][0] + Vt[1][1] + Vt[2][2]);

    count = sweeps = 0;
    do
    {
        orthsum = 0.0;
        for (j = 0; j < 2; j++)
        {
            for (k = j+1; k < 3; k++)
            {
                p = q = r = 0.0; /* step 4 */
                for (i = 0; i < 3; ++i) /* step 5 */
                {
                    g = Vt[j][i];
                    h = Vt[k][i];
                    p += g * h;
                    q += g * g;
                    r += h * h;
                }

                pp = p*p;
                qr = q*r;
                orth = pp / qr;
                orthsum += orth;

                if (r > eps && qr > eps && orth > eps)
                {
                    if (q < r)
                    {
                        c = 0.0;
                        s = 1.0;
                    }
                    else
                    {
                        /* step 8 - calculation of sine and cosine of rotation angle */
                        q -= r;
                        v = sqrt((4.0 * pp) + (q * q));
                        // v = hypot(2.0*p, q);
                        c = sqrt(0.5 * (v + q) / v);
                        s = p / (v * c);
                    }

                    /* step 9 - apply rotation to a */
                    for (i = 0; i < 3; ++i)
                    {
                        r = Vt[j][i];
                        h = Vt[k][i];
                        Vt[j][i] =  (r * c) + (h * s);
                        Vt[k][i] = -(r * s) + (h * c);
                    }

                    /* step 10 - apply rotation to V */
                    for (i = 0; i < 3; ++i)
                    {
                        r = U[i][j];
                        h = U[i][k];
                        U[i][j] =  (r * c) + (h * s);
                        U[i][k] = -(r * s) + (h * c);
                    }
                }
                /* step 12 */
                count++;
            }
        }

        sweeps++;
        /* printf("\n> [%3d] orthsum: %e   %e", count, orthsum, eps); */
    } while (orthsum > eps && count < 200);

    for (j = 0; j < 3; j++)
    {
        q = 0;
        for (i = 0; i < 3; ++i)
            q += Vt[j][i] * Vt[j][i];

        z[j] = q = sqrt(q);

        if (q <= DBL_EPSILON)
            break;

        for (i = 0; i < 3; ++i)
            Vt[j][i] /= q;
    }

/*     printf("\n> count: %3d   sweeps: %3d", count, sweeps); */
/*     printf("\n singvals:     % f   % f   % f\nJacobi\n", z[0], z[1], z[2]); */
/*     Mat3Print(U); */
/*     Mat3Print(Vt); */
/*  */
/*     long int        info = 0; */
/*     char            jobu = 'A'; */
/*     char            jobvt = 'A'; */
/*     long int        m = 3, n = 3, lda = 3, ldu = 3, ldvt = 3; */
/*     long int        lwork = 100; */
/*     double         *work; */
/*   */
/*     work = (double *) malloc(lwork * sizeof(double)); */
/*  */
/*     DGESVD(&jobvt, &jobu, &n, &m, */
/*             &A[0][0], &lda, &z[0], */
/*             &Vt[0][0], &ldvt, */
/*             &U[0][0], &ldu, */
/*             work, &lwork, &info); */
/*  */
/*     printf("\n singvals:     % f   % f   % f\nLAPACK:\n", z[0], z[1], z[2]); */
/*     Mat3Print(U); */
/*     Mat3Print(Vt); */
/*  */
/*     free(work); */
/*     exit(0); */
    return(count);
}


/* Same as CalcJacobiSVD(), but with partial pivoting on row with largest norm.
   Supposedly improves convergence, but probably not for 3x3 matrices.
   Adapted from:

   De Rijk, P.P.M. (1989) "A one-sided Jacobi algorithm for computing the
   singular value decomposition on a vector computer."
   SIAM J. Sci. Stat. Comput. 10(2):359-371.

   See section 2.4 pp. 363-364.
   */
int
CalcJacobiSVDpp(double **A, double **U, double *z, double **Vt, double tol)
{
    int             i, j, k, ss, count, sweeps, pivot;
    double          eps, orth, orthsum, tmp;
    double          p, q, r, v, g, h, pp, qr, c = 0.0, s = 1.0;

    /* U = V = I_3 */
    memset(&U[0][0], 0, 9 * sizeof(double));
    for (i = 0; i < 3; i++)
        U[i][i] = 1.0;

    MatCpySym(Vt, (const double **) A, 3);

    eps = tol * tol * fabs(Vt[0][0] + Vt[1][1] + Vt[2][2]);

    /* test for orthonormality */
    orthsum = 0.0;
    for (j = 0; j < 2; j++)
    {
        for (k = j+1; k < 3; k++)
        {
            p = q = r = 0.0;
            for (i = 0; i < 3; ++i)
            {
                g = Vt[j][i];
                h = Vt[k][i];
                p += g * h;
                q += g * g;
                r += h * h;
            }

            orthsum += (p*p) / (q*r);
        }
    }

    count = sweeps = pivot = 0;
    while (orthsum > eps && count < 100)
    {
        for (ss = 0; ss < 2; ++ss) /* subsweeps 1 & 2 */
        {
            p = 0.0;
            for (i = 0; i < 3; ++i)
                p += Vt[ss][i] * Vt[ss][i];
    
            q = 0.0;
            for (i = 0; i < 3; ++i)
                q += Vt[ss+1][i] * Vt[ss+1][i];
    
            if (p < q)
            {
                for (i = 0; i < 3; ++i)
                {
                    tmp = U[ss][i];
                    U[ss][i] = U[ss+1][i];
                    U[ss+1][i] = tmp;

                    tmp = Vt[ss][i];
                    Vt[ss][i] = Vt[ss+1][i];
                    Vt[ss+1][i] = tmp;
                    pivot++;
                }
            }

            for (j = ss; j < 2; j++)
            {
                for (k = j+1; k < 3; k++)
                {
                    p = q = r = 0.0; /* step 4 */
                    for (i = 0; i < 3; ++i) /* step 5 */
                    {
                        g = Vt[j][i];
                        h = Vt[k][i];
                        p += g * h;
                        q += g * g;
                        r += h * h;
                    }
    
                    pp = p*p;
                    qr = q*r;
                    orth = pp / qr;
                    /* orthsum += orth; */
    
                    if (r > eps && qr > eps && orth > eps)
                    {
                        if (q < r)
                        {
                            c = 0.0;
                            s = 1.0;
                        }
                        else
                        {
                            /* step 8 - calculation of sine and cosine of rotation angle */
                            q -= r;
                            v = sqrt((4.0 * pp) + (q * q));
                            c = sqrt(0.5 * (v + q) / v);
                            s = p / (v * c);
                        }
    
                        /* step 9 - apply rotation to a */
                        for (i = 0; i < 3; ++i)
                        {
                            r = Vt[j][i];
                            h = Vt[k][i];
                            Vt[j][i] =  (r * c) + (h * s);
                            Vt[k][i] = -(r * s) + (h * c);
                        }
    
                        /* step 10 - apply rotation to V */
                        for (i = 0; i < 3; ++i)
                        {
                            r = U[i][j];
                            h = U[i][k];
                            U[i][j] =  (r * c) + (h * s);
                            U[i][k] = -(r * s) + (h * c);
                        }
                    }
                    /* step 12 */
                    count++;
                }
            } /* end subsweep */
            sweeps++;

            /* test for orthonormality */
            orthsum = 0.0;
            for (j = 0; j < 2; j++)
            {
                for (k = j+1; k < 3; k++)
                {
                    p = q = r = 0.0;
                    for (i = 0; i < 3; ++i)
                    {
                        g = Vt[j][i];
                        h = Vt[k][i];
                        p += g * h;
                        q += g * g;
                        r += h * h;
                    }
    
                    orthsum += (p*p) / (q*r);
                }
            }

            if (orthsum <= eps)
                break;
        } /* end sweep */
/*         printf("\n> [%3d] orthsum: %e   %e", count, orthsum, eps); */
/*         fflush(NULL); */
    }

    for (j = 0; j < 3; j++)
    {
        q = 0;
        for (i = 0; i < 3; ++i)
            q += Vt[j][i] * Vt[j][i];

        z[j] = q = sqrt(q);

        if (q <= DBL_EPSILON)
            break;

        for (i = 0; i < 3; ++i)
            Vt[j][i] /= q;
    }

/*     printf("\n> count: %3d   sweeps: %3d   pivot: %3d", count, sweeps, pivot); */
/*     fflush(NULL); */

    return(count);
}


/* Takes U and V^t on input, calculates R = VU^t */
static int
CalcRotMat(double **rotmat, double **Umat, double **Vtmat)
{   
    int         i, j, k;
    double      det;

    memset(&rotmat[0][0], 0, 9 * sizeof(double));

    det = Mat3Det((const double **)Umat) * Mat3Det((const double **)Vtmat);

    if (det > 0)
    {
        for (i = 0; i < 3; ++i)
            for (j = 0; j < 3; ++j)
                for (k = 0; k < 3; ++k)
                    rotmat[i][j] += (Vtmat[k][i] * Umat[j][k]);

        return(1);
    }
    else
    {
        /* printf("\n * determinant of SVD U or V matrix = %f", det); */

        for (i = 0; i < 3; ++i)
        {
            for (j = 0; j < 3; ++j)
            {
                for (k = 0; k < 2; ++k)
                    rotmat[i][j] += (Vtmat[k][i] * Umat[j][k]);

                rotmat[i][j] -= (Vtmat[2][i] * Umat[j][2]);
            }
        }

        return(-1);
    }
}


/* returns sum of squared residuals, E
   rmsd = sqrt(E/atom_num)  */
double
ProcJacobiSVD(const Coords *coords1, const Coords *coords2, double **rotmat,
                 const double *weights, const double *axesw,
                 double **Rmat, double **Umat, double **VTmat, double *sigma)
{
    double          det, sumdev;

    sumdev = CalcE0(coords1, coords2, weights, axesw);
    /* printf("\n # sumdev = %8.2f ", sumdev); */
    CalcR(coords1, coords2, Rmat, weights, axesw);
    CalcJacobiSVD(Rmat, Umat, sigma, VTmat, 1e-8);
    det = CalcRotMat(rotmat, Umat, VTmat);

/*     VerifyRotMat(rotmat, 1e-5); */
/*     printf("\n\n rotmat:"); */
/*     write_C_mat((const double **)rotmat, 3, 8, 0); */

    if (det < 0)
        sumdev -= 2.0 * (sigma[0] + sigma[1] - sigma[2]);
    else
        sumdev -= 2.0 * (sigma[0] + sigma[1] + sigma[2]);

/*     printf("\n\n Rmat:"); */
/*     write_C_mat((const double **)Rmat, 3, 8, 0); */
/*     printf("\n\n Umat:"); */
/*     write_C_mat((const double **)Umat, 3, 8, 0); */
/*     printf("\n\n VTmat:"); */
/*     write_C_mat((const double **)VTmat, 3, 8, 0); */
/*     int i; */
/*     for (i = 0; i < 3; ++i) */
/*         printf("\n sigma[%d] = %8.2f ", i, sigma[i]); */

    return(sumdev);
}


// static double
// CalcInnProdNorm(const Coords *coords)
// {   
//     int             i;
//     double          sum;
//     const double   *x = (const double *) coords->x,
//                    *y = (const double *) coords->y,
//                    *z = (const double *) coords->z;
//     double          xi, yi, zi;
// 
//     sum = 0.0;
//     i = coords->vlen;
//     while(i-- > 0)
//     {
//         xi = *x++;
//         yi = *y++;
//         zi = *z++;
// 
//         sum += xi * xi + yi * yi + zi * zi;
//     }
// 
//     return(sum);
// }
// 
// 
// static double
// CalcInnProdNormRot(const Coords *coords, const double **rmat)
// {   
//     int             i;
//     double          sum;
//     const double   *x = (const double *) coords->x,
//                    *y = (const double *) coords->y,
//                    *z = (const double *) coords->z;
//     double          xi, yi, zi, xir, yir, zir;
//     const double    rmat00 = rmat[0][0], rmat01 = rmat[0][1], rmat02 = rmat[0][2],
//                     rmat10 = rmat[1][0], rmat11 = rmat[1][1], rmat12 = rmat[1][2],
//                     rmat20 = rmat[2][0], rmat21 = rmat[2][1], rmat22 = rmat[2][2];
// 
//     sum = 0.0;
//     i = coords->vlen;
//     while(i-- > 0)
//     {
//         xi = *x++;
//         yi = *y++;
//         zi = *z++;
// 
//         xir = xi * rmat00 + yi * rmat10 + zi * rmat20;
//         yir = xi * rmat01 + yi * rmat11 + zi * rmat21;
//         zir = xi * rmat02 + yi * rmat12 + zi * rmat22;
// 
//         sum += xir * xir + yir * yir + zir * zir;
//     }
// 
//     return(sum);
// }


double
ProcJacobiSVDvan(const Coords *coords1, const Coords *coords2, double **rotmat,
                 double **Rmat, double **Umat, double **VTmat, double *sigma)
{
    double          det, sumdev, term1, term2;

    // term1 = CalcInnProdNorm(coords2);
    CalcRvan(coords1, coords2, Rmat);
    CalcJacobiSVD(Rmat, Umat, sigma, VTmat, 1e-12);
    det = CalcRotMat(rotmat, Umat, VTmat);
    // term2 = CalcInnProdNormRot(coords1, (const double **) rotmat);
    term1 = term2 = 0.0; /* this is just a fudge to make this faster - DLT */
    sumdev = term1 + term2;

/*     VerifyRotMat(rotmat, 1e-5); */
/*     printf("\n*************** sumdev = %8.2f ", sumdev); */
/*     printf("\nrotmat:"); */
/*     write_C_mat((const double **)rotmat, 3, 8, 0); */

    if (det < 0)
        sumdev -= 2.0 * (sigma[0] + sigma[1] - sigma[2]);
    else
        sumdev -= 2.0 * (sigma[0] + sigma[1] + sigma[2]);

/*     printf("\nRmat:"); */
/*     write_C_mat((const double **)Rmat, 3, 8, 0); */
/*     printf("\nUmat:"); */
/*     write_C_mat((const double **)Umat, 3, 8, 0); */
/*     printf("\nVTmat:"); */
/*     write_C_mat((const double **)VTmat, 3, 8, 0); */
/*     int i; */
/*     for (i = 0; i < 3; ++i) */
/*         printf("\nsigma[%d] = %8.2f ", i, sigma[i]); */

    return(sumdev);
}


double
ProcJacobiSVDFrag(const FragCoords *coords1, const FragCoords *coords2, double **rotmat,
                  double **Rmat, double **Umat, double **VTmat, double *sigma)
{
    double          det, sumdev, term1, term2;

    CalcRFrag(coords1, coords2, Rmat);
    CalcJacobiSVD(Rmat, Umat, sigma, VTmat, 1e-12);
    det = CalcRotMat(rotmat, Umat, VTmat);
    term1 = term2 = 0.0; /* this is just a fudge to make this faster - DLT */
    sumdev = term1 + term2;

    if (det < 0)
        sumdev -= 2.0 * (sigma[0] + sigma[1] - sigma[2]);
    else
        sumdev -= 2.0 * (sigma[0] + sigma[1] + sigma[2]);

    return(sumdev);
}


/* returns sum of squared residuals, E
   rmsd = sqrt(E/atom_num)  */
double
ProcJacobiSVDCov(Coords *coords1, Coords *coords2, double **rotmat,
                    const double **covmat, const double *axesw, double **Rmat,
                    double **Umat, double **VTmat, double *sigma)
{
    double          det, sumdev = 0.0;

    CalcCovCoords(coords1, covmat);
    CalcCovCoords(coords2, covmat);

    sumdev = CalcE0Cov(coords1, coords2, axesw);
    CalcRCov(coords1, coords2, Rmat, covmat, axesw);
    CalcJacobiSVD(Rmat, Umat, sigma, VTmat, DBL_EPSILON);
    det = CalcRotMat(rotmat, Umat, VTmat);

    if (det < 0)
        sumdev -= 2.0 * (sigma[0] + sigma[1] - sigma[2]);
    else
        sumdev -= 2.0 * (sigma[0] + sigma[1] + sigma[2]);

    return(sumdev);
}
