/* otfdescrip.cc -- descriptions for OpenType tags
 *
 * Copyright (c) 2002-2006 Eddie Kohler
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version. This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <efont/otf.hh>

struct Mapping {
    uint32_t tag;
    const char *description;
};

#define T(str) ((str)[0]<<24 | (str)[1]<<16 | (str)[2]<<8 | (str)[3])

static const Mapping scripts[] = {
    { T("DFLT"), "Default" },
    { T("arab"), "Arabic" },
    { T("armn"), "Armenian" },
    { T("beng"), "Bengali" },
    { T("bopo"), "Bopomofo" },
    { T("brai"), "Braille" },
    { T("byzm"), "Byzantine Music" },
    { T("cans"), "Canadian Syllabics" },
    { T("cher"), "Cherokee" },
    { T("cyrl"), "Cyrillic" },
    { T("deva"), "Devanagari" },
    { T("ethi"), "Ethiopic" },
    { T("geor"), "Georgian" },
    { T("grek"), "Greek" },
    { T("gujr"), "Gujarati" },
    { T("guru"), "Gurmukhi" },
    { T("hang"), "Hangul" },
    { T("hani"), "CJK Ideographic" },
    { T("hebr"), "Hebrew" },
    { T("jamo"), "Hangul Jamo" },
    { T("kana"), "Hiragana/Katakana" },
    { T("khmr"), "Khmer" },
    { T("knda"), "Kannada" },
    { T("lao "), "Lao" },
    { T("latn"), "Latin" },
    { T("mlym"), "Malayalam" },
    { T("mong"), "Mongolian" },
    { T("mymr"), "Myanmar" },
    { T("ogam"), "Ogham" },
    { T("orya"), "Oriya" },
    { T("runr"), "Runic" },
    { T("sinh"), "Sinhala" },
    { T("syrc"), "Syriac" },
    { T("taml"), "Tamil" },
    { T("telu"), "Telugu" },
    { T("thaa"), "Thaana" },
    { T("thai"), "Thai" },
    { T("tibt"), "Tibetan" },
    { T("yi  "), "Yi" }
};

static const Mapping languages[] = {
    { T("ABA "), "Abaza" },
    { T("ABK "), "Abkhazian" },
    { T("ADY "), "Adyghe" },
    { T("AFK "), "Afrikaans" },
    { T("AFR "), "Afar" },
    { T("AGW "), "Agaw" },
    { T("ALT "), "Altai" },
    { T("AMH "), "Amharic" },
    { T("ARA "), "Arabic" },
    { T("ARI "), "Aari" },
    { T("ARK "), "Arakanese" },
    { T("ASM "), "Assamese" },
    { T("ATH "), "Athapaskan" },
    { T("AVR "), "Avar" },
    { T("AWA "), "Awadhi" },
    { T("AYM "), "Aymara" },
    { T("AZE "), "Azeri" },
    { T("BAD "), "Badaga" },
    { T("BAG "), "Baghelkhandi" },
    { T("BAL "), "Balkar" },
    { T("BAU "), "Baule" },
    { T("BBR "), "Berber" },
    { T("BCH "), "Bench" },
    { T("BCR "), "Bible Cree" },
    { T("BEL "), "Belarussian" },
    { T("BEM "), "Bemba" },
    { T("BEN "), "Bengali" },
    { T("BGR "), "Bulgarian" },
    { T("BHI "), "Bhili" },
    { T("BHO "), "Bhojpuri" },
    { T("BIK "), "Bikol" },
    { T("BIL "), "Bilen" },
    { T("BKF "), "Blackfoot" },
    { T("BLI "), "Balochi" },
    { T("BLN "), "Balante" },
    { T("BLT "), "Balti" },
    { T("BMB "), "Bambara" },
    { T("BML "), "Bamileke" },
    { T("BRE "), "Breton" },
    { T("BRH "), "Brahui" },
    { T("BRI "), "Braj Bhasha" },
    { T("BRM "), "Burmese" },
    { T("BSH "), "Bashkir" },
    { T("BTI "), "Beti" },
    { T("CAT "), "Catalan" },
    { T("CEB "), "Cebuano" },
    { T("CHE "), "Chechen" },
    { T("CHG "), "Chaha Gurage" },
    { T("CHH "), "Chattisgarhi" },
    { T("CHI "), "Chichewa" },
    { T("CHK "), "Chukchi" },
    { T("CHP "), "Chipewyan" },
    { T("CHR "), "Cherokee" },
    { T("CHU "), "Chuvash" },
    { T("CMR "), "Comorian" },
    { T("COP "), "Coptic" },
    { T("CRE "), "Cree" },
    { T("CRR "), "Carrier" },
    { T("CRT "), "Crimean Tatar" },
    { T("CSL "), "Church Slavonic" },
    { T("CSY "), "Czech" },
    { T("DAN "), "Danish" },
    { T("DAR "), "Dargwa" },
    { T("DCR "), "Woods Cree" },
    { T("DEU "), "German (Standard)" },
    { T("DGR "), "Dogri" },
    { T("DHV "), "Dhivehi" },
    { T("DJR "), "Djerma" },
    { T("DNG "), "Dangme" },
    { T("DNK "), "Dinka" },
    { T("DUN "), "Dungan" },
    { T("DZN "), "Dzongkha" },
    { T("EBI "), "Ebira" },
    { T("ECR "), "Eastern Cree" },
    { T("EDO "), "Edo" },
    { T("EFI "), "Efik" },
    { T("ELL "), "Greek" },
    { T("ENG "), "English" },
    { T("ERZ "), "Erzya" },
    { T("ESP "), "Spanish" },
    { T("ETI "), "Estonian" },
    { T("EUQ "), "Basque" },
    { T("EVK "), "Evenki" },
    { T("EVN "), "Even" },
    { T("EWE "), "Ewe" },
    { T("FAN "), "French Antillean" },
    { T("FAR "), "Farsi" },
    { T("FIN "), "Finnish" },
    { T("FJI "), "Fijian" },
    { T("FLE "), "Flemish" },
    { T("FNE "), "Forest Nenets" },
    { T("FON "), "Fon" },
    { T("FOS "), "Faroese" },
    { T("FRA "), "French (Standard)" },
    { T("FRI "), "Frisian" },
    { T("FRL "), "Friulian" },
    { T("FTA "), "Futa" },
    { T("FUL "), "Fulani" },
    { T("GAD "), "Ga" },
    { T("GAE "), "Gaelic" },
    { T("GAG "), "Gagauz" },
    { T("GAL "), "Galician" },
    { T("GAR "), "Garshuni" },
    { T("GAW "), "Garhwali" },
    { T("GEZ "), "Ge'ez" },
    { T("GIL "), "Gilyak" },
    { T("GMZ "), "Gumuz" },
    { T("GON "), "Gondi" },
    { T("GRN "), "Greenlandic" },
    { T("GRO "), "Garo" },
    { T("GUA "), "Guarani" },
    { T("GUJ "), "Gujarati" },
    { T("HAI "), "Haitian" },
    { T("HAL "), "Halam" },
    { T("HAR "), "Harauti" },
    { T("HAU "), "Hausa" },
    { T("HAW "), "Hawaiian" },
    { T("HBN "), "Hammer-Banna" },
    { T("HIL "), "Hiligaynon" },
    { T("HIN "), "Hindi" },
    { T("HMA "), "High Mari" },
    { T("HND "), "Hindko" },
    { T("HO  "), "Ho" },
    { T("HRI "), "Harari" },
    { T("HRV "), "Croatian" },
    { T("HUN "), "Hungarian" },
    { T("HYE "), "Armenian" },
    { T("IBO "), "Igbo" },
    { T("IJO "), "Ijo" },
    { T("ILO "), "Ilokano" },
    { T("IND "), "Indonesian" },
    { T("ING "), "Ingush" },
    { T("INU "), "Inuktitut" },
    { T("IRI "), "Irish" },
    { T("IRT "), "Irish Traditional" },
    { T("ISL "), "Icelandic" },
    { T("ISM "), "Inari Sami" },
    { T("ITA "), "Italian" },
    { T("IWR "), "Hebrew" },
    { T("JAN "), "Japanese" },
    { T("JAV "), "Javanese" },
    { T("JII "), "Yiddish" },
    { T("JUD "), "Judezmo" },
    { T("JUL "), "Jula" },
    { T("KAB "), "Kabardian" },
    { T("KAC "), "Kachchi" },
    { T("KAL "), "Kalenjin" },
    { T("KAN "), "Kannada" },
    { T("KAR "), "Karachay" },
    { T("KAT "), "Georgian" },
    { T("KAZ "), "Kazakh" },
    { T("KEB "), "Kebena" },
    { T("KGE "), "Khutsuri Georgian" },
    { T("KHA "), "Khakass" },
    { T("KHK "), "Khanty-Kazim" },
    { T("KHM "), "Khmer" },
    { T("KHS "), "Khanty-Shurishkar" },
    { T("KHV "), "Khanty-Vakhi" },
    { T("KHW "), "Khowar" },
    { T("KIK "), "Kikuyu" },
    { T("KIR "), "Kirghiz" },
    { T("KIS "), "Kisii" },
    { T("KKN "), "Kokni" },
    { T("KLM "), "Kalmyk" },
    { T("KMB "), "Kamba" },
    { T("KMN "), "Kumaoni" },
    { T("KMO "), "Komo" },
    { T("KMS "), "Komso" },
    { T("KNR "), "Kanuri" },
    { T("KOD "), "Kodagu" },
    { T("KOK "), "Konkani" },
    { T("KON "), "Kikongo" },
    { T("KOP "), "Komi-Permyak" },
    { T("KOR "), "Korean" },
    { T("KOZ "), "Komi-Zyrian" },
    { T("KPL "), "Kpelle" },
    { T("KRI "), "Krio" },
    { T("KRK "), "Karakalpak" },
    { T("KRL "), "Karelian" },
    { T("KRM "), "Karaim" },
    { T("KRN "), "Karen" },
    { T("KRT "), "Koorete" },
    { T("KSH "), "Kashmiri" },
    { T("KSI "), "Khasi" },
    { T("KSM "), "Kildin Sami" },
    { T("KUI "), "Kui" },
    { T("KUL "), "Kulvi" },
    { T("KUM "), "Kumyk" },
    { T("KUR "), "Kurdish" },
    { T("KUU "), "Kurukh" },
    { T("KUY "), "Kuy" },
    { T("KYK "), "Koryak" },
    { T("LAD "), "Ladin" },
    { T("LAH "), "Lahuli" },
    { T("LAK "), "Lak" },
    { T("LAM "), "Lambani" },
    { T("LAO "), "Lao" },
    { T("LAT "), "Latin" },
    { T("LAZ "), "Laz" },
    { T("LCR "), "L-Cree" },
    { T("LDK "), "Ladakhi" },
    { T("LEZ "), "Lezgi" },
    { T("LIN "), "Lingala" },
    { T("LMA "), "Low Mari" },
    { T("LMB "), "Limbu" },
    { T("LMW "), "Lomwe" },
    { T("LSB "), "Lower Sorbian" },
    { T("LSM "), "Lule Sami" },
    { T("LTH "), "Lithuanian" },
    { T("LUB "), "Luba" },
    { T("LUG "), "Luganda" },
    { T("LUH "), "Luhya" },
    { T("LUO "), "Luo" },
    { T("LVI "), "Latvian" },
    { T("MAJ "), "Majang" },
    { T("MAK "), "Makua" },
    { T("MAL "), "Malayalam Traditional" },
    { T("MAN "), "Mansi" },
    { T("MAR "), "Marathi" },
    { T("MAW "), "Marwari" },
    { T("MBN "), "Mbundu" },
    { T("MCH "), "Manchu" },
    { T("MCR "), "Moose Cree" },
    { T("MDE "), "Mende" },
    { T("MEN "), "Me'en" },
    { T("MIZ "), "Mizo" },
    { T("MKD "), "Macedonian" },
    { T("MLE "), "Male" },
    { T("MLG "), "Malagasy" },
    { T("MLN "), "Malinke" },
    { T("MLR "), "Malayalam Reformed" },
    { T("MLY "), "Malay" },
    { T("MND "), "Mandinka" },
    { T("MNG "), "Mongolian" },
    { T("MNI "), "Manipuri" },
    { T("MNK "), "Maninka" },
    { T("MNX "), "Manx Gaelic" },
    { T("MOK "), "Moksha" },
    { T("MOL "), "Moldavian" },
    { T("MON "), "Mon" },
    { T("MOR "), "Moroccan" },
    { T("MRI "), "Maori" },
    { T("MTH "), "Maithili" },
    { T("MTS "), "Maltese" },
    { T("MUN "), "Mundari" },
    { T("NAG "), "Naga-Assamese" },
    { T("NAN "), "Nanai" },
    { T("NAS "), "Naskapi" },
    { T("NCR "), "N-Cree" },
    { T("NDB "), "Ndebele" },
    { T("NDG "), "Ndonga" },
    { T("NEP "), "Nepali" },
    { T("NEW "), "Newari" },
    { T("NHC "), "Norway House Cree" },
    { T("NIS "), "Nisi" },
    { T("NIU "), "Niuean" },
    { T("NKL "), "Nkole" },
    { T("NLD "), "Dutch" },
    { T("NOG "), "Nogai" },
    { T("NOR "), "Norwegian" },
    { T("NSM "), "Northern Sami" },
    { T("NTA "), "Northern Tai" },
    { T("NTO "), "Esperanto" },
    { T("NYN "), "Nynorsk" },
    { T("OCR "), "Oji-Cree" },
    { T("OJB "), "Ojibway" },
    { T("ORI "), "Oriya" },
    { T("ORO "), "Oromo" },
    { T("OSS "), "Ossetian" },
    { T("PAA "), "Palestinian Aramaic" },
    { T("PAL "), "Pali" },
    { T("PAN "), "Punjabi" },
    { T("PAP "), "Palpa" },
    { T("PAS "), "Pashto" },
    { T("PGR "), "Polytonic Greek" },
    { T("PIL "), "Pilipino" },
    { T("PLG "), "Palaung" },
    { T("PLK "), "Polish" },
    { T("PRO "), "Provencal" },
    { T("PTG "), "Portuguese" },
    { T("QIN "), "Chin" },
    { T("RAJ "), "Rajasthani" },
    { T("RBU "), "Russian Buriat" },
    { T("RCR "), "R-Cree" },
    { T("RIA "), "Riang" },
    { T("RMS "), "Rhaeto-Romanic" },
    { T("ROM "), "Romanian" },
    { T("ROY "), "Romany" },
    { T("RSY "), "Rusyn" },
    { T("RUA "), "Ruanda" },
    { T("RUS "), "Russian" },
    { T("SAD "), "Sadri" },
    { T("SAN "), "Sanskrit" },
    { T("SAT "), "Santali" },
    { T("SAY "), "Sayisi" },
    { T("SEK "), "Sekota" },
    { T("SEL "), "Selkup" },
    { T("SGO "), "Sango" },
    { T("SHN "), "Shan" },
    { T("SIB "), "Sibe" },
    { T("SID "), "Sidamo" },
    { T("SIG "), "Silte Gurage" },
    { T("SKS "), "Skolt Sami" },
    { T("SKY "), "Slovak" },
    { T("SLA "), "Slavey" },
    { T("SLV "), "Slovenian" },
    { T("SML "), "Somali" },
    { T("SMO "), "Samoan" },
    { T("SNA "), "Sena" },
    { T("SND "), "Sindhi" },
    { T("SNH "), "Sinhalese" },
    { T("SNK "), "Soninke" },
    { T("SOG "), "Sodo Gurage" },
    { T("SOT "), "Sotho" },
    { T("SQI "), "Albanian" },
    { T("SRB "), "Serbian" },
    { T("SRK "), "Saraiki" },
    { T("SRR "), "Serer" },
    { T("SSL "), "South Slavey" },
    { T("SSM "), "Southern Sami" },
    { T("SUR "), "Suri" },
    { T("SVA "), "Svan" },
    { T("SVE "), "Swedish" },
    { T("SWA "), "Swadaya Aramaic" },
    { T("SWK "), "Swahili" },
    { T("SWZ "), "Swazi" },
    { T("SXT "), "Sutu" },
    { T("SYR "), "Syriac" },
    { T("TAB "), "Tabasaran" },
    { T("TAJ "), "Tajiki" },
    { T("TAM "), "Tamil" },
    { T("TAT "), "Tatar" },
    { T("TCR "), "TH-Cree" },
    { T("TEL "), "Telugu" },
    { T("TGN "), "Tongan" },
    { T("TGR "), "Tigre" },
    { T("TGY "), "Tigrinya" },
    { T("THA "), "Thai" },
    { T("THT "), "Tahitian" },
    { T("TIB "), "Tibetan" },
    { T("TKM "), "Turkmen" },
    { T("TMN "), "Temne" },
    { T("TNA "), "Tswana" },
    { T("TNE "), "Tundra Nenets" },
    { T("TNG "), "Tonga" },
    { T("TOD "), "Todo" },
    { T("TRK "), "Turkish" },
    { T("TSG "), "Tsonga" },
    { T("TUA "), "Turoyo Aramaic" },
    { T("TUL "), "Tulu" },
    { T("TUR "), "Turkish" },
    { T("TUV "), "Tuvin" },
    { T("TWI "), "Twi" },
    { T("UDM "), "Udmurt" },
    { T("UKR "), "Ukrainian" },
    { T("URD "), "Urdu" },
    { T("USB "), "Upper Sorbian" },
    { T("UYG "), "Uyghur" },
    { T("UZB "), "Uzbek" },
    { T("VEN "), "Venda" },
    { T("VIT "), "Vietnamese" },
    { T("WA  "), "Wa" },
    { T("WAG "), "Wagdi" },
    { T("WCR "), "West-Cree" },
    { T("WEL "), "Welsh" },
    { T("WLF "), "Wolof" },
    { T("XHS "), "Xhosa" },
    { T("YAK "), "Yakut" },
    { T("YBA "), "Yoruba" },
    { T("YCR "), "Y-Cree" },
    { T("YIC "), "Yi Classic" },
    { T("YIM "), "Yi Modern" },
    { T("ZHP "), "Chinese Phonetic" },
    { T("ZHS "), "Chinese Simplified" },
    { T("ZHT "), "Chinese Traditional" },
    { T("ZND "), "Zande" },
    { T("ZUL "), "Zulu" }
};

static const Mapping features[] = {
    { T("aalt"), "Access All Alternates" },
    { T("abvf"), "Above-base Forms" },
    { T("abvm"), "Above-base Mark Positioning" },
    { T("abvs"), "Above-base Substitutions" },
    { T("afrc"), "Alternative Fractions" },
    { T("akhn"), "Akhands" },
    { T("blwf"), "Below-base Forms" },
    { T("blwm"), "Below-base Mark Positioning" },
    { T("blws"), "Below-base Substitutions" },
    { T("c2pc"), "Petite Capitals From Capitals" },
    { T("c2sc"), "Small Capitals From Capitals" },
    { T("calt"), "Contextual Alternates" },
    { T("case"), "Case-Sensitive Forms" },
    { T("ccmp"), "Glyph Composition/Decomposition" },
    { T("clig"), "Contextual Ligatures" },
    { T("cpsp"), "Capital Spacing" },
    { T("cswh"), "Contextual Swash" },
    { T("curs"), "Cursive Positioning" },
    { T("dist"), "Distances" },
    { T("dlig"), "Discretionary Ligatures" },
    { T("dnom"), "Denominators" },
    { T("expt"), "Expert Forms" },
    { T("falt"), "Final Glyph on Line Alternates" },
    { T("fin2"), "Terminal Forms #2" },
    { T("fin3"), "Terminal Forms #3" },
    { T("fina"), "Terminal Forms" },
    { T("frac"), "Fractions" },
    { T("fwid"), "Full Widths" },
    { T("half"), "Half Forms" },
    { T("haln"), "Halant Forms" },
    { T("halt"), "Alternate Half Widths" },
    { T("hist"), "Historical Forms" },
    { T("hkna"), "Horizontal Kana Alternates" },
    { T("hlig"), "Historical Ligatures" },
    { T("hngl"), "Hangul" },
    { T("hwid"), "Half Widths" },
    { T("init"), "Initial Forms" },
    { T("isol"), "Isolated Forms" },
    { T("ital"), "Italics" },
    { T("jalt"), "Justification Alternates" },
    { T("jp78"), "JIS78 Forms" },
    { T("jp83"), "JIS83 Forms" },
    { T("jp90"), "JIS90 Forms" },
    { T("kern"), "Kerning" },
    { T("lfbd"), "Left Bounds" },
    { T("liga"), "Standard Ligatures" },
    { T("ljmo"), "Leading Jamo Forms" },
    { T("lnum"), "Lining Figures" },
    { T("locl"), "Localized Forms" },
    { T("mark"), "Mark Positioning" },
    { T("med2"), "Medial Forms #2" },
    { T("medi"), "Medial Forms" },
    { T("mgrk"), "Mathematical Greek" },
    { T("mkmk"), "Mark to Mark Positioning" },
    { T("mset"), "Mark Positioning via Substitution" },
    { T("nalt"), "Alternate Annotation Forms" },
    { T("nlck"), "NLC Kanji Forms" },
    { T("nukt"), "Nukta Forms" },
    { T("numr"), "Numerators" },
    { T("onum"), "Oldstyle Figures" },
    { T("opbd"), "Optical Bounds" },
    { T("ordn"), "Ordinals" },
    { T("ornm"), "Ornaments" },
    { T("palt"), "Proportional Alternate Widths" },
    { T("pcap"), "Petite Capitals" },
    { T("pnum"), "Proportional Figures" },
    { T("pref"), "Pre-Base Forms" },
    { T("pres"), "Pre-base Substitutions" },
    { T("pstf"), "Post-base Forms" },
    { T("psts"), "Post-base Substitutions" },
    { T("pwid"), "Proportional Widths" },
    { T("qwid"), "Quarter Widths" },
    { T("rand"), "Randomize" },
    { T("rlig"), "Required Ligatures" },
    { T("rphf"), "Reph Forms" },
    { T("rtbd"), "Right Bounds" },
    { T("rtla"), "Right-to-left Alternates" },
    { T("ruby"), "Ruby Notation Forms" },
    { T("salt"), "Stylistic Alternates" },
    { T("sinf"), "Scientific Inferiors" },
    { T("size"), "Optical Size" },
    { T("smcp"), "Small Capitals" },
    { T("smpl"), "Simplified Forms" },
    { T("ss01"), "Stylistic Set 1" },
    { T("ss02"), "Stylistic Set 2" },
    { T("ss03"), "Stylistic Set 3" },
    { T("ss04"), "Stylistic Set 4" },
    { T("ss05"), "Stylistic Set 5" },
    { T("ss06"), "Stylistic Set 6" },
    { T("ss07"), "Stylistic Set 7" },
    { T("ss08"), "Stylistic Set 8" },
    { T("ss09"), "Stylistic Set 9" },
    { T("ss10"), "Stylistic Set 10" },
    { T("ss11"), "Stylistic Set 11" },
    { T("ss12"), "Stylistic Set 12" },
    { T("ss13"), "Stylistic Set 13" },
    { T("ss14"), "Stylistic Set 14" },
    { T("ss15"), "Stylistic Set 15" },
    { T("ss16"), "Stylistic Set 16" },
    { T("ss17"), "Stylistic Set 17" },
    { T("ss18"), "Stylistic Set 18" },
    { T("ss19"), "Stylistic Set 19" },
    { T("ss20"), "Stylistic Set 20" },
    { T("subs"), "Subscript" },
    { T("sups"), "Superscript" },
    { T("swsh"), "Swash" },
    { T("titl"), "Titling" },
    { T("tjmo"), "Trailing Jamo Forms" },
    { T("tnam"), "Traditional Name Forms" },
    { T("tnum"), "Tabular Figures" },
    { T("trad"), "Traditional Forms" },
    { T("twid"), "Third Widths" },
    { T("unic"), "Unicase" },
    { T("valt"), "Alternate Vertical Metrics" },
    { T("vatu"), "Vattu Variants" },
    { T("vert"), "Vertical Writing" },
    { T("vhal"), "Alternate Vertical Half Metrics" },
    { T("vjmo"), "Vowel Jamo Forms" },
    { T("vkna"), "Vertical Kana Alternates" },
    { T("vkrn"), "Vertical Kerning" },
    { T("vpal"), "Proportional Alternate Vertical Metrics" },
    { T("vrt2"), "Vertical Alternates and Rotation" },
    { T("zero"), "Slashed Zero" }
};

namespace Efont { namespace OpenType {

static const char *
find_description(uint32_t tag, const Mapping *maps, int n)
{
    // should use STL...
    int l = 0, r = n - 1;
    while (l <= r) {
	int m = (l + r) / 2;
	if (maps[m].tag < tag)
	    l = m + 1;
	else if (maps[m].tag == tag)
	    return maps[m].description;
	else
	    r = m - 1;
    }
    return 0;
}

const char *
Tag::script_description() const
{
    return find_description(value(), scripts, sizeof(scripts) / sizeof(Mapping));
}

const char *
Tag::language_description() const
{
    return find_description(value(), languages, sizeof(languages) / sizeof(Mapping));
}

const char *
Tag::feature_description() const
{
    return find_description(value(), features, sizeof(features) / sizeof(Mapping));
}

}}
