/*
 * ui-preview.c - a module for preview window UI
 * by Hirotsugu Kakugawa
 */
/*
 * Copyright (C) 1996-1997 Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/param.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>
#include <X11/Xresource.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/ScrolledW.h>
#include <Xm/ScrollBar.h>
#include <Xm/DrawingA.h>
#include <Xm/Scale.h>

#include "cf-xmdvi.h"
#include "dvi-2_6.h"
#include "dev.h"
#include "defs.h"
#include "resource.h"
#include "paper.h"
#include "window.h"
#include "key.h"
#include "ui.h"


Private Widget   x_toplevel          = NULL;
Private Widget   x_previewer         = NULL;
Private Widget   x_scroll            = NULL;
Private Widget   x_preview_window    = NULL;
Private Widget   x_display_area      = NULL;
Private Widget   x_vertical_slider   = NULL;
Private Widget   x_horizontal_slider = NULL;
Private Cursor   x_c_reading, x_c_ready, x_c_drawing, x_c_ps; 
#define GC_OPS  (GCFunction|GCForeground|GCBackground)
#define XDISP  XtDisplay(x_toplevel)

Private void  make_preview_paper(int,int);
Private void  preview_window_enlarge_shrink(int);
Private void  cb_preview_expose(Widget,caddr_t,caddr_t);
Private void  cb_preview_user_cmd(Widget,caddr_t,XEvent*);
Private void  do_move_v(int,int,int,int,int,int);
Private void  do_move_h(int,int,int,int,int,int);


Public int
x_make_preview(Widget top)
{
  char    name[80];
  int     w, h;
  XEvent  xev;
  Arg     args[10];
  int     i;

  x_toplevel = top;

  i = 0;
  sprintf(name, "XMDVI Preview");
  XtSetArg(args[i], XmNiconName, name); i++;
  XtSetArg(args[i], XmNmappedWhenManaged, True); i++;
  x_previewer 
    = XtAppCreateShell(PROG_NAME_RESOURCE, "xmdvi preview", 
		       topLevelShellWidgetClass, XDISP, args, i);

  i = 0;
  XtSetArg(args[i], XmNwidth,  RES(DviDev,prevwin_w)); i++;
  XtSetArg(args[i], XmNheight, RES(DviDev,prevwin_h)); i++;
  XtSetArg(args[i], XmNscrollBarPlacement, XmTOP_LEFT); i++;
  XtSetArg(args[i], XmNscrollBarDisplayPolicy, XmAUTOMATIC); i++;
  XtSetArg(args[i], XmNscrollingPolicy, XmAUTOMATIC); i++;
  XtSetArg(args[i], XmNhighlightThickness, 0); i++;
  XtSetArg(args[i], XmNtraversalOn, False); i++;
  XtSetArg(args[i], XmNkeyboardFocusPolicy, XmPOINTER); i++;
  x_scroll = XmCreateScrolledWindow(x_previewer, "preview", args, i);
  x_widget_short_cut(x_scroll, KeyPressMask|ButtonPressMask);
    
  XtManageChild(x_scroll);
  XtRealizeWidget(x_previewer);
  XSync(XDISP, False);

  XtSetArg(args[0], XmNhorizontalScrollBar, &x_horizontal_slider);
  XtGetValues(x_scroll, args, 1);
  XtSetArg(args[0], XmNtraversalOn, False);
  XtSetValues(x_horizontal_slider, args, 1);
  XtSetArg(args[0], XmNverticalScrollBar, &x_vertical_slider);
  XtGetValues(x_scroll, args, 1);
  XtSetArg(args[0], XmNtraversalOn, False);
  XtSetValues(x_vertical_slider, args, 1);
  XtSetArg(args[0], XmNclipWindow, &x_display_area);
  XtGetValues(x_scroll, args, 1);
  x_widget_short_cut(x_display_area, KeyPressMask|ButtonPressMask);

  x_c_reading = XCreateFontCursor(XDISP, XC_shuttle);
  x_c_drawing = XCreateFontCursor(XDISP, XC_watch);
  x_c_ready   = XCreateFontCursor(XDISP, XC_circle);
  x_c_ps      = XCreateFontCursor(XDISP, XC_coffee_mug);

  x_preview_window = NULL;
  ui_preview_paper_change(&w, &h);

  XSync(XDISP, False);
  while (XtPending()){
    XtNextEvent(&xev);
    XtDispatchEvent(&xev);
  }
  XSync(XDISP, False);

  return 0;
}

Public void
ui_preview_mag_changed(int *wp, int *hp)
{
  ui_preview_paper_change(wp, hp);
}

Public void
ui_preview_paper_change(int *wp, int *hp)
{
  int   w, h;

  if (RES(DviDev,orient) == PAPER_ORIENTAION_PORT){
    paper_pixel_size(PAPER_BASIC_ID(RES(DviDev,paper)), 
		     &w, &h, DviShrinkFactor);
  } else {
    paper_pixel_size(PAPER_BASIC_ID(RES(DviDev,paper)), 
		     &h, &w, DviShrinkFactor);
  }
  XVAR(DviDev,page_width)  = w;
  XVAR(DviDev,page_height) = h;
  XVAR(DviDev,offset_x) = RES(DviDev,offset_x)*RES(DviDev,dpi)/DviShrinkFactor;
  XVAR(DviDev,offset_y) = RES(DviDev,offset_y)*RES(DviDev,dpi)/DviShrinkFactor;

  make_preview_paper(w, h);
  x_alloc_page_cache(DviShrinkFactor, w, h);

  if (wp != NULL)
    *wp = w;
  if (hp != NULL)
    *hp = h;
}

Private void
make_preview_paper(int w, int h)
{
  Arg         args[10];
  int         i;
  Window        root;
  unsigned int  dw, dh, pw, ph, b, d; 
  int           x, y, pos;

  if (x_preview_window != NULL){
    XGetGeometry(XtDisplay(x_preview_window), XtWindow(x_preview_window),
		 &root, &x, &y, &pw, &ph, &b, &d);
    XGetGeometry(XtDisplay(x_display_area), XtWindow(x_display_area),
		 &root, &x, &y, &dw, &dh, &b, &d);
    XtSetArg(args[0], XmNvalue, &pos);
    XtGetValues(x_vertical_slider, args, 1);
    do_move_v(w, h, dw, dh, pos, 0);
    XtSetArg(args[0], XmNvalue, &pos);
    XtGetValues(x_horizontal_slider, args, 1);
    do_move_h(w, h, dw, dh, pos, 0);

    i = 0;
    XtSetArg(args[i], XmNwidth,  w); i++;
    XtSetArg(args[i], XmNheight, h); i++;
    XtSetValues(x_preview_window, args, i);
    return;
  }

  i = 0;
  XtSetArg(args[i], XmNwidth, w); i++;
  XtSetArg(args[i], XmNheight, h); i++;
  XtSetArg(args[i], XmNtraversalOn, False); i++;
  XtSetArg(args[i], XmNkeyboardFocusPolicy, XmPOINTER); i++;
  XtSetArg(args[i], XmNforeground, XVAR(DviDev,color_char)); i++;
  XtSetArg(args[i], XmNbackground, XVAR(DviDev,color_paper)); i++;
  x_preview_window = XmCreateDrawingArea(x_scroll, "Window", args, i);
  XtAddCallback(x_preview_window, XmNexposeCallback, 
		(XtCallbackProc)cb_preview_expose, 0);
  x_widget_short_cut(x_preview_window, KeyPressMask|ButtonPressMask);
  XtManageChild(x_preview_window);
#if 0
  XmScrolledWindowSetAreas(x_scroll, NULL, NULL, x_preview_window);
#endif
  XVAR(DviDev,view) = XtWindow(x_preview_window);

  XVAR(DviDev,gc_copy)  = XCreateGC(XDISP, XVAR(DviDev,view), 0, NULL);
  XVAR(DviDev,gc_clear) = XCreateGC(XDISP, XVAR(DviDev,view), 0, NULL);
  XSetForeground(XDISP, XVAR(DviDev,gc_clear), XVAR(DviDev,color_paper));
}

Public void
x_widget_short_cut(Widget w, EventMask mask)
{
  XtAddEventHandler(w, mask, FALSE, (XtEventHandler)cb_preview_user_cmd, NULL);
}

Public void
x_cursor_change(int curs_id)
{
  switch (curs_id){
  case CURSOR_READING:
    XDefineCursor(XDISP, XVAR(DviDev,view), x_c_reading);
    XDefineCursor(XDISP, XtWindow(x_display_area), x_c_reading);
    break;
  case CURSOR_READY:  
    XDefineCursor(XDISP, XVAR(DviDev,view), x_c_ready);
    XDefineCursor(XDISP, XtWindow(x_display_area), x_c_ready);
    break;
  case CURSOR_DRAWING:  
    XDefineCursor(XDISP, XVAR(DviDev,view), x_c_drawing);
    XDefineCursor(XDISP, XtWindow(x_display_area), x_c_drawing);
    break;
  case CURSOR_DRAWING_PS: 
    XDefineCursor(XDISP, XVAR(DviDev,view), x_c_ps);
    XDefineCursor(XDISP, XtWindow(x_display_area), x_c_ps);
    break;
  }
  XVAR(DviDev,cursor) = curs_id;
  XFlush(XDISP); 
}

Private void
cb_preview_expose(Widget w, caddr_t d1, caddr_t d2)
{
  PAGE_CACHE pc; 

  if ((DviFile == NULL) || (XVAR(DviDev,page_cache) == NULL)
      || (!XtIsRealized(x_preview_window)))
    return;

  pc = x_get_cache(DviCurrentPage);
  XCopyArea(XVAR(DviDev,xdisp), pc->pix, 
	    XVAR(DviDev,view), XVAR(DviDev,gc_copy), 
	    0, 0, XVAR(DviDev,page_width), XVAR(DviDev,page_height), 
	    0, 0);
}

Private void
cb_preview_user_cmd(Widget w, caddr_t d1, XEvent *xev)
{
  key_command(xev);
}


Private void
do_move_h(int pw, int ph, int dw, int dh, int pos, int mv)
{
  int           new_pos;
  Arg           args[5];
  XmScaleCallbackStruct  cbs;

  if ((SCROLL_LEFTMOST < mv) && (mv < SCROLL_RIGHTMOST)){
    new_pos = (int)pos + (int)((double)dw*((double)mv/100.0));
    if (new_pos > ((int)pw-(int)dw))
      new_pos = (int)pw-(int)dw;
    else if (new_pos < 0)
      new_pos = 0;
  } else if (mv <= SCROLL_LEFTMOST){
    new_pos = 0;
  } else {
    new_pos = (int)pw-(int)dw;
  }

  if (new_pos < 0)
    new_pos = 0;
  if (new_pos != pos){
    cbs.reason = XmCR_DRAG;
    cbs.event  = NULL;
    cbs.value  = new_pos;
    XtSetArg(args[0], XmNvalue, new_pos);
    XtSetValues(x_horizontal_slider, args, 1);
    XtCallCallbacks(x_horizontal_slider, XmNdragCallback, &cbs);
  }
}

Public void
cmd_preview_move_h(int mv)
{
  int           pos;
  Arg           args[5];
  Window        root;
  unsigned int  dw, dh, pw, ph, b, d; 
  int           x, y, i;

  if (DviDisableCommand == 1)
    ;  /* no return */
  if (x_preview_window == NULL)
    return;

  XGetGeometry(XtDisplay(x_preview_window), XtWindow(x_preview_window),
	       &root, &x, &y, &pw, &ph, &b, &d);
  XGetGeometry(XtDisplay(x_display_area), XtWindow(x_display_area),
	       &root, &x, &y, &dw, &dh, &b, &d);
  i = 0;
  XtSetArg(args[i], XmNvalue, &pos); i++;
  XtSetArg(args[i], XmNmaximum, &pw); i++;
  XtSetArg(args[i], XmNsliderSize, &dw); i++;
  XtGetValues(x_horizontal_slider, args, i);

  do_move_h(pw, ph, dw, dh, pos, mv);
}

Private void 
do_move_v(int pw, int ph, int dw, int dh, int pos, int mv)
{
  int           new_pos;
  Arg           args[5];
  XmScaleCallbackStruct  cbs;

  if ((SCROLL_TOP < mv) && (mv < SCROLL_BOTTOM)){
    new_pos = pos + (int)((double)dw*((double)mv/100.0));
    if (new_pos > (int)ph-(int)dh)
      new_pos = (int)ph-(int)dh;
    else if (new_pos < 0)
      new_pos = 0;
  } else if (mv <= SCROLL_TOP){
    new_pos = 0;
  } else {
    new_pos = (int)ph-(int)dh;
  }

  if (new_pos < 0)
    new_pos = 0;
  if (new_pos != pos){
    cbs.reason = XmCR_DRAG;
    cbs.event  = NULL;
    cbs.value  = new_pos;
    XtSetArg(args[0], XmNvalue, new_pos);
    XtSetValues(x_vertical_slider, args, 1);
    XtCallCallbacks(x_vertical_slider, XmNdragCallback, &cbs);
  }
}

Public void
cmd_preview_move_v(int mv)
{
  int           pos;
  Arg           args[5];
  Window        root;
  unsigned int  dw, dh, pw, ph, b, d; 
  int           x, y;

  if (DviDisableCommand == 1)
    ;  /* no return */
  if (x_preview_window == NULL)
    return;

  XGetGeometry(XtDisplay(x_preview_window), XtWindow(x_preview_window),
	       &root, &x, &y, &pw, &ph, &b, &d);
  XGetGeometry(XtDisplay(x_display_area), XtWindow(x_display_area),
	       &root, &x, &y, &dw, &dh, &b, &d);
  XtSetArg(args[0], XmNvalue, &pos);
  XtGetValues(x_vertical_slider, args, 1);

  do_move_v(pw, ph, dw, dh, pos, mv);
}



Public void
cmd_preview_enlarge(void)
{
  if (DviDisableCommand == 1)
    ; /* no return */
  preview_window_enlarge_shrink(1);
}

Public void
cmd_preview_shrink(void)
{
  if (DviDisableCommand == 1)
    ; /* no return */
  preview_window_enlarge_shrink(0);
}

Private void
preview_window_enlarge_shrink(int enlarge)
{
  Arg         args[5];
  int         i;
  Dimension   w0, h0, w1, h1;

  i = 0;
  XtSetArg(args[i], XmNwidth, &w0); i++;
  XtSetArg(args[i], XmNheight, &h0); i++;
  XtGetValues(x_previewer, args, i);
  if (enlarge == 1){
    w1 = (w0 * PREVIEW_MAG_DEN)/PREVIEW_MAG_NUM;
    h1 = (h0 * PREVIEW_MAG_DEN)/PREVIEW_MAG_NUM;
  } else {
    w1 = (w0 * PREVIEW_MAG_NUM)/PREVIEW_MAG_DEN;
    h1 = (h0 * PREVIEW_MAG_NUM)/PREVIEW_MAG_DEN;
  }

  i = 0;
  XtSetArg(args[i], XmNwidth, w1); i++;
  XtSetArg(args[i], XmNheight, h1); i++;
  XtSetValues(x_previewer, args, i);
}

