/*   FILE: new_proj.c -- A program to create a new project.
 * AUTHOR: W. Michael Petullo <new@flyn.org>
 *   DATE: 01 MAY 1 999
 *
 * Copyright (c) 1999 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <template.h>
#include <fmt_ptrn.h>
#ifdef HAVE_LIBZ
#include <zlib.h>
#endif
#include <limits.h>
#include <string.h>
#include <common.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <config.h>
#ifdef HAVE_PWDB_PWDB_PUBLIC_H
#include <pwdb/pwdb_public.h>
#else
#include <pwd.h>
#endif

/* Bit mask values for arguments. */
#define FORCE 0x1		/* Overwrite existing file. */
#define USE_GLOBAL 0x2		/* Ignore local templates. */

typedef struct settings_t {
    int force;
    int list_only;
    int use_global;
    char *proj_name;
    char *proj_type;
} settings_t;

/* ============================ usage () ==================================== */
void usage(const int exitcode, const char *error, const char *more)
{
    fprintf(stderr, "Usage: new proj_name proj_type [option]\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "  -h, --help          print this message\n");
    fprintf(stderr, "  -f, --force         overwite any existing file\n");
    fprintf(stderr, "  -l, --list          list projects available\n");
    fprintf(stderr, "  -g, --global        Use global projects instead "
	    "of personal ones\n\n");
    if (error)
	fprintf(stderr, "%s: %s.\n\n", error, more);
    exit(exitcode);
}

/* ============================ parse_args () =============================== */
void parse_args(int argc, char *argv[], settings_t * settings)
{
    char c;
    int opt_index = 0;
    struct option opts[] = {
	{"help", 0, NULL, 'h'},
	{"force", 0, NULL, 'f'},
	{"list", 0, NULL, 'l'},
	{"global", 0, NULL, 'g'},
	{0, 0, 0, 0}
    };
    while ((c = getopt_long(argc, argv, "hflg", opts, &opt_index)) >= 0) {
	switch (c) {
	case 'h':
	    usage(EXIT_SUCCESS, NULL, NULL);
	case 'f':
	    settings->force = 1;
	    break;
	case 'l':
	    settings->list_only = 1;
	    break;
	case 'g':
	    settings->use_global = 1;
	    break;
	default:
	    exit(EXIT_FAILURE);
	}
    }
    if (settings->list_only)
	return;
    if (!(settings->proj_name = argv[optind++]))
	usage(EXIT_FAILURE, "new", "No project name specified");
    if (!(settings->proj_type = argv[optind++]))
	usage(1, "new", "No project type specified");
}

/* ============================ build_local_proj_dir () ===================== */
char *build_local_proj_dir(char *path)
{
    char hd[PATH_MAX + 1];
#ifdef HAVE_PWDB_PWDB_PUBLIC_H
    if (pwdb_start() != PWDB_SUCCESS)
	fprintf(stderr, "could not start pwdb lib\n");
#endif
    if (!homedir(hd)) {
	fprintf(stderr, "error determining home directory\n");
	strcpy(path, "");
    } else
	strcpy(path, hd);
    strcat(path, "/.new_proj/projects/");
#ifdef HAVE_PWDB_PWDB_PUBLIC_H
    pwdb_end();
#endif
    return path;
}

/* ============================ list_project_types () ======================= */
void list_project_types(void)
{
    char template_path[PATH_MAX + 1];
    DIR *dp;
    printf("Personal projects:\n");
    /* Try $HOME/.new/templates first. */
    if (!(dp = opendir(build_local_proj_dir(template_path))))
	printf("  <none>\n");
    else
	print_dir(dp);
    closedir(dp);
    printf("\nGlobal projects:\n");
    /* Try global location. */
    if (!(dp = opendir(GLOBAL_PROJECT_DIR)))
	printf("  <none>\n");
    else
	print_dir(dp);
    closedir(dp);
}

/* ============================ at_path () ================================== */
int at_path(char *path)
/* Make sure path is big enough to have .gz added to the end! */
{
    struct stat stat_buf;
    if (stat(path, &stat_buf) != -1)
	return 1;
    else {
	strcat(path, ".gz");
	if (stat(path, &stat_buf) != -1)
	    return 1;
    }
    return 0;
}

/* ============================ find_project () ============================= */
int find_project(char *path, char *type, int use_global)
{
    char hd[PATH_MAX + 1];
    if (!use_global) {
	/* Try $HOME/.new/templates first. */
	strcpy(path, homedir(hd));
	strcat(path, "/.new/projects/");
	strcat(path, type);
    }
    if (!use_global && !at_path(path)) {
	strcpy(path, GLOBAL_PROJECT_DIR);
	strcat(path, "/");
	strcat(path, type);
    }
    if (!at_path(path)) {
	return 0;
    }
    return 1;
}

/* ============================ write_file () =============================== */
int write_file(const settings_t settings, char *conf_line)
{
    fmt_ptrn_t x;
    char *argv[ARG_MAX], *delim;
    int status, argc = 0, i;
    char str[ARG_MAX];
    fmt_ptrn_init(&x);
    fmt_ptrn_update_kv(&x, strdup("PROJ_NAME"),
                               strdup(settings.proj_name));
    argv[argc] = (char *) malloc (4);
    strcpy (argv[argc++], "new");
    argv[argc] = (char *) malloc (3);
    strcpy (argv[argc++], "-s");
    argv[argc] = (char *) malloc (10 + strlen (settings.proj_name) + 1);
    strcpy (argv[argc], "PROJ_NAME=");
    strcat (argv[argc++], settings.proj_name);
    while (*conf_line && (delim = strchr (conf_line, ' '))) {
        memset (str, 0x00, ARG_MAX);
        strncpy (str, conf_line, delim - conf_line);
        argv[argc++] = fmt_ptrn_filled(&x, str);
	conf_line = delim + 1;
    }
    if (*conf_line) {
        memset (str, 0x00, ARG_MAX);
        strcpy (str, conf_line);
	str[strlen(str) - 1] = 0x00; /* Get rid of '\n'. */
        argv[argc++] = fmt_ptrn_filled(&x, str);
    }
    argv[argc] = 0x00;
    if (! fork ()) {
        execvp ("new", argv);
	perror("error running new");
	return 0;
    }
    wait (&status);
    for (i = 0; i < argc; i++)
        free (argv[i]);
    fmt_ptrn_close(&x);
    if (WEXITSTATUS(status))
        return 0;
    return 1;
}

/* ============================ write_files () ============================== */
int write_files(settings_t settings)
{
#ifdef HAVE_LIBZ
    gzFile *conf;
#else
    FILE *conf;
#endif
    char conf_line[PATH_MAX + 1], conf_path[PATH_MAX + 1];
    if (!find_project(conf_path, settings.proj_type, settings.use_global)) {
	fprintf(stderr, "project type %s not found\n", settings.proj_type);
	exit(EXIT_FAILURE);
    }
#ifdef HAVE_LIBZ
    if (!(conf = gzopen(conf_path, "r"))) {
#else
    if (!(conf = fopen(conf_path, "r"))) {
#endif
	fprintf(stderr, "error opening %s\n", conf_path);
	exit(EXIT_FAILURE);
    }
#ifdef HAVE_LIBZ
    while (gzgets(conf, conf_line, PATH_MAX + 1) != Z_NULL) {
#else
    while (fgets(conf_line, PATH_MAX + 1, conf) != NULL) {
#endif
	if (!write_file(settings, conf_line))
	    return 0;
    }
#ifdef HAVE_LIBZ
    gzclose(conf);
#else
    fclose(conf);
#endif
    return 1;
}

/* ============================ main () ===================================== */
int main(int argc, char *argv[], char *env[])
{
    settings_t settings = { 0, 0, 0, NULL };
    parse_args(argc, argv, &settings);
    if (settings.list_only) {
	list_project_types();
	exit(EXIT_SUCCESS);
    }
    template_init();
    if (!template_set_global_dir(GLOBAL_TEMPLATE_DIR)) {
	template_perror("error setting global template dir");
	exit(EXIT_FAILURE);
    }
    if (!template_set_local_dir(".new/templates")) {
	template_perror("error setting local template dir");
	exit(EXIT_FAILURE);
    }
    if (!write_files(settings)) {
	/* Error message already written by new_proj or new. */
	exit(EXIT_FAILURE);
    }
    template_destroy();
    exit(EXIT_SUCCESS);
}
