/*
 * StreamedMedia page for channel notebook.
 *
 * Copyright (C) 2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2008 Nokia Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "config.h"

#include "page-streamed-media.h"

#include <glade/glade.h>

#include "page-priv.h"

struct _TIPageStreamedMediaClass {
    TIPageClass parent;
    gpointer priv;
};


struct _TIPageStreamedMediaPrivate {
    TpChannel *channel;

    /* UI for RequestStreams */
    GtkSpinButton *handle_spin;
    GtkToggleButton *audio_toggle;
    GtkToggleButton *video_toggle;
    GtkButton *request_streams_button;
};


G_DEFINE_TYPE (TIPageStreamedMedia, ti_page_streamed_media, TI_TYPE_PAGE);

enum {
    PROP_CHANNEL = 1
};


static void
ti_page_streamed_media_init (TIPageStreamedMedia *self)
{
  self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self, TI_TYPE_PAGE_STREAMED_MEDIA,
      TIPageStreamedMediaPrivate);
}


static void
ti_page_streamed_media_dispose (GObject *object)
{
  TIPageStreamedMedia *self = TI_PAGE_STREAMED_MEDIA (object);

  if (self->priv->channel != NULL)
    {
      g_object_unref (self->priv->channel);
      self->priv->channel = NULL;
    }

  G_OBJECT_CLASS (ti_page_streamed_media_parent_class)->dispose (object);
}


static void
ti_page_streamed_media_get_property (GObject *object,
                                     guint property_id,
                                     GValue *value,
                                     GParamSpec *pspec)
{
  TIPageStreamedMedia *self = TI_PAGE_STREAMED_MEDIA (object);

  switch (property_id)
    {
    case PROP_CHANNEL:
      g_value_set_object (value, self->priv->channel);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}


static void
ti_page_streamed_media_set_property (GObject *object,
                                     guint property_id,
                                     const GValue *value,
                                     GParamSpec *pspec)
{
  TIPageStreamedMedia *self = TI_PAGE_STREAMED_MEDIA (object);

  switch (property_id)
    {
    case PROP_CHANNEL:
      g_return_if_fail (self->priv->channel == NULL);
      self->priv->channel = TP_CHANNEL (g_value_dup_object (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}


static void
requested_streams (TpChannel *channel,
                   const GPtrArray *out0,
                   const GError *error,
                   gpointer unused G_GNUC_UNUSED,
                   GObject *object)
{
  TIPageStreamedMedia *self = TI_PAGE_STREAMED_MEDIA (object);

  if (error != NULL)
    {
      g_warning ("%p: RequestStreams(): %s", self, error->message);
      return;
    }
}


static void
request_streams_clicked (TIPageStreamedMedia *self)
{
  guint handle;
  GArray *types;

  handle = gtk_spin_button_get_value (self->priv->handle_spin);

  types = g_array_sized_new (FALSE, FALSE, sizeof (guint), 2);

  if (gtk_toggle_button_get_active (self->priv->audio_toggle))
    {
      guint i = TP_MEDIA_STREAM_TYPE_AUDIO;

      g_array_append_val (types, i);
    }

  if (gtk_toggle_button_get_active (self->priv->video_toggle))
    {
      guint i = TP_MEDIA_STREAM_TYPE_VIDEO;

      g_array_append_val (types, i);
    }

  tp_cli_channel_type_streamed_media_call_request_streams (
      self->priv->channel, -1, handle, types, requested_streams,
      NULL, NULL, (GObject *) self);

  g_array_free (types, TRUE);
}


static void
ti_page_streamed_media_setup_page (TIPage *base,
                                   GladeXML *glade_xml)
{
  TIPageStreamedMedia *self = TI_PAGE_STREAMED_MEDIA (base);
  GtkWidget *page = glade_xml_get_widget (glade_xml, "page");
  GList *kids;
  GtkBox *hbox;
  GtkWidget *widget;

  g_assert (GTK_IS_VBOX (page));

  /* get rid of placeholder content in the Glade file */
  kids = gtk_container_get_children (GTK_CONTAINER (page));
  g_list_foreach (kids, (GFunc) gtk_widget_destroy, NULL);
  g_list_free (kids);

  hbox = GTK_BOX (gtk_hbox_new (FALSE, 0));

  widget = gtk_label_new ("Handle: ");
  gtk_box_pack_start (hbox, widget, FALSE, FALSE, 0);

  widget = gtk_spin_button_new_with_range (0.0, G_MAXUINT32, 1.0);
  self->priv->handle_spin = GTK_SPIN_BUTTON (widget);
  gtk_spin_button_set_digits (self->priv->handle_spin, 0);
  gtk_spin_button_set_numeric (self->priv->handle_spin, TRUE);
  gtk_box_pack_start (hbox, widget, FALSE, FALSE, 0);

  widget = gtk_check_button_new_with_label ("Audio");
  self->priv->audio_toggle = GTK_TOGGLE_BUTTON (widget);
  gtk_box_pack_start (hbox, widget, FALSE, FALSE, 0);

  widget = gtk_check_button_new_with_label ("Video");
  self->priv->video_toggle = GTK_TOGGLE_BUTTON (widget);
  gtk_box_pack_start (hbox, widget, FALSE, FALSE, 0);

  widget = gtk_button_new_with_label ("RequestStreams");
  self->priv->request_streams_button = GTK_BUTTON (widget);
  gtk_box_pack_start (hbox, widget, FALSE, FALSE, 0);

  g_signal_connect_swapped (self->priv->request_streams_button, "clicked",
      G_CALLBACK (request_streams_clicked), self);

  gtk_box_pack_start (GTK_BOX (page), (GtkWidget *) hbox, FALSE, FALSE, 0);
}


static void
ti_page_streamed_media_class_init (TIPageStreamedMediaClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  TIPageClass *page_class = TI_PAGE_CLASS (klass);
  GParamSpec *param_spec;

  object_class->dispose = ti_page_streamed_media_dispose;
  object_class->get_property = ti_page_streamed_media_get_property;
  object_class->set_property = ti_page_streamed_media_set_property;

  page_class->setup_page = ti_page_streamed_media_setup_page;

  g_type_class_add_private (klass, sizeof (TIPageStreamedMediaPrivate));

  param_spec = g_param_spec_object ("channel", "TpChannel",
      "This page's associated channel", TP_TYPE_CHANNEL,
      G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE |
      G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB | G_PARAM_STATIC_NICK);
  g_object_class_install_property (object_class, PROP_CHANNEL, param_spec);
}


TIPageStreamedMedia *
ti_page_streamed_media_new (GtkNotebook *parent_notebook,
                            TpChannel *channel)
{
  TIPage *page = NULL;

  g_return_val_if_fail (channel != NULL, NULL);

  page = g_object_new (TI_TYPE_PAGE_STREAMED_MEDIA,
      "channel", channel,
      NULL);

  _ti_page_new (&page, parent_notebook, "page-streamed-media.xml");

  return TI_PAGE_STREAMED_MEDIA (page);
}
