/*
 * tclcurl.c --
 *
 * Implementation of the TclCurl extension that creates the curl namespace
 * so that Tcl interpreters can access libcurl.
 *
 * Copyright (c) 2001-2002 Andrs Garca Garca.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include "tclcurl.h"

/*
 *----------------------------------------------------------------------
 *
 * Tclcurl_Init --
 *
 *	This procedure initializes the package
 *
 * Results:
 *	A standard Tcl result.
 *
 *----------------------------------------------------------------------
 */

int
Tclcurl_Init (Tcl_Interp *interp) {

    if(Tcl_InitStubs(interp,"8.1",0)==NULL) {
        return TCL_ERROR;
    }

    Tcl_CreateObjCommand (interp,"::curl::init",curlInitObjCmd,
            (ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
    Tcl_CreateObjCommand (interp,"::curl::version",curlVersion,
            (ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
    Tcl_CreateObjCommand (interp,"::curl::escape",curlEscape,
            (ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
    Tcl_CreateObjCommand (interp,"::curl::unescape",curlUnescape,
            (ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);

    Tcl_PkgProvide(interp,"TclCurl","0.9.5");

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * curlCreateObjCmd --
 *
 *	Looks for the first free handle (curl1, curl2,...) and creates a
 *	Tcl command for it.
 *
 * Results:
 *  A string with the name of the handle, don't forget to free it.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

char *
curlCreateObjCmd (Tcl_Interp *interp,struct curlObjData  *curlData) {
    char                *handleName;
    int                 i;
    Tcl_CmdInfo         info;
    Tcl_Command         cmdToken;

    // We try with curl1, if it already exists with curl2...
    handleName=(char *)ckalloc(10);
    for (i=1;;i++) {
        sprintf(handleName,"curl%d",i);
        if (!Tcl_GetCommandInfo(interp,handleName,&info)) {
            cmdToken=Tcl_CreateObjCommand(interp,handleName,curlObjCmd,
                                (ClientData)curlData, 
                                (Tcl_CmdDeleteProc *)curlDeleteCmd);
            break;
        }
    }

    curlData->token=cmdToken;

    return handleName;
}

/*
 *----------------------------------------------------------------------
 *
 * curlIniTObjCmd --
 *
 *	This procedure is invoked to process the "curl::init" Tcl command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
curlInitObjCmd (ClientData clientData, Tcl_Interp *interp,
        int objc,Tcl_Obj *CONST objv[]) {

    Tcl_Obj		      *result;
    CURL                *curlHandle;
    struct curlObjData  *curlData;
    char                *handleName;

    curlData=(struct curlObjData *)ckalloc(sizeof(struct curlObjData));
    if (curlData==NULL) {
        result=Tcl_NewStringObj("Couldn't allocate memory",-1);
        Tcl_SetObjResult(interp,result); 
        return TCL_ERROR;
    }

    memset(curlData, 0, sizeof(struct curlObjData));
    curlData->interp=interp;

    curlHandle=curl_easy_init();

    if (curlHandle==NULL) {
        result=Tcl_NewStringObj("Couldn't open curl handle",-1);
        Tcl_SetObjResult(interp,result); 
        return TCL_ERROR;
    }
    
    handleName=curlCreateObjCmd(interp,curlData);

    curlData->curl=curlHandle;
   
    result=Tcl_NewStringObj(handleName,-1);
    Tcl_SetObjResult(interp,result);
    ckfree(handleName);

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * curlObjCmd --
 *
 *	This procedure is invoked to process the "curl" commands.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int
curlObjCmd (ClientData clientData, Tcl_Interp *interp,
    int objc,Tcl_Obj *CONST objv[]) {

    Tcl_Obj                *result;
    struct curlObjData     *curlData=(struct curlObjData *)clientData;
    CURL                   *curlHandle=curlData->curl;
    Tcl_Command            cmdToken=curlData->token;
    int                    tableIndex;
    int                    exitCode;

    if (objc<2) {
        Tcl_WrongNumArgs(interp,1,objv,"option arg ?arg?");
        return TCL_ERROR;
    }
    if (Tcl_GetIndexFromObj(interp, objv[1], commandTable, "option",
            TCL_EXACT,&tableIndex)==TCL_ERROR) {
        return TCL_ERROR;
    }
    switch(tableIndex) {
        case 0:
            if (curlSetOptsTransfer(interp,curlData,objc,objv)==TCL_ERROR) {
                return TCL_ERROR;
            }
            break;
        case 1:
//            fprintf(stdout,"Perform\n");
            if (curlPerform(interp,curlHandle,curlData)) {
                if (curlData->errorBuffer!=NULL) {
                    if (curlData->errorBufferKey==NULL) {
                        Tcl_SetVar(interp,curlData->errorBufferName,
                                curlData->errorBuffer,0);
                    } else {
                        Tcl_SetVar2(interp,curlData->errorBufferName,
                                curlData->errorBufferKey,
                                curlData->errorBuffer,0);
                    }
                }
                return TCL_ERROR;
            }
            break;
        case 2:
//            fprintf(stdout,"Getinfo\n");
            if (Tcl_GetIndexFromObj(interp,objv[2],getInfoTable,
                    "getinfo option",TCL_EXACT,&tableIndex)==TCL_ERROR) {
                return TCL_ERROR;
            }
            if (curlGetInfo(interp,curlHandle,tableIndex)) {
                return TCL_ERROR;
            }
            return TCL_OK;
            break;
        case 3:
//            fprintf(stdout,"Cleanup\n");
            Tcl_DeleteCommandFromToken(interp,curlData->token);
            break;
        case 4:
//            fprintf(stdout,"Configure\n");
            if (curlConfigTransfer(interp,curlData,objc,objv)==TCL_ERROR) {
                return TCL_ERROR;
            }
            break;
        case 5:
//            fprintf(stdout,"DupHandle\n");
            if (curlDupHandle(interp,curlData,objc,objv)==TCL_ERROR) {
                return TCL_ERROR;
            }
            break;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * curlDeleteCmd --
 *
 *	This procedure is invoked when curl handle is deleted.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Cleans the curl handle and frees the memory.
 *
 *----------------------------------------------------------------------
 */
int
curlDeleteCmd(ClientData clientData) {
    struct curlObjData     *curlData=(struct curlObjData *)clientData;
    CURL                   *curlHandle=curlData->curl;

    curlFreeSpace(curlHandle,curlData);

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * curlPerform --
 *
 *	Invokes the libcurl function 'curl_easy_perform'
 *
 *  Parameter:
 *      interp: Pointer to the interpreter we are using.
 *      curlHandle: the curl handle for which the option is set.
 *      objc and objv: The usual in Tcl.
 *
 * Results:
 *  '0' all went well.
 *  'non-zero' in case of error.
 *----------------------------------------------------------------------
 */
int
curlPerform(Tcl_Interp *interp,CURL *curlHandle,
            struct curlObjData *curlData) {
    FILE        *outFile;
    FILE        *inFile;
    FILE        *headerFile;
    FILE        *stderrFile;

    Tcl_Obj     *bodyVarObjPtr;
    Tcl_Obj     *bodyVarNameObjPtr;

    Tcl_Obj     *errorMsgObjPtr;
    char        errorMsg[300];

    int         exitCode;
    Tcl_Obj     *resultPtr;

    if (curlData->outFile!=NULL) {
        if (curlData->transferText==1) {
            outFile=fopen(curlData->outFile,"w");
        } else {
            outFile=fopen(curlData->outFile,"wb");
        }
        if (outFile==NULL) {
            snprintf(errorMsg,300,"Couldn't open file %s for writing.",
                    curlData->outFile);
            errorMsgObjPtr=Tcl_NewStringObj(errorMsg,-1);
            Tcl_SetObjResult(interp,errorMsgObjPtr);
            return TCL_ERROR;
        }
        if (curl_easy_setopt(curlHandle,CURLOPT_FILE,outFile)) {
            curlErrorSetOpt(interp,1,curlData->outFile);
            fclose(outFile);   
            return TCL_ERROR;
        }
    }
    if (curlData->headerFile!=NULL) {
        headerFile=fopen(curlData->headerFile,"w");
        if (headerFile==NULL) {
            snprintf(errorMsg,300,"Couldn't open file %s for writing.",
                    curlData->headerFile);
            errorMsgObjPtr=Tcl_NewStringObj(errorMsg,-1);
            Tcl_SetObjResult(interp,errorMsgObjPtr);
            return TCL_ERROR;
        }
        if (curl_easy_setopt(curlHandle,CURLOPT_WRITEHEADER,headerFile)) {
            curlErrorSetOpt(interp,1,curlData->headerFile);
            fclose(headerFile);
            return TCL_ERROR;
        }
    }
    if (curlData->inFile!=NULL) {
        if (!(strcmp(curlData->inFile,""))) {
            curl_easy_setopt(curlHandle,CURLOPT_INFILE,NULL);
        } else {
            if (curlData->transferText==1) {
                inFile=fopen(curlData->inFile,"r");
            } else {
                inFile=fopen(curlData->inFile,"rb");
            }
            if (inFile==NULL) {
                snprintf(errorMsg,300,"Couldn't open file %s for reading.",
                        curlData->inFile);
                errorMsgObjPtr=Tcl_NewStringObj(errorMsg,-1);
                Tcl_SetObjResult(interp,errorMsgObjPtr);
                return TCL_ERROR;
            }
            if (curl_easy_setopt(curlHandle,CURLOPT_INFILE,inFile)) {
                curlErrorSetOpt(interp,2,curlData->inFile);
                fclose(inFile);
                return TCL_ERROR;
            }
        }
    }
    if (curlData->stderrFile!=NULL) {
        if((!(strcmp(curlData->stderrFile,"stderr")))
                ||(!(strcmp(curlData->stderrFile,"")))) {
            if (curl_easy_setopt(curlHandle,CURLOPT_STDERR,stderr)) {
                curlErrorSetOpt(interp,48,curlData->stderrFile);
                return TCL_ERROR;
            }
            ckfree(curlData->stderrFile);
            curlData->stderrFile=NULL;
        } else {            
            if(!(strcmp(curlData->stderrFile,"stdout"))) {
                if (curl_easy_setopt(curlHandle,CURLOPT_STDERR,stdout)) {
                    curlErrorSetOpt(interp,48,curlData->stderrFile);
                    return TCL_ERROR;
                }
                ckfree(curlData->stderrFile);
                curlData->stderrFile=NULL;
            } else {
                stderrFile=fopen(curlData->stderrFile,"w");
                if (stderrFile==NULL) {
                    snprintf(errorMsg,300,"Couldn't open file %s for writing",
                            curlData->stderrFile);
                    errorMsgObjPtr=Tcl_NewStringObj(errorMsg,-1);
                    Tcl_SetObjResult(interp,errorMsgObjPtr);
                    return TCL_ERROR;
                }
                if (curl_easy_setopt(curlHandle,CURLOPT_STDERR,stderrFile)) {
                    curlErrorSetOpt(interp,48,curlData->stderrFile);
                    fclose(stderrFile);
                    return TCL_ERROR;
                }
            }
        }
    }
    if (curlData->postListFirst!=NULL) {
        if (curl_easy_setopt(curlHandle,CURLOPT_HTTPPOST,curlData->postListFirst)) {
            curl_formfree(curlData->postListFirst);
            errorMsgObjPtr=Tcl_NewStringObj("Error setting the data to post",-1);
            Tcl_SetObjResult(interp,errorMsgObjPtr);
            return TCL_ERROR;
        }
    }

    exitCode=curl_easy_perform(curlHandle);
    resultPtr=Tcl_NewIntObj(exitCode);
    Tcl_SetObjResult(interp,resultPtr);

    if (curlData->outFile!=NULL) {
        fclose(outFile);
    }
    if (curlData->inFile!=NULL) {
        fclose(inFile);
    }
    if (curlData->headerFile!=NULL) {
        fclose(headerFile);
    }
    if (curlData->stderrFile!=NULL) {
        fclose(stderrFile);
    }
    if (curlData->bodyVarName) {
        bodyVarNameObjPtr=Tcl_NewStringObj(curlData->bodyVarName,-1);
        bodyVarObjPtr=Tcl_NewByteArrayObj((unsigned char *)curlData->bodyVar.memory,
                curlData->bodyVar.size);
        Tcl_ObjSetVar2(interp,bodyVarNameObjPtr,(Tcl_Obj *)NULL,bodyVarObjPtr,0);
        curlData->bodyVar.size=0;
        ckfree(curlData->bodyVarName);
        curlData->bodyVarName=NULL;
    }
    if (curlData->postListFirst) {
        curl_formfree(curlData->postListFirst);
        curlData->postListFirst=NULL;
        curlData->postListLast=NULL;
        curl_easy_setopt(curlHandle,CURLOPT_HTTPPOST,NULL);
    }
    return exitCode;
}

/*
 *----------------------------------------------------------------------
 *
 * curlSetOptsTransfer --
 *
 *	This procedure is invoked when the user invokes the 'setopt'
 *  command, it is used to set the 'curl' options 
 *
 *  Parameter:
 *      interp: Pointer to the interpreter we are using.
 *      curlHandle: the curl handle for which the option is set.
 *      objc and objv: The usual in Tcl.
 *
 * Results:
 *  A standard Tcl result.
 *----------------------------------------------------------------------
 */
int
curlSetOptsTransfer(Tcl_Interp *interp, struct curlObjData *curlData,
        int objc, Tcl_Obj *CONST objv[]) {

    int            tableIndex;

    if (Tcl_GetIndexFromObj(interp, objv[2], optionTable, "option", 
            TCL_EXACT, &tableIndex)==TCL_ERROR) {
        return TCL_ERROR;
    }
    return  curlSetOpts(interp,curlData,objv[3],tableIndex);
}

/*
 *----------------------------------------------------------------------
 *
 * curlConfigTransfer --
 *
 *  This procedure is invoked by the user command 'configure', it reads 
 *  the options passed by the user to configure a transfer, and passes
 *  then, one by one to 'curlSetOpts'.
 *
 *  Parameter:
 *      interp: Pointer to the interpreter we are using.
 *      curlHandle: the curl handle for which the option is set.
 *      objc and objv: The usual in Tcl.
 *
 * Results:
 *  A standard Tcl result.
 *----------------------------------------------------------------------
 */
int
curlConfigTransfer(Tcl_Interp *interp, struct curlObjData *curlData,
        int objc, Tcl_Obj *CONST objv[]) {

    int            tableIndex;
    int            i,j;

    Tcl_Obj     *resultPtr;
    char        errorMsg[500];

    for(i=2,j=3;i<objc;i=i+2,j=j+2) {
        if (Tcl_GetIndexFromObj(interp, objv[i], configTable, "option", 
                TCL_EXACT, &tableIndex)==TCL_ERROR) {
            return TCL_ERROR;
        }
        if (i==objc-1) {
            snprintf(errorMsg,500,"Empty value for %s",configTable[tableIndex]);
            resultPtr=Tcl_NewStringObj(errorMsg,-1);
            Tcl_SetObjResult(interp,resultPtr);            
            return TCL_ERROR;
        }
        if (curlSetOpts(interp,curlData,objv[j],tableIndex)==TCL_ERROR) {
            return TCL_ERROR;
        }
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * curlSetOpts --
 *
 *	This procedure takes care of setting the transfer options.
 *
 *  Parameter:
 *      interp: Pointer to the interpreter we are using.
 *      curlHandle: the curl handle for which the option is set.
 *      objv: A pointer to the object where the data to set is stored.
 *      tableIndex: The index of the option in the options table.
 *
 * Results:
 *  A standard Tcl result.
 *----------------------------------------------------------------------
 */
int
curlSetOpts(Tcl_Interp *interp, struct curlObjData *curlData,
        Tcl_Obj *CONST objv,int tableIndex) {

    CURLcode       exitCode;
    CURL           *curlHandle=curlData->curl;
    int            i,j,k;

    Tcl_Obj        *resultObjPtr;

    Tcl_RegExp     regExp;
    char           *startPtr;
    char           *endPtr;

    int            charLength;
    long           longNumber;
    int            intNumber;
    char           *tmpStr;

    Tcl_Obj           **httpPostData;
    int               curlFormIndex,postFile,postContentHeader,formaddError,formArrayIndex;
    struct curl_forms *formArray;
    char              *postName,*postContents,*postContentType;

    switch(tableIndex) {
        case 0:
            if (SetoptChar(interp,&curlData->URL,curlHandle,CURLOPT_URL,
                    tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 1:
            ckfree(curlData->outFile);
            curlData->outFile=strdup(Tcl_GetString(objv));
            if (!strcmp(curlData->outFile,"")) {
                ckfree(curlData->outFile);
                curl_easy_setopt(curlHandle,CURLOPT_FILE,stdout);
                curlData->outFile=NULL;
            }
            break;
        case 2:
            ckfree(curlData->inFile);
            curlData->inFile=strdup(Tcl_GetString(objv));
            if (!strcmp(curlData->inFile,"")) {
                ckfree(curlData->inFile);
                curl_easy_setopt(curlHandle,CURLOPT_INFILE,stdin);
                curlData->inFile=NULL;
            }
            break;
        case 3:
            if (SetoptChar(interp,&curlData->useragent,curlHandle,
                    CURLOPT_USERAGENT,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 4:
            if (SetoptChar(interp,&curlData->referer,curlHandle,
                    CURLOPT_REFERER,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 5:
            if (SetoptInt(interp,curlHandle,CURLOPT_VERBOSE,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 6:
            if (SetoptInt(interp,curlHandle,CURLOPT_HEADER,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 7:
            if (SetoptInt(interp,curlHandle,CURLOPT_NOBODY,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 8:
            if (SetoptChar(interp,&curlData->proxy,curlHandle,
                    CURLOPT_PROXY,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 9:
            if (SetoptLong(interp,curlHandle,CURLOPT_PROXYPORT,tableIndex,
                        objv)) {
                return TCL_ERROR;
            }
            break;
        case 10:
            if (SetoptInt(interp,curlHandle,CURLOPT_HTTPPROXYTUNNEL,tableIndex,
                    objv)) {
                return TCL_ERROR;
            }
            break;
        case 11:
            if (SetoptInt(interp,curlHandle,CURLOPT_FAILONERROR,tableIndex,
                    objv)) {
                return TCL_ERROR;
            }
            break;
        case 12:
            if (SetoptLong(interp,curlHandle,CURLOPT_TIMEOUT,tableIndex,
                        objv)) {
                return TCL_ERROR;
            }
            break;
        case 13:
            if (SetoptLong(interp,curlHandle,CURLOPT_LOW_SPEED_LIMIT,tableIndex,
                        objv)) {
                return TCL_ERROR;
            }
            break;
        case 14:
            if (SetoptLong(interp,curlHandle,CURLOPT_LOW_SPEED_TIME,tableIndex,
                        objv)) {
                return TCL_ERROR;
            }
            break;
        case 15:
            if (SetoptLong(interp,curlHandle,CURLOPT_RESUME_FROM,tableIndex,
                        objv)) {
                return TCL_ERROR;
            }
            break;
        case 16:
            if (SetoptLong(interp,curlHandle,CURLOPT_INFILESIZE,tableIndex,
                        objv)) {
                return TCL_ERROR;
            }
            break;
        case 17:
            if (SetoptInt(interp,curlHandle,CURLOPT_UPLOAD,tableIndex,
                    objv)) {
                return TCL_ERROR;
            }
            break;
        case 18:
            if (SetoptInt(interp,curlHandle,CURLOPT_FTPLISTONLY,tableIndex,
                    objv)) {
                return TCL_ERROR;
            }
            break;
        case 19:
            if (SetoptInt(interp,curlHandle,CURLOPT_FTPAPPEND,tableIndex,
                    objv)) {
                return TCL_ERROR;
            }
            break;
        case 20:
            if (SetoptInt(interp,curlHandle,CURLOPT_NETRC,tableIndex,
                    objv)) {
                return TCL_ERROR;
            }
            break;
        case 21:
            if (SetoptInt(interp,curlHandle,CURLOPT_FOLLOWLOCATION,tableIndex,
                    objv)) {
                return TCL_ERROR;
            }
            break;
        case 22:
            if (SetoptInt(interp,curlHandle,CURLOPT_TRANSFERTEXT,tableIndex,
                    objv)) {
                return TCL_ERROR;
            }
            Tcl_GetIntFromObj(interp,objv,&curlData->transferText);
            break;
        case 23:
            if (SetoptInt(interp,curlHandle,CURLOPT_PUT,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 24: /* The CURLOPT_MUTE option no longer does anything.*/
            break;
        case 25:
            if (SetoptChar(interp,&curlData->userpwd,curlHandle,
                    CURLOPT_USERPWD,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 26:
            if (SetoptChar(interp,&curlData->proxyuserpwd,curlHandle,
                    CURLOPT_PROXYUSERPWD,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 27:
            if (SetoptChar(interp,&curlData->range,curlHandle,
                    CURLOPT_RANGE,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 28:
            tmpStr=strdup(Tcl_GetString(objv));
            regExp=Tcl_RegExpCompile(interp,"(.*)(?:\\()(.*)(?:\\))");
            exitCode=Tcl_RegExpExec(interp,regExp,tmpStr,tmpStr);
            switch(exitCode) {
                case -1:
                    ckfree((char *)tmpStr);
                    return TCL_ERROR;
                    break;
                case 0:
                    if (*tmpStr!=0) {
                        curlData->errorBufferName=strdup(tmpStr);
                    } else {
                        curlData->errorBuffer=NULL;
                    }
                    curlData->errorBufferKey=NULL;
                    break;
                case 1:
                    Tcl_RegExpRange(regExp,1,&startPtr,&endPtr);
                    charLength=endPtr-startPtr;
                    curlData->errorBufferName=ckalloc(charLength+1);
                    strncpy(curlData->errorBufferName,startPtr,charLength);
                    curlData->errorBufferName[charLength]=0;
                    Tcl_RegExpRange(regExp,2,&startPtr,&endPtr);
                    charLength=endPtr-startPtr;
                    curlData->errorBufferKey=ckalloc(charLength+1);
                    strncpy(curlData->errorBufferKey,startPtr,charLength);
                    curlData->errorBufferKey[charLength]=0;
                    break;
            }
            ckfree((char *)tmpStr);
            if (curlData->errorBufferName!=NULL) {
                curlData->errorBuffer=ckalloc(CURL_ERROR_SIZE);
                if (curl_easy_setopt(curlHandle,CURLOPT_ERRORBUFFER,
                        curlData->errorBuffer)) {
                    ckfree((char *)curlData->errorBuffer);
                    curlData->errorBuffer=NULL;
                    return TCL_ERROR;
                }
            } else {
                ckfree(curlData->errorBuffer);
            }
            break;
        case 29:
            if (SetoptLong(interp,curlHandle,CURLOPT_HTTPGET,tableIndex,
                        objv)) {
                return TCL_ERROR;
            }
            break;
        case 30:
            if (SetoptInt(interp,curlHandle,CURLOPT_POST,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 31:
            if (SetoptChar(interp,&curlData->postFields,curlHandle,
                    CURLOPT_POSTFIELDS,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 33:
            if (SetoptChar(interp,&curlData->ftpport,curlHandle,
                    CURLOPT_FTPPORT,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 34:
            if (SetoptChar(interp,&curlData->cookie,curlHandle,
                    CURLOPT_COOKIE,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 35:
            if (SetoptChar(interp,&curlData->cookiefile,curlHandle,
                    CURLOPT_COOKIEFILE,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 36:
            if(SetoptsList(interp,&curlData->headerList,objv)) {
                curlErrorSetOpt(interp,tableIndex,"Header list invalid");
                return TCL_ERROR;
            }
            if (curl_easy_setopt(curlHandle,CURLOPT_HTTPHEADER,curlData->headerList)) {
                curl_slist_free_all(curlData->headerList);
                return TCL_ERROR;
            }
            return TCL_OK;
            break;
        case 37:
            if (Tcl_ListObjGetElements(interp,objv,&k,&httpPostData)
                    ==TCL_ERROR) {
                return TCL_ERROR;
            }
            postName=NULL;
            postContents=NULL;
            postContentType=NULL;
            postFile=0;
            postContentHeader;
            formaddError=0;
            formArray=(struct curl_forms *)malloc(k*(sizeof(struct curl_forms)));
            formArrayIndex=0;
            for(i=0,j=0;i<k;i+=2,j+=1) {
                if (Tcl_GetIndexFromObj(interp,httpPostData[i],curlFormTable,
                        "CURLFORM option",TCL_EXACT,&curlFormIndex)==TCL_ERROR) {
                    return TCL_ERROR;
                }
                switch(curlFormIndex) {
                    case 0:
//                        fprintf(stdout,"Section name: %s\n",Tcl_GetString(httpPostData[i+1]));
                        postName=strdup(Tcl_GetString(httpPostData[i+1]));
                        break;
                    case 1:
//                        fprintf(stdout,"Section contents: %s\n",Tcl_GetString(httpPostData[i+1]));
                        postContents=strdup(Tcl_GetString(httpPostData[i+1]));
                        break;
                    case 2:
                        postFile=1;
//                        fprintf(stdout,"File name %d: %s\n",formArrayIndex,Tcl_GetString(httpPostData[i+1]));
                        formArray[formArrayIndex].option=CURLFORM_FILE;
                        formArray[formArrayIndex].value=strdup(Tcl_GetString(httpPostData[i+1]));
                        formArrayIndex++;
                        break;
                    case 3:
//                        fprintf(stdout,"Data type: %s\n",Tcl_GetString(httpPostData[i+1]));
                        postContentType=strdup(Tcl_GetString(httpPostData[i+1]));
                        break;
                    case 4:
//                        fprintf(stdout,"ContentHeader: %s\n",Tcl_GetString(httpPostData[i+1]));
                        postContentHeader=1;
                        if(SetoptsList(interp,&curlData->formHeaderList,httpPostData[i+1])) {
                            curlErrorSetOpt(interp,tableIndex,"Header list invalid");
                            return TCL_ERROR;
                        }
                        break;
                }
            }
            if (postName==NULL) {
                resultObjPtr=Tcl_NewStringObj("No section name",-1);
                Tcl_SetObjResult(interp,resultObjPtr);
                formaddError=1;
            } else {
                if (postFile==0) {
                    if (postContentType==NULL) {
                        if (postContentHeader==0) {
                            if (curl_formadd(&(curlData->postListFirst), &(curlData->postListLast)
	                                , CURLFORM_COPYNAME, postName, CURLFORM_COPYCONTENTS, postContents
                                    , CURLFORM_END)) {
                                formaddError=1;
                            }
                        } else {
                            if (curl_formadd(&(curlData->postListFirst), &(curlData->postListLast)
	                                , CURLFORM_COPYNAME, postName, CURLFORM_COPYCONTENTS, postContents
                                    , CURLFORM_CONTENTHEADER, curlData->formHeaderList, CURLFORM_END)) {
                                formaddError=1;
                            }
                        }
                    } else {
                        if (postContentHeader==0) {
                            if (curl_formadd(&(curlData->postListFirst), &(curlData->postListLast)
	                                , CURLFORM_COPYNAME, postName, CURLFORM_COPYCONTENTS, postContents
                                    , CURLFORM_CONTENTTYPE, postContentType, CURLFORM_END)) {
                                formaddError=1;
                            }
                        } else {
                            if (curl_formadd(&(curlData->postListFirst), &(curlData->postListLast)
	                                , CURLFORM_COPYNAME, postName, CURLFORM_COPYCONTENTS, postContents
                                    , CURLFORM_CONTENTTYPE, postContentType
                                    , CURLFORM_CONTENTHEADER, curlData->formHeaderList, CURLFORM_END)) {
                                formaddError=1;
                            }
                        }
                    }
                } else {
                    formArray[formArrayIndex].option=CURLFORM_END;
                    if (postContentType==NULL) {
                        if (postContentHeader==0) {
                            if (curl_formadd(&(curlData->postListFirst), &(curlData->postListLast)
                                , CURLFORM_COPYNAME, "filesToPost", CURLFORM_ARRAY, formArray
                                , CURLFORM_END)) {
                                formaddError=1;
                            }
                        } else {
                            if (curl_formadd(&(curlData->postListFirst), &(curlData->postListLast)
                                , CURLFORM_COPYNAME, "filesToPost", CURLFORM_ARRAY, formArray
                                , CURLFORM_CONTENTHEADER, curlData->formHeaderList, CURLFORM_END)) {
                                formaddError=1;
                            }
                        }
                    } else {
                        if (postContentHeader==0) {
                            if (curl_formadd(&(curlData->postListFirst), &(curlData->postListLast)
                                , CURLFORM_COPYNAME, "filesToPost", CURLFORM_ARRAY, formArray
                                , CURLFORM_CONTENTTYPE, postContentType, CURLFORM_END)) {
                                formaddError=1;
                            }
                        } else {
                            if (curl_formadd(&(curlData->postListFirst), &(curlData->postListLast)
                                , CURLFORM_COPYNAME, "filesToPost", CURLFORM_ARRAY, formArray
                                , CURLFORM_CONTENTTYPE, postContentType
                                , CURLFORM_CONTENTHEADER, curlData->formHeaderList, CURLFORM_END)) {
                                formaddError=1;
                            }
                        }
                    }
                }
                if (formaddError==1) {
                    resultObjPtr=Tcl_NewStringObj("Couldn't set data to post",-1);
                    Tcl_SetObjResult(interp,resultObjPtr);
                }
            }
            free(postName);
            free(postContents);
            free(postContentType);
            for (i=0;i<formArrayIndex-1;i++) {
                free(formArray[i].value);
            }
            free((char *)formArray);
 
            if (formaddError==1) {
                return TCL_ERROR;
            }
            return TCL_OK;
            break;
        case 38:
            if (SetoptChar(interp,&curlData->sslcert,curlHandle,
                    CURLOPT_SSLCERT,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 39:
            if (SetoptChar(interp,&curlData->sslcertpasswd,curlHandle,
                    CURLOPT_SSLCERTPASSWD,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 40:
            if (SetoptLong(interp,curlHandle,CURLOPT_SSLVERSION,tableIndex,
                        objv)) {
                return TCL_ERROR;
            }
            break;
        case 41:
            if (SetoptInt(interp,curlHandle,CURLOPT_CRLF,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 42:
            if(SetoptsList(interp,&curlData->quote,objv)) {
                curlErrorSetOpt(interp,tableIndex,"quote list invalid");
                return TCL_ERROR;
            }
            if (curl_easy_setopt(curlHandle,CURLOPT_QUOTE,curlData->quote)) {
                curl_slist_free_all(curlData->quote);
                return TCL_ERROR;
            }
            return TCL_OK;
            break;
        case 43:
            if(SetoptsList(interp,&curlData->postquote,objv)) {
                curlErrorSetOpt(interp,tableIndex,"postqoute invalid");
                return TCL_ERROR;
            }
            if (curl_easy_setopt(curlHandle,CURLOPT_POSTQUOTE,curlData->postquote)) {
                curlErrorSetOpt(interp,tableIndex,"postqoute invalid");
                curl_slist_free_all(curlData->postquote);
                return TCL_ERROR;
            }
            return TCL_OK;
            break;
        case 44:
            if (SetoptChar(interp,&curlData->headerFile,curlHandle,
                    CURLOPT_WRITEHEADER,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 45:
            if (Tcl_GetIndexFromObj(interp, objv, timeCond,
                "time cond option",TCL_EXACT, &intNumber)==TCL_ERROR) {
                return TCL_ERROR;
            }
            if (intNumber==0) {
                longNumber=TIMECOND_IFMODSINCE;
            } else {
                longNumber=TIMECOND_IFUNMODSINCE;
            }
            if (curl_easy_setopt(curlHandle,CURLOPT_TIMECONDITION,longNumber)) {
                return TCL_ERROR;
            }
            break;
        case 46:
            if (SetoptLong(interp,curlHandle,CURLOPT_TIMEVALUE,tableIndex,
                        objv)) {
                return TCL_ERROR;
            }
            break;
        case 47:
            if (SetoptChar(interp,&curlData->customrequest,curlHandle,
                    CURLOPT_CUSTOMREQUEST,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 48:
            ckfree(curlData->stderrFile);
            curlData->stderrFile=strdup(Tcl_GetString(objv));
            break;
        case 49:
            if (SetoptChar(interp,&curlData->interface,curlHandle,
                    CURLOPT_INTERFACE,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 50:
            if (SetoptChar(interp,&curlData->krb4level,curlHandle,
                    CURLOPT_KRB4LEVEL,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 51:
            if (SetoptLong(interp,curlHandle,CURLOPT_SSL_VERIFYPEER,tableIndex,
                        objv)) {
                return TCL_ERROR;
            }
            break;
        case 52:
            if (SetoptChar(interp,&curlData->cainfo,curlHandle,
                    CURLOPT_CAINFO,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 53:
            if (SetoptLong(interp,curlHandle,CURLOPT_FILETIME,tableIndex,
                        objv)) {
                return TCL_ERROR;
            }
            break;
        case 54:
            if (SetoptLong(interp,curlHandle,CURLOPT_MAXREDIRS,tableIndex,
                        objv)) {
                return TCL_ERROR;
            }
            break;
        case 55:
            if (SetoptLong(interp,curlHandle,CURLOPT_MAXCONNECTS,tableIndex,
                        objv)) {
                return TCL_ERROR;
            }
            break;
        case 56:
            if (Tcl_GetIndexFromObj(interp, objv, closePolicy,
                    "close policy option",TCL_EXACT, &intNumber)==TCL_ERROR) {
                return TCL_ERROR;
            }
            switch(intNumber) {
                case 1: longNumber=CURLCLOSEPOLICY_OLDEST;
                        break;
                case 2: longNumber=CURLCLOSEPOLICY_LEAST_RECENTLY_USED;
                        break;
                case 3: longNumber=CURLCLOSEPOLICY_LEAST_TRAFFIC;
                        break;
                case 4: longNumber=CURLCLOSEPOLICY_SLOWEST;
                        break;
            }
            if (curl_easy_setopt(curlHandle,CURLOPT_CLOSEPOLICY,longNumber)) {
                return TCL_ERROR;
            }
            break;
        case 57:
            if (SetoptChar(interp,&curlData->randomFile,curlHandle,
                    CURLOPT_RANDOM_FILE,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 58:
            if (SetoptChar(interp,&curlData->egdsocket,curlHandle,
                    CURLOPT_EGDSOCKET,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 59:
            if (SetoptLong(interp,curlHandle,CURLOPT_CONNECTTIMEOUT,
                        tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 60:
            if (SetoptLong(interp,curlHandle,CURLOPT_NOPROGRESS,
                        tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 61:
            if (curl_easy_setopt(curlHandle,CURLOPT_HEADERFUNCTION,
                    curlHeaderReader)) {
                return TCL_ERROR;
            }
            ckfree(curlData->headerVar);
            curlData->headerVar=strdup(Tcl_GetString(objv));
            if (curl_easy_setopt(curlHandle,CURLOPT_WRITEHEADER,
                    (FILE *)curlData)) {
                return TCL_ERROR;
            }
            break;
        case 62:
            ckfree(curlData->bodyVarName);
            curlData->bodyVarName=strdup(Tcl_GetString(objv));
            if (curl_easy_setopt(curlHandle,CURLOPT_WRITEFUNCTION,
                    curlBodyReader)) {
                return TCL_ERROR;
            }
            if (curl_easy_setopt(curlHandle,CURLOPT_FILE,curlData)) {
                return TCL_ERROR;
            }
            break;
        case 63:
            ckfree(curlData->progressProc);
            curlData->progressProc=strdup(Tcl_GetString(objv));
            if (curl_easy_setopt(curlHandle,CURLOPT_PROGRESSFUNCTION,
                    curlProgressCallback)) {
                return TCL_ERROR;
            }
            if (curl_easy_setopt(curlHandle,CURLOPT_PROGRESSDATA,
                    curlData)) {
                return TCL_ERROR;
            }
            break;
        case 64:
            if (curlData->cancelTransVarName) {
                Tcl_UnlinkVar(curlData->interp,curlData->cancelTransVarName);
                ckfree(curlData->cancelTransVarName);
            }
            curlData->cancelTransVarName=strdup(Tcl_GetString(objv));
            Tcl_LinkVar(interp,curlData->cancelTransVarName,
                    (char *)&(curlData->cancelTrans),TCL_LINK_INT);
            break;
        case 65:
            curlData->writeProc=strdup(Tcl_GetString(objv));
            if (curl_easy_setopt(curlHandle,CURLOPT_WRITEFUNCTION,
                    curlWriteProcInvoke)) {    
                return TCL_ERROR;
            }
            if (curl_easy_setopt(curlHandle,CURLOPT_FILE,curlData)) {
                return TCL_ERROR;
            }
            break;
        case 66:
            curlData->readProc=strdup(Tcl_GetString(objv));
            if (curl_easy_setopt(curlHandle,CURLOPT_READFUNCTION,
                    curlReadProcInvoke)) {    
                return TCL_ERROR;
            }
            if (curl_easy_setopt(curlHandle,CURLOPT_INFILE,curlData)) {
                return TCL_ERROR;
            }
            break;
        case 67:
            if (SetoptLong(interp,curlHandle,CURLOPT_SSL_VERIFYHOST,
                        tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 68:
            if (SetoptChar(interp,&curlData->cookiejar,curlHandle,
                    CURLOPT_COOKIEJAR,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 69:
            if (SetoptChar(interp,&curlData->sslcipherlist,curlHandle,
                    CURLOPT_SSL_CIPHER_LIST,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 70:
            if (Tcl_GetIndexFromObj(interp, objv, httpVersionTable,
                "http version",TCL_EXACT,&tableIndex)==TCL_ERROR) {
                return TCL_ERROR;
            }
            if (curl_easy_setopt(curlHandle,CURLOPT_HTTP_VERSION,
                        tableIndex)) {
                tmpStr=strdup(Tcl_GetString(objv));
                curlErrorSetOpt(interp,70,tmpStr);
                free(tmpStr);
                return TCL_ERROR;
            }
            break;
        case 71:
            if (SetoptLong(interp,curlHandle,CURLOPT_FTP_USE_EPSV,
                        tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 72:
            if (SetoptChar(interp,&curlData->sslcerttype,curlHandle,
                    CURLOPT_SSLCERTTYPE,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 73:
            if (SetoptChar(interp,&curlData->sslkey,curlHandle,
                    CURLOPT_SSLKEY,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 74:
            if (SetoptChar(interp,&curlData->sslkeytype,curlHandle,
                    CURLOPT_SSLKEYTYPE,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 75:
            if (SetoptChar(interp,&curlData->sslkeypasswd,curlHandle,
                    CURLOPT_SSLKEYPASSWD,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 76:
            if (SetoptChar(interp,&curlData->sslengine,curlHandle,
                    CURLOPT_SSLENGINE,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 77:
            if (SetoptChar(interp,&curlData->sslenginedefault,curlHandle,
                    CURLOPT_SSLENGINE_DEFAULT,tableIndex,objv)) {
                return TCL_ERROR;
            }
            break;
        case 78:
            if(SetoptsList(interp,&curlData->prequote,objv)) {
                curlErrorSetOpt(interp,tableIndex,"pretqoute invalid");
                return TCL_ERROR;
            }
            if (curl_easy_setopt(curlHandle,CURLOPT_PREQUOTE,curlData->prequote)) {
                curlErrorSetOpt(interp,tableIndex,"preqoute invalid");
                curl_slist_free_all(curlData->prequote);
                return TCL_ERROR;
            }
            return TCL_OK;
            break;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * curlSetoptint --
 *
 *	Set the curl options that require an int
 *
 *  Parameter:
 *      interp: The interpreter we are working with.
 *      curlHandle: and the curl handle
 *      opt: the option to set
 *      tclObj: The Tcl with the value for the option.
 *
 * Results:
 *  0 if all went well.
 *  1 in case of error.
 *----------------------------------------------------------------------
 */
int
SetoptInt(Tcl_Interp *interp,CURL *curlHandle,CURLoption opt,
        int tableIndex,Tcl_Obj *tclObj) {
    int        intNumber;
    char       *parPtr;

    if (Tcl_GetIntFromObj(interp,tclObj,&intNumber)) {
        parPtr=strdup(Tcl_GetString(tclObj));
        curlErrorSetOpt(interp,tableIndex,parPtr);
        free(parPtr);
        return 1;
    }
    if (curl_easy_setopt(curlHandle,opt,intNumber)) {
        parPtr=strdup(Tcl_GetString(tclObj));
        curlErrorSetOpt(interp,tableIndex,parPtr);
        free(parPtr);
        return 1;
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * curlSetoptLong --
 *
 *	Set the curl options that require a long
 *
 *  Parameter:
 *      interp: The interpreter we are working with.
 *      curlHandle: and the curl handle
 *      opt: the option to set
 *      tclObj: The Tcl with the value for the option.
 *
 * Results:
 *  0 if all went well.
 *  1 in case of error.
 *----------------------------------------------------------------------
 */
int
SetoptLong(Tcl_Interp *interp,CURL *curlHandle,CURLoption opt,
        int tableIndex,Tcl_Obj *tclObj) {
    long        longNumber;
    char        *parPtr;

    if (Tcl_GetLongFromObj(interp,tclObj,&longNumber)) {
        parPtr=strdup(Tcl_GetString(tclObj));
        curlErrorSetOpt(interp,tableIndex,parPtr);
        free(parPtr);
        return 1;
    }
    if (curl_easy_setopt(curlHandle,opt,longNumber)) {
        parPtr=strdup(Tcl_GetString(tclObj));
        curlErrorSetOpt(interp,tableIndex,parPtr);
        free(parPtr);
        return 1;
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * curlSetoptChar --
 *
 *	Set the curl options that require a string
 *
 *  Parameter:
 *      interp: The interpreter we are working with.
 *      charPtr: Pointer in the curlData structure to keep the value.
 *      curlHandle: and the curl handle
 *      opt: the option to set
 *      tclObj: The Tcl with the value for the option.
 *
 * Results:
 *  0 if all went well.
 *  1 in case of error.
 *----------------------------------------------------------------------
 */
int
SetoptChar(Tcl_Interp *interp,char **charPtr,CURL *curlHandle,
        CURLoption opt,int tableIndex,Tcl_Obj *tclObj) {

    ckfree(*charPtr);
    *charPtr=strdup(Tcl_GetString(tclObj));

    if (!(strcmp(*charPtr,""))) {
        ckfree(*charPtr);
        *charPtr=NULL;
    }
    if (curl_easy_setopt(curlHandle,opt,*charPtr)) {
        curlErrorSetOpt(interp,tableIndex,*charPtr);
        ckfree(*charPtr);
        *charPtr=NULL;
        return 1;
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * SetoptsList --
 *
 *	Prepares a slist for future use.
 *
 *  Parameter:
 *      slistPtr: Pointer to the slist to prepare.
 *      objv: Tcl object with a list of the data.
 *
 * Results:
 *  0 if all went well.
 *  1 in case of error.
 *----------------------------------------------------------------------
 */
int
SetoptsList(Tcl_Interp *interp,struct curl_slist **slistPtr,
        Tcl_Obj *CONST objv) {
    int         i,headerNumber;
    Tcl_Obj     **headers;

    if (slistPtr!=NULL) {
        curl_slist_free_all(*slistPtr);
    }
    if (Tcl_ListObjGetElements(interp,objv,&headerNumber,&headers)
            ==TCL_ERROR) {
        return 1;
    }
    for (i=0;i<headerNumber;i++) {
        *slistPtr=curl_slist_append(*slistPtr,Tcl_GetString(headers[i]));
        if (slistPtr==NULL) {
            curl_slist_free_all(*slistPtr);
            return 1;
        }
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * curlErrorSetOpt --
 *
 *	When an error happens when setting an options, this function
 *  takes cares of reporting it
 *
 *  Parameter:
 *      interp: Pointer to the interpreter we are using.
 *      option: The index of the option in 'optionTable'
 *
 *----------------------------------------------------------------------
 */

void
curlErrorSetOpt(Tcl_Interp *interp, int option, char *parPtr) {
    Tcl_Obj     *resultPtr;
    char        errorMsg[500];

    snprintf(errorMsg,500,"Error setting option %s: %s",configTable[option],parPtr);
    resultPtr=Tcl_NewStringObj(errorMsg,-1);
    Tcl_SetObjResult(interp,resultPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * curlHeaderVar --
 *
 *	This is the function that will be invoked if the user wants to put
 *  the headers into a variable
 *
 *  Parameter:
 *      header: string with the header line.
 *      size and nmemb: it so happens size * nmemb if the size of the
 *      header string.
 *      curlData: A pointer to the curlData structure for the transfer.
 *
 *  Returns
 *      The number of bytes actually written or -1 in case of error, in
 *      which case 'libcurl' will abort the transfer.
 *-----------------------------------------------------------------------
 */
size_t
curlHeaderReader(void *ptr,size_t size,size_t nmemb,FILE *curlDataPtr) {

    char                *header=ptr;
    struct curlObjData  *curlData=(struct curlObjData *)curlDataPtr;
    Tcl_RegExp          regExp;

    char                *startPtr;
    char                *endPtr;

    char                *headerName;
    char                *headerContent;
    char                *httpStatus;

    int                 match,charLength;

    regExp=Tcl_RegExpCompile(curlData->interp,"(.*?)(?::\\s*)(.*)(\\n)");
    match=Tcl_RegExpExec(curlData->interp,regExp,header,header);

    if (match) {
        Tcl_RegExpRange(regExp,1,&startPtr,&endPtr);
        charLength=endPtr-startPtr;
        headerName=ckalloc(charLength+1);
        strncpy(headerName,startPtr,charLength);
        headerName[charLength]=0;

        Tcl_RegExpRange(regExp,2,&startPtr,&endPtr);
        charLength=endPtr-startPtr;
        headerContent=ckalloc(charLength+1);
        strncpy(headerContent,startPtr,charLength);
        headerContent[charLength]=0;

        Tcl_SetVar2(curlData->interp,curlData->headerVar,headerName,
                headerContent,0);
    } else {
        regExp=Tcl_RegExpCompile(curlData->interp,"(HTTP/.*)(\\n)");
        match=Tcl_RegExpExec(curlData->interp,regExp,header,header);

        if (match) {
            Tcl_RegExpRange(regExp,1,&startPtr,&endPtr);
            charLength=endPtr-startPtr;
            httpStatus=ckalloc(charLength+1);
            strncpy(httpStatus,startPtr,charLength);
            httpStatus[charLength]=0;

            Tcl_SetVar2(curlData->interp,curlData->headerVar,"http",
                httpStatus,0);
        }
    }
    return size*nmemb;
}

/*
 *----------------------------------------------------------------------
 *
 * curlBodyReader --
 *
 *	This is the function that will be invoked as a callback while 
 *  transferring the body of a request into a Tcl variable.
 * 
 *  This function has been adapted from an example in libcurl's FAQ.
 *
 *  Parameter:
 *      header: string with the header line.
 *      size and nmemb: it so happens size * nmemb if the size of the
 *      header string.
 *      curlData: A pointer to the curlData structure for the transfer.
 *
 *  Returns
 *      The number of bytes actually written or -1 in case of error, in
 *      which case 'libcurl' will abort the transfer.
 *-----------------------------------------------------------------------
 */
size_t
curlBodyReader(void *ptr,size_t size,size_t nmemb,FILE *curlDataPtr) {

    register int realsize = size * nmemb;
    struct MemoryStruct *mem=&(((struct curlObjData *)curlDataPtr)->bodyVar);

    mem->memory = (char *)Tcl_Realloc(mem->memory,mem->size + realsize);
    if (mem->memory) {
        memcpy(&(mem->memory[mem->size]), ptr, realsize);
        mem->size += realsize;
    }
    return realsize;
}

/*
 *----------------------------------------------------------------------
 *
 * curlProgressCallback --
 *
 *  This is the function that will be invoked as a callback during a  
 *  transfer.
 * 
 *  This function has been adapted from an example in libcurl's FAQ.
 *
 *  Parameter:
 *      clientData: The curlData struct for the transfer.
 *      dltotal: Total amount of bytes to download.
 *      dlnow: Bytes downloaded so far.
 *      ultotal: Total amount of bytes to upload.
 *      ulnow: Bytes uploaded so far.
 *
 *  Returns
 *      Returning a non-zero value will make 'libcurl' abort the transfer
 *      and return 'CURLE_ABORTED_BY_CALLBACK'.
 *-----------------------------------------------------------------------
 */
int
curlProgressCallback(void *clientData,size_t dltotal,size_t dlnow,
        size_t ultotal,size_t ulnow) {

    struct curlObjData    *curlData=(struct curlObjData *)clientData;
    Tcl_Obj               *tclProcPtr;
    char                  tclCommand[300];

    Tcl_DoOneEvent(TCL_ALL_EVENTS);
    snprintf(tclCommand,300,"%s %ld %ld %ld %ld",curlData->progressProc,dltotal,
            dlnow,ultotal,ulnow);
    tclProcPtr=Tcl_NewStringObj(tclCommand,-1);
    if (curlData->cancelTransVarName) {
        if (curlData->cancelTrans) {
            curlData->cancelTrans=0;
            return -1;
        }
    }
    if (Tcl_EvalObjEx(curlData->interp,tclProcPtr,TCL_EVAL_GLOBAL)!=TCL_OK) {
        return -1;
    }
    Tcl_DoOneEvent(TCL_ALL_EVENTS);
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * curlWriteProcInvoke --
 *
 *	This is the function that will be invoked as a callback when the user
 *  wants to invoke a Tcl procedure to write the recieved data.
 * 
 *  This function has been adapted from an example in libcurl's FAQ.
 *
 *  Parameter:
 *      ptr: A pointer to the data.
 *      size and nmemb: it so happens size * nmemb if the size of the
 *      data read.
 *      curlData: A pointer to the curlData structure for the transfer.
 *
 *  Returns
 *      The number of bytes actually written or -1 in case of error, in
 *      which case 'libcurl' will abort the transfer.
 *-----------------------------------------------------------------------
 */
size_t
curlWriteProcInvoke(void *ptr,size_t size,size_t nmemb,FILE *curlDataPtr) {
    register int realsize = size * nmemb;
    struct curlObjData  *curlData=(struct curlObjData *)curlDataPtr;
    Tcl_Obj             *objv[2];


    objv[0]=Tcl_NewStringObj(curlData->writeProc,-1);
    objv[1]=Tcl_NewByteArrayObj(ptr,realsize);

    if (curlData->cancelTransVarName) {
        if (curlData->cancelTrans) {
            curlData->cancelTrans=0;
            return -1;
        }
    }

    if (Tcl_EvalObjv(curlData->interp,2,objv,TCL_EVAL_GLOBAL)!=TCL_OK) {
        return -1;
    }
    return realsize;
}

/*
 *----------------------------------------------------------------------
 *
 * curlReadProcInvoke --
 *
 *	This is the function that will be invoked as a callback when the user
 *  wants to invoke a Tcl procedure to read the data to send.
 *
 *  Parameter:
 *      header: string with the header line.
 *      size and nmemb: it so happens size * nmemb if the size of the
 *      header string.
 *      curlData: A pointer to the curlData structure for the transfer.
 *
 *  Returns
 *      The number of bytes actually read or -1 in case of error, in
 *      which case 'libcurl' will abort the transfer.
 *-----------------------------------------------------------------------
 */
size_t
curlReadProcInvoke(void *ptr,size_t size,size_t nmemb,FILE *curlDataPtr) {
    register int realsize = size * nmemb;
    struct curlObjData  *curlData=(struct curlObjData *)curlDataPtr;
    Tcl_Obj             *tclProcPtr;
    Tcl_Obj             *readDataPtr;
    char                tclCommand[300];
    unsigned char       *readBytes;
    int                 sizeRead;

    snprintf(tclCommand,300,"%s %d",curlData->readProc,realsize);
    tclProcPtr=Tcl_NewStringObj(tclCommand,-1);

    if (curlData->cancelTransVarName) {
        if (curlData->cancelTrans) {
            curlData->cancelTrans=0;
            return -1;
        }
    }
    if (Tcl_EvalObjEx(curlData->interp,tclProcPtr,TCL_EVAL_GLOBAL)!=TCL_OK) {
        return -1;
    }
    readDataPtr=Tcl_GetObjResult(curlData->interp);
    readBytes=Tcl_GetByteArrayFromObj(readDataPtr,&sizeRead);
    memcpy(ptr,readBytes,sizeRead);

    return sizeRead;
}

 
/*
 *----------------------------------------------------------------------
 *
 * curlGetInfo --
 *
 *	Invokes the 'curl_easy_getinfo' function in libcurl.
 *
 * Parameter:
 *
 * Results:
 *  0 if all went well.
 *  The CURLcode for the error.
 *----------------------------------------------------------------------
 */
CURLcode
curlGetInfo(Tcl_Interp *interp,CURL *curlHandle,int tableIndex) {
    char        *charPtr;
    long        longNumber;
    double      doubleNumber;

    CURLcode    exitCode;

    Tcl_Obj    *resultObj;

    switch(tableIndex) {
        case 0:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_EFFECTIVE_URL,&charPtr);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewStringObj(charPtr,-1);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 1:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_HTTP_CODE,&longNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewLongObj(longNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 2:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_FILETIME,&longNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewLongObj(longNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 3:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_TOTAL_TIME,&doubleNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewDoubleObj(doubleNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 4:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_NAMELOOKUP_TIME,
                    &doubleNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewDoubleObj(doubleNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 5:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_CONNECT_TIME,
                    &doubleNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewDoubleObj(doubleNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 6:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_PRETRANSFER_TIME,
                    &doubleNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewDoubleObj(doubleNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 7:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_SIZE_UPLOAD,
                    &doubleNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewDoubleObj(doubleNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 8:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_SIZE_DOWNLOAD,
                    &doubleNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewDoubleObj(doubleNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 9:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_SPEED_DOWNLOAD,
                    &doubleNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewDoubleObj(doubleNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 10:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_SPEED_UPLOAD,
                    &doubleNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewDoubleObj(doubleNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 11:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_HEADER_SIZE,
                    &longNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewLongObj(longNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 12:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_REQUEST_SIZE,
                    &longNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewLongObj(longNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 13:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_SSL_VERIFYRESULT,
                    &longNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewLongObj(longNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 14:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_CONTENT_LENGTH_DOWNLOAD,
                    &doubleNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewDoubleObj(doubleNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 15:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_CONTENT_LENGTH_UPLOAD,
                    &doubleNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewDoubleObj(doubleNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 16:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_STARTTRANSFER_TIME,
                    &doubleNumber);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewDoubleObj(doubleNumber);
            Tcl_SetObjResult(interp,resultObj);
            break;
        case 17:
            exitCode=curl_easy_getinfo(curlHandle,CURLINFO_CONTENT_TYPE,\
                    &charPtr);
            if (exitCode) {
                return exitCode;
            }
            resultObj=Tcl_NewStringObj(charPtr,-1);
            Tcl_SetObjResult(interp,resultObj);
            break;
    }
    return 0;            
}

/*
 *----------------------------------------------------------------------
 *
 * curlFreeSpace --
 *
 *	Frees the space taken by a curlObjData struct.
 *
 *  Parameter:
 *      interp: Pointer to the interpreter we are using.
 *      curlHandle: the curl handle for which the option is set.
 *      objc and objv: The usual in Tcl.
 *
 * Results:
 *  A standard Tcl result.
 *----------------------------------------------------------------------
 */
void
curlFreeSpace(CURL *curlHandle,struct curlObjData *curlData) {

    curl_easy_cleanup(curlHandle);
    curl_slist_free_all(curlData->headerList);
    curl_slist_free_all(curlData->quote);
    curl_slist_free_all(curlData->prequote);
    curl_slist_free_all(curlData->postquote);
    curl_slist_free_all(curlData->formHeaderList);

    if (curlData->postListFirst!=NULL) {
        curl_formfree(curlData->postListFirst);
        curlData->postListLast=NULL;
    }
    ckfree(curlData->URL);
    ckfree(curlData->outFile);
    ckfree(curlData->inFile);
    ckfree(curlData->useragent);
    ckfree(curlData->referer);
    ckfree(curlData->proxy);
    ckfree(curlData->userpwd);
    ckfree(curlData->proxyuserpwd);
    ckfree(curlData->range);
    ckfree(curlData->errorBuffer);
    ckfree(curlData->errorBufferName);
    ckfree(curlData->errorBufferKey);
    ckfree(curlData->postFields);
    ckfree(curlData->ftpport);
    ckfree(curlData->cookie);
    ckfree(curlData->cookiefile);
    ckfree(curlData->sslcert);
    ckfree(curlData->sslcertpasswd);
    ckfree(curlData->customrequest);
    ckfree(curlData->stderrFile);
    ckfree(curlData->interface);
    ckfree(curlData->krb4level);
    ckfree(curlData->cainfo);
    ckfree(curlData->randomFile);
    ckfree(curlData->egdsocket);
    ckfree(curlData->headerVar);
    ckfree(curlData->bodyVarName);
    if (curlData->bodyVar.memory) {
        ckfree(curlData->bodyVar.memory);
    }
    ckfree(curlData->progressProc);
    if (curlData->cancelTransVarName) {
        Tcl_UnlinkVar(curlData->interp,curlData->cancelTransVarName);
        ckfree(curlData->cancelTransVarName);
    }

    ckfree(curlData->writeProc);
    ckfree(curlData->readProc);
    ckfree(curlData->cookiejar);
    ckfree(curlData->sslcipherlist);
    ckfree(curlData->sslcerttype);
    ckfree(curlData->sslkey);
    ckfree(curlData->sslkeytype);

    ckfree((char *)curlData);
}

/*
 *----------------------------------------------------------------------
 *
 * curlDupHandle --
 *
 *	This function is invoked by the 'duphandle' command, it will 
 *	create a duplicate of the given handle.
 *
 * Parameters:
 *  The stantard parameters for Tcl commands
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int
curlDupHandle(Tcl_Interp *interp, struct curlObjData *curlData,
        int objc, Tcl_Obj *CONST objv[]) {

    CURL                *newCurlHandle;
    Tcl_Obj             *result;
    struct curlObjData  *newCurlData;
    char                *handleName;

    newCurlHandle=curl_easy_duphandle(curlData->curl);
    if (newCurlHandle==NULL) {
        result=Tcl_NewStringObj("Couldn't create new handle.",-1);
        Tcl_SetObjResult(interp,result);
        return TCL_ERROR;
    }

    newCurlData=(struct curlObjData *)ckalloc(sizeof(struct curlObjData));    

    curlCopyCurlData(curlData,newCurlData);

    handleName=curlCreateObjCmd(interp,newCurlData);

    newCurlData->curl=newCurlHandle;

    result=Tcl_NewStringObj(handleName,-1);
    Tcl_SetObjResult(interp,result);
    ckfree(handleName);

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * curlVersion --
 *
 *	This procedure is invoked to process the "curl::init" Tcl command.
 *	See the user documentation for details on what it does.
 *
 * Parameters:
 *  The stantard parameters for Tcl commands
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int
curlVersion (ClientData clientData, Tcl_Interp *interp,
    int objc,Tcl_Obj *CONST objv[]) {

    Tcl_Obj     *versionPtr;
    char        tclversion[200];

    sprintf(tclversion,"TclCurl Version %s (%s)",TclCurlVersion,
                                                 curl_version());
    versionPtr=Tcl_NewStringObj(tclversion,-1);
    Tcl_SetObjResult(interp,versionPtr);

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * curlEscape --
 *
 *	This function is invoked to process the "curl::escape" Tcl command.
 *	See the user documentation for details on what it does.
 *
 *
 * Parameters:
 *  The stantard parameters for Tcl commands
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int
curlEscape(ClientData clientData, Tcl_Interp *interp,
    int objc,Tcl_Obj *CONST objv[]) {

    Tcl_Obj        *resultObj;
    char           *escapedStr;
    int            len;
    char           *toEscapeStr;

    len=Tcl_GetCharLength(objv[1]);
    toEscapeStr=ckalloc(len+1);
    strncpy(toEscapeStr,Tcl_GetString(objv[1]),len+1);
    escapedStr=curl_escape(toEscapeStr,0);

    if(!escapedStr) {
        resultObj=Tcl_NewStringObj("curl::escape bad parameter",-1);
        Tcl_SetObjResult(interp,resultObj);
        return TCL_ERROR;
    }
    resultObj=Tcl_NewStringObj(escapedStr,-1);
    Tcl_SetObjResult(interp,resultObj);
    ckfree((char *)escapedStr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * curlUnescape --
 *
 *	This function is invoked to process the "curl::Unescape" Tcl command.
 *	See the user documentation for details on what it does.
 *
 *
 * Parameters:
 *  The stantard parameters for Tcl commands
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int
curlUnescape(ClientData clientData, Tcl_Interp *interp,
    int objc,Tcl_Obj *CONST objv[]) {

    Tcl_Obj        *resultObj;
    char           *unescapedStr;

    unescapedStr=curl_unescape(Tcl_GetString(objv[1]),0);
    if(!unescapedStr) {
        resultObj=Tcl_NewStringObj("curl::unescape bad parameter",-1);
        Tcl_SetObjResult(interp,resultObj);
        return TCL_ERROR;
    }
    resultObj=Tcl_NewStringObj(unescapedStr,-1);
    Tcl_SetObjResult(interp,resultObj);
    ckfree((char *)unescapedStr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * curlCopyCurlData --
 *
 *	This function copies the contents of a curlData struct into another.
 *
 * Parameters:
 *  curlDataOld: The original one.
 *  curlDataNew: The new one
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int
curlCopyCurlData (struct curlObjData *curlDataOld,
                      struct curlObjData *curlDataNew) {

    /* This takes care of the int and long values */
    memcpy(curlDataNew, curlDataOld, sizeof(struct curlObjData));

    /* Some of the data doesn't get copied */

    curlDataNew->headerList=NULL;
    curlDataNew->quote=NULL;
    curlDataNew->prequote=NULL;
    curlDataNew->postquote=NULL;
    curlDataNew->postListFirst=NULL;
    curlDataNew->postListLast=NULL;
    curlDataNew->formHeaderList=NULL;

    /* The strings need a special treatment. */

    curlDataNew->outFile=curlstrdup(curlDataOld->outFile);
    curlDataNew->inFile=curlstrdup(curlDataOld->inFile);
    curlDataNew->URL=curlstrdup(curlDataOld->URL);
    curlDataNew->useragent=curlstrdup(curlDataOld->useragent);
    curlDataNew->referer=curlstrdup(curlDataOld->referer);
    curlDataNew->proxy=curlstrdup(curlDataOld->proxy);
    curlDataNew->userpwd=curlstrdup(curlDataOld->userpwd);
    curlDataNew->proxyuserpwd=curlstrdup(curlDataOld->proxyuserpwd);
    curlDataNew->range=curlstrdup(curlDataOld->range);
    curlDataNew->errorBuffer=curlstrdup(curlDataOld->errorBuffer);
    curlDataNew->errorBufferName=curlstrdup(curlDataOld->errorBufferName);
    curlDataNew->errorBufferKey=curlstrdup(curlDataOld->errorBufferKey);
    curlDataNew->postFields=curlstrdup(curlDataOld->postFields);
    curlDataNew->ftpport=curlstrdup(curlDataOld->ftpport);
    curlDataNew->cookie=curlstrdup(curlDataOld->cookie);
    curlDataNew->cookiefile=curlstrdup(curlDataOld->cookiefile);
    curlDataNew->sslcert=curlstrdup(curlDataOld->sslcert);
    curlDataNew->sslcertpasswd=curlstrdup(curlDataOld->sslcertpasswd);
    curlDataNew->headerFile=curlstrdup(curlDataOld->headerFile);
    curlDataNew->customrequest=curlstrdup(curlDataOld->customrequest);
    curlDataNew->stderrFile=curlstrdup(curlDataOld->stderrFile);
    curlDataNew->interface=curlstrdup(curlDataOld->interface);
    curlDataNew->krb4level=curlstrdup(curlDataOld->krb4level);
    curlDataNew->cainfo=curlstrdup(curlDataOld->cainfo);
    curlDataNew->randomFile=curlstrdup(curlDataOld->randomFile);
    curlDataNew->egdsocket=curlstrdup(curlDataOld->egdsocket);
    curlDataNew->headerVar=curlstrdup(curlDataOld->headerVar);
    curlDataNew->bodyVarName=curlstrdup(curlDataOld->bodyVarName);
    curlDataNew->progressProc=curlstrdup(curlDataOld->progressProc);
    curlDataNew->cancelTransVarName=curlstrdup(curlDataOld->cancelTransVarName);
    curlDataNew->writeProc=curlstrdup(curlDataOld->writeProc);
    curlDataNew->readProc=curlstrdup(curlDataOld->readProc);
    curlDataNew->cookiejar=curlstrdup(curlDataOld->cookiejar);
    curlDataNew->sslcipherlist=curlstrdup(curlDataOld->sslcipherlist);
    curlDataNew->sslcerttype=curlstrdup(curlDataOld->sslcerttype);
    curlDataNew->sslkey=curlstrdup(curlDataOld->sslkey);
    curlDataNew->sslkey=curlstrdup(curlDataOld->sslkeytype);

    memcpy(curlDataNew->bodyVar.memory,curlDataOld->bodyVar.memory
            ,curlDataOld->bodyVar.size);
    curlDataNew->bodyVar.size=curlDataOld->bodyVar.size;

    return TCL_OK;
}

/*----------------------------------------------------------------------
 *
 * curlstrdup --
 *
 *	The same as strdup, but won't seg fault if the string to copy is NULL.
 *
 * Parameter:
 *  old: The original one.
 *
 * Results:
 *	Returns a pointer to the new string.
 *----------------------------------------------------------------------
 */
char
*curlstrdup (char *old) {

    if (old==NULL) {
        return NULL;
    }
    return strdup(old);
}



