/*
 * Implementation of some test utility functions
 *
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifndef INSTANTIATING_TEMPLATES
#include <string>

namespace Tagcoll {
	//template class TextFormat<std::string, std::string>;
}
#endif


#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>
#include <tagcoll/StringParserInput.h>
#include <tagcoll/TextFormat.h>
#include <tagcoll/InputMerger.h>
#include <tagcoll/Patches.h>

namespace tut_tagcoll {

using namespace std;
using namespace Tagcoll;
using namespace tut;
	
void _gen_ensure(const Location& loc, bool res)
{
	if (!res)
		throw failure(loc.locstr());
}

void outputCollection(const std::string& str, Tagcoll::Consumer<string, string>& cons)
{
	StringParserInput input(str);
	TrivialConverter<string, string> a;
	TextFormat<string, string>::parse(a, a, input, cons);
}

void __tc_ensure_coll_equals(const Location& loc,
		const Tagcoll::ReadonlyCollection<string, string>& c1, const Tagcoll::ReadonlyCollection<string, string>& c2)
{
	PatchList<string, string> p;
	p.addPatch(c1, c2);
	if (!p.empty())
	{
		string f = "collections differ.  Patch:\n";
		for (PatchList<string, string>::const_iterator i = p.begin();
				i != p.end(); i++)
		{
			f += i->first;
			f += ": ";
			bool first = true;
			for (OpSet<string>::const_iterator j = i->second.getAdded().begin();
					j != i->second.getAdded().end(); j++)
				if (first)
				{
					f += "+" + *j;
					first = false;
				} else {
					f += ", +" + *j;
				}
			for (OpSet<string>::const_iterator j = i->second.getRemoved().begin();
					j != i->second.getRemoved().end(); j++)
				if (first)
				{
					f += "-" + *j;
					first = false;
				} else {
					f += ", -" + *j;
				}
			f += "\n";
		}
		
		throw failure(loc.msg(f));
	}
}

void __output_test_collection(const Location& loc, Consumer<string, string>& tc)
{
	OpSet<string> tagset;

	tagset.clear();
	tc.consume("gnocco");

	tagset.clear();
	tagset += "tomato"; tagset += "mozzarella";
	tc.consume("margherita", tagset);

	tagset.clear();
	tagset += "tomato"; tagset += "mozzarella"; tagset += "mushrooms";
	tc.consume("funghi", tagset);

	tagset.clear();
	tagset += "garlic"; tagset += "rosemerry";
	tc.consume("rosmarino", tagset);

	tagset.clear();
	tagset += "garlic"; tagset += "tomato";
	tc.consume("marinara", tagset);
}

void __test_readonly_collection(const Location& loc, ReadonlyCollection<string, string>& tc)
{
	OpSet<string> s, s1;

	// hasTag
	inner_ensure(tc.hasTag("tomato"));
	inner_ensure(tc.hasTag("mozzarella"));
	inner_ensure(tc.hasTag("mushrooms"));
	inner_ensure(tc.hasTag("garlic"));
	inner_ensure(tc.hasTag("rosemerry"));
	inner_ensure(!tc.hasTag("ketchup"));

	// getTags(item)
	s = tc.getTags("funghi");
	inner_ensure_contains(s, string("tomato"));
	inner_ensure_contains(s, string("mozzarella"));
	inner_ensure_contains(s, string("mushrooms"));
	inner_ensure_not_contains(s, string("garlic"));
	inner_ensure_not_contains(s, string("rosemerry"));

	s = tc.getTags("margherita");
	inner_ensure_contains(s, string("tomato"));
	inner_ensure_contains(s, string("mozzarella"));
	inner_ensure_not_contains(s, string("mushrooms"));
	inner_ensure_not_contains(s, string("garlic"));
	inner_ensure_not_contains(s, string("rosemerry"));

	s = tc.getTags("rosmarino");
	inner_ensure(!s.contains("tomato"));
	inner_ensure(!s.contains("mozzarella"));
	inner_ensure(!s.contains("mushrooms"));
	inner_ensure(s.contains("garlic"));
	inner_ensure(s.contains("rosemerry"));

	s = tc.getTags("marinara");
	inner_ensure_contains(s, string("tomato"));
	inner_ensure_not_contains(s, string("mozzarella"));
	inner_ensure_not_contains(s, string("mushrooms"));
	inner_ensure_contains(s, string("garlic"));
	inner_ensure_not_contains(s, string("rosemerry"));

	s = tc.getTags("gnocco");
	inner_ensure(s.empty());

	// getTags(items)
	s1.clear();
	s1 += "funghi"; s1 += "margherita";
	s = tc.getTags(s1);
	inner_ensure_contains(s, string("tomato"));
	inner_ensure_contains(s, string("mozzarella"));
	inner_ensure_contains(s, string("mushrooms"));
	inner_ensure_not_contains(s, string("garlic"));
	inner_ensure_not_contains(s, string("rosemerry"));

	s1.clear();
	s1 += "rosmarino"; s1 += "margherita";
	s = tc.getTags(s1);
	inner_ensure_contains(s, string("tomato"));
	inner_ensure_contains(s, string("mozzarella"));
	inner_ensure_not_contains(s, string("mushrooms"));
	inner_ensure_contains(s, string("garlic"));
	inner_ensure_contains(s, string("rosemerry"));

	s1.clear();
	s1 += "funghi"; s1 += "margherita"; s1 += "marinara";
	s = tc.getTags(s1);
	inner_ensure_contains(s, string("tomato"));
	inner_ensure_contains(s, string("mozzarella"));
	inner_ensure_contains(s, string("mushrooms"));
	inner_ensure_contains(s, string("garlic"));
	inner_ensure_not_contains(s, string("rosemerry"));

	// getItems(tag)
	s = tc.getItems("tomato");
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_contains(s, string("margherita"));
	inner_ensure_contains(s, string("funghi"));
	inner_ensure_not_contains(s, string("rosmarino"));
	inner_ensure_contains(s, string("marinara"));

	s = tc.getItems("mozzarella");
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_contains(s, string("margherita"));
	inner_ensure_contains(s, string("funghi"));
	inner_ensure_not_contains(s, string("rosmarino"));
	inner_ensure_not_contains(s, string("marinara"));

	s = tc.getItems("mushrooms");
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_not_contains(s, string("margherita"));
	inner_ensure_contains(s, string("funghi"));
	inner_ensure_not_contains(s, string("rosmarino"));
	inner_ensure_not_contains(s, string("marinara"));

	s = tc.getItems("garlic");
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_not_contains(s, string("margherita"));
	inner_ensure_not_contains(s, string("funghi"));
	inner_ensure_contains(s, string("rosmarino"));
	inner_ensure_contains(s, string("marinara"));

	s = tc.getItems("rosemerry");
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_not_contains(s, string("margherita"));
	inner_ensure_not_contains(s, string("funghi"));
	inner_ensure_contains(s, string("rosmarino"));
	inner_ensure_not_contains(s, string("marinara"));

	// getItems(tags)
	s1.clear();
	s1 += "tomato"; s1 += "mozzarella";
	s = tc.getItems(s1);
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_contains(s, string("margherita"));
	inner_ensure_contains(s, string("funghi"));
	inner_ensure_not_contains(s, string("rosmarino"));
	inner_ensure_not_contains(s, string("marinara"));

	s1.clear();
	s1 += "garlic"; s1 += "rosemerry";
	s = tc.getItems(s1);
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_not_contains(s, string("margherita"));
	inner_ensure_not_contains(s, string("funghi"));
	inner_ensure_contains(s, string("rosmarino"));
	inner_ensure_not_contains(s, string("marinara"));

	s1.clear();
	s1 += "tomato"; s1 += "rosemerry";
	s = tc.getItems(s1);
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_not_contains(s, string("margherita"));
	inner_ensure_not_contains(s, string("funghi"));
	inner_ensure_not_contains(s, string("rosmarino"));
	inner_ensure_not_contains(s, string("marinara"));

	// TODO:    virtual OpSet<ITEM> getTaggedItems() const = 0;
	// TODO:    virtual OpSet<TAG> getAllTags() const = 0;
	// TODO:    virtual int getCardinality(const TAG& tag) const
	// TODO:    virtual OpSet<TAG> getCompanionTags(const OpSet<TAG>& tags) const
	// TODO:    virtual OpSet<ITEM> getRelatedItems(const OpSet<TAG>& tags, int maxdistance = 1) const

	// void output(Consumer<ITEM, TAG>& consumer) const
	InputMerger<string, string> coll1;
	tc.output(coll1);
	inner_ensure_coll_equals(tc, coll1);

	// TODO:    virtual void outputHavingTags(const OpSet<TAG>& tags, Consumer<ITEM, TAG>& consumer) const
}

void __test_collection(const Location& loc, Collection<string, string>& tc)
{
	// Test handling of untagged items (they are not stored)
	tc.consume("untagged");
	inner_ensure(tc.getTags("untagged").empty());
	
	// Test handling of tagged items
	OpSet<string> tagset;
	tagset += "tag1"; tagset += "tag2";
	tc.consume("tagged", tagset);
	inner_ensure(tc.getTaggedItems().contains("tagged"));
	//inner_ensure(tc.hasTag("tag1"));
	//inner_ensure(tc.hasTag("tag2"));
	tagset = tc.getTags("tagged");
	inner_ensure(tagset.contains("tag1"));
	inner_ensure(tagset.contains("tag2"));
	OpSet<string> itemset = tc.getItems("tag1");
	inner_ensure(itemset.contains("tagged"));
	itemset = tc.getItems("tag2");
	inner_ensure(itemset.contains("tagged"));
	tagset = tc.getAllTags();
	inner_ensure(tagset.contains("tag1"));
	inner_ensure(tagset.contains("tag2"));
	tagset.clear();
	tagset += "tag1";
	tagset = tc.getCompanionTags(tagset);
	inner_ensure(!tagset.contains("tag1"));
	inner_ensure(tagset.contains("tag2"));

	// Test handling of changes
	PatchList<string, string> change;
	Patch<string, string> p("tagged");
	tagset.clear();
	p.remove("tag1");
	p.remove("tag2");
	change.addPatch(p);

	tc.applyChange(change);
	
	// "tagged" should now be untagged
	inner_ensure(tc.getTags("tagged").empty());

	tc.applyChange(change.getReverse());

	// "tagged" should now be as before
	//inner_ensure(tc.hasTag("tag1"));
	//inner_ensure(tc.hasTag("tag2"));
	tagset = tc.getTags("tagged");
	inner_ensure(tagset.contains("tag1"));
	inner_ensure(tagset.contains("tag2"));
	itemset = tc.getItems("tag1");
	inner_ensure(itemset.contains("tagged"));
	itemset = tc.getItems("tag2");
	inner_ensure(itemset.contains("tagged"));
	tagset = tc.getAllTags();
	inner_ensure(tagset.contains("tag1"));
	inner_ensure(tagset.contains("tag2"));
	tagset.clear();
	tagset += "tag1";
	tagset = tc.getCompanionTags(tagset);
	inner_ensure(!tagset.contains("tag1"));
	inner_ensure(tagset.contains("tag2"));

	// Try a patch that adds a tag
	change = PatchList<string, string>();
	p = Patch<string, string>("tagged");
	p.add("tag3");
	change.addPatch(p);
	tc.applyChange(change);

	//inner_ensure(tc.hasTag("tag1"));
	//inner_ensure(tc.hasTag("tag2"));
	//inner_ensure(tc.hasTag("tag3"));
	tagset = tc.getTags("tagged");
	inner_ensure(tagset.contains("tag1"));
	inner_ensure(tagset.contains("tag2"));
	inner_ensure(tagset.contains("tag3"));
	itemset = tc.getItems("tag1");
	inner_ensure(itemset.contains("tagged"));
	itemset = tc.getItems("tag2");
	inner_ensure(itemset.contains("tagged"));
	itemset = tc.getItems("tag3");
	inner_ensure(itemset.contains("tagged"));
	tagset = tc.getAllTags();
	inner_ensure(tagset.contains("tag1"));
	inner_ensure(tagset.contains("tag2"));
	inner_ensure(tagset.contains("tag3"));
	tagset.clear();
	tagset += "tag1";
	tagset = tc.getCompanionTags(tagset);
	inner_ensure(!tagset.contains("tag1"));
	inner_ensure(tagset.contains("tag2"));
	inner_ensure(tagset.contains("tag3"));

	// Try a patch that adds some items
	change = PatchList<string, string>();
	p = Patch<string, string>("tagged1");
	p.add("tag1");
	p.add("tag2");
	p.add("tag4");
	change.addPatch(p);
	tc.applyChange(change);

	tagset = tc.getTags("tagged1");
	inner_ensure(tagset.contains("tag1"));
	inner_ensure(tagset.contains("tag2"));
	inner_ensure(!tagset.contains("tag3"));
	inner_ensure(tagset.contains("tag4"));
	itemset = tc.getItems("tag1");
	inner_ensure(itemset.contains("tagged1"));
	itemset = tc.getItems("tag2");
	inner_ensure(itemset.contains("tagged1"));
	itemset = tc.getItems("tag3");
	inner_ensure(!itemset.contains("tagged1"));
	itemset = tc.getItems("tag4");
	inner_ensure(!itemset.contains("tagged"));
	inner_ensure(itemset.contains("tagged1"));
	tagset = tc.getAllTags();
	inner_ensure(tagset.contains("tag1"));
	inner_ensure(tagset.contains("tag2"));
	inner_ensure(tagset.contains("tag3"));
	inner_ensure(tagset.contains("tag4"));
	tagset.clear();
	tagset += "tag1";
	tagset = tc.getCompanionTags(tagset);
	inner_ensure(!tagset.contains("tag1"));
	inner_ensure(tagset.contains("tag2"));
	inner_ensure(tagset.contains("tag3"));
	inner_ensure(tagset.contains("tag4"));

	// And reverse it
	tc.applyChange(change.getReverse());

	itemset = tc.getItems("tag1");
	inner_ensure(!itemset.contains("tagged1"));
	itemset = tc.getItems("tag2");
	inner_ensure(!itemset.contains("tagged1"));
	itemset = tc.getItems("tag3");
	inner_ensure(!itemset.contains("tagged1"));
	inner_ensure(tc.getItems("tag4") == OpSet<string>());
	tagset = tc.getAllTags();
	inner_ensure(tagset.contains("tag1"));
	inner_ensure(tagset.contains("tag2"));
	inner_ensure(tagset.contains("tag3"));
	inner_ensure(!tagset.contains("tag4"));
	tagset.clear();
	tagset += "tag1";
	tagset = tc.getCompanionTags(tagset);
	inner_ensure(!tagset.contains("tag1"));
	inner_ensure(tagset.contains("tag2"));
	inner_ensure(tagset.contains("tag3"));
	inner_ensure(!tagset.contains("tag4"));
}

}

#endif

// vim:set ts=4 sw=4:
