#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
This module contains code generation tools for the ufc::form class.
"""

__author__ = "Martin Sandve Alnes"
__date__   = "2008-08-13 -- 2009-03-05"
__copyright__ = "(C) 2008-2009 Martin Sandve Alnes and Simula Resarch Laboratory"
__license__  = "GNU GPL Version 2, or (at your option) any later version"

from sfc.codegeneration.codeformatting import indent, CodeFormatter

class FormCG:
    "Code generator for an ufc::form."
    def __init__(self, formrep):
        self.formrep = formrep

    def hincludes(self):
        l = []
        return l

    def cincludes(self):
        l = []
        return l

    def generate_code_dict(self):
        form_vars = {
             'classname'                       : self.formrep.classname,
             'constructor'                     : indent(self.gen_constructor()),
             'destructor'                      : indent(self.gen_destructor()),
             'members'                         : indent(self.gen_members()),
             'rank'                            : indent(self.gen_rank()),
             'num_coefficients'                : indent(self.gen_num_coefficients()),
             'signature'                       : indent(self.gen_signature()),
             'create_dof_map'                  : indent(self.gen_create_dof_map()),
             'create_finite_element'           : indent(self.gen_create_finite_element()),
             'create_cell_integral'            : indent(self.gen_create_cell_integral()),
             'create_exterior_facet_integral'  : indent(self.gen_create_exterior_facet_integral()),
             'create_interior_facet_integral'  : indent(self.gen_create_interior_facet_integral()),
             'num_cell_integrals'              : indent(self.gen_num_cell_integrals()),
             'num_exterior_facet_integrals'    : indent(self.gen_num_exterior_facet_integrals()),
             'num_interior_facet_integrals'    : indent(self.gen_num_interior_facet_integrals()),
            }
        return form_vars
    
    def generate_support_code(self):
        return ""

    def gen_constructor(self):
        return ""

    def gen_destructor(self):
        return ""

    def gen_members(self):
        return ""

    def gen_rank(self):
        return 'return %d;'   % self.formrep.rank

    def gen_num_coefficients(self):
        return 'return %d;'   % self.formrep.num_coefficients

    def gen_signature(self):
        return 'return "%s";' % self.formrep.signature
    
    def _gen_num_integrals(self, names):
        if names:
            n = 1 + max(list(names.keys()))
        else:
            n = 0
        return 'return %d;' % n

    def gen_num_cell_integrals(self):
        return self._gen_num_integrals(self.formrep.citg_names)

    def gen_num_exterior_facet_integrals(self):
        return self._gen_num_integrals(self.formrep.eitg_names)

    def gen_num_interior_facet_integrals(self):
        return self._gen_num_integrals(self.formrep.iitg_names)

    def _gen_create_switch(self, classnames, name, indices = None):
        "Utility function for creating factory functions."
        code = CodeFormatter()
        #if options.add_debug_code: # FIXME
        #    code.begin_debug()
        #    code += 'std::cout << "Creating object in %s, index is " << i << std::endl;' % name
        #    code.end_debug()
        if classnames:
            if isinstance(classnames, dict):
                indices = classnames.keys()
                classnames = classnames.values()
            elif indices is None:
                indices = range(len(classnames))
            
            code.begin_switch("i")
            for i, c in zip(indices, classnames):
                code.begin_case(i)
                code += "return new %s();" % c
                code.end_case()
            code.end_switch()
        code += 'throw std::runtime_error("Invalid index in %s()");' % name
        return str(code)

    def gen_create_dof_map(self):
        return self._gen_create_switch(self.formrep.dm_names, "create_dof_map")

    def gen_create_finite_element(self):
        return self._gen_create_switch(self.formrep.fe_names, "create_finite_element")

    def gen_create_cell_integral(self):
        return self._gen_create_switch(self.formrep.citg_names, "create_cell_integral")

    def gen_create_exterior_facet_integral(self):
        return self._gen_create_switch(self.formrep.eitg_names, "create_exterior_facet_integral")

    def gen_create_interior_facet_integral(self):
        return self._gen_create_switch(self.formrep.iitg_names, "create_interior_facet_integral")

if __name__ == '__main__':
    print "form.py, no test implemented"

