// Copyright (C) 2009-2009 Martin Sandve Alnæs
// Licensed under the GNU LGPL Version 3.0.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program computes a projection in 1D.

#include <dolfin.h>
#include "generated_code/Projection1D.h"

using namespace dolfin;
using namespace Projection1D;

class Weight: public Function
{
public:
  Weight(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    values[0] = 1.0;
  }
};

class Source: public Function
{
public:
  Source(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    double dx = x[0] - 0.5;
    values[0] = dx*dx;
  }
};

int main()
{
  // Geometry
  UnitInterval mesh(20);

  // Function spaces
  BilinearForm::TrialSpace V(mesh);
  CoefficientSpace_c C(mesh);
  CoefficientSpace_f F(mesh);

  // Functions
  Weight c(C);
  Source f(F);
  Function u(V);

  // Forms
  BilinearForm a(V, V);
  LinearForm L(V);

  // Form coefficients
  a.c = c;
  L.f = f;

  // Assemble linear system
  Matrix A;
  Vector b;
  assemble(A, a);
  assemble(b, L);

  // Solve linear system
  LUSolver solver;
  solver.solve(A, u.vector(), b);

  // Write solution to file
  File u_file("u.pvd");
  u_file << u;

  // Interpolate f onto the discrete space V
  Source source(V);
  Function finterpolated(V);
  source.interpolate(finterpolated.vector(), V);

  // Write interpolated solution to file
  File f_file("f.pvd");
  f_file << finterpolated;
  
  //plot(u);
  
  return 0;
}

