/* Storm Package Manager
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <limits.h>

#include "../libstormpkg/libstormpkg_app_include.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "stormpkg.h"
#include "depend.h"
#include "match.h"
#include "list.h"
#include "main.h"


/* Icky dpkg stuff */
const char printforhelp[] = "<Magic string match here>.\n";
const char thisname[] = "stormpkg";

/* Globals */
LIST_HEAD(dependencies_input_anchor);
LIST_HEAD(dependencies_output_anchor);
LIST_HEAD(selected_packages_anchor);
LIST_HEAD(dr_selected_packages_anchor);

extern int maxID;
extern char *MainState;
extern char *package_s[];
extern pkgCache *MainPkgCache;
extern pkgRecords *MainPkgRecords;
extern pkgDepCache *MainPkgDepCache;

package_node **nodes=NULL;      //main node list

int base_pkg_count=0;
int should_apply=0;

static char *global_search_expression;
static gboolean global_search_case_sensitive;
static int global_search_where_mask;
static int global_search_packages_mask;
static int global_show_mask = SHOW_PRIORITIES | SHOW_SECTIONS;

static int resolve_initial_dependencies;
static struct package_node *topleft_node = NULL;
static struct package_node *global_last_match = NULL;
static char base_node_name[] = "DEB Packages";

char nullstring[] = "";
static char tempstring[512+1];

int msg_dlg_exit=0;

struct units {
   double multiplier;
   char unit_name[8];
};
/*
struct units byte_units[] = {
   { 1024,              "KB", },
   { 1048576,           "MB", },
   { 1073741824,        "GB", },
   { 0,                 "", },
};
*/
struct units byte_units[] = {
   { 1000,              "KB", },
   { 1000000,           "MB", },
   { 1000000000,        "GB", },
   { 0,                 "", },
};

/* Functions */

static void collapse_path(char *s){
   char *d = s;
   char c = '\0';

   for (;*s;s++){
      if (*s == '/')
         if (c == '/')
            continue;
      c = *s;
      *d++ = c;
   }
   *d = '\0';
}



/*
 * Automatically select an appropriate unit and print number into strings
 *
 * Assume we have DESIRED_OUTPUT_SIZE characters to present something useful in,
 * excluding the unit.
 */

#define DESIRED_OUTPUT_SIZE 3

static char *auto_unit(struct units *up,double v){
   double multiplier;
   char *unitp;  
   int vlen,dlen;
   char lenstring[32+1];
   static char outstring[DESIRED_OUTPUT_SIZE + 32+1];

   if (v < 0)
      v = -v;

   multiplier = up->multiplier;
   unitp = up->unit_name;
   for (;up->multiplier;up++){
      if (v < up->multiplier)
         break;
      multiplier = up->multiplier;
      unitp = up->unit_name;
   }
   v/= multiplier;
   sprintf(lenstring,"%.0f",v);
   vlen = strlen(lenstring);
   dlen = DESIRED_OUTPUT_SIZE - 1 - vlen;
   if (dlen > 0)
      sprintf(outstring,"%.*f %.24s",dlen,v,unitp);
   else
      sprintf(outstring,"%s %.24s",lenstring,unitp);   /* No decimals */
   return outstring;
}

char * autosize(double v){
    return (char *)auto_unit(byte_units, v);
    }

/* choke
 *
 * Die cleanly with error message
 */

void choke(char *s){
   fprintf(stderr,"%s\n",s);
   gtk_exit(1);
}

/* sync_display
 *
 * Flush updates to screen before returning to mainloop
 */
static void sync_display(){
#if 0
   while (gtk_events_pending())
      gtk_main_iteration();
   gdk_flush();
   while (gtk_events_pending())
      gtk_main_iteration();
   gdk_flush();
#else
    int i=1;
    while (gtk_events_pending())
        gtk_main_iteration();
    while (i) {
        gdk_flush();
        i=0;
        while (gtk_events_pending()) {
            gtk_main_iteration();
            i=1;
            }
        }
#endif
    }

/* set_progress
 *
 * Set progress meter to specified percentage
 */

void set_progress(GtkWidget *window,gfloat percentage){
   static GtkWidget *progressbar = NULL;
#if 0
   if (!progressbar)
      progressbar = lookup_widget(window,"progressbar");
   assert(progressbar);

   gtk_progress_bar_update(GTK_PROGRESS_BAR(progressbar),percentage);

   sync_display();
#endif
}


/* set_statusbar
 *
 * Set status bar to specified message
 */

void set_statusbar(GtkWidget *window,gchar *message){
   static GtkStatusbar *statusbar = NULL;
//   static guint message_id = -1;
   static int message_id = -1;

   if (!statusbar)
      statusbar = GTK_STATUSBAR(lookup_widget(window,"statusbar"));
   assert(statusbar);

   if (message_id != -1)
      gtk_statusbar_remove(statusbar,1,message_id);
   message_id = gtk_statusbar_push(statusbar,1,message);
}

static __inline__ void gtktext_append_string(GtkText *x,char *s){
   gtk_text_insert(x,NULL,NULL,NULL,s,strlen(s));
}

static __inline__ void gtktext_append_buffer(GtkText *x,char *s,size_t l){
   gtk_text_insert(x,NULL,NULL,NULL,s,l);
}

static __inline__ void gtktext_clear(GtkText *x){
   gtk_text_set_point(x,0);
   gtk_text_forward_delete(x,gtk_text_get_length(x));
}

static GdkColor *get_color_for_background(struct package_node *node){

    switch(node->Selected){
        case pkgCache::State::Install:
            return install_background_color;
        case pkgCache::State::DeInstall:
            return uninstall_background_color;
        case pkgCache::State::Purge:
            return purge_background_color;
        case pkgCache::State::Hold:
            return hold_background_color;
        default:break;
        }
    if  (node->pkg->SelectedState == pkgCache::State::Unknown)
        return new_background_color;
    return unchanged_background_color;
    }

static GdkColor *get_color_for_foreground(struct package_node *node){

    if (!node->pkg)
        return unchanged_foreground_color;
    if  (!node->Selected)
        return unchanged_foreground_color;
    return changed_foreground_color;

	return NULL;
}

static void update_node_background_color(GtkCTree *ctree,struct package_node *node){
   gtk_ctree_node_set_background(ctree,node->gtk_ctree_node,get_color_for_background(node));
   gtk_ctree_node_set_foreground(ctree,node->gtk_ctree_node,get_color_for_foreground(node));
}

static void update_dependency_node_background_color(GtkCList *clist,struct package_node *node){
   gtk_clist_set_background(clist,node->gtk_clist_row,get_color_for_background(node));
   gtk_clist_set_foreground(clist,node->gtk_clist_row,get_color_for_foreground(node));
}

static int search_matches(struct package_node *node){

   assert(node->pkg);

   if (global_search_packages_mask == PACKAGES_UNINSTALLED)
      if (node->pkg->CurrentState==pkgCache::State::Installed)//(PACKAGE_INSTALLED(pkg))
         return 0;
   if (global_search_packages_mask == PACKAGES_INSTALLED)
      if (node->pkg->CurrentState!=pkgCache::State::Installed)//(!PACKAGE_INSTALLED(pkg))
         return 0;

   if (!*global_search_expression)
      return 1;

   if (global_search_where_mask & SEARCH_NAMES)
      if (match(global_search_expression,node->Name,global_search_case_sensitive))
         return 1;

   if (global_search_where_mask & SEARCH_DESCRIPTIONS){
		if	(node->CurrentLongDesc)
            if (match(global_search_expression, node->CurrentLongDesc, global_search_case_sensitive))
				return 2;
		if	(node->AvailableLongDesc)
            if (match(global_search_expression, node->AvailableLongDesc, global_search_case_sensitive))
				return 2;
		}
	return 0;
}

/* populate_package_tree
 *
 * Populate the package columned tree
 */

enum {
   CTREE_PACKAGE_NAME,
   CTREE_STATUS,
   CTREE_SIZE,
   CTREE_INSTALLED,
   CTREE_NEWEST,
   CTREE_DESCRIPTION,
   CTREE_FIELDS /* counter */
};

enum {
   CLIST_OLD_STATUS,
   CLIST_NEW_STATUS,
   CLIST_PRIORITY,
   CLIST_SECTION,
   CLIST_PACKAGE_NAME,
   CLIST_DESCRIPTION,
   CLIST_FIELDS /* counter */
};

#define PKGSTATUS_MAX (stat_unknown + 1)
#define PKGPRIORITY_MAX (pri_unknown + 1)

static char namestring[1024+1];

/* recurse_populate_by_path
 *
 * Recursively populate the two dimensional tree list.
 */

static void recurse_populate_by_path(struct package_node *from_node,char *path, pkgCache::PkgIterator *ppi){

   struct list_head *lp=NULL;
   struct package_node *node;
   char *s ; // ,*sortkey ;

   // Are we at the actual package or at a path component?
   if ((s = strchr(path,'/'))){
      // Path component: find or create and recurse
      *s++ = '\0';
      if (from_node->right){
         lp = &from_node->right->up_down;
         do {
            node = list_entry(lp,struct package_node,up_down);
            if (!strcmp(node->Name,path)){
               // This node already exists
               recurse_populate_by_path(node,s,ppi);
               return;
            }
            lp = lp->next;
         } while (lp != &from_node->right->up_down);
         // Must allocate this node
      }
   }
   node = alloc_structure_stretchable(struct package_node,strlen(path));
   strcpy(node->Name,path);
	node->right = NULL;
	node->left = from_node;

    node->relations=new string();

	if (!from_node->right){
		from_node->right = node;
		INIT_LIST_HEAD(&node->up_down);	// No external anchor
		}
	else
		list_add_tail(&node->up_down,&from_node->right->up_down);

   if (s)
      recurse_populate_by_path(node,s,ppi);
   else {
      nodes[(*ppi)->ID]=node;

      node->pkg=ppi->operator->();
	
      node->Section=ppi->Section()==0?_("No Section"):ppi->Section();

	  node->InstalledVersionStr=(*ppi)->CurrentVer==0?"":ppi->CurrentVer().VerStr();
	  node->AvailableVersionStr=(*ppi)->VersionList==0?"":ppi->VersionList().VerStr();
	
	  pkgCache::VerIterator vi;
	
	  vi=ppi->VersionList();
	  const char *p;
	  p=vi.PriorityType();
	  node->PriorityStr = *p==0?_("No Priority"):p;
	
      vi=ppi->CurrentVer();

      node->CurrentShortDesc=node->CurrentLongDesc=node->AvailableShortDesc=node->AvailableLongDesc=node->Name;

      if (vi.end()==false)	{
		node->CurrentShortDesc=strdup((MainPkgRecords->Lookup(vi.FileList())).ShortDesc().c_str());
		node->CurrentLongDesc=strdup((MainPkgRecords->Lookup(vi.FileList())).LongDesc().c_str());
	  	}
	
//	  vi=MainPkgCache->GetCandidateVer(*ppi);
	  vi=MainPkgDepCache->GetCandidateVer(*ppi);
      if (vi.end()==false)	{
		node->AvailableShortDesc=strdup((MainPkgRecords->Lookup(vi.FileList())).ShortDesc().c_str());
	  	node->AvailableLongDesc=strdup((MainPkgRecords->Lookup(vi.FileList())).LongDesc().c_str());
		}	
		
	  node->pkgInstalledSize=vi->InstalledSize;
	  node->pkgDebSize=vi->Size;
	
	  node->Selected=pkgCache::State::Unknown;
   }

}

static int global_package_displayed_count;

static void recurse_gtk_populate(struct package_node *from_node,GtkCTree *ctree,GtkCTreeNode *ctree_node_parent,gboolean expand_leaves){

	struct list_head *lp;
	struct package_node *node,*leftnode;
	GtkCTreeNode *gtk_node;
	static char *s;
	static gchar *ctree_fields[CTREE_FIELDS];
	static int recurse = 0;
	static char package_name_new[256+8+1];
	int local_displayed_count;

	recurse++;

	lp = &from_node->up_down;               	

	do {
      node = list_entry(lp,struct package_node,up_down);

      if ((!node->pkg) || (node->pkg->SelectedState != pkgCache::State::Unknown)){
         ctree_fields[CTREE_PACKAGE_NAME] = node->Name;
      	}
      else { // New
        	 sprintf(package_name_new,_("%.256s (NEW)"),node->Name);
         	ctree_fields[CTREE_PACKAGE_NAME] = package_name_new;
      	}

	if (node->pkg){
       	 ctree_fields[CTREE_STATUS] = nullstring;
         ctree_fields[CTREE_STATUS] = get_selection_string(node);
         ctree_fields[CTREE_DESCRIPTION] = nullstring;
         ctree_fields[CTREE_SIZE] = nullstring;
         ctree_fields[CTREE_INSTALLED] = (char *)node->InstalledVersionStr;
         ctree_fields[CTREE_NEWEST] = (char *)node->AvailableVersionStr;

	 //	 ctree_fields[CTREE_DESCRIPTION]=node->CurrentShortDesc==0?(node->AvailableShortDesc==0?nullstring:node->AvailableShortDesc):node->CurrentShortDesc;
       	 ctree_fields[CTREE_DESCRIPTION]=node->AvailableShortDesc;
	   
       	 if	(node->pkgDebSize)
			ctree_fields[CTREE_SIZE] = auto_unit(byte_units,node->pkgDebSize);

         s = strchr(ctree_fields[CTREE_DESCRIPTION],'\n');
         if (s)
            *s = '\0';
       	}
      else {
         ctree_fields[CTREE_STATUS] = nullstring;
         ctree_fields[CTREE_SIZE] = nullstring;
         ctree_fields[CTREE_INSTALLED] = nullstring;
         ctree_fields[CTREE_NEWEST] = nullstring;
         ctree_fields[CTREE_DESCRIPTION] = nullstring;
         s = NULL;
      	}

      gtk_node = NULL;
      if (node->pkg){
         if (search_matches(node)){
            global_package_displayed_count++;
            gtk_node = gtk_ctree_insert_node(ctree,ctree_node_parent,NULL,ctree_fields,0,NULL,NULL,NULL,NULL,
             TRUE ,FALSE);
            gtk_ctree_node_set_row_data(ctree,gtk_node,(gpointer)node);
            if (expand_leaves){
               for (leftnode = node->left;leftnode;leftnode = leftnode->left)
                  if (leftnode->gtk_ctree_node)
                     gtk_ctree_expand(ctree,leftnode->gtk_ctree_node);
            	}
	 		}
      	}
      else {
         do {
            gtk_node = ctree_node_parent;
            if (!(global_show_mask & SHOW_PRIORITIES))
               if (recurse == 3)
                  break;
            if (!(global_show_mask & SHOW_SECTIONS))
               if (recurse >= 4)
                  break;
            gtk_node = gtk_ctree_insert_node(ctree,ctree_node_parent,NULL,ctree_fields,0,
            								NULL,NULL,NULL,NULL,FALSE ,FALSE);
            gtk_ctree_node_set_selectable(ctree,gtk_node,FALSE);
         	} while (0);
      	}
      if (s)
         *s = '\n';

      node->gtk_ctree_node = gtk_node;

      if (node->pkg)
         if (gtk_node)
            update_node_background_color(ctree,node);

      local_displayed_count = global_package_displayed_count;

      if (node->right){
         recurse_gtk_populate(node->right,ctree,gtk_node,expand_leaves);

         if (local_displayed_count == global_package_displayed_count){
            // Recursion added no new packages -- zap this trunk
            if (node != topleft_node){
               if (gtk_node != ctree_node_parent){	// Not ignored level
                  gtk_ctree_remove_node(ctree,gtk_node);
                  node->gtk_ctree_node = NULL;
               		}
            	}
         	}
      	}

   	lp = lp->next;
   	} while (lp != &from_node->up_down);
   recurse--;

}


struct package_node** alloc_nodes(int i){
    struct package_node **p;
    size_t size;

    size=i*sizeof(struct package_node*);
    p=(struct package_node**)malloc(size);
      if (!p){
         printf(_("Can't alloc memory for nodes\n"));
         exit (1);
         }
    memset((void *)p, 0, size);
    return p;
    }



static void construct_main_nodes(){

   if (!base_pkg_count){
      topleft_node = alloc_structure_stretchable(struct package_node,strlen(base_node_name));
      strcpy(topleft_node->Name,base_node_name);
      topleft_node->left =
      topleft_node->right = NULL;

        topleft_node->relations=new string();

      topleft_node = topleft_node;
			
      INIT_LIST_HEAD(&topleft_node->up_down);

      resolve_initial_dependencies = 0;
      INIT_LIST_HEAD(&dependencies_input_anchor);

      base_pkg_count = get_package_list();

      if    (base_pkg_count<=0)
      {
        _error->DumpErrors();
        MsgDlg(_("Sorry, There's error, I will Exit!"), 1, 1);
        return;
      }
      printf(_("Construct main nodes... "));

      if (nodes)  free(nodes);
      nodes=alloc_nodes(maxID);

      MainPkgRecords=new pkgRecords(*MainPkgCache);
      if (base_pkg_count){
		pkgCache::PkgIterator pi=MainPkgCache->PkgBegin();
        int abc=0;
        for (;pi.end()==false;pi++){
        	pkgCache::VerIterator vi=pi.VersionList();
         	pkgDepCache::StateCache &State=(*MainPkgDepCache)[pi];
         	if	(!State.CandidateVer){
         	    struct package_node * node;
                node = alloc_structure_stretchable(struct package_node,strlen(pi.Name()));
                strcpy(node->Name,pi.Name());
                node->relations=new string();
         	    nodes[pi->ID]=node;
         		continue;
         		}
         	char *p;
         	if  (vi.end())  continue;
         	p=(char *)vi.PriorityType();
         	if  (!p)
         	{
                printf("name=%s\n", pi.Name());
                continue;
            }
			sprintf(namestring,"%.200s/%.256s/%.256s/%.256s",
                    package_s[pi->CurrentState],
                    *p==0?_("No Priority"):p,
    	    		pi.Section()==0?_("No Section"):pi.Section(),
    	    		pi.Name());
            collapse_path(namestring);
            recurse_populate_by_path(topleft_node,namestring,&pi);
            }
      printf(_("Done.\n"));
      } else
         choke(_("No packages available?"));
   }
}

static void recurse_free_nodes(struct package_node *from_node){
   struct list_head *lp,*nextlp;
   struct package_node *node;

   for (lp = &from_node->up_down;lp != &from_node->up_down;lp = nextlp){
      nextlp = lp->next;
      node = list_entry(lp,struct package_node,up_down);
      if (node->right)
         recurse_free_nodes(node->right);
      free_structure(node);
   }
}

void free_main_nodes(){
   static GtkCTree *ctree = NULL;

   if (!ctree)
      ctree = GTK_CTREE(lookup_widget(main_window,"package_tree"));

   gtk_ctree_remove_node(ctree,topleft_node->gtk_ctree_node);	/* Recursive zap */
   /* (Callbacks to unselect stuff, etc., may happen here) */

   recurse_free_nodes(topleft_node);
   free_structure(topleft_node);
   topleft_node = NULL;

   INIT_LIST_HEAD(&dependencies_input_anchor);
   INIT_LIST_HEAD(&dependencies_output_anchor);
   INIT_LIST_HEAD(&selected_packages_anchor);
   INIT_LIST_HEAD(&dr_selected_packages_anchor);
}

/* populate_package_tree_nodes
 *
 * Populate a package columned tree (two dimensional linked list),
 * using "search_expression" to filter out unwanted packages.
 * Assumes the lists have been emptied or freshly initialized before
 * calling.
 */
void populate_package_tree_nodes(GtkCTree *ctree){

   gtk_clist_freeze(GTK_CLIST(ctree));
   set_progress(main_window,0);

   if (topleft_node)
      gtk_ctree_remove_node(ctree,topleft_node->gtk_ctree_node);	/* Recursive zap */

   construct_main_nodes();

#if 0
   DEBUG_CODE(fprintf(stderr,"Sorting...\n"));
   recurse_sort_with_compare_func(topleft_node,
#endif

   global_package_displayed_count = 0;

   global_last_match = NULL;

   if (*global_search_expression){
      recurse_gtk_populate(topleft_node,ctree,NULL,TRUE /* expand to leaves */);
   } else {
      recurse_gtk_populate(topleft_node,ctree,NULL,FALSE /* don't expand to leaves */);
      gtk_ctree_expand_to_depth(ctree,topleft_node->gtk_ctree_node,2);
   }

   set_progress(main_window,0.9);

   sprintf(tempstring,_("%u package%s."),global_package_displayed_count,(global_package_displayed_count == 1) ? "" : "s");
   set_statusbar(main_window,_(tempstring));

   gtk_clist_thaw(GTK_CLIST(ctree));
}

static void populate_package_tree(GtkWidget *main_window){
   static GtkWidget *ctree = NULL;

   if (!ctree)
      ctree = lookup_widget(main_window,"package_tree");

   sync_display();
   gtk_widget_hide(ctree); /* Faster updating */

   gtk_ctree_set_expander_style(GTK_CTREE(ctree),GTK_CTREE_EXPANDER_SQUARE);

   gtk_ctree_set_line_style(GTK_CTREE(ctree),GTK_CTREE_LINES_SOLID);

   gtk_ctree_set_indent(GTK_CTREE(ctree),12);
   gtk_clist_set_column_auto_resize(GTK_CLIST(ctree),CTREE_FIELDS - 1,TRUE);
//   gtk_clist_set_auto_sort(GTK_CLIST(ctree),FALSE);

   /* On first call, form a two dimensional node structure independent from GTK,
    * and then regardless of first call or not, rescan to populate GTK ctree.
    */
   resolve_initial_dependencies = 0;
   populate_package_tree_nodes(GTK_CTREE(ctree));
//   gtk_clist_set_auto_sort(GTK_CLIST(ctree),TRUE);


#if 0
     gtk_ctree_sort_recursive(GTK_CTREE(ctree),gtk_main_node);
#endif

   gtk_widget_show(ctree);
// clear up package_information text  added by vicky_
	show_package_information(main_window,NULL, 1); 
	show_package_information(dependency_window,NULL, 1); 

//   if (resolve_initial_dependencies)
//      check_dependencies_and_interactively_resolve(&dependencies_input_anchor,dp_none);
}

/*
 * Do package search
 */
void populate_package_tree_search(char *search_expression,
                                           gboolean search_case_sensitive,
                                           int search_where_mask,
                                           int search_packages_mask){
   static char match_search_expression[132+1];

   if (*search_expression)
      sprintf(match_search_expression,"*%.128s*",search_expression);
   else
      *match_search_expression = '\0';

   global_search_expression = match_search_expression;
   global_search_case_sensitive = search_case_sensitive;
   global_search_where_mask = search_where_mask;
   global_search_packages_mask = search_packages_mask;

   populate_package_tree(main_window);
}

/*
 * Do package search
 */
void populate_package_tree_toggle_show_parameters(int toggle_mask){

   global_show_mask^= toggle_mask;

   populate_package_tree(main_window);
}

static struct package_node *recurse_find_next(struct package_node *from_node,GtkCTree *ctree,struct package_node **last_match,char *find_expression,gboolean case_sensitive){

   struct list_head *lp;
   struct package_node *node;
   static gchar *field;
   static int recurse = 0;
   static int i;

   recurse++;

   lp = &from_node->up_down;
   do {
      node = list_entry(lp,struct package_node,up_down);

      if (!*last_match){
         if (node->gtk_ctree_node){
            if (node->pkg)
               if (match(find_expression,node->Name,case_sensitive))
                  return node;
//            if (match(find_expression,node->Name,case_sensitive))
//               return node;
            for (i = 1;i <= CTREE_FIELDS;i++){
               if (!gtk_ctree_node_get_text(ctree,node->gtk_ctree_node,i,&field))
                  break;
               if (match(find_expression,field,case_sensitive))
                  return node;
            }
         }
      }

      if (*last_match == node)
         *last_match = NULL;

      if (node->right)
         if ((node = recurse_find_next(node->right,ctree,last_match,find_expression,case_sensitive))){
            recurse--;
            return node;
         }

      lp = lp->next;
   } while (lp != &from_node->up_down);
   recurse--;

   return NULL;
}

/*
 * Do "find next"
 */
void find_next(char *find_expression,gboolean case_sensitive){
   struct package_node *node;
   struct package_node *leftnode;
   struct package_node *last_match,*save_last_match;
   static GtkCTree *ctree = NULL;
   int wrapped = 0, cnt, i;
   static char match_find_expression[132+1];

   if (!ctree)
      ctree = GTK_CTREE(lookup_widget(main_window,"package_tree"));

   if (*find_expression)
      sprintf(match_find_expression,"*%.128s*",find_expression);
   else {
//      *match_find_expression = '\0';
        gtk_widget_grab_focus(lookup_widget(main_window,"find_textentry"));
        return;
        }

   save_last_match = global_last_match;

   if (global_last_match){
      if (gtk_ctree_node_get_selectable(ctree,global_last_match->gtk_ctree_node))
         gtk_ctree_unselect(ctree,global_last_match->gtk_ctree_node);
      else
         update_node_background_color(ctree,global_last_match);
   }

   global_last_match = save_last_match;

#if 1
   for (;;){
      last_match = global_last_match;
      node = recurse_find_next(topleft_node,ctree,&global_last_match,match_find_expression,case_sensitive);
      if (!node)
         if (last_match){
            global_last_match = NULL;
            wrapped++;
            continue;
         }
      break;
   }

#else
    if  (global_last_match) {
        cnt=1+global_last_match->index;
        if (cnt==maxID) cnt=0;
        }
    else    cnt=0;
    i=cnt;
    while(1)    {
        node=nodes[i];
        if (node->gtk_ctree_node)
            if (node->pkg)
                if (!match(find_expression,node->Name,case_sensitive))
                    break;
        i++;
        if  (i >= maxID){
            wrapped=1;
            i=0;
            }
        if  (i == cnt)   {
            node=NULL;
            break;
            }
        }
#endif


   if (node){
      global_last_match = node;
      DEBUG_CODE(fprintf(stderr,"Find match: %p (%s)\n",node,node->name));
//      gtk_ctree_node_moveto(ctree,node->gtk_ctree_node,-1,0.5,0.5);
      if (gtk_ctree_node_get_selectable(ctree,node->gtk_ctree_node))
         gtk_ctree_select(ctree,node->gtk_ctree_node);
      else
         gtk_ctree_node_set_background(ctree,node->gtk_ctree_node,find_background_color);
      for (leftnode = node;leftnode;leftnode = leftnode->left)
         if (leftnode->gtk_ctree_node)
            gtk_ctree_expand(ctree,leftnode->gtk_ctree_node);
      gtk_ctree_node_moveto(ctree,node->gtk_ctree_node,-1,0.5,0.5);
      if (wrapped)
         set_statusbar(main_window,_("Search wrapped.  Next match selected."));
      else
         set_statusbar(main_window,_("Next match selected."));
   } else {
      DEBUG_CODE(fprintf(stderr,"No match.\n"));
      set_statusbar(main_window, _("No match."));
   }
}

static void update_node_status_text(struct package_node *node){
   static GtkCTree *ctree = NULL;

   if (!ctree)
      ctree = GTK_CTREE(lookup_widget(main_window,"package_tree"));

   // Package updated may not be displayed (eg: in dependency resolving),
   // so next condition being false is common.

   if (node->gtk_ctree_node){
      gtk_ctree_node_set_text(ctree,node->gtk_ctree_node,CTREE_STATUS, get_selection_string(node));
      update_node_background_color(ctree,node);
   }

}

static void update_dependency_node_status_text(struct package_node *node){
   static GtkCList *clist = NULL;

   if (!clist)
      clist = GTK_CLIST(lookup_widget(dependency_window,"dependency_clist"));

   gtk_clist_set_text(clist,node->gtk_clist_row,CLIST_NEW_STATUS,get_selection_string(node));
   update_dependency_node_background_color(clist,node);

}


/* free_dependency_nodes_and_optionally_copy_selections_out_to_main_nodes
 *
 * It slices, it dices, and it even repopulates the dependency_input
 * list with the base nodes for the nodes it frees, even though the title
 * doesn't say so.  Sorry, I thought it was a bit long already...
 */
void free_dependency_nodes_and_optionally_copy_selections_out_to_main_nodes(gboolean copy_selections){

   struct list_head *lp,*nextlp;
   struct package_node *node;


   INIT_LIST_HEAD(&dependencies_input_anchor);

   for (lp = dependencies_output_anchor.next;lp != &dependencies_output_anchor;lp = nextlp){
      nextlp = lp->next;
      node = list_entry(lp,struct package_node,up_down);

      if (node->gtk_ctree_node)
         fprintf(stderr,_("While freeing dependency resolving nodes: Oops, GtkCTree widget node on dependency list!\n"));

      if (node->dep_up){
         // Untie us as the first reference
         nodes[node->pkg->ID] = node->dep_up;
         list_add(&node->dep_up->dependency_list,&dependencies_input_anchor);
         if (copy_selections){
            node->dep_up->Selected = node->Selected;
            update_node_status_text(node->dep_up);
         }

      (*node->relations)="";    //can't free it, it was shared with node->dep_up
      free_structure(node);
      } else
         fprintf(stderr,_("While freeing dependency resolving nodes: No dep_up on this record!\n"));

   }

   should_apply=1;
   gtk_widget_show(apply_button);
   gtk_widget_hide(apply_button0);

   // Empty list
   INIT_LIST_HEAD(&dependencies_output_anchor);
}

void close_dependency_window_and_optionally_copy_selections(gboolean copy_selections){
   static GtkCList *dependency_clist = NULL;

   if (!dependency_clist)
      dependency_clist = GTK_CLIST(lookup_widget(dependency_window,"dependency_clist"));

   gtk_widget_hide(dependency_window);

   if (copy_selections);
      free_dependency_nodes_and_optionally_copy_selections_out_to_main_nodes(copy_selections);
}

static void interactively_resolve_dependencies(struct list_head *anchor){
   struct list_head *lp;
   struct package_node *node;
   GtkNotebook *notebook;
   static GtkCList *dependency_clist = NULL;
   static gchar *clist_fields[CLIST_FIELDS];
   static char package_name_new[256+8+1];
   char *s;
   int row, cnt=0;

   if (!dependency_clist)
      dependency_clist = GTK_CLIST(lookup_widget(dependency_window,"dependency_clist"));

   gtk_clist_freeze(dependency_clist);
   gtk_clist_clear(dependency_clist);

   INIT_LIST_HEAD(&dr_selected_packages_anchor);

   for (lp = anchor->next;lp != anchor;lp = lp->next){

      node = list_entry(lp,struct package_node,up_down);
//yema
      cnt++;
      if (!node->dep_up){
          fprintf(stderr, _("This node has not dep_up?name=%s\n"), node->Name);
          exit(1);
          }

      clist_fields[CLIST_OLD_STATUS] = get_selection_string(node->dep_up);
      clist_fields[CLIST_NEW_STATUS] = get_selection_string(node);
      clist_fields[CLIST_PRIORITY] = (char *)node->dep_up->PriorityStr;
      clist_fields[CLIST_SECTION] = (char *)node->dep_up->Section;

      if (node->pkg->SelectedState != pkgCache::State::Unknown){
         clist_fields[CLIST_PACKAGE_NAME] = node->dep_up->Name;                 //maybe problem
         }
      else { // New
         sprintf(package_name_new,_("%.256s (NEW)"),node->dep_up->Name);
         clist_fields[CLIST_PACKAGE_NAME] = package_name_new;
         }

      clist_fields[CLIST_DESCRIPTION] = nullstring;

      if (node->pkg->CurrentState==pkgCache::State::Installed)
            clist_fields[CLIST_DESCRIPTION] = (char *)node->dep_up->CurrentShortDesc;
      else
            clist_fields[CLIST_DESCRIPTION] = (char *)node->dep_up->AvailableShortDesc;

      s = strchr(clist_fields[CTREE_DESCRIPTION],'\n');
      if (s)
         *s = '\0';

     row = gtk_clist_append(dependency_clist,clist_fields);
      if (s)
         *s = '\n';

      node->gtk_clist_row = row;
      gtk_clist_set_row_data(dependency_clist,row,(gpointer)node);
      update_dependency_node_background_color(dependency_clist,node);

    }


    gtk_clist_select_row(dependency_clist, 0, 0);

    gtk_clist_set_column_auto_resize(dependency_clist,CLIST_FIELDS - 1,TRUE);
    gtk_clist_thaw(dependency_clist);

    if  (cnt>1){
        notebook = (GtkNotebook *)lookup_widget(dependency_window,"dependencies_notebook");
        gtk_notebook_set_page(notebook,2);
        // Show main window in case we're just starting up and main window
        // hasn't been unhidden yet.  We want it to be under the dependency
        // window, so show it first.
        gtk_widget_show(main_window);
        gtk_widget_show(dependency_window);
        }
    else
        free_dependency_nodes_and_optionally_copy_selections_out_to_main_nodes(TRUE);
//        close_dependency_window_and_optionally_copy_selections(TRUE);
    }

void check_dependencies_and_interactively_resolve(struct list_head *dependencies_input_anchor,
                                                  enum showpriority minimum_showpriority){
//    printf("------check_dependencies_and_interactively_resolve-------\n");

    close_dependency_window_and_optionally_copy_selections(TRUE);

}

static void check_dependencies_for_selected_packages_and_interactively_resolve(enum pkgCache::State::PkgSelectedState want){

    struct list_head *lp;
    struct package_node *node;//, *up, *nod;

    INIT_LIST_HEAD(&dependencies_input_anchor);
    INIT_LIST_HEAD(&dependencies_output_anchor);

    // First, mirror selected_list into dependencies_list for first run
    for (lp = selected_packages_anchor.next;lp != &selected_packages_anchor;lp = lp->next){

        node = list_entry(lp,struct package_node,selected_list);
        struct pkgCache::PkgIterator pi=pkgCache::PkgIterator(*MainPkgCache, node->pkg);

        switch(want){
            case pkgCache::State::Install:
                Mark_Install(pi);
                break;
            case pkgCache::State::Hold:
                Mark_Keep(pi);
                break;
            case pkgCache::State::DeInstall:
                Mark_Delete(pi);
                break;
            case pkgCache::State::Purge:
                Mark_Delete(pi, true);
            default: break;
            }
        }
    interactively_resolve_dependencies(&dependencies_output_anchor);
    set_toolbar_button_state(1);
}



static void recurse_find_or_activate_changed_wants(struct package_node *from_node,GtkText *changes_textbox){

   struct list_head *lp;
   struct package_node *node;
   static int recurse = 0;

   recurse++;
   lp = &from_node->up_down;
   do {
      node = list_entry(lp,struct package_node,up_down);

      if (node->Selected){
          gtktext_append_string(changes_textbox,get_selection_string(node));
          gtktext_append_string(changes_textbox," ");
          gtktext_append_string(changes_textbox,node->Name);
          gtktext_append_string(changes_textbox,"\n");
          }

      if (node->right)
         recurse_find_or_activate_changed_wants(node->right,changes_textbox);

      lp = lp->next;
   } while (lp != &from_node->up_down);
   recurse--;

}

void populate_changes_list(){
   GtkText *changes_textbox;

   changes_textbox = (GtkText *)lookup_widget(apply_changes_window,"changes_textbox");

   gtktext_clear(changes_textbox);
   recurse_find_or_activate_changed_wants(topleft_node,changes_textbox);
}

int gquit=0, gclean=0;
void apply_changes_and_possibly_quit(){

//    MsgDlg(_("Wait..."), 0, 0);
    gtk_widget_hide(filter_window);
    gtk_widget_hide(apply_changes_window);
    gtk_widget_hide(main_window);
    sync_display();

    apply_change(gclean);

    should_apply=0;
    gtk_widget_show(apply_button0);
    gtk_widget_hide(apply_button);

    free_main_nodes();
    base_pkg_count = 0;
    if (gquit)
        gtk_exit(0);
    else {
       populate_package_tree_nodes(GTK_CTREE(lookup_widget(main_window,"package_tree")));
//       gtk_widget_hide(msg_dlg);
// clear up package infomation,  added by vicky
	 show_package_information(main_window,NULL, 1);  
         show_package_information(dependency_window,NULL, 1); 

       gtk_widget_show(main_window);
       }
    }




void update_packages(){

    free_main_nodes();
    base_pkg_count=0;	
//    MsgDlg(_("Wait..."), 0, 0);
    gtk_widget_hide(filter_window);
    gtk_widget_hide(main_window);
    sync_display();
    DoUpdate();
//    gtk_widget_hide(msg_dlg);
	show_package_information(main_window,NULL, 1);
	show_package_information(dependency_window,NULL, 1);
    gtk_widget_show(main_window);
    gray_buttons(1);
    gray_buttons(0);
    }



void gray_buttons(int main)
{
    if  (main)
    {
        gtk_widget_show(install_button0);
        gtk_widget_hide(install_button);
        gtk_widget_show(remove_button0);
        gtk_widget_hide(remove_button);
        gtk_widget_show(purge_button0);
        gtk_widget_hide(purge_button);
        gtk_widget_show(hold_button0);
        gtk_widget_hide(hold_button);
        if  (should_apply)  gtk_widget_show(apply_button), gtk_widget_hide(apply_button0);
        else                gtk_widget_show(apply_button0), gtk_widget_hide(apply_button);
    }
    else
    {
        gtk_widget_show(dr_install_button0);
        gtk_widget_hide(dr_install_button);
        gtk_widget_show(dr_remove_button0);
        gtk_widget_hide(dr_remove_button);
        gtk_widget_show(dr_purge_button0);
        gtk_widget_hide(dr_purge_button);
        gtk_widget_show(dr_hold_button0);
        gtk_widget_hide(dr_hold_button);
    }
}


void select_main_toolbar_button(int install, int remove, int purge, int hold, int main, int flag)
{
    if  (main)
    {
        if  (install)   gtk_widget_show(install_button), gtk_widget_hide(install_button0);
        else if (flag)  gtk_widget_show(install_button0),gtk_widget_hide(install_button);

        if  (remove)    gtk_widget_show(remove_button),  gtk_widget_hide(remove_button0);
        else if (flag)  gtk_widget_show(remove_button0), gtk_widget_hide(remove_button);

        if  (purge)     gtk_widget_show(purge_button),   gtk_widget_hide(purge_button0);
        else if (flag)  gtk_widget_show(purge_button0),  gtk_widget_hide(purge_button);

        if  (hold)      gtk_widget_show(hold_button),   gtk_widget_hide(hold_button0);
        else if (flag)  gtk_widget_show(hold_button0),   gtk_widget_hide(hold_button);
    }
    else
    {
        if  (install)   gtk_widget_show(dr_install_button), gtk_widget_hide(dr_install_button0);
        else if (flag)  gtk_widget_show(dr_install_button0),gtk_widget_hide(dr_install_button);

        if  (remove)    gtk_widget_show(dr_remove_button),  gtk_widget_hide(dr_remove_button0);
        else if (flag)  gtk_widget_show(dr_remove_button0), gtk_widget_hide(dr_remove_button);

        if  (purge)     gtk_widget_show(dr_purge_button),   gtk_widget_hide(dr_purge_button0);
        else if (flag)  gtk_widget_show(dr_purge_button0),  gtk_widget_hide(dr_purge_button);

        if  (hold)      gtk_widget_show(dr_hold_button),gtk_widget_hide(dr_hold_button0);
        else if (flag)  gtk_widget_show(dr_hold_button0),gtk_widget_hide(dr_hold_button);
    }

}


void set_toolbar_button_state_by_node(struct package_node *node, int main, int flag)
{

    switch((int)node->Selected){
        case pkgCache::State::Unknown:
            switch(node->pkg->CurrentState){
                case    pkgCache::State::Installed:
                    select_main_toolbar_button(0, 1, 1, 1, main, flag);
                    break;
                case    pkgCache::State::ConfigFiles:
                    select_main_toolbar_button(1, 0, 1, 1, main, flag);
                    break;
                case    pkgCache::State::NotInstalled:
                    select_main_toolbar_button(1, 0, 0, 1, main, flag);
                default: break;
                }
            break;
        case pkgCache::State::Install:
            select_main_toolbar_button(1, 1, 1, 0, main, flag);
            break;
        case pkgCache::State::Purge:
            select_main_toolbar_button(1, 1, 1, 0, main, flag);
            break;
        case pkgCache::State::DeInstall:
            select_main_toolbar_button(1, 1, 1, 0, main, flag);
            break;
        case pkgCache::State::Hold:
            select_main_toolbar_button(1, 1, 1, 0, main, flag);
        default:
            break;
        }

    }

void set_toolbar_button_state(int main){

    int i=1;
    struct list_head *lp;
    struct list_head *head;
    struct package_node *node;

    if  (main)  head=&selected_packages_anchor;
    else        head=&dr_selected_packages_anchor;

    for (lp = head->next;lp != head;lp = lp->next)
    {
        node = list_entry(lp,struct package_node,selected_list);
       	set_toolbar_button_state_by_node(node, main, i);
       	i=0;
    }

}


void link_selected_package(GtkWidget *window,struct package_node *node){
	static GtkCTree *ctree = NULL;
    static GtkCList *dependency_clist = NULL;

    if (!dependency_clist)
        dependency_clist = GTK_CLIST(lookup_widget(dependency_window,"dependency_clist"));
	if (!ctree)
		ctree = GTK_CTREE(lookup_widget(main_window,"package_tree"));

	if (!node->selected_list.next && !node->selected_list.prev){
		if (window == main_window)
			list_add_tail(&node->selected_list,&selected_packages_anchor);
		else
			list_add_tail(&node->selected_list,&dr_selected_packages_anchor);
		}
	show_package_information(window,node, 0);
	if (window == main_window){
		if (global_last_match)
			update_node_background_color(ctree,global_last_match);
	    global_last_match = node;
       	set_toolbar_button_state(1);
	    }
    else {
        update_dependency_node_background_color(dependency_clist,node);
       	set_toolbar_button_state(0);
        }
}

void unlink_selected_package(struct package_node *node, int main){
   static GtkCTree *ctree = NULL;
   if (!ctree)
      ctree = GTK_CTREE(lookup_widget(main_window,"package_tree"));

   if (node->selected_list.next || node->selected_list.prev)
      list_del(&node->selected_list);
   else
      fprintf(stderr,_("ERROR: double remove!\n"));

   node->selected_list.next =
   node->selected_list.prev = NULL;
   set_statusbar(main_window,"");
   if (global_last_match == node)
      global_last_match = NULL;

    gray_buttons(main);
   	set_toolbar_button_state(main);

}

void mark_selected_packages(enum pkgCache::State::PkgSelectedState want){
   if (!list_empty(&selected_packages_anchor))
      check_dependencies_for_selected_packages_and_interactively_resolve(want);
}

void mark_selected_dependency_resolver_packages(enum pkgCache::State::PkgSelectedState want){
    struct list_head *lp;
    static struct package_node *node;

    for (lp = dr_selected_packages_anchor.next;lp != &dr_selected_packages_anchor;lp = lp->next){
        node = list_entry(lp,struct package_node,selected_list);
        if  (node->pkg->CurrentState==selection_to_state(node, want))
            node->Selected=pkgCache::State::Unknown;
        else
            node->Selected=want;
        update_dependency_node_status_text(node);
        }
    set_toolbar_button_state(0);
    }

void show_package_information(GtkWidget *window,struct package_node *node, int clear){


   GtkText *description_textbox;
   GtkText *details_textbox;
   GtkText *dependencies_textbox;

   description_textbox = (GtkText *)lookup_widget(window,"description_textbox");
   details_textbox = (GtkText *)lookup_widget(window,"details_textbox");
   dependencies_textbox = (GtkText *)lookup_widget(window,"dependencies_textbox");


   gtk_text_freeze(description_textbox);
   gtk_text_freeze(details_textbox);
   gtk_text_freeze(dependencies_textbox);

   gtk_text_set_word_wrap(description_textbox,TRUE);

   gtktext_clear(description_textbox);
   gtktext_clear(details_textbox);
   gtktext_clear(dependencies_textbox);

   if   (clear) goto goback;

	{ // Description

	char *description = nullstring;
	char *s,*line;

	if  (node->dep_up)
	    node=node->dep_up;
	
    description=node->AvailableLongDesc;
    line = NULL;
    for (s = description;*s;s++){
    	if (*s == '\n'){
        	if (line)
            	gtktext_append_buffer(description_textbox,line,s - line + 1);
            if (s[1] == ' '){
            	s++;
            	if (s[1] == '.')
                	s++;
            	}
            line = s + 1;
         	}
		}
	if ((line) && (line != s + 1))	// No trailing newline
    	gtktext_append_string(description_textbox,line);

	while ((*description) && (*description != '\n'))
    	description++;
	if (*description)
    	description++;
	} // Description

	{ // Details

      gtktext_append_string(details_textbox,_("Name:  "));
      gtktext_append_string(details_textbox,(char *)node->Name);
      gtktext_append_string(details_textbox,_("\nSection:  "));
      gtktext_append_string(details_textbox,(char *)node->Section);
      gtktext_append_string(details_textbox,_("\nStatus:  "));
      gtktext_append_string(details_textbox,package_s[node->pkg->CurrentState]);
      gtktext_append_string(details_textbox,_("\nInstalled version:  "));
      gtktext_append_string(details_textbox, (char *)node->InstalledVersionStr);
      gtktext_append_string(details_textbox,_("\nNewest available version:  "));
      gtktext_append_string(details_textbox, (char *)node->AvailableVersionStr);
      gtktext_append_string(details_textbox,_("\nInstalled size:  "));
      gtktext_append_string(details_textbox, auto_unit(byte_units,node->pkgInstalledSize));
      gtktext_append_string(details_textbox,"\n");
   } // Details

   { // Dependencies
	if	(window==main_window){
   		pkgCache::DepIterator	dpi;
   		
   		pkgCache::VerIterator	vi;
   		pkgCache::PkgIterator	pi;
   		pi=pkgCache::PkgIterator(*MainPkgCache, node->pkg);
   		
        if  (node->pkg->CurrentState==pkgCache::State::Installed)
            vi=pi.CurrentVer();
        else
            vi=pi.VersionList();
   		if  (vi.end()==false) {
       		for	(dpi=vi.DependsList();dpi.end()==false;)	{
       		    pkgCache::DepIterator start, end;
       		    dpi.GlobOr(start, end);
  		    	gtktext_append_string(dependencies_textbox, (char *)start.ParentPkg().Name());
    		    gtktext_append_string(dependencies_textbox," ");
   	    		gtktext_append_string(dependencies_textbox, (char *)start.DepType());
    	    	gtktext_append_string(dependencies_textbox," ");
       		    char *s=" ";
       		    while  (1)  {
    			    gtktext_append_string(dependencies_textbox, (char *)start.TargetPkg().Name());
	    	    	if  (start.TargetVer()){
    	    	    	gtktext_append_string(dependencies_textbox, " (");
    		        	gtktext_append_string(dependencies_textbox, (char *)start.CompType());
	    		        gtktext_append_string(dependencies_textbox, (char *)start.TargetVer());
    		    	    gtktext_append_string(dependencies_textbox, ")");
    			        }
    			    if  (start!=end)    {
    			        start++;
    	    		    s=", ";
    	    		    if  (start==end) s=_(" or ");
    	    		    gtktext_append_string(dependencies_textbox,s);
    			        }
    			    else    break;
    			    }
                gtktext_append_string(dependencies_textbox,"\n");
 		    	}
		    for  (pkgCache::PrvIterator	pvi=vi.ProvidesList();pvi.end()==false;pvi++)	{
	    	    gtktext_append_string(dependencies_textbox, (char *)pi.Name());
    		    gtktext_append_string(dependencies_textbox,_(" provides "));
   	    		gtktext_append_string(dependencies_textbox, (char *)pvi.Name());
    	    	gtktext_append_string(dependencies_textbox," ");
                gtktext_append_string(dependencies_textbox,"\n");
        	    }
 			}
		}else{
        gtktext_append_string(dependencies_textbox,(char *)node->relations->c_str());		
		}
		
		
	} // Dependencies
	
goback:
   gtk_text_thaw(description_textbox);
   gtk_text_thaw(details_textbox);
   gtk_text_thaw(dependencies_textbox);
}

void select_want(struct package_node *node, void (*mark)(enum pkgCache::State::PkgSelectedState)){

    switch((int)node->Selected){
        case pkgCache::State::Unknown:
        case pkgCache::State::Hold:
            if  (node->pkg->CurrentState==pkgCache::State::Installed)
                mark(pkgCache::State::Purge);
            else
                mark(pkgCache::State::Install);
            break;
        case pkgCache::State::Install:
            mark(pkgCache::State::Purge);
            break;
        case pkgCache::State::Purge:
        case pkgCache::State::DeInstall:
                mark(pkgCache::State::Install);
        default:
            break;
        }
    }

void package_tree_row_double_clicked_event(struct package_node *node){
   /* Should we install or deinstall? */
    select_want(node, mark_selected_packages);
    }

void package_list_row_double_clicked_event(struct package_node *node){
   /* Should we install or deinstall? */
    select_want(node, mark_selected_dependency_resolver_packages);
    }

void MsgDlg(char *str, int ok, int exit){

    static GtkWidget *msg_ok=NULL;
    static GtkWidget *msg=NULL;

    if (!msg)   msg=lookup_widget(msg_dlg, "msg");

    if (!msg_ok)msg_ok=lookup_widget(msg_dlg, "msg_ok");

    msg_dlg_exit=exit;

    if  (ok)    gtk_widget_show(msg_ok);
    else        gtk_widget_hide(msg_ok);

    gtk_label_set_text(GTK_LABEL(msg), str);

    gtk_widget_show(msg_dlg);
    }
