/*
 * Storm Package Manager
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>

#include "../libstormpkg/libstormpkg_app_include.h"

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "stormpkg.h"
#include "sources.h"
#include "match.h"
#include "list.h"
#include "main.h"

/* Defines */

#define APT_SOURCES_LIST_PATH "/etc/apt/sources.list"
#define MAX_LINE_LENGTH 2048

/* Globals */

static LIST_HEAD(sources_anchor);
static struct source_line *selected_source_line;

/* Functions */

static int contains_whitespace(char *s){
   for (;*s;s++)
//    if ((*s == ' ') || (*s == '\t'))
      if (((*s == ' ') || (*s == '\t'))&&((*(s+1) != ' ') && (*(s+1) != '\t') && (*(s+1) != 0)))
         return 1;
   return 0;
}

static char *chomp(char *from){
   char *s;

   for (s = from;*s;s++){
      if ((*s == '\n') || (*s == '\r')){
         *s = '\0';
         break;
      }
   }
   return from;
}

static void collapse_repeated_spaces(char *from){
   char *s = from;
   char *d = from;
   char lc = '\0';

   for (;*s;s++){
      if (((*s != ' ') && (*s != '\t')) ||
          ((lc != ' ') && (lc != '\t'))){
         if (*s != '\t')
            *d++ = *s;
         else
            *d++ = ' ';
      }
      lc = *s;
   }
   *d = '\0';
   while (d > from){
      d--;
      if (*d == ' ')
         *d = '\0';
      else
         break;
   }
}

static char *strdup_possibly_quoted_word_and_return_next(char *from,char **to){
   char *s;
   char c;

   if (*from != '"'){
      for (s = from;*s;s++)
         if ((*s == ' ') || (*s == '\t') || (*s == '\n') || (*s == '\r'))
            break;
   } else {
      for (s = ++from;*s;s++)
         if (*s == '"')
            break;
   }

   c = *s;
   if (c)
      *s = '\0';
   *to = strdup(from);
   if (c)
      *s++ = c;

   while ((*s == ' ') || (*s == '\t') || (*s == '\n') || (*s == '\r'))
      s++;
   return s;
}

#define NUM_SOURCE_STATUSES 3

static char *source_statuses[NUM_SOURCE_STATUSES] = {
   N_("Disabled"),
   N_("Enabled"),
   N_("Comment")
};

static char *get_source_status_string(int status){
   
   if (status <= NUM_SOURCE_STATUSES){
      return _(source_statuses[status]);
   }
   return _("Unknown");
}

static void free_sources_line_data(struct source_line *sp){
   DEBUG_CODE(fprintf(stderr,"type:%p uri:%p dist:%p sect:%p com:%p (rec:%p)\n",sp->type,sp->source_uri,sp->distribution,sp->sections,sp->comment,sp));
   free(sp->type);
   free(sp->source_uri);
   free(sp->distribution);
   free(sp->sections);
   free(sp->comment);
   free(sp);
}

static void clear_sources_list_data(){
   struct list_head *lp,*nextlp;
   struct source_line *sp;
   static GtkCList *sources_clist = NULL;

   if (!sources_clist)
      sources_clist = GTK_CLIST(lookup_widget(sources_window,"sources_clist"));

   gtk_clist_freeze(sources_clist);
   gtk_clist_clear(sources_clist);
   gtk_clist_thaw(sources_clist);

   for (lp = sources_anchor.next;lp != &sources_anchor;lp = nextlp){
      nextlp = lp->next;
      sp = list_entry(lp,struct source_line,list);
      free_sources_line_data(sp);
   }

   INIT_LIST_HEAD(&sources_anchor);
   selected_source_line = NULL;
}

static void ec_write_string(FILE *fp,char *s){
   if (fputs(s,fp) != EOF)
      return;
   perror("fputs()");
   choke(_("Cannot write to file."));
}

static void ec_write_autoquoted_string(FILE *fp,char *s){
   if (contains_whitespace(s)){
      if (fprintf(fp,"\"%s\"",s) != EOF)
         return;
      perror("fputs()");
      choke(_("Cannot write to file."));
   } else
      ec_write_string(fp,s);
}

static void write_sources_list_data(){
   struct list_head *lp;
   struct source_line *sp;
   static GtkCList *sources_clist = NULL;
   FILE *fp;

   if (!sources_clist)
      sources_clist = GTK_CLIST(lookup_widget(sources_window,"sources_clist"));

   gtk_clist_freeze(sources_clist);
   gtk_clist_clear(sources_clist);
   gtk_clist_thaw(sources_clist);

   fp = fopen(APT_SOURCES_LIST_PATH ".stormpkg","w");
   if (!fp){
      perror("fopen()");
      fprintf(stderr,"%s \"%s\" %s\n", _("Unable to open"), _("for writing."),APT_SOURCES_LIST_PATH);
      clear_sources_list_data();
      return;
   }

   clearerr(fp);

   for (lp = sources_anchor.next;lp != &sources_anchor;lp = lp->next){
      sp = list_entry(lp,struct source_line,list);

      if (sp->status == SOURCE_COMMENT){
         ec_write_string(fp,sp->comment);
      } else {
         if (sp->status == SOURCE_DISABLED)
            ec_write_string(fp,"#");
         ec_write_autoquoted_string(fp,sp->type);
         ec_write_string(fp," ");
         ec_write_autoquoted_string(fp,sp->source_uri);
         ec_write_string(fp," ");
         ec_write_autoquoted_string(fp,sp->distribution);
         if (*sp->sections){
            ec_write_string(fp," ");
            ec_write_string(fp,sp->sections);
         }
         ec_write_string(fp,"\n");
      }
   }

   if (rename(APT_SOURCES_LIST_PATH ".stormpkg",APT_SOURCES_LIST_PATH) != 0)
      perror("rename()");

   fclose(fp);
}

static void parse_sources_list_file(){
   static char linebuf[MAX_LINE_LENGTH+1];
   struct source_line *sp;
   FILE *fp;

   if (!list_empty(&sources_anchor))
      clear_sources_list_data();

   fp = fopen(APT_SOURCES_LIST_PATH,"r");
   if (!fp){
      perror("fopen()");
      fprintf(stderr,"Unable to open \"%s\" for reading.\n",APT_SOURCES_LIST_PATH);
      return;
   }

   clearerr(fp);

   {
      char *s;
      enum source_status status;

      while (fgets(linebuf,MAX_LINE_LENGTH,fp)){
         sp = alloc_structure(struct source_line);
         list_add_tail(&sp->list,&sources_anchor);

         status = SOURCE_ENABLED;

         for (s = linebuf;*s;s++){
            if ((*s == ' ') || (*s == '\t') || (*s == '\n') || (*s == '\r'))
               continue;
            if (*s == '#'){
               status = SOURCE_DISABLED;
               continue;
            }
            break;
         }
         if (*s){
            if (strncmp("deb",s,3) == 0){
               s = strdup_possibly_quoted_word_and_return_next(s,&sp->type);
               s = strdup_possibly_quoted_word_and_return_next(s,&sp->source_uri);
               s = strdup_possibly_quoted_word_and_return_next(s,&sp->distribution);
               sp->sections = chomp(strdup(s));
               collapse_repeated_spaces(sp->sections);
               
               if (*sp->distribution){
                  if (strcmp("file:/your/mirror/here/debian",sp->source_uri) != 0){
                     sp->status = status;
                     continue;
                  }
               }
            }
         }

         /* Fall through: Handle this as a comment */
         sp->status = SOURCE_COMMENT;
         sp->comment = strdup(linebuf);
      }
      if (!feof(fp)){
         if (ferror(fp)){
            perror("fgets()");
            fprintf(stderr,_("Unable to read from \"%s\".\n"), APT_SOURCES_LIST_PATH);
            return;
         }
      }
   }
}

enum {
   SOURCES_CLIST_STATUS,
   SOURCES_CLIST_TYPE,
   SOURCES_CLIST_SOURCE_URI,
   SOURCES_CLIST_DISTRIBUTION,
   SOURCES_CLIST_SECTIONS,
   SOURCES_CLIST_FIELDS /* counter */
};

static void populate_clist_fields(char *clist_fields[],struct source_line *sp){
   clist_fields[SOURCES_CLIST_STATUS] = get_source_status_string(sp->status);
   clist_fields[SOURCES_CLIST_TYPE] = sp->type;
   clist_fields[SOURCES_CLIST_SOURCE_URI] = sp->source_uri;
   clist_fields[SOURCES_CLIST_DISTRIBUTION] = sp->distribution;
   clist_fields[SOURCES_CLIST_SECTIONS] = sp->sections;
}

void interactively_edit_sources_list(){
   struct list_head *lp;
   struct source_line *sp;
   static GtkCList *sources_clist = NULL;
   static gchar *clist_fields[SOURCES_CLIST_FIELDS];
   int row;

   if (!sources_clist)
      sources_clist = GTK_CLIST(lookup_widget(sources_window,"sources_clist"));

   gtk_clist_freeze(sources_clist);
   gtk_clist_clear(sources_clist);

   parse_sources_list_file();

   for (lp = sources_anchor.next;lp != &sources_anchor;lp = lp->next){
      sp = list_entry(lp,struct source_line,list);

      if (sp->status == SOURCE_COMMENT)
         continue;

      populate_clist_fields(clist_fields,sp);

      row = gtk_clist_append(sources_clist,clist_fields);
      sp->gtk_clist_row = row;
      gtk_clist_set_row_data(sources_clist,row,(gpointer)sp);
   }

   gtk_clist_select_row(sources_clist,/* row */ 0,/* column */ 0);
   gtk_clist_set_column_auto_resize(sources_clist,SOURCES_CLIST_FIELDS - 1,TRUE);

   gtk_clist_thaw(sources_clist);
   gtk_widget_show(sources_window);
}

void ok_edit_sources_list(){
   write_sources_list_data();
   selected_source_line = NULL;
   gtk_widget_hide(sources_window);
}

void cancel_edit_sources_list(){
   clear_sources_list_data();
   selected_source_line = NULL;
   gtk_widget_hide(sources_window);
}

static void show_source_line_details(struct source_line *sp){
   static GtkCheckButton *enabled_checkbutton = NULL;
   static GtkEntry *type_entry = NULL;
   static GtkEntry *uri_entry = NULL;
   static GtkEntry *distribution_entry = NULL;
   static GtkCheckButton *main_section_checkbutton = NULL;
   static GtkCheckButton *contrib_section_checkbutton = NULL;
   static GtkCheckButton *nonfree_section_checkbutton = NULL;
   static GtkCheckButton *nonus_section_checkbutton = NULL;
   static GtkEntry *other_sections_entry = NULL;

   if (!enabled_checkbutton)
      enabled_checkbutton = GTK_CHECK_BUTTON(lookup_widget(sources_window,"enabled_checkbutton"));

   if (!type_entry)
      type_entry = GTK_ENTRY(lookup_widget(sources_window,"type_entry"));

   if (!uri_entry)
      uri_entry = GTK_ENTRY(lookup_widget(sources_window,"uri_entry"));

   if (!distribution_entry)
      distribution_entry = GTK_ENTRY(lookup_widget(sources_window,"distribution_entry"));

   if (!main_section_checkbutton)
      main_section_checkbutton = GTK_CHECK_BUTTON(lookup_widget(sources_window,"main_section_checkbutton"));

   if (!contrib_section_checkbutton)
      contrib_section_checkbutton = GTK_CHECK_BUTTON(lookup_widget(sources_window,"contrib_section_checkbutton"));

   if (!nonfree_section_checkbutton)
      nonfree_section_checkbutton = GTK_CHECK_BUTTON(lookup_widget(sources_window,"nonfree_section_checkbutton"));

   if (!nonus_section_checkbutton)
      nonus_section_checkbutton = GTK_CHECK_BUTTON(lookup_widget(sources_window,"nonus_section_checkbutton"));

   if (!other_sections_entry)
      other_sections_entry = GTK_ENTRY(lookup_widget(sources_window,"other_sections_entry"));

   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enabled_checkbutton),
                                (sp->status == SOURCE_ENABLED) ? TRUE : FALSE);
   gtk_entry_set_text(type_entry,sp->type);
   gtk_entry_set_text(uri_entry,sp->source_uri);
   gtk_entry_set_text(distribution_entry,sp->distribution);

   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(main_section_checkbutton),FALSE);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(contrib_section_checkbutton),FALSE);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(nonfree_section_checkbutton),FALSE);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(nonus_section_checkbutton),FALSE);

   gtk_entry_set_text(other_sections_entry,"");
   {
      char *sections = strdup(sp->sections);
      char *s,*e;

      e = sections;
      for (s = sections;*s;s = e){
         for (;*e;e++){
            if ((*e == ' ') || (*e == '\t'))
               break;
         }
         if (*e)
            *e++ = '\0';
         if (*s){
            if (strcasecmp("main",s) == 0)
               gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(main_section_checkbutton),TRUE);
            else if (strcasecmp("contrib",s) == 0)
               gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(contrib_section_checkbutton),TRUE);
            else if (strcasecmp("non-free",s) == 0)
               gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(nonfree_section_checkbutton),TRUE);
            else {
               gtk_entry_append_text(other_sections_entry,s);
               gtk_entry_append_text(other_sections_entry," ");
            }
         }
         while ((*e == ' ') && (*e == '\t'))
            e++;
      }
     
      free(sections);
   }
}

static void update_sources_line_data(struct source_line *sp){
   static GtkCheckButton *enabled_checkbutton = NULL;
   static GtkEntry *type_entry = NULL;
   static GtkEntry *uri_entry = NULL;
   static GtkEntry *distribution_entry = NULL;
   static GtkCheckButton *main_section_checkbutton = NULL;
   static GtkCheckButton *contrib_section_checkbutton = NULL;
   static GtkCheckButton *nonfree_section_checkbutton = NULL;
   static GtkCheckButton *nonus_section_checkbutton = NULL;
   static GtkEntry *other_sections_entry = NULL;
   static char sections[1024+64+1];

   if (!enabled_checkbutton)
      enabled_checkbutton = GTK_CHECK_BUTTON(lookup_widget(sources_window,"enabled_checkbutton"));

   if (!type_entry)
      type_entry = GTK_ENTRY(lookup_widget(sources_window,"type_entry"));

   if (!uri_entry)
      uri_entry = GTK_ENTRY(lookup_widget(sources_window,"uri_entry"));

   if (!distribution_entry)
      distribution_entry = GTK_ENTRY(lookup_widget(sources_window,"distribution_entry"));

   if (!main_section_checkbutton)
      main_section_checkbutton = GTK_CHECK_BUTTON(lookup_widget(sources_window,"main_section_checkbutton"));

   if (!contrib_section_checkbutton)
      contrib_section_checkbutton = GTK_CHECK_BUTTON(lookup_widget(sources_window,"contrib_section_checkbutton"));

   if (!nonfree_section_checkbutton)
      nonfree_section_checkbutton = GTK_CHECK_BUTTON(lookup_widget(sources_window,"nonfree_section_checkbutton"));

   if (!nonus_section_checkbutton)
      nonus_section_checkbutton = GTK_CHECK_BUTTON(lookup_widget(sources_window,"nonus_section_checkbutton"));

   if (!other_sections_entry)
      other_sections_entry = GTK_ENTRY(lookup_widget(sources_window,"other_sections_entry"));

   sp->status = (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(enabled_checkbutton)) == TRUE) ?
                SOURCE_ENABLED : SOURCE_DISABLED;
   free(sp->type);
   sp->type = strdup(gtk_entry_get_text(type_entry));
   free(sp->source_uri);
   sp->source_uri = strdup(gtk_entry_get_text(uri_entry));
   free(sp->distribution);
   sp->distribution = strdup(gtk_entry_get_text(distribution_entry));
   free(sp->sections);
   *sections = '\0';
   if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(main_section_checkbutton)))
      strcat(sections,"main ");
   if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(contrib_section_checkbutton)))
      strcat(sections,"contrib ");
   if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(nonfree_section_checkbutton)))
      strcat(sections,"non-free ");
   if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(nonus_section_checkbutton)))
      strcat(sections,"non-US ");
   sprintf(sections + strlen(sections),"%.1024s",gtk_entry_get_text(other_sections_entry));
   collapse_repeated_spaces(sections);
   sp->sections = strdup(sections);
}

void sources_list_row_double_clicked_event(struct source_line *sp){
   static GtkCList *sources_clist = NULL;

   if (!sources_clist)
      sources_clist = GTK_CLIST(lookup_widget(sources_window,"sources_clist"));

   sp->status = (sp->status == SOURCE_ENABLED) ? SOURCE_DISABLED : SOURCE_ENABLED;
   gtk_clist_set_text(sources_clist,sp->gtk_clist_row,SOURCES_CLIST_STATUS,
                      get_source_status_string(sp->status));
   show_source_line_details(sp);
}

void link_selected_sources_line(struct source_line *sp){
   selected_source_line = sp;
   show_source_line_details(sp);
}

void unlink_selected_sources_line(struct source_line *sp){
   selected_source_line = NULL;
}

void remove_selected_sources_line(){
   static GtkCList *sources_clist = NULL;
   struct list_head *lp;
   struct source_line *sp;

   if (!sources_clist)
      sources_clist = GTK_CLIST(lookup_widget(sources_window,"sources_clist"));

   sp = selected_source_line;
   if (!sp)
      return;

   gtk_clist_unselect_row(sources_clist,sp->gtk_clist_row,0);
   for (lp = sp->list.next;lp != &sources_anchor;lp = lp->next)
      (list_entry(lp,struct source_line,list))->gtk_clist_row--;
   gtk_clist_remove(sources_clist,sp->gtk_clist_row);
   list_del(&sp->list);
   free_sources_line_data(sp);
}

static void update_sources_line(struct source_line *sp){
   static GtkCList *sources_clist = NULL;
   static gchar *clist_fields[SOURCES_CLIST_FIELDS];
   int row,i;

   if (!sources_clist)
      sources_clist = GTK_CLIST(lookup_widget(sources_window,"sources_clist"));

   update_sources_line_data(sp);

   populate_clist_fields(clist_fields,sp);

   if (sp->gtk_clist_row != -1){
      for (i = 0;i < SOURCES_CLIST_FIELDS;i++)
         gtk_clist_set_text(sources_clist,sp->gtk_clist_row,i,clist_fields[i]);
   } else {
      row = gtk_clist_append(sources_clist,clist_fields);
      sp->gtk_clist_row = row;
      gtk_clist_set_row_data(sources_clist,row,(gpointer)sp);
   }

   if (sp)
      gtk_clist_unselect_row(sources_clist,sp->gtk_clist_row,/* column */ 0);
   gtk_clist_select_row(sources_clist,sp->gtk_clist_row,/* column */ 0);
}

void update_selected_sources_line(){

   if (check_method())  return;

   if (!selected_source_line)
      return;

   update_sources_line(selected_source_line);
}

void add_sources_line(){
   struct source_line *sp;

   if (check_method())  return;

   sp = alloc_structure(struct source_line);

   list_add_tail(&sp->list,&sources_anchor);	/* Append */

   sp->gtk_clist_row = -1;
   update_sources_line(sp);
}


int
check_method(){
    char* source_uri;
    char Method[200];
    char *p, c, bt=1;
    static GtkEntry *uri_entry = NULL;
    static GtkEntry *distribution_entry = NULL;
    static GtkCheckButton *main_section_checkbutton = NULL;
    static GtkCheckButton *contrib_section_checkbutton = NULL;
    static GtkCheckButton *nonfree_section_checkbutton = NULL;
    static GtkCheckButton *nonus_section_checkbutton = NULL;
    static GtkEntry *other_sections_entry = NULL;
    static GtkWidget *msg_ok=NULL;
    static GtkWidget *msg=NULL;

    struct stat methodstat;

    if (!msg)
        msg=lookup_widget(msg_dlg, "msg");

    if (!msg_ok)
        msg_ok=lookup_widget(msg_dlg, "msg_ok");

    if (!uri_entry)
        uri_entry = GTK_ENTRY(lookup_widget(sources_window,"uri_entry"));

    source_uri = strdup(gtk_entry_get_text(uri_entry));

    for (p=source_uri;*p;p++)
        if  (*p==':'){
            c=*p;
            *p=0;
            break;
            }

    if  (c!=':')  
    {  
	//added by vicky to show the message when the url is not correct

        gtk_label_set_text(GTK_LABEL(msg), _("You should set the correct URL!"));
        gtk_widget_show (msg_ok);
        gtk_widget_show(msg_dlg);
        goto errback;
    }
    snprintf(Method, 200, "%s%s", (_config->FindDir("Dir::Bin::Methods")).c_str(), source_uri);

    if (FileExists(Method)==false){
        snprintf(Method, 200, _("Error:\nThe method driver %s%s could not be found"),
            (_config->FindDir("Dir::Bin::Methods")).c_str(), source_uri);
        gtk_label_set_text(GTK_LABEL(msg), Method);
        gtk_widget_show (msg_ok);
        gtk_widget_show(msg_dlg);
        goto errback;
        }

    if (!distribution_entry)
        distribution_entry = GTK_ENTRY(lookup_widget(sources_window,"distribution_entry"));

        p=gtk_entry_get_text(distribution_entry);

    if  (*p==0)
    {
        gtk_label_set_text(GTK_LABEL(msg), _("You should set the distribution!"));
        gtk_widget_show (msg_ok);
        gtk_widget_show(msg_dlg);
        goto errback;
    }

    c=p[strlen(p)-1];

    if (!main_section_checkbutton)
        main_section_checkbutton = GTK_CHECK_BUTTON(lookup_widget(sources_window,"main_section_checkbutton"));
    if (!contrib_section_checkbutton)
        contrib_section_checkbutton = GTK_CHECK_BUTTON(lookup_widget(sources_window,"contrib_section_checkbutton"));
    if (!nonfree_section_checkbutton)
        nonfree_section_checkbutton = GTK_CHECK_BUTTON(lookup_widget(sources_window,"nonfree_section_checkbutton"));
    if (!nonus_section_checkbutton)
        nonus_section_checkbutton = GTK_CHECK_BUTTON(lookup_widget(sources_window,"nonus_section_checkbutton"));
    if (!other_sections_entry)
        other_sections_entry = GTK_ENTRY(lookup_widget(sources_window,"other_sections_entry"));

    if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(main_section_checkbutton)))
    if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(contrib_section_checkbutton)))
    if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(nonfree_section_checkbutton)))
    if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(nonus_section_checkbutton)))
        bt=0;

    p=gtk_entry_get_text(other_sections_entry);

    if  (c!='/')
        if  ((bt==0)&&(*p==0))
        {
            gtk_label_set_text(GTK_LABEL(msg), _("You should set the section!"));
            gtk_widget_show (msg_ok);
            gtk_widget_show(msg_dlg);
            goto  errback;
        }

    free(source_uri);
    return 0;
errback:
    free(source_uri);
    return 1;
    }
