/*
 * gtk-event.c	-- GTK + Event Management
 * 
 * Copyright  2000 Erick Gallesio - I3S-CNRS/ESSI <eg@unice.fr>
 * 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 * USA.
 * 
 *           Author: Erick Gallesio [eg@unice.fr]
 *    Creation date: 21-Oct-2000 14:16 (eg)
 * Last file update: 10-Nov-2000 12:59 (eg)
 */

#include <signal.h>
#include "stklos.h"
#include "gtk-glue.h"

/*===========================================================================*\
 * 
 * 				Event Loop Management
 * 
\*===========================================================================*/

static int input_handler_id;	/* the tag of the current input handler */
static int seen_control_C = 0;	/* will be one if a ^C is hit while in event loop */


static void seen_a_char(void)
{
  gtk_main_quit();
}


static void sigint_handler(int sig)
{
  /* This handler is effective only while managing GTK events */
  seen_control_C = 1;
  if (input_handler_id >= 0) gtk_main_quit();
}



DEFINE_PRIMITIVE("", gtk_idle, subr0, (void)) /* "anonymous" primitive !! */
{
  struct sigaction sigact, old_sigact;

  /* The normal handler for ^C cannot be used, since it makes the gtk-main 
   * reentrant. => We hae a special handler while in the GTk loop */
  sigfillset(&(sigact.sa_mask));
  sigact.sa_handler = sigint_handler;
  sigact.sa_flags   = 0;
  sigaction(SIGINT, &sigact, &old_sigact);
  
  /* Call the GTk+ event loop handler */
  seen_control_C = 0;
  gtk_main();
  
  /* Reset normal STklos handler */
  sigaction(SIGINT, &old_sigact, NULL);

  if (seen_control_C) /* propagate  */ old_sigact.sa_handler(SIGINT);
  
  /* old code:  if (gtk_events_pending()) gtk_main_iteration(); */
  return STk_void;
}


DEFINE_PRIMITIVE("%gtk-main", gtk_main, subr0, (void))
{
  /* to avoid premature termination when a script is completely read */
  gtk_main();
  return STk_void;
}

static void set_input_handler(void)
{
  /* Set a handler for stdin to interact cleanly with the event loop */
  input_handler_id = gdk_input_add(0, GDK_INPUT_READ | GDK_INPUT_EXCEPTION, 
				   (GdkInputFunction) seen_a_char, NULL);
}


/*===========================================================================*\
 * 
 *		 		After 
 *
\*===========================================================================*/

static SCM after_handlers = STk_nil;

static void register_proc(SCM arg)
{
  /* Register the function to avoid GC problems */
  after_handlers = STk_cons((SCM) arg, after_handlers);
}

static int apply_after(gpointer data)
{
  SCM tmp, prev; 

  if (data) {
    /* apply the function */
    STk_C_apply((SCM) data, 0);
    
    /* Unregister the function from the global list */
    for (prev=(SCM) NULL, tmp=after_handlers; !NULLP(tmp); prev=tmp, tmp=CDR(tmp)) {
      if (CAR(tmp) == (SCM) data) {
	if (prev)
	  CDR(prev) = CDR(tmp);
	else
	  after_handlers = CDR(tmp);
      }
    }
  }
  return 0;
}

//static void sigint_handler_on_wait(int sig)
//{
//  seen_control_C = 1;
//}


static void wait_ms(int ms)		/* For (after ms) without procedure */
{
  GTimeVal now, end;
  struct timespec ts;
  struct sigaction sigact, old_sigact;

  g_get_current_time(&now);
  end.tv_sec  = now.tv_sec  + ms/1000;
  end.tv_usec = now.tv_usec + (ms%1000)*1000;

  /* Unregister the character handler on standard input since it interacts
   * badly with our wait to manage the waiting 
   */
  gdk_input_remove(input_handler_id);
  input_handler_id = -1;

  
  sigfillset(&(sigact.sa_mask));
  sigact.sa_handler = sigint_handler;
  sigact.sa_flags   = 0;
  sigaction(SIGINT, &sigact, &old_sigact);
  
  seen_control_C = 0;
  
  for ( ; ; ) {
     /* if current time is > to end exit the loop */
    g_get_current_time(&now);
    if (now.tv_sec > end.tv_sec) break;
    else if (now.tv_sec == end.tv_sec)
      if (now.tv_usec >= end.tv_usec) break;
    
    /* If we have seen a ^C abort the loop too */
    if (seen_control_C) break;

    /* Otherwise treat events */
    while (gtk_events_pending())
      gtk_main_iteration();

    /* wait a little befor looping again */
    ts.tv_sec=0; ts.tv_nsec = 1000; /* 1 s */
    nanosleep(&ts, NULL);
  }
  
  /* restore the input  handler */
  set_input_handler();

  sigaction(SIGINT, &old_sigact, NULL);
  if (seen_control_C) /* propagate  */ old_sigact.sa_handler(SIGINT);
}
  
  

DEFINE_PRIMITIVE("%after", after, subr23, (SCM what, SCM arg1, SCM arg2))
{
  /* WARNING: no control on the type of arguments. This is done in Scheme */
  switch (what) {
    case MAKE_INT(0): {				/* after ms action */
      long ms = STk_integer_value(arg1);

      register_proc(arg2);
      return MAKE_INT(gtk_timeout_add(ms, apply_after, (gpointer) arg2));
    }
    case MAKE_INT(1): {				/* after ms */	 //FIXME
      wait_ms(STk_integer_value(arg1));
      return STk_void;
    }
    case MAKE_INT(2): 				/* after 'idle ... */
      register_proc(arg1);
      return MAKE_INT(gtk_idle_add(apply_after, (gpointer) arg1));
    case MAKE_INT(3): 				/* after 'cancel # */
      gtk_timeout_remove(STk_integer_value(arg1));
      return STk_void;
  }
  return STk_void;
}



/*===========================================================================*\
 * 
 *		 		Update 
 *
\*===========================================================================*/

/*
<doc update
 * (update)
 * 
 *  |Update| is used to bring the application ``up to date'' by
 *  entering the event loop repeated until all pending events have
 *  been processed. This procedure is useful in scripts where you are
 *  performing a long-running computation but you still want the
 *  application to respond to events such as user interactions.
doc>
 */
DEFINE_PRIMITIVE("update", update, subr0, (void))
{
  while (gtk_events_pending())
    gtk_main_iteration();
  return STk_void;
}


void STk_init_gtk_event(void)
{
  set_input_handler();

  /* Add the event loop */
  STk_set_port_idle(STk_stdin, THE_PRIMITIVE(gtk_idle));

  ADD_PRIMITIVE(gtk_main);
  ADD_PRIMITIVE(after);
  ADD_PRIMITIVE(update);
}
