/*
 * gtk-cont.c					-- GTk+ containers
 * 
 * Copyright  2000-2001 Erick Gallesio - I3S-CNRS/ESSI <eg@unice.fr>
 * 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 * USA.
 * 
 *           Author: Erick Gallesio [eg@unice.fr]
 *    Creation date: 19-Aug-2000 11:56 (eg)
 * Last file update:  3-Jan-2001 13:19 (eg)
 */

#include "stklos.h"
#include "gtk-glue.h"

/*===========================================================================*\
 * 
 * Utilities
 *
\*===========================================================================*/

static void error_bad_integer(SCM obj, char *opt)
{
  STk_error("~S is a bad integer value for %S option", obj, opt);
}

static Inline long cont_integer_value(SCM obj, char *option)
{
  long val;
  
  val = STk_integer_value(obj);
  if (val==LONG_MIN) error_bad_integer(obj, option);
  return val;
}


/*===========================================================================*\
 * 
 * 			General Containers Functions
 *
\*===========================================================================*/

static void cont_children_helper(gpointer p, gpointer data)
{
  SCM *l     = (SCM*) data;
  SCM widget = (SCM) gtk_object_get_user_data(GTK_OBJECT(p));

  if (widget && widget != STk_void)
    *l = STk_cons(widget, *l);
}


DEFINE_PRIMITIVE("%cont-children", cont_children, subr1, (SCM w))
{
  GList *gl;
  SCM l = STk_nil;

  ENTER_PRIMITIVE(cont_children);
  if (!WIDGETP(w)) STk_error_bad_widget(w);
  
  gl = gtk_container_children(GTK_CONTAINER(WIDGET_ID(w)));
  g_list_foreach(gl, cont_children_helper, &l);
  g_list_free(gl);

  return l;
}


/*===========================================================================*\
 * 
 * 				  B O X 
 *
\*===========================================================================*/
SIMPLE_WIDGET("%hbox", hbox, (gtk_hbox_new(FALSE, 0)));
SIMPLE_WIDGET("%vbox", vbox, (gtk_vbox_new(FALSE, 0)));


/*===========================================================================*\
 * 
 * 				 P A C K E R
 *
\*===========================================================================*/
SIMPLE_WIDGET("%packer", packer, (gtk_packer_new()));

/*===========================================================================*\
 * 
 * 				T A B L E
 *
\*===========================================================================*/
SIMPLE_WIDGET("%table", table, (gtk_table_new(2, 2, TRUE)));


/*===========================================================================*\
 * 
 * 				F R A M E
 *
\*===========================================================================*/
SIMPLE_WIDGET("%frame", frame, (gtk_frame_new("")));


/*===========================================================================*\
 * 
 * 				T O P L E V E L
 *
\*===========================================================================*/
SIMPLE_WIDGET("%toplevel", toplevel, (gtk_window_new(GTK_WINDOW_TOPLEVEL)));


/*===========================================================================*\
 * 
 * 			S C R O L L E D - W I N D O W
 *
\*===========================================================================*/
SIMPLE_WIDGET("%scrolled-window", scrollwin, (gtk_scrolled_window_new(NULL, NULL)));


/*===========================================================================*\
 * 
 *		 	H A N D L E - W I N D O W
 *
\*===========================================================================*/
SIMPLE_WIDGET("%handle-window", handlewin, (gtk_handle_box_new()));



/*===========================================================================*\
 * 
 * 				I M A G E
 *
\*===========================================================================*/
static GtkWidget *fake_toplevel;

static gchar **scm_list2argv(SCM l)
{
  SCM tmp;
  gchar **res, **p;
  int len = STk_int_length(l);

  if (len < 0) 
    STk_error("bad list ~S", l);

  p = res = STk_must_malloc((len+1) * sizeof(gchar*));

  for (tmp = l; !NULLP(tmp); tmp = CDR(tmp)) {
    if (!STRINGP(CAR(tmp)))
      STk_error("bad data string for image ~S", CAR(tmp));
    *p++ = STRING_CHARS(CAR(tmp));
  }
  *p = NULL;
  return res;
}
  

DEFINE_PRIMITIVE("%image-file", image_file, subr2, (SCM self, SCM filename))
{
  GdkPixmap *pixmap, *mask;
  GtkStyle  style;
  
  ENTER_PRIMITIVE(image_file);
  if (!STRINGP(filename))
    STk_error("bad file name for image ~S", filename);
  
  /* create the image */
  pixmap = gdk_pixmap_create_from_xpm(fake_toplevel->window,
				      &mask,
				      &style.bg[GTK_STATE_NORMAL],
				      STRING_CHARS(filename));
  if (!pixmap) STk_error("cannot load image in file ~A", filename);
  
  return STk_gtk_widget2scm(gtk_pixmap_new(pixmap, mask), self);
}

DEFINE_PRIMITIVE("%image-data", image_data, subr2, (SCM self, SCM data))
{
  GdkPixmap *pixmap, *mask;
  GtkStyle  style;
  gchar **argv;

  ENTER_PRIMITIVE(image_data);
 
  argv = scm_list2argv(data);
  
  /* create the image */
  pixmap = gdk_pixmap_create_from_xpm_d(fake_toplevel->window,
					&mask,
					&style.bg[GTK_STATE_NORMAL],
					argv);
  if (!pixmap) STk_error("cannot load from given data");
  
  return STk_gtk_widget2scm(gtk_pixmap_new(pixmap, mask), self);
}


/* ====================================================================== */



/* ====================================================================== */
void STk_init_gtk_cont(void)
{
  ADD_PRIMITIVE(hbox);			/* Constructors */
  ADD_PRIMITIVE(vbox);
  ADD_PRIMITIVE(packer);
  ADD_PRIMITIVE(table);
  ADD_PRIMITIVE(frame);

  ADD_PRIMITIVE(toplevel);
  ADD_PRIMITIVE(scrollwin);
  ADD_PRIMITIVE(handlewin);


  ADD_PRIMITIVE(cont_children);



  /* When image pixmaps are read in a window which is not realized,
   * GTK issue a warning. To avoid this we create such images in a
   * fake window which is only accessible in this file
   */
  fake_toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_realize(fake_toplevel);


  ADD_PRIMITIVE(image_file);
  ADD_PRIMITIVE(image_data);
}
