/*  GNU SED, a batch stream editor.
    Copyright (C) 1999
    Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */

#include "config.h"

#include <ctype.h>
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#ifndef NULL
# include <stdio.h>
#endif

#include "basicdefs.h"
#include "utils.h"
#include "sed.h"
#include "regexp.h"
#ifndef NULL
# define NULL CAST(VOID *)0
#endif

#ifdef gettext_noop
# define N_(String) gettext_noop(String)
#else
# define N_(String) (String)
#endif

extern flagT use_extended_syntax_p;

#define TRANSLATE_LENGTH 256 /*XXX shouldn't this be (UCHAR_MAX+1)?*/

static const char NO_REGEX[] = N_("No previous regular expression");
static const char BAD_MODIF[] = N_("Cannot specify modifiers on empty regexp");



regex_t *
compile_regex(b, flags, needed_sub)
  struct buffer *b;
  int flags;
  int needed_sub;
{
  regex_t *new_regex;

  char *last_re = NULL;
  size_t last_re_len;
  char error[200];
  int errcode;

  /* My reading of IEEE Std 1003.2-1992 is that // means the empty RE.
     But historical and common practice is that // "matches the last RE";
     thus this use of POSIXLY_CORRECT. */
  if (size_buffer(b) == 0 && !POSIXLY_CORRECT)
    {
      if (flags > 0)
	bad_prog(_(BAD_MODIF));
      return NULL;
    }

  new_regex = MALLOC(1, regex_t);

  last_re_len = size_buffer(b);
  last_re = ck_memdup(get_buffer(b), last_re_len);

#ifdef REG_PERL
  errcode = regncomp(new_regex, last_re, last_re_len,
		     (needed_sub ? 0 : REG_NOSUB)
		     | flags
		     | extended_regexp_flags);
#else
  /* Only PCRE processes \t & co. */
  last_re_len = normalize_text(last_re, last_re_len);

  errcode = regcomp(new_regex, last_re,
		     (needed_sub ? 0 : REG_NOSUB)
		     | flags
		     | extended_regexp_flags);
#endif

  FREE(last_re);
  if (errcode)
    {
      regerror(errcode, NULL, error, 200);
      bad_prog(gettext(error));
    }

  /* Just to be sure, I mark this as not POSIXLY_CORRECT behavior */
  if (new_regex->re_nsub < needed_sub && !POSIXLY_CORRECT)
    {
      char buf[200];
      sprintf(buf, _("Invalid reference \\%d on `s' command's RHS"),
	      needed_sub);
      bad_prog(buf);
    }

  return new_regex;
}


int
match_regex(regex, buf, buflen, buf_start_offset, regarray, regsize)
  regex_t *regex;
  char *buf;
  size_t buflen;
  size_t buf_start_offset;
  regmatch_t *regarray;
  int regsize;
{
  static regex_t *regex_last;
  int ret;

  /* Keep track of the last regexp matched. */
  if (!regex)
    {
      regex = regex_last;
      if (!regex_last)
	bad_prog(_(NO_REGEX));
    }
  else
    regex_last = regex;

#ifdef REG_PERL
  ret = regexec2(regex, 
		 buf, CAST(int)buflen, CAST(int)buf_start_offset,
		 regsize, regarray, 0);
#else
  ret = regexec(regex, buf + buf_start_offset, regsize, regarray, 0);
#endif

  return (ret == 0);
}


#ifdef DEBUG_LEAKS
void
release_regex(regex)
  regex_t *regex;
{
    regfree(regex);
    FREE(regex);
}
#endif /*DEBUG_LEAKS*/
