/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

#include <kproc/q-extern.h>
#include <kproc/queue.h>
#include <kproc/lock.h>
#include <kproc/sem.h>
#include <klib/rc.h>
#include <atomic32.h>
#include <os-native.h>
#include <sysalloc.h>

#include <stdlib.h>
#include <assert.h>

/*--------------------------------------------------------------------------
 * KQueue
 *  a simple thread-safe queue structure supporting push/pop operation
 *  makes use of semaphore objects for synchronization
 */
struct KQueue
{
    KSemaphore *rc;
    KSemaphore *wc;

    KLock *rl;
    KLock *wl;

    uint32_t capacity;
    uint32_t bmask, imask;
    volatile uint32_t read, write;
    atomic32_t refcount;
    volatile bool sealed;
    uint8_t align [ 7 ];
    void *buffer [ 16 ];
};


/* Whack
 */
static
rc_t KQueueWhack ( KQueue *self )
{
    rc_t rc = KSemaphoreRelease ( self -> wc );
    if ( rc == 0 )
    {
        KSemaphoreRelease ( self -> rc );
        KLockRelease ( self -> wl );
        KLockRelease ( self -> rl );
        free ( self );
    }
    return rc;
}

/* AddRef
 * Release
 *  ignores NULL references
 */
LIB_EXPORT rc_t CC KQueueAddRef ( const KQueue *cself )
{
    if ( cself != NULL )
        atomic32_inc ( & ( ( KQueue* ) cself ) -> refcount );
    return 0;
}

LIB_EXPORT rc_t CC KQueueRelease ( const KQueue *cself )
{
    KQueue *self = ( KQueue* ) cself;
    if ( cself != NULL )
    {
        if ( atomic32_dec_and_test ( & self -> refcount ) )
            return KQueueWhack ( self );
    }
    return 0;
}

/* Make
 * create an empty queue object
 *
 *  "capacity" [ IN ] - minimum queue length
 *  always expands to a power of 2, i.e. providing
 *  a length of 10 will result in a length of 16.
 */
LIB_EXPORT rc_t CC KQueueMake ( KQueue **qp, uint32_t capacity )
{
    rc_t rc;
    if ( qp == NULL )
        rc = RC ( rcCont, rcQueue, rcConstructing, rcParam, rcNull );
    else
    {
        KQueue *q;

        uint32_t cap = 1;
        while ( cap < capacity )
            cap += cap;

        q = malloc ( sizeof * q - sizeof q -> buffer + cap * sizeof q -> buffer [ 0 ] );
        if ( q == NULL )
            rc = RC ( rcCont, rcQueue, rcConstructing, rcMemory, rcExhausted );
        else
        {
            rc = KSemaphoreMake ( & q -> rc, 0 );
            if ( rc == 0 )
            {
                rc = KSemaphoreMake ( & q -> wc, cap );
                if ( rc == 0 )
                {
                    rc = KLockMake ( & q -> rl );
                    if ( rc == 0 )
                    {
                        rc = KLockMake ( & q -> wl );
                        if ( rc == 0 )
                        {
                            q -> capacity = cap;
                            q -> bmask = cap - 1;
                            q -> imask = ( cap + cap ) - 1;
                            q -> read = q -> write = 0;
                            atomic32_set ( & q -> refcount, 1 );
                            q -> sealed = false;

                            * qp = q;
                            return 0;
                        }

                        KLockRelease ( q -> rl );
                    }

                    KSemaphoreRelease ( q -> wc );
                }

                KSemaphoreRelease ( q -> rc );
            }
            free ( q );
        }
        * qp = NULL;
    }
    return rc;
}

/* Push
 *  add an object to the queue
 *
 *  "item" [ IN, OPAQUE ] - pointer to item being queued
 *
 *  "tm" [ IN, NULL OKAY ] - pointer to system specific timeout
 *  structure. if the queue is full, wait for indicated period
 *  of time for space to become available, or return status
 *  code indicating a timeout. when NULL and queue is full,
 *  Push will time out immediately and return status code.
 */
LIB_EXPORT rc_t CC KQueuePush ( KQueue *self, const void *item, timeout_t *tm )
{
    rc_t rc;

    if ( self == NULL )
        return RC ( rcCont, rcQueue, rcInserting, rcSelf, rcNull );
    if ( self -> sealed )
        return RC ( rcCont, rcQueue, rcInserting, rcQueue, rcReadonly );
    if ( item == NULL )
        return RC ( rcCont, rcQueue, rcInserting, rcTimeout, rcNull );

    rc = KLockAcquire ( self -> wl );
    if ( rc == 0 )
    {
        rc = KSemaphoreTimedWait ( self -> wc, self -> wl, tm );
        KLockUnlock ( self -> wl );

        if ( rc == 0 )
        {
            uint32_t w;

            /* re-check the seal */
            if ( self -> sealed )
            {
                /* not a disaster if semaphore not signaled */
                if ( ! KLockAcquire ( self -> wl ) )
                {
                    KSemaphoreSignal ( self -> wc );
                    KLockUnlock ( self -> wl );
                }
                return RC ( rcCont, rcQueue, rcInserting, rcQueue, rcReadonly );
            }

            /* insert item */
            w = self -> write & self -> imask;
            self -> buffer [ w & self -> bmask ] = ( void* ) item;
            self -> write = w + 1;

            /* let listeners know about item */
            if ( KLockAcquire ( self -> rl ) == 0 )
            {
                KSemaphoreSignal ( self -> rc );
                KLockUnlock ( self -> rl );
            }
        }
    }

    return rc;
}

/* Pop
 *  pop an object from queue
 *
 *  "item" [ OUT, OPAQUE* ] - return parameter for popped item
 *
 *  "tm" [ IN, NULL OKAY ] - pointer to system specific timeout
 *  structure. if the queue is empty, wait for indicated period
 *  of time for an object to become available, or return status
 *  code indicating a timeout. when NULL and queue is empty,
 *  Pop will time out immediately and return status code.
 */
LIB_EXPORT rc_t CC KQueuePop ( KQueue *self, void **item, timeout_t *tm )
{
    rc_t rc;

    if ( item == NULL )
        rc = RC ( rcCont, rcQueue, rcRemoving, rcParam, rcNull );
    else
    {
        * item = NULL;

        if ( self == NULL )
            rc = RC ( rcCont, rcQueue, rcRemoving, rcSelf, rcNull );
        else
        {
            rc = KLockAcquire ( self -> rl );
            if ( rc == 0 )
            {
                rc = KSemaphoreTimedWait ( self -> rc, self -> rl, self -> sealed ? NULL : tm );
                KLockUnlock ( self -> rl );

                if ( rc == 0 )
                {
                    uint32_t r, idx;

                    /* got an element */
                    assert ( self -> read != self -> write );

                    /* read element */
                    r = self -> read & self -> imask;
                    idx = r & self -> bmask;
                    * item = self -> buffer [ idx ];
                    self -> buffer [ idx ] = NULL;
                    self -> read = r + 1;

                    /* let write know there's a free slot available */
                    if ( KLockAcquire ( self -> wl ) == 0 )
                    {
                        KSemaphoreSignal ( self -> wc );
                        KLockUnlock ( self -> wl );
                    }
                }
                else if ( self -> sealed && GetRCState ( rc ) == rcTimeout )
                    rc = RC ( rcCont, rcQueue, rcRemoving, rcData, rcDone );
            }
        }
    }

    return rc;
}

/* Sealed
 *  ask if the queue has been closed off
 *  meaning there will be no further push operations
 *
 *  NB - if "self" is NULL, the return value is "true"
 *  since a NULL queue cannot accept items via push
 */
LIB_EXPORT bool CC KQueueSealed ( const KQueue *self )
{
    if ( self != NULL )
        return self -> sealed;
    return false;
}


/* Seal
 *  indicate that the queue has been closed off
 *  meaning there will be no further push operations
 */
LIB_EXPORT rc_t CC KQueueSeal ( KQueue *self )
{
    if ( self == NULL )
        return RC ( rcCont, rcQueue, rcFreezing, rcSelf, rcNull );

    self -> sealed = true;
    return 0;
}
