-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with SLI;

separate (Sem)
package body Subprogram_Specification is

   procedure Wf_Subprogram_Specification_From_Body
     (Node          : in     STree.SyntaxNode;
      Hidden        : in     Boolean;
      Current_Scope : in out Dictionary.Scopes;
      Subprog_Sym   :    out Dictionary.Symbol;
      First_Seen    :    out Boolean) is
      Ident_Node, Return_Type_Node     : STree.SyntaxNode;
      Ident_Str                        : LexTokenManager.Lex_String;
      Type_Sym, First_Sym_Found        : Dictionary.Symbol;
      Grand_Parent, Great_Grand_Parent : SP_Symbols.SP_Symbol;
      Adding_Proper_Body               : Boolean;

      -----------------------------------------------

      function In_Package_Body (Current_Scope : Dictionary.Scopes) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Dictionary.IsLocalScope (Current_Scope) and then Dictionary.IsPackage (Dictionary.GetRegion (Current_Scope));
      end In_Package_Body;

      -----------------------------------------------

      function In_Protected_Body (Current_Scope : Dictionary.Scopes) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Dictionary.IsLocalScope (Current_Scope)
           and then Dictionary.IsType (Dictionary.GetRegion (Current_Scope))
           and then Dictionary.TypeIsProtected (Dictionary.GetRegion (Current_Scope));
      end In_Protected_Body;

      -----------------------------------------------

      procedure Do_Add
        (Add_Subprog, Add_Body, Hidden : in     Boolean;
         Ident_Str                     : in     LexTokenManager.Lex_String;
         Node_Pos                      : in     LexTokenManager.Token_Position;
         First_Seen                    : in out Boolean;
         Current_Scope                 : in out Dictionary.Scopes;
         Subprog_Sym                   : in out Dictionary.Symbol)
      --# global in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives Current_Scope     from *,
      --#                                Add_Body,
      --#                                Add_Subprog,
      --#                                ContextManager.Ops.Unit_Stack,
      --#                                Dictionary.Dict,
      --#                                Node_Pos,
      --#                                Subprog_Sym &
      --#         Dictionary.Dict   from *,
      --#                                Add_Body,
      --#                                Add_Subprog,
      --#                                ContextManager.Ops.Unit_Stack,
      --#                                Current_Scope,
      --#                                Hidden,
      --#                                Ident_Str,
      --#                                Node_Pos,
      --#                                Subprog_Sym &
      --#         First_Seen        from *,
      --#                                Add_Subprog &
      --#         SPARK_IO.File_Sys from *,
      --#                                Add_Body,
      --#                                Add_Subprog,
      --#                                ContextManager.Ops.Unit_Stack,
      --#                                Current_Scope,
      --#                                Dictionary.Dict,
      --#                                Hidden,
      --#                                Ident_Str,
      --#                                LexTokenManager.State,
      --#                                Node_Pos,
      --#                                Subprog_Sym &
      --#         Subprog_Sym       from *,
      --#                                Add_Subprog,
      --#                                ContextManager.Ops.Unit_Stack,
      --#                                Dictionary.Dict,
      --#                                Node_Pos;
      is
      begin
         if Add_Subprog then
            Dictionary.AddSubprogram
              (Name          => Ident_Str,
               Comp_Unit     => ContextManager.Ops.Current_Unit,
               Specification => Dictionary.Location'(Start_Position => Node_Pos,
                                                     End_Position   => Node_Pos),
               Scope         => Current_Scope,
               Context       => Dictionary.ProgramContext,
               Subprogram    => Subprog_Sym);
         else
            First_Seen := False;
         end if;

         if Add_Body then
            Dictionary.AddBody
              (CompilationUnit => Subprog_Sym,
               Comp_Unit       => ContextManager.Ops.Current_Unit,
               TheBody         => Dictionary.Location'(Start_Position => Node_Pos,
                                                       End_Position   => Node_Pos),
               Hidden          => Hidden);
            Current_Scope := Dictionary.LocalScope (Subprog_Sym);
         else
            Dictionary.AddBodyStub
              (CompilationUnit => Subprog_Sym,
               Comp_Unit       => ContextManager.Ops.Current_Unit,
               BodyStub        => Dictionary.Location'(Start_Position => Node_Pos,
                                                       End_Position   => Node_Pos));
         end if;
      end Do_Add;

      -----------------------------------------------

      procedure Check_For_Child (Ident_Node    : in STree.SyntaxNode;
                                 Current_Scope : in Dictionary.Scopes)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         Current_Scope,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         Ident_Node,
      --#                                         LexTokenManager.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table;
      --# pre STree.Syntax_Node_Type (Ident_Node, STree.Table) = SP_Symbols.identifier;
      is
      begin
         if Dictionary.IsPackage (Dictionary.GetRegion (Current_Scope))
           and then not Dictionary.IsEmbeddedPackage (Dictionary.GetRegion (Current_Scope))
           and then Dictionary.LookupSelectedItem
           (Prefix   => Dictionary.GetRegion (Current_Scope),
            Selector => STree.Node_Lex_String (Node => Ident_Node),
            Scope    => Dictionary.GlobalScope,
            Context  => Dictionary.ProofContext) /=
           Dictionary.NullSymbol then
            -- name exists as child
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => STree.Node_Position (Node => Ident_Node),
               Id_Str    => STree.Node_Lex_String (Node => Ident_Node));
         end if;
      end Check_For_Child;

      -----------------------------------------------

      function Declared_In_Same_Or_Related_Scope (Sym           : Dictionary.Symbol;
                                                  Current_Scope : Dictionary.Scopes) return Boolean
      --# global in Dictionary.Dict;
      --  return true if Sym is declared in Current_Scope or in the
      --  visible/private scope of the region associate with Current_Scope
      is
      begin
         return Dictionary.GetScope (Sym) = Current_Scope
           or else Dictionary.GetScope (Sym) = Dictionary.VisibleScope (Dictionary.GetRegion (Current_Scope))
           or else Dictionary.GetScope (Sym) = Dictionary.PrivateScope (Dictionary.GetRegion (Current_Scope));
      end Declared_In_Same_Or_Related_Scope;

   begin -- Wf_Subprogram_Specification_From_Body
      Ident_Node := STree.Child_Node (Current_Node => STree.Child_Node (Current_Node => Node));
      -- ASSUME Ident_Node = identifier
      SystemErrors.RT_Assert
        (C       => STree.Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.identifier,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Ident_Node = identifier in Wf_Subprogram_Specification_From_Body");
      Ident_Str          := STree.Node_Lex_String (Node => Ident_Node);
      Grand_Parent       :=
        STree.Syntax_Node_Type (Node => STree.Parent_Node (Current_Node => STree.Parent_Node (Current_Node => Node)));
      Great_Grand_Parent :=
        STree.Syntax_Node_Type
        (Node => STree.Parent_Node (Current_Node => STree.Parent_Node (Current_Node => STree.Parent_Node (Current_Node => Node))));
      Adding_Proper_Body := Great_Grand_Parent = SP_Symbols.abody or else In_Protected_Body (Current_Scope => Current_Scope);
      -- in prot bod we can't be adding a stub
      First_Seen  := True; -- default value in case all checks below fail
      Subprog_Sym :=
        Dictionary.LookupItem
        (Name              => Ident_Str,
         Scope             => Current_Scope,
         Context           => Dictionary.ProofContext,
         Full_Package_Name => False);

      if Subprog_Sym = Dictionary.NullSymbol then
         if STree.Syntax_Node_Type (Node => STree.Parent_Node (Current_Node => Node)) = SP_Symbols.body_stub then
            Check_For_Child (Ident_Node    => Ident_Node,
                             Current_Scope => Current_Scope);
            Do_Add
              (Add_Subprog   => True,
               Add_Body      => False,
               Hidden        => Hidden,
               Ident_Str     => Ident_Str,
               Node_Pos      => STree.Node_Position (Node => Node),
               First_Seen    => First_Seen,
               Current_Scope => Current_Scope,
               Subprog_Sym   => Subprog_Sym);
         elsif Grand_Parent = SP_Symbols.main_program_declaration then
            Do_Add
              (Add_Subprog   => True,
               Add_Body      => True,
               Hidden        => Hidden,
               Ident_Str     => Ident_Str,
               Node_Pos      => STree.Node_Position (Node => Node),
               First_Seen    => First_Seen,
               Current_Scope => Current_Scope,
               Subprog_Sym   => Subprog_Sym);
         elsif Great_Grand_Parent /= SP_Symbols.subunit then
            Do_Add
              (Add_Subprog   => True,
               Add_Body      => True,
               Hidden        => Hidden,
               Ident_Str     => Ident_Str,
               Node_Pos      => STree.Node_Position (Node => Node),
               First_Seen    => First_Seen,
               Current_Scope => Current_Scope,
               Subprog_Sym   => Subprog_Sym);
         else -- no stub for subunit
            Subprog_Sym := Dictionary.NullSymbol;
            ErrorHandler.Semantic_Error
              (Err_Num   => 15,
               Reference => ErrorHandler.No_Reference,
               Position  => STree.Node_Position (Node => Ident_Node),
               Id_Str    => Ident_Str);
         end if;
      else -- symbol found so further checks needed

         --  if the Subprog_Sym is an implicit proof function associated
         --  with the declaration of an Ada function then we want to
         --  recover the associated Ada function before proceding (since
         --  that is the thing we are going to add a body to).  The
         --  GetAdaFunction call is guarded to meet its precondition.
         if Dictionary.IsImplicitProofFunction (Subprog_Sym) then
            Subprog_Sym := Dictionary.GetAdaFunction (Subprog_Sym);
         end if;

         if Great_Grand_Parent = SP_Symbols.subunit then
            if Dictionary.IsSubprogram (Subprog_Sym)
              and then Dictionary.HasBodyStub (Subprog_Sym)
              and then not Dictionary.HasBody (Subprog_Sym) then
               STree.Set_Node_Lex_String (Sym  => Subprog_Sym,
                                          Node => Ident_Node);
               Do_Add
                 (Add_Subprog   => False,
                  Add_Body      => True,
                  Hidden        => Hidden,
                  Ident_Str     => Ident_Str,
                  Node_Pos      => STree.Node_Position (Node => Node),
                  First_Seen    => First_Seen,
                  Current_Scope => Current_Scope,
                  Subprog_Sym   => Subprog_Sym);
            else
               Subprog_Sym := Dictionary.NullSymbol;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 10,
                  Reference => ErrorHandler.No_Reference,
                  Position  => STree.Node_Position (Node => Ident_Node),
                  Id_Str    => Ident_Str);
            end if;
         elsif -- a place where completion of declaration may be allowed
           (In_Package_Body (Current_Scope => Current_Scope)
              -- another place where completion of declaration may be allowed
              or else In_Protected_Body (Current_Scope => Current_Scope))
           and then
           --  check that we are in a place where the the declaration can
           --  be legally completed (i.e. if subprog declared in a
           --  package spec it can only be completed in the package body
           --  (ditto protected type/body)
           Declared_In_Same_Or_Related_Scope (Sym           => Subprog_Sym,
                                              Current_Scope => Current_Scope) then
            First_Sym_Found := Subprog_Sym;
            Subprog_Sym     :=
              Dictionary.LookupImmediateScope
              (Name    => Ident_Str,
               Scope   => Dictionary.VisibleScope (Dictionary.GetRegion (Current_Scope)),
               Context => Dictionary.ProgramContext);
            --  Above looked for declaration in spec vis part, if not
            --  found, try again in private part
            if Subprog_Sym = Dictionary.NullSymbol and then Dictionary.IsPackage (Dictionary.GetRegion (Current_Scope)) then
               Subprog_Sym :=
                 Dictionary.LookupImmediateScope
                 (Name    => Ident_Str,
                  Scope   => Dictionary.PrivateScope (Dictionary.GetRegion (Current_Scope)),
                  Context => Dictionary.ProgramContext);
            end if;

            if Subprog_Sym = Dictionary.NullSymbol then -- something definitely wrong
               if not Dictionary.IsSubprogram (First_Sym_Found) or else Dictionary.IsProofFunction (First_Sym_Found) then
                  -- Name in use for something other than a subprogram or in use for an explicit proof function.
                  -- Report "illegal redec" rather than "already has body" for these cases
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 10,
                     Reference => ErrorHandler.No_Reference,
                     Position  => STree.Node_Position (Node => Ident_Node),
                     Id_Str    => Ident_Str);
                  -- add anyway to prevent scope problems later
                  Do_Add
                    (Add_Subprog   => True,
                     Add_Body      => Adding_Proper_Body,
                     Hidden        => Hidden,
                     Ident_Str     => Ident_Str,
                     Node_Pos      => STree.Node_Position (Node => Node),
                     First_Seen    => First_Seen,
                     Current_Scope => Current_Scope,
                     Subprog_Sym   => Subprog_Sym);
               else -- it is a subprogram which must be a duplicate
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 13,
                     Reference => ErrorHandler.No_Reference,
                     Position  => STree.Node_Position (Node => Ident_Node),
                     Id_Str    => Ident_Str);
                  if Adding_Proper_Body then
                     if Dictionary.HasBody (First_Sym_Found) then
                        -- add complete duplicate subprogram to dict
                        Do_Add
                          (Add_Subprog   => True,
                           Add_Body      => True,
                           Hidden        => Hidden,
                           Ident_Str     => Ident_Str,
                           Node_Pos      => STree.Node_Position (Node => Node),
                           First_Seen    => First_Seen,
                           Current_Scope => Current_Scope,
                           Subprog_Sym   => Subprog_Sym);
                     else
                        -- add body to duplicate procedure stub in dict
                        Subprog_Sym := First_Sym_Found;
                        Do_Add
                          (Add_Subprog   => False,
                           Add_Body      => True,
                           Hidden        => Hidden,
                           Ident_Str     => Ident_Str,
                           Node_Pos      => STree.Node_Position (Node => Node),
                           First_Seen    => First_Seen,
                           Current_Scope => Current_Scope,
                           Subprog_Sym   => Subprog_Sym);
                     end if;
                  end if;
               end if;
            else -- Subprog_Sym was found in package's visible part
               if not Dictionary.IsSubprogram (First_Sym_Found) then
                  -- name in use for something other than a subprogram
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 10,
                     Reference => ErrorHandler.No_Reference,
                     Position  => STree.Node_Position (Node => Ident_Node),
                     Id_Str    => Ident_Str);
                  -- add anyway to prevent scope problems later
                  Do_Add
                    (Add_Subprog   => True,
                     Add_Body      => Adding_Proper_Body,
                     Hidden        => Hidden,
                     Ident_Str     => Ident_Str,
                     Node_Pos      => STree.Node_Position (Node => Node),
                     First_Seen    => First_Seen,
                     Current_Scope => Current_Scope,
                     Subprog_Sym   => Subprog_Sym);
               else -- it is a subprogram which may be a duplicate
                  if Dictionary.HasBody (Subprog_Sym) then
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 13,
                        Reference => ErrorHandler.No_Reference,
                        Position  => STree.Node_Position (Node => Ident_Node),
                        Id_Str    => Ident_Str);
                     if Adding_Proper_Body then
                        -- add complete duplicate procedure to dict
                        Do_Add
                          (Add_Subprog   => True,
                           Add_Body      => True,
                           Hidden        => Hidden,
                           Ident_Str     => Ident_Str,
                           Node_Pos      => STree.Node_Position (Node => Node),
                           First_Seen    => First_Seen,
                           Current_Scope => Current_Scope,
                           Subprog_Sym   => Subprog_Sym);
                     end if;
                  elsif Dictionary.HasBodyStub (Subprog_Sym) then
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 13,
                        Reference => ErrorHandler.No_Reference,
                        Position  => STree.Node_Position (Node => Ident_Node),
                        Id_Str    => Ident_Str);
                     if Adding_Proper_Body then
                        -- add body to duplicate procedure stub in dict
                        Do_Add
                          (Add_Subprog   => False,
                           Add_Body      => True,
                           Hidden        => Hidden,
                           Ident_Str     => Ident_Str,
                           Node_Pos      => STree.Node_Position (Node => Node),
                           First_Seen    => First_Seen,
                           Current_Scope => Current_Scope,
                           Subprog_Sym   => Subprog_Sym);
                     end if;
                  else -- the non-error case of pre-declaration of subprogram
                     STree.Set_Node_Lex_String (Sym  => Subprog_Sym,
                                                Node => Ident_Node);
                     Do_Add
                       (Add_Subprog   => False,
                        Add_Body      => Adding_Proper_Body,
                        Hidden        => Hidden,
                        Ident_Str     => Ident_Str,
                        Node_Pos      => STree.Node_Position (Node => Node),
                        First_Seen    => First_Seen,
                        Current_Scope => Current_Scope,
                        Subprog_Sym   => Subprog_Sym);
                  end if;
               end if;
            end if;
         else -- not in a package so duplicate is definitely error
            if Dictionary.IsSubprogram (Subprog_Sym) and then Dictionary.HasBody (Subprog_Sym) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 13,
                  Reference => ErrorHandler.No_Reference,
                  Position  => STree.Node_Position (Node => Ident_Node),
                  Id_Str    => Ident_Str);
            else
               ErrorHandler.Semantic_Error
                 (Err_Num   => 10,
                  Reference => ErrorHandler.No_Reference,
                  Position  => STree.Node_Position (Node => Ident_Node),
                  Id_Str    => Ident_Str);
            end if;
            if Adding_Proper_Body then
               Do_Add
                 (Add_Subprog   => True,
                  Add_Body      => True,
                  Hidden        => Hidden,
                  Ident_Str     => Ident_Str,
                  Node_Pos      => STree.Node_Position (Node => Node),
                  First_Seen    => First_Seen,
                  Current_Scope => Current_Scope,
                  Subprog_Sym   => Subprog_Sym);
            else
               Subprog_Sym := Dictionary.NullSymbol;
            end if;
         end if;
      end if;

      if STree.Syntax_Node_Type (Node => Node) = SP_Symbols.function_specification
        and then Subprog_Sym /= Dictionary.NullSymbol then
         -- ASSUME Node = function_specification
         Return_Type_Node := STree.Last_Sibling_Of (Start_Node => STree.Child_Node (Current_Node => Node));
         -- ASSUME Return_Type_Node = type_mark
         SystemErrors.RT_Assert
           (C       => STree.Syntax_Node_Type (Node => Return_Type_Node) = SP_Symbols.type_mark,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = Return_Type_Node = type_mark in Wf_Subprogram_Specification_From_Body");
         Sem.Wf_Type_Mark
           (Node          => Return_Type_Node,
            Current_Scope => Current_Scope,
            Context       => Dictionary.ProgramContext,
            Type_Sym      => Type_Sym);
         if First_Seen then
            if Dictionary.IsUnconstrainedArrayType (Type_Sym) then
               Type_Sym := Dictionary.GetUnknownTypeMark;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 39,
                  Reference => ErrorHandler.No_Reference,
                  Position  => STree.Node_Position (Node => Return_Type_Node),
                  Id_Str    => STree.Node_Lex_String (Node => Return_Type_Node));
            elsif Dictionary.IsPredefinedSuspensionObjectType (Type_Sym) or Dictionary.TypeIsProtected (Type_Sym) then
               Type_Sym := Dictionary.GetUnknownTypeMark;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 905,
                  Reference => ErrorHandler.No_Reference,
                  Position  => STree.Node_Position (Node => Return_Type_Node),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
            Dictionary.AddReturnType
              (TheFunction   => Subprog_Sym,
               TypeMark      => Type_Sym,
               Comp_Unit     => ContextManager.Ops.Current_Unit,
               TypeReference => Dictionary.Location'(Start_Position => STree.Node_Position (Node => Return_Type_Node),
                                                     End_Position   => STree.Node_Position (Node => Return_Type_Node)));
         else -- not First_Seen so check consistency of return type
            if Type_Sym /= Dictionary.GetType (Subprog_Sym) then
               if Dictionary.GetType (Subprog_Sym) = Dictionary.GetUnknownTypeMark then
                  -- remind user that return type on spec was illegal
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 841,
                     Reference => ErrorHandler.No_Reference,
                     Position  => STree.Node_Position (Node => Return_Type_Node),
                     Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
               else
                  -- report inconsistency
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 22,
                     Reference => ErrorHandler.No_Reference,
                     Position  => STree.Node_Position (Node => Return_Type_Node),
                     Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
               end if;
            end if;
         end if;
      end if;
   end Wf_Subprogram_Specification_From_Body;

   ------------------------------------------------------------------------

   procedure Wf_Subprogram_Specification
     (Spec_Node, Anno_Node, Constraint_Node : in     STree.SyntaxNode;
      Current_Scope                         : in     Dictionary.Scopes;
      Current_Context                       : in     Dictionary.Contexts;
      Component_Data                        : in out ComponentManager.ComponentData;
      The_Heap                              : in out Heap.HeapRecord;
      Subprog_Sym                           :    out Dictionary.Symbol) is
      Formal_Part_Node, Null_Constraint_Node : STree.SyntaxNode;

      procedure Wf_Subprogram_Specification_From_Declaration
        (Node            : in     STree.SyntaxNode;
         Current_Scope   : in     Dictionary.Scopes;
         Current_Context : in     Dictionary.Contexts;
         Subprog_Sym     :    out Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --# derives Dictionary.Dict,
      --#         STree.Table                from CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Current_Context,
      --#                                         Current_Scope,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Current_Context,
      --#                                         Current_Scope,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table &
      --#         Subprog_Sym                from ContextManager.Ops.Unit_Stack,
      --#                                         Current_Scope,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table;
      --# pre STree.Syntax_Node_Type (Node, STree.Table) = SP_Symbols.function_specification or
      --#   STree.Syntax_Node_Type (Node, STree.Table) = SP_Symbols.procedure_specification or
      --#   STree.Syntax_Node_Type (Node, STree.Table) = SP_Symbols.entry_specification;
      --# post STree.Table = STree.Table~;
      is
         Return_Type_Node : STree.SyntaxNode;
         Ident_Str        : LexTokenManager.Lex_String;
         Type_Sym         : Dictionary.Symbol;

         ------------------------------------------------------------------------

         -- If we are declaring a subprogram in a package spec and the spec contains
         -- protected types we search each of these to detect re-use of the subprogram
         -- name.  If we don't trap such re-use at this point then we end up with a
         -- legal package spec for which no legal body could be written (since its
         -- implementation would inevitably involve overload resolution of calls made from
         -- within the protected body.  e.g. type PT in package P declares operation K.  Package
         -- P also declares an operation K.  From inside the body of PT, a call to K could refer
         -- to either of the two Ks since both are directly visible.
         function Is_Defined_In_Protected_Type
           (Name          : LexTokenManager.Lex_String;
            Current_Scope : Dictionary.Scopes;
            Is_Private    : Boolean)
           return          Boolean
         --# global in Dictionary.Dict;
         --#        in LexTokenManager.State;
         is
            Result : Boolean := False;
            It     : Dictionary.Iterator;
         begin
            if Dictionary.IsPackage (Dictionary.GetRegion (Current_Scope)) then
               if Is_Private then
                  It := Dictionary.FirstPrivateProtectedType (Dictionary.GetRegion (Current_Scope));
               else
                  It := Dictionary.FirstVisibleProtectedType (Dictionary.GetRegion (Current_Scope));
               end if;
               while not Dictionary.IsNullIterator (It) loop
                  Result :=
                    Dictionary.IsDirectlyDefined
                    (Name    => Name,
                     Scope   => Dictionary.VisibleScope (Dictionary.CurrentSymbol (It)),
                     Context => Dictionary.ProofContext);
                  exit when Result;
                  It := Dictionary.NextSymbol (It);
               end loop;
            end if;
            return Result;
         end Is_Defined_In_Protected_Type;

      begin -- Wf_Subprogram_Specification_From_Declaration
         Ident_Str := STree.Node_Lex_String (Node => STree.Child_Node (Current_Node => STree.Child_Node (Current_Node => Node)));
         if Dictionary.IsDefined
           (Name              => Ident_Str,
            Scope             => Current_Scope,
            Context           => Dictionary.ProofContext,
            Full_Package_Name => False) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => STree.Node_Position (Node => Node),
               Id_Str    => Ident_Str);
            Subprog_Sym := Dictionary.NullSymbol;
         elsif Is_Defined_In_Protected_Type (Name          => Ident_Str,
                                             Current_Scope => Current_Scope,
                                             Is_Private    => False)
           or else Is_Defined_In_Protected_Type (Name          => Ident_Str,
                                                 Current_Scope => Current_Scope,
                                                 Is_Private    => True) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 988,
               Reference => ErrorHandler.No_Reference,
               Position  => STree.Node_Position (Node => Node),
               Id_Str    => Ident_Str);
            Subprog_Sym := Dictionary.NullSymbol;
         else
            Dictionary.AddSubprogram
              (Name          => Ident_Str,
               Comp_Unit     => ContextManager.Ops.Current_Unit,
               Specification => Dictionary.Location'(Start_Position => STree.Node_Position (Node => Node),
                                                     End_Position   => STree.Node_Position (Node => Node)),
               Scope         => Current_Scope,
               Context       => Current_Context,
               Subprogram    => Subprog_Sym);
            if STree.Syntax_Node_Type (Node => Node) = SP_Symbols.entry_specification then
               Dictionary.SetSubprogramIsEntry (Subprog_Sym);
            end if;
         end if;
         if STree.Syntax_Node_Type (Node => Node) = SP_Symbols.function_specification then
            -- ASSUME Node = function_specification
            Return_Type_Node := STree.Last_Sibling_Of (Start_Node => STree.Child_Node (Current_Node => Node));
            -- ASSUME Return_Type_Node = type_mark
            SystemErrors.RT_Assert
              (C       => STree.Syntax_Node_Type (Node => Return_Type_Node) = SP_Symbols.type_mark,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Return_Type_Node = type_mark in Wf_Subprogram_Specification_From_Declaration");
            Sem.Wf_Type_Mark
              (Node          => Return_Type_Node,
               Current_Scope => Current_Scope,
               Context       => Current_Context,
               Type_Sym      => Type_Sym);
            if Dictionary.IsUnconstrainedArrayType (Type_Sym) then
               Type_Sym := Dictionary.GetUnknownTypeMark;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 39,
                  Reference => ErrorHandler.No_Reference,
                  Position  => STree.Node_Position (Node => Return_Type_Node),
                  Id_Str    => STree.Node_Lex_String (Node => Return_Type_Node));
            elsif Dictionary.IsPredefinedSuspensionObjectType (Type_Sym) or else Dictionary.TypeIsProtected (Type_Sym) then
               Type_Sym := Dictionary.GetUnknownTypeMark;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 905,
                  Reference => ErrorHandler.No_Reference,
                  Position  => STree.Node_Position (Node => Return_Type_Node),
                  Id_Str    => LexTokenManager.Null_String);
            elsif Current_Context = Dictionary.ProgramContext
              and then Dictionary.TypeIsTagged (Type_Sym)
              and then (Dictionary.GetScope (Type_Sym) = Current_Scope) then
               -- attempt to declare primitive function with controlling return result
               Type_Sym := Dictionary.GetUnknownTypeMark;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 840,
                  Reference => ErrorHandler.No_Reference,
                  Position  => STree.Node_Position (Node => Return_Type_Node),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
            if Subprog_Sym /= Dictionary.NullSymbol then
               Dictionary.AddReturnType
                 (TheFunction   => Subprog_Sym,
                  TypeMark      => Type_Sym,
                  Comp_Unit     => ContextManager.Ops.Current_Unit,
                  TypeReference => Dictionary.Location'(Start_Position => STree.Node_Position (Node => Return_Type_Node),
                                                        End_Position   => STree.Node_Position (Node => Return_Type_Node)));
               -- mark signature as not wellformed if wf_type_mark has returned the unknown type
               if Type_Sym = Dictionary.GetUnknownTypeMark then
                  Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, Subprog_Sym);
               end if;
            end if;
         end if;
      end Wf_Subprogram_Specification_From_Declaration;

   begin -- Wf_Subprogram_Specification
      Wf_Subprogram_Specification_From_Declaration
        (Node            => Spec_Node,
         Current_Scope   => Current_Scope,
         Current_Context => Current_Context,
         Subprog_Sym     => Subprog_Sym);
      if Subprog_Sym /= Dictionary.NullSymbol then
         Formal_Part_Node := STree.Next_Sibling (Current_Node => STree.Child_Node (Current_Node => Spec_Node));
         -- ASSUME Formal_Part_Node = formal_part OR type_mark OR NULL
         if STree.Syntax_Node_Type (Node => Formal_Part_Node) = SP_Symbols.formal_part then
            -- ASSUME Formal_Part_Node = formal_part
            Sem.Wf_Formal_Part
              (Node             => Formal_Part_Node,
               Current_Scope    => Current_Scope,
               Subprog_Sym      => Subprog_Sym,
               First_Occurrence => True,
               Context          => Current_Context);
         elsif Formal_Part_Node /= STree.NullNode
           and then STree.Syntax_Node_Type (Node => Formal_Part_Node) /= SP_Symbols.type_mark then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Formal_Part_Node = formal_part OR type_mark OR NULL in Wf_Subprogram_Specification");
         end if;
         if STree.Syntax_Node_Type (Node => Anno_Node) = SP_Symbols.procedure_annotation
           or else STree.Syntax_Node_Type (Node => Anno_Node) = SP_Symbols.function_annotation then
            -- ASSUME Anno_Node = procedure_annotation OR function_annotation
            Sem.Wf_Subprogram_Annotation
              (Node          => Anno_Node,
               Current_Scope => Current_Scope,
               Subprog_Sym   => Subprog_Sym,
               First_Seen    => True,
               The_Heap      => The_Heap);
         end if;

         -- Synthesise 'all from all' dependency if necessary.
         if (STree.Syntax_Node_Type (Node => Spec_Node) = SP_Symbols.procedure_specification
                or else STree.Syntax_Node_Type (Node => Spec_Node) = SP_Symbols.entry_specification)
           and then Sem.Needs_Synthetic_Dependency (Subprog_Sym) then
            Sem.Create_Full_Subprog_Dependency
              (Node_Pos    => STree.Node_Position (Node => STree.Parent_Node (Current_Node => Spec_Node)),
               Subprog_Sym => Subprog_Sym,
               Abstraction => Dictionary.IsAbstract,
               The_Heap    => The_Heap);
         end if;

         if STree.Syntax_Node_Type (Node => Constraint_Node) = SP_Symbols.procedure_constraint
           or else STree.Syntax_Node_Type (Node => Constraint_Node) = SP_Symbols.function_constraint then
            -- ASSUME Constraint_Node = procedure_constraint OR function_constraint
            case Current_Context is
               when Dictionary.ProgramContext =>
                  Sem.Wf_Subprogram_Constraint
                    (Node           => Constraint_Node,
                     Scope          => Dictionary.LocalScope (Subprog_Sym),
                     First_Seen     => True,
                     Component_Data => Component_Data,
                     The_Heap       => The_Heap);
               when Dictionary.ProofContext =>
                  Null_Constraint_Node := STree.Child_Node (Current_Node => Constraint_Node);
                  -- ASSUME Null_Constraint_Node = precondition OR return_expression OR NULL
                  if STree.Syntax_Node_Type (Node => Null_Constraint_Node) = SP_Symbols.precondition
                    or else STree.Syntax_Node_Type (Node => Null_Constraint_Node) = SP_Symbols.return_expression then
                     -- ASSUME Null_Constraint_Node = precondition OR return_expression
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 315,
                        Reference => ErrorHandler.No_Reference,
                        Position  => STree.Node_Position (Node => Constraint_Node),
                        Id_Str    => LexTokenManager.Null_String);
                  elsif Null_Constraint_Node /= STree.NullNode then
                     SystemErrors.Fatal_Error
                       (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                        Msg     => "Expect Null_Constraint_Node = precondition OR return_expression OR NULL " &
                          "in Wf_Subprogram_Specification");
                  end if;
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Proof_Function
                       (Comp_Unit   => ContextManager.Ops.Current_Unit,
                        Parse_Tree  => Spec_Node,
                        Scope       => Current_Scope,
                        Subprog_Sym => Subprog_Sym);
                  end if;
            end case;
         end if;
      end if;
   end Wf_Subprogram_Specification;

end Subprogram_Specification;
