-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Dictionary)
procedure InstantiateSubprogramParameters
  (GenericSubprogramSym : in Symbol;
   ActualSubprogramSym  : in Symbol;
   Comp_Unit            : in ContextManager.UnitDescriptors;
   Declaration          : in Location) is

   function SubstituteType (PossiblyGenericType : Symbol;
                            Subprogram          : Symbol) return Symbol
   --# global in Dict;
   is
      Result : Symbol;
   begin
      if TypeIsGeneric (PossiblyGenericType) then
         Result := ActualOfGenericFormal (PossiblyGenericType, Subprogram);
      else
         Result := PossiblyGenericType;
      end if;
      return Result;
   end SubstituteType;

   procedure SubstituteReturnValue
     (GenericSubprogramSym : in Symbol;
      ActualSubprogramSym  : in Symbol;
      Comp_Unit            : in ContextManager.UnitDescriptors;
      Declaration          : in Location)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ActualSubprogramSym,
   --#                   Comp_Unit,
   --#                   Declaration,
   --#                   GenericSubprogramSym;
   is
      GenericReturnType, ActualReturnType, ProofFunction : Symbol;
   begin
      GenericReturnType := RawDict.GetSubprogramReturnType (GenericSubprogramSym);
      if GenericReturnType /= NullSymbol then
         ActualReturnType := SubstituteType (PossiblyGenericType => GenericReturnType,
                                             Subprogram          => ActualSubprogramSym);
         -- set actual return type for the instantiation
         RawDict.SetSubprogramReturnType (ActualSubprogramSym, ActualReturnType);
         -- create an implcit proof function to go with the new Ada function
         RawDict.CreateImplicitProofFunction
           (AdaFunction   => ActualSubprogramSym,
            Comp_Unit     => Comp_Unit,
            Loc           => Declaration.Start_Position,
            ProofFunction => ProofFunction);
         RawDict.SetSubprogramImplicitProofFunction (IsAbstract, ActualSubprogramSym, ProofFunction);
      end if;
   end SubstituteReturnValue;

   ----------------------

   procedure SubstituteParameters (GenericSubprogramSym : in Symbol;
                                   ActualSubprogramSym  : in Symbol)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ActualSubprogramSym,
   --#                   GenericSubprogramSym;
   is
      GenericParameter, GenericParameterType, ActualParameterType : Symbol;
      GenericParameterName                                        : LexTokenManager.Lex_String;
      GenericParameterMode                                        : Modes;
      ThePreviousParameter, TheNewParameter                       : Symbol;
      Comp_Unit                                                   : ContextManager.UnitDescriptors;
      Declaration                                                 : LexTokenManager.Token_Position;
   begin -- SubstituteParameters
      GenericParameter := RawDict.GetSubprogramFirstParameter (GenericSubprogramSym);
      while GenericParameter /= NullSymbol loop
         -- get generic parameter details
         GenericParameterName := RawDict.GetSubprogramParameterName (GenericParameter);
         GenericParameterType := RawDict.GetSubprogramParameterType (GenericParameter);
         GenericParameterMode := RawDict.GetSubprogramParameterMode (GenericParameter);
         Comp_Unit            := RawDict.Get_Symbol_Compilation_Unit (GenericParameter);
         Declaration          := RawDict.Get_Symbol_Location (GenericParameter);
         -- substitute type
         ActualParameterType := SubstituteType (PossiblyGenericType => GenericParameterType,
                                                Subprogram          => ActualSubprogramSym);
         -- create new parameter for instantiation
         RawDict.CreateSubprogramParameter
           (Name       => GenericParameterName,
            Subprogram => ActualSubprogramSym,
            TypeMark   => ActualParameterType,
            Mode       => GenericParameterMode,
            Comp_Unit  => Comp_Unit,
            Loc        => Declaration,
            Parameter  => TheNewParameter);
         -- link it in
         ThePreviousParameter := RawDict.GetSubprogramLastParameter (ActualSubprogramSym);
         if ThePreviousParameter = NullSymbol then
            RawDict.SetSubprogramFirstParameter (ActualSubprogramSym, TheNewParameter);
         else
            RawDict.SetNextSubprogramParameter (ThePreviousParameter, TheNewParameter);
         end if;
         RawDict.SetSubprogramLastParameter (ActualSubprogramSym, TheNewParameter);

         -- move on to next parameter
         GenericParameter := RawDict.GetNextSubprogramParameter (GenericParameter);
      end loop;
   end SubstituteParameters;

   ----------------------

   procedure SubstituteDerives
     (GenericSubprogramSym : in Symbol;
      ActualSubprogramSym  : in Symbol;
      Comp_Unit            : in ContextManager.UnitDescriptors)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                ActualSubprogramSym,
   --#                                Comp_Unit,
   --#                                GenericSubprogramSym &
   --#         SPARK_IO.File_Sys from *,
   --#                                ActualSubprogramSym,
   --#                                Comp_Unit,
   --#                                Dict,
   --#                                GenericSubprogramSym,
   --#                                LexTokenManager.State;
   is
      ExportIt, DependencyIt : Iterator;
      TheExport              : Symbol;
   begin
      ExportIt := FirstExport (IsAbstract, GenericSubprogramSym);
      while not IsNullIterator (ExportIt) loop
         TheExport := CurrentSymbol (ExportIt);
         AddExport
           (Abstraction     => IsAbstract,
            TheProcedure    => ActualSubprogramSym,
            TheExport       => ActualParameterOfGenericParameter (TheExport, ActualSubprogramSym),
            ExportReference => Null_Location,
            Annotation      => Null_Location);

         DependencyIt := FirstDependency (IsAbstract, GenericSubprogramSym, TheExport);
         while not IsNullIterator (DependencyIt) loop
            AddDependency
              (Abstraction     => IsAbstract,
               Comp_Unit       => Comp_Unit,
               TheProcedure    => ActualSubprogramSym,
               TheExport       => ActualParameterOfGenericParameter (TheExport, ActualSubprogramSym),
               TheImport       => ActualParameterOfGenericParameter (CurrentSymbol (DependencyIt), ActualSubprogramSym),
               ImportReference => Null_Location);
            DependencyIt := NextSymbol (DependencyIt);
         end loop;
         ExportIt := NextSymbol (ExportIt);
      end loop;
   end SubstituteDerives;

   ----------------------

begin -- InstantiateSubprogramParameters
   SubstituteReturnValue
     (GenericSubprogramSym => GenericSubprogramSym,
      ActualSubprogramSym  => ActualSubprogramSym,
      Comp_Unit            => Comp_Unit,
      Declaration          => Declaration);

   SubstituteParameters (GenericSubprogramSym => GenericSubprogramSym,
                         ActualSubprogramSym  => ActualSubprogramSym);

   SubstituteDerives
     (GenericSubprogramSym => GenericSubprogramSym,
      ActualSubprogramSym  => ActualSubprogramSym,
      Comp_Unit            => Comp_Unit);

   -- copy pre/posts - just copy pointers
   RawDict.SetSubprogramPrecondition
     (IsAbstract,
      ActualSubprogramSym,
      RawDict.GetSubprogramPrecondition (IsAbstract, GenericSubprogramSym));
   RawDict.SetSubprogramPostcondition
     (IsAbstract,
      ActualSubprogramSym,
      RawDict.GetSubprogramPostcondition (IsAbstract, GenericSubprogramSym));
end InstantiateSubprogramParameters;
