/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * wsdl-headers.c: Emit code for header files
 *
 * Authors:
 *	Dick Porter (dick@ximian.com)
 *
 * Copyright (C) 2001, Ximian, Inc.
 */

#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <libwsdl/wsdl-typecodes.h>
#include <libwsdl/wsdl-typecodes-c.h>

#include "wsdl-parse.h"
#include "wsdl-soap-headers.h"
#include "wsdl-trace.h"
#include "wsdl-locate.h"
#include "wsdl-soap-emit.h"

static void
wsdl_emit_soap_header_typecodes (const wsdl_typecode * const tc,
				 gpointer                    user_data)
{
	FILE *out = (FILE *) user_data;

	wsdl_typecode_write_c_declaration (out, tc);
}

static void
wsdl_emit_soap_header_mm (const wsdl_typecode * const tc, gpointer user_data)
{
	FILE *out = (FILE *) user_data;

	wsdl_typecode_write_c_mm_decl (out, tc);
}

static void
wsdl_emit_soap_headers_binding_operation (
				FILE                               *out, 
				const guchar                       *opns,
				const wsdl_binding_operation *const op)
{
	wsdl_porttype_operation *porttype_op;

	g_assert (op->name != NULL);
	g_assert (op->soap_operation != NULL);

	fprintf (out, "/* BEGIN Binding Operation %s */\n", op->name);

	porttype_op = op->thread_soap_porttype_operation;
	g_assert (porttype_op != NULL);
	g_assert (porttype_op->input != NULL);

	if (op->documentation != NULL) {
		fprintf (out, "/* %s */\n", op->documentation->str);
	}

	if (porttype_op->documentation != NULL) {
		fprintf (out, "/* %s */\n", porttype_op->documentation->str);
	}

	/* Emit the user callback function type */
	fprintf (out,
		 "typedef void (*%s_%s_callback)(SoupErrorCode err, guint response_code, const guchar *response_phrase, SoupEnv *env,\n",
		 opns, op->name);

	if (porttype_op->output != NULL) {
		fprintf (out, "\t/* output args go here */\n");

		wsdl_emit_part_list (out,
				     porttype_op->output->thread_soap_parts,
				     "\t%t\t/* %p */,\n");
	}

	fprintf (out, "\tgpointer user_data);\n");
	fprintf (out, "\n");

	/* And now do the client stub prototype */
	fprintf (out, "extern void %s_%s(SoupEnv *env,\n", opns, op->name);

	if (porttype_op->input != NULL) {
		fprintf (out, "\t/* input parameters go here */\n");

		wsdl_emit_part_list (out, 
				     porttype_op->input->thread_soap_parts,
				     "\t%t %p,\n");
	}

	fprintf (out, "\t%s_%s_callback,\n", opns, op->name);
	fprintf (out, "\tgpointer user_data);\n");
	fprintf (out, "\n");

	/* The client synchronous stub prototype */
	fprintf (out, "extern void %s_%s_sync(SoupEnv *env\n", opns, op->name);

	if (porttype_op->input != NULL) {
		fprintf (out, "\t/* input parameters go here */\n");
		wsdl_emit_part_list (out, 
				     porttype_op->input->thread_soap_parts,
				     "\t, %t\t/* %p */\n");
	}
	if (porttype_op->output != NULL) {
		fprintf (out, "\t/* output parameters go here */\n");
		wsdl_emit_part_list (out,
				     porttype_op->output->thread_soap_parts,
				     "\t,%t *\t/* %p */\n");
	}
	fprintf (out, "\t);\n\n");


	/* And the server callback function type */
	fprintf (out, "typedef void (*%s_%s_server_callback)(SoupEnv *env\n",
		 opns, op->name);

	if (porttype_op->input != NULL) {
		fprintf (out, "\t/* input args go here */\n");

		wsdl_emit_part_list (out, 
				     porttype_op->input->thread_soap_parts,
				     "\t,%t\t/* %p */\n");
	}

	if (porttype_op->output != NULL) {
		fprintf (out, "\t/* output args go here */\n");

		wsdl_emit_part_list (out,
				     porttype_op->output->thread_soap_parts,
				     "\t,%t *\t/* %p */\n");
	}

	fprintf (out, "\t,gpointer user_data);\n");
	fprintf (out, "\n");

	/* And the server skel register prototypes */
	fprintf (out,
		 "extern void %s_%s_soup_register(%s_%s_server_callback callback, gpointer user_data);\n",
		 opns, 
		 op->name, 
		 opns, 
		 op->name);
	fprintf (out, "\n");
	fprintf (out,
		 "extern void %s_%s_soup_register_with_auth(%s_%s_server_callback callback, gpointer user_data, gint allow_types, SoupServerAuthorizeFn auth_cb, gpointer auth_user_data);\n",
		 opns, 
		 op->name, 
		 opns, 
		 op->name);

	fprintf (out, "/* END Binding Operation %s */\n\n", op->name);
}

static void
wsdl_emit_soap_headers_binding (FILE                      *out, 
				const guchar              *opns,
				const wsdl_binding * const binding)
{
	GSList *iter;
	wsdl_porttype *porttype;

	g_assert (binding->name != NULL);

	fprintf (out, "/* BEGIN Binding %s */\n", binding->name);

	if (binding->documentation != NULL) {
		fprintf (out, "/* %s */\n", binding->documentation->str);
	}

	porttype = binding->thread_soap_porttype;
	g_assert (porttype != NULL);

	/* 
	 * For each binding operation, output a function prototype and
	 * a context structure
	 */
	iter = binding->operations;
	while (iter != NULL) {
		wsdl_emit_soap_headers_binding_operation (out, 
							  opns,
							  iter->data);

		iter = iter->next;
	}

	fprintf (out, "/* END Binding %s */\n\n", binding->name);
}

static void
wsdl_emit_soap_headers_service (FILE                      *out, 
				const guchar              *opns,
				const wsdl_service * const service)
{
	GSList *iter;

	g_assert (service->name != NULL);

	fprintf (out, "/* BEGIN Service %s */\n", service->name);

	if (service->documentation != NULL) {
		fprintf (out, "/* %s */\n", service->documentation->str);
	}

	iter = service->thread_soap_ports;
	while (iter != NULL) {
		wsdl_service_port *port = iter->data;

		g_assert (port->thread_soap_binding != NULL);

		wsdl_emit_soap_headers_binding (out, 
						opns,
						port->thread_soap_binding);

		iter = iter->next;
	}

	fprintf (out, "/* END Service %s */\n\n", service->name);
}

/**
 * wsdl_emit_soap_headers:
 * @outdir: a string containing the path to a directory.  This
 * function expects the string to have a trailing '/'.
 * @fileroot: a string containing the root of a filename.  ".h" will
 * be appended to this name.
 * @definitions: a pointer to a #wsdl_definitions structure,
 * containing a set of WSDL elements.
 *
 * Creates the file @outdir/@fileroot.h, and writes C code containing
 * typecode declarations, and function prototypes for client stubs,
 * server skeletons and common code.
 */
void
wsdl_emit_soap_headers (const guchar                  *outdir, 
			const guchar                  *fileroot,
			const wsdl_definitions * const definitions)
{
	FILE *out;
	GSList *iter;
	guchar *filename;
	const guchar *opns;

	filename = g_strconcat (outdir, fileroot, ".h", NULL);
	wsdl_debug (WSDL_LOG_DOMAIN_HEADERS, 
		    G_LOG_LEVEL_DEBUG, 
		    "file: [%s]",
		    filename);

	out = fopen (filename, "w");
	g_free (filename);

	if (out == NULL) {
		g_warning ("Couldn't open %s for writing: %s", 
			   filename,
			   strerror (errno));
		return;
	}

	fprintf (out, "/*\n");
	if (definitions->name != NULL) {
		fprintf (out, " * %s\n", definitions->name);
		opns = definitions->name;
	} else {
		opns = "m";
	}

	fprintf (out, " *\n");
	fprintf (out, " * Automatically generated by soup-wsdl.\n");
	fprintf (out, " */\n");
	fprintf (out, "\n");
	fprintf (out, "#ifndef _%s_H_\n", opns);
	fprintf (out, "#define _%s_H_\n", opns);
	fprintf (out, "\n");

	fprintf (out, "#include <glib.h>\n");
	fprintf (out, "#include <libsoup/soup.h>\n");
	fprintf (out, "#include <libsoup/soup-server.h>\n");
	fprintf (out, "#include <libwsdl/wsdl.h>\n");

	if (definitions->documentation != NULL) {
		fprintf (out, "/* %s */\n", definitions->documentation->str);
		fprintf (out, "\n");
	}

	/* 
	 * call wsdl_typecode_write_c_declaration() for each non-simple
	 * typecode known.
	 */

	wsdl_typecode_foreach (FALSE, wsdl_emit_soap_header_typecodes, out);
	wsdl_typecode_foreach (FALSE, wsdl_emit_soap_header_mm, out);

	iter = definitions->thread_soap_services;
	while (iter != NULL) {
		wsdl_emit_soap_headers_service (out, opns, iter->data);

		iter = iter->next;
	}

	fprintf (out, "#endif /* _%s_H_ */\n", opns);

	fclose (out);
}
