/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * soup-ssl-tunnel.c: RFC 2817 implementation to tunnel SSL through
 * HTTP proxies
 *
 * Authors:
 *      Joe Shaw (joe@ximian.com)
 *
 * Copyright (C) 2002, Ximian, Inc.
 */

#include <glib.h>
#include <string.h>

#include "soup-ssl-tunnel.h"
#include "soup-auth.h"
#include "soup-context.h"
#include "soup-misc.h"
#include "soup-socket.h"
#include "soup-transfer.h"
#include "soup-private.h"

typedef struct {
	SoupConnection        *src_conn;
	
	SoupContext           *dest_ctx;
	SoupMessage           *req;
	GByteArray            *recv_buf;
	guint                  header_len;
	guint                  read_tag;
	SoupConnectCallbackFn  cb;
} SoupSSLData;

static void
ssl_tunnel_data_free (SoupSSLData *sd)
{
	if (sd->dest_ctx)
		soup_context_unref (sd->dest_ctx);

	if (sd->recv_buf)
		g_byte_array_free(sd->recv_buf, TRUE);

	while (g_source_remove_by_user_data (sd))
		continue;

	g_free (sd);
}

/* FIXME: Cut and pasted from soup-queue.c */
static void
ssl_tunnel_encode_http_auth (SoupMessage *msg, GString *header, gboolean proxy_auth)
{
	SoupContext *ctx;
	char *token;

	ctx = proxy_auth ? soup_get_proxy () : msg->context;

	if (ctx->auth) {
		token = soup_auth_authorize (ctx->auth, msg);
		if (token) {
			g_string_sprintfa (
				header, 
				"%s: %s\r\n",
				proxy_auth ? 
				        "Proxy-Authorization" : 
				        "Authorization",
				token);

			g_free (token);
		}
 	}
}

#if 0
static void
soup_ssl_write_done_cb(gpointer user_data)
{
	printf("data written\n");
}

static void
soup_ssl_read_done_cb(const SoupDataBuffer *data,
 		      gpointer user_data)
{
	SoupSSLData *sd = user_data;
	
	printf("data read\n");

	(*sd->cb) (sd->dest_ctx,
		   SOUP_CONNECT_ERROR_NONE,
		   sd->src_conn,
		   sd->req);
	ssl_tunnel_data_free(sd);
}

static SoupTransferDone
soup_ssl_headers_done_cb(const GString *headers,
			 SoupTransferEncoding *encoding,
			 gint *content_len, gpointer data)
{
	SoupSSLData *sd = data;
	SoupHttpVersion version;
	int response_code;
	char *response_phrase;

	printf("headers done\n");

	if (!soup_headers_parse_response(
		    headers->str,
		    headers->len,
		    NULL,
		    &version,
		    &response_code,
		    &response_phrase))
		goto THROW_MALFORMED_HEADER;

	*encoding = SOUP_TRANSFER_CONTENT_LENGTH;
	*content_len = 0;

	return SOUP_TRANSFER_CONTINUE;

THROW_MALFORMED_HEADER:
	soup_message_issue_callback(sd->req, SOUP_ERROR_MALFORMED_HEADER);
	return SOUP_TRANSFER_END;
}

static gboolean
soup_ssl_error_cb(gboolean body_started,
		  SoupMessage *req)
{
	printf("An error happened\n");

	return FALSE;
}
#endif

static void
soup_ssl_tunnel_send_request(SoupSSLData *sd)
{
	SoupMessage *req = sd->req;
	GString *header;
	const SoupUri *suri;
	char *uri;
	SoupContext *proxy;
	GIOChannel *channel;
	gint total_bytes = 0;
	guint bytes_written;
	GIOError error;
	
	header = g_string_new(NULL);
	suri = soup_context_get_uri(req->context);
	uri = g_strdup_printf(
		"%s:%d", suri->host,
		(suri->port != -1) ? suri->port :
		soup_uri_get_default_port(suri->protocol));
	
	g_string_sprintfa(header,
			  req->priv->http_version == SOUP_HTTP_1_1 ?
			  "CONNECT %s HTTP/1.1\r\n" :
			  "CONNECT %s HTTP/1.0\r\n",
			  uri);

	g_free(uri);


	if (!soup_message_get_request_header(req, "Host"))
		g_string_sprintfa(header, "Host: %s\r\n", suri->host);

	if (!soup_message_get_request_header(req, "User-Agent")) {
		g_string_sprintfa(header,
				  "User-Agent: Soup/"VERSION"\r\n");
	}

	proxy = soup_get_proxy();
	if (proxy && soup_context_get_uri(proxy)->user &&
	    !soup_message_get_request_header(req, "Proxy-Authorization"))
		ssl_tunnel_encode_http_auth(req, header, TRUE);

	g_string_append(header, "\r\n");

	channel = soup_connection_get_iochannel(sd->src_conn);

WRITE_DATA:

#if 0
	soup_transfer_write(
		channel,
		header,
		&req->request,
		NULL,
		(SoupWriteDoneFn) soup_ssl_write_done_cb,
		(SoupWriteErrorFn) soup_ssl_error_cb,
		req);

	g_io_channel_unref(channel);
#endif

#if 1
	error = g_io_channel_write(
		channel, header->str + total_bytes, header->len - total_bytes, 
		&bytes_written);

	total_bytes += bytes_written;

	if (error == G_IO_ERROR_AGAIN) goto WRITE_DATA;
	if (error != G_IO_ERROR_NONE) goto CONNECT_ERROR;
#endif

	return;

CONNECT_ERROR:
	(*sd->cb) (sd->dest_ctx,
		   SOUP_CONNECT_ERROR_NETWORK,
		   NULL,
		   sd->req);
	ssl_tunnel_data_free(sd);
}

static SoupConnection *
soup_ssl_tunnel_get_secure_connection(SoupConnection *src_conn, 
				      SoupContext *dest_ctx)
{
	SoupConnection *new_conn;

	new_conn = g_new0(SoupConnection, 1);
	new_conn->server = dest_ctx->server;
	new_conn->context = dest_ctx;
	new_conn->socket = src_conn->socket;
	new_conn->port = dest_ctx->uri->port;
	new_conn->keep_alive = src_conn->keep_alive;
	new_conn->in_use = TRUE;
	new_conn->last_used_id = 0;

	dest_ctx->server->connections =
		g_slist_prepend(dest_ctx->server->connections, new_conn);

	return new_conn;
}

static gboolean
soup_ssl_tunnel_read (GIOChannel* iochannel, 
		      GIOCondition condition, 
		      SoupSSLData *sd)
{
	guchar buf[RESPONSE_BLOCK_SIZE];
	guint bytes_read = 0, total_read = 0;
	GIOError error;
	SoupHttpVersion version;
	int response_code;
	char *response_phrase;

 READ_AGAIN:
	error = g_io_channel_read (iochannel, buf, sizeof (buf), &bytes_read);

	if (error == G_IO_ERROR_AGAIN) {
		if (total_read) goto PROCESS_READ;
		else return TRUE;
	}

	if (error != G_IO_ERROR_NONE) {
		if (total_read) goto PROCESS_READ;
		else goto CONNECT_ERROR;
	}

	if (bytes_read) {
		g_byte_array_append(sd->recv_buf, buf, bytes_read);

		total_read += bytes_read;

		goto READ_AGAIN;
	}

 PROCESS_READ:
	if (!sd->header_len) {
		int index = soup_substring_index(
			sd->recv_buf->data,
			sd->recv_buf->len,
			"\r\n\r\n");

		if (index < 0) return TRUE;

		sd->header_len = index;
	}

	if (sd->header_len) {
		SoupConnection *ssl_conn;

		if (sd->req->response_headers)
			g_hash_table_destroy(sd->req->response_headers);

		sd->req->response_headers = g_hash_table_new(
			soup_str_case_hash, soup_str_case_equal);

		if (!soup_headers_parse_response(
			    sd->recv_buf->data,
			    sd->header_len,
			    sd->req->response_headers,
			    &version,
			    &response_code,
			    &response_phrase))
			goto CONNECT_ERROR;

		sd->req->response_code = response_code;
		sd->req->response_phrase = response_phrase;

#if 0
		printf("PROXY RESPONSE CODE: %d\n"
		       "PROXY RESPONSE PHRASE: %s\n",
		       response_code, response_phrase);
#endif

		if (response_code == 401 || response_code == 407) {
			const char *auth_header;
			SoupAuth *auth;
			SoupContext *ctx;

#if 0
			printf("Auth required.\n");
#endif

			auth_header = soup_message_get_response_header(
				sd->req,
				response_code == 407 ? "Proxy-Authenticate" :
				"WWW-Authenticate");
			if (!auth_header)
				goto CONNECT_ERROR;

			ctx = soup_connection_get_context(sd->src_conn);

			auth = soup_auth_new_from_header(ctx, auth_header);
			if (!auth)
				goto CONNECT_ERROR;

			if (ctx->auth) {
				if (soup_auth_invalidates_prior(
					    auth, ctx->auth))
					soup_auth_free(ctx->auth);
				else {
					soup_auth_free(auth);
					goto CONNECT_ERROR;
				}
			}
			ctx->auth = auth;

			soup_message_queue(sd->req, sd->req->priv->callback,
					   sd->req->priv->user_data);

			g_source_remove(sd->read_tag);
			sd->read_tag = 0;
			ssl_tunnel_data_free (sd);
			return FALSE;
		}

		/* 400 or 500 error codes should abort the tunnel */
		if (response_code > 400) {
			sd->req->response_code = response_code;
			sd->req->response_phrase = response_phrase;
			goto CONNECT_ERROR;
		}

		ssl_conn = soup_ssl_tunnel_get_secure_connection(
			sd->src_conn, sd->dest_ctx);

#if 0
		printf("calling the callback, baby\n");
#endif
		(*sd->cb) (sd->dest_ctx, 
			   SOUP_CONNECT_ERROR_NONE, 
			   ssl_conn,
			   sd->req);
		ssl_tunnel_data_free (sd);
		return FALSE;
	}

	return TRUE;

 CONNECT_ERROR:
	g_source_remove(sd->read_tag);
	sd->read_tag = 0;
	(*sd->cb) (sd->dest_ctx, 
		   SOUP_CONNECT_ERROR_NETWORK, 
		   NULL, 
		   sd->req);
	ssl_tunnel_data_free (sd);
	return FALSE;
}

static gboolean
soup_ssl_tunnel_error (GIOChannel* iochannel, 
		       GIOCondition condition, 
		       SoupSSLData *sd)
{
	(*sd->cb) (sd->dest_ctx, 
		   SOUP_CONNECT_ERROR_NETWORK, 
		   NULL, 
		   sd->req);

	ssl_tunnel_data_free (sd);
	return FALSE;	
}

void
soup_connect_ssl_proxy (SoupConnection        *conn, 
			SoupContext           *dest_ctx, 
			SoupConnectCallbackFn  cb,
			SoupMessage           *req)
{
	SoupSSLData *sd = NULL;
	SoupContext *proxy_ctx;
	const SoupUri *dest_uri, *proxy_uri;
	GIOChannel *channel;

	if (!soup_connection_is_new (conn)) goto CONNECT_SUCCESS;
	
	soup_context_ref (dest_ctx);
	dest_uri = soup_context_get_uri (dest_ctx);

	proxy_ctx = soup_connection_get_context (conn);
	proxy_uri = soup_context_get_uri (proxy_ctx);

	sd = g_new0 (SoupSSLData, 1);
	sd->src_conn = conn;
	sd->dest_ctx = dest_ctx;
	sd->req = req;
	sd->recv_buf = g_byte_array_new();
	sd->cb = cb;

	channel = soup_connection_get_iochannel(conn);

#if 1
	sd->read_tag = g_io_add_watch(
		channel,
		G_IO_IN,
		(GIOFunc) soup_ssl_tunnel_read,
		sd);
	g_io_add_watch(
		channel,
		G_IO_ERR | G_IO_HUP | G_IO_NVAL,
		(GIOFunc) soup_ssl_tunnel_error,
		sd);
#else
	soup_transfer_read(channel,
			   FALSE,
			   (SoupReadHeadersDoneFn) soup_ssl_headers_done_cb,
			   NULL,
			   (SoupReadDoneFn) soup_ssl_read_done_cb,
			   (SoupReadErrorFn) soup_ssl_error_cb,
			   sd);

	g_io_channel_unref(channel);
#endif

	soup_ssl_tunnel_send_request(sd);

	return;
	
 CONNECT_SUCCESS:
	(*cb) (dest_ctx, SOUP_CONNECT_ERROR_NONE, conn, req); 
	g_free (sd);
}
