<?php
/**
 * Forwards_Driver_plesk implements the Forwards_Driver API for Plesk control
 * panel servers.
 *
 * Plesk 8.1 or later is required.
 *
 * $Horde: forwards/lib/Driver/plesk.php,v 1.2.2.2 2009/01/06 15:22:46 jan Exp $
 *
 * Copyright 2008-2009 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (BSDL). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Jan Schneider <jan@horde.org>
 * @since   Forwards 3.1
 * @package Forwards
 */
class Forwards_Driver_plesk extends Forwards_Driver {

    /**
     * The curl resource handler
     *
     * @var resource
     */
    var $_curl;

    /**
     * The Plesk domain id of the current realm.
     *
     * @var integer
     */
    var $_domain_id;

    /**
     * The current forwards details.
     *
     * @var array
     */
    var $_details = null;

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     */
    function enableForwarding($password, $target, $keeplocal = false)
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig($password), 'PEAR_Error')) {
            return $checked;
        }

        // Query the server.
        @list($user,) = explode('@', $this->_user);
        $request = '<mail><update><set><filter><domain_id>'
            . $this->_domain_id . '</domain_id><mailname><name>'
            . htmlspecialchars($user)
            . '</name><redirect><enabled>true</enabled>'
            . '<address>' . htmlspecialchars($target) . '</address>'
            . '</redirect></mailname></filter>'
            . '</set></update></mail>';
        $result = $this->_request($password, $request);

        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        if (isset($result['mail']['update']['set']['result'])) {
            $result = $result['mail']['update']['set']['result'];
        } else {
            $result = false;
        }
        if (isset($result['status']['_']) &&
            $result['status']['_'] == 'error') {
            return PEAR::raiseError(sprintf(_("Cannot set forwarding for mail user %s: %s"),
                                            $this->_user, $result['errtext']['_']));
        }
        if (!isset($result['status']['_']) ||
            $result['status']['_'] != 'ok' ||
            empty($result)) {
            return PEAR::raiseError(sprintf(_("Cannot set forwarding for mail user %s."),
                                            $this->_user));
        }
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $password  The password of the user.
     */
    function disableForwarding($password)
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig($password), 'PEAR_Error')) {
            return $checked;
        }

        // Get current forward address.
        if (is_a($target = $this->_getUserDetails($password), 'PEAR_Error')) {
            return $target;
        }
        if ($target === false) {
           return PEAR::raiseError(_("Cannot determine the current forward address."));
        }

        // Query the server.
        @list($user,) = explode('@', $this->_user);
        $request = '<mail><update><set><filter><domain_id>'
            . $this->_domain_id . '</domain_id><mailname><name>'
            . htmlspecialchars($user)
            . '</name><redirect><enabled>false</enabled>'
            . '<address>' . htmlspecialchars($target) . '</address>'
            . '</redirect></mailname></filter>'
            . '</set></update></mail>';
        $result = $this->_request($password, $request);

        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        if (isset($result['mail']['update']['set']['result'])) {
            $result = $result['mail']['update']['set']['result'];
        } else {
            $result = false;
        }
        if (isset($result['status']['_']) &&
            $result['status']['_'] == 'error') {
            return PEAR::raiseError(sprintf(_("Cannot remove vacation notice for mail user %s: %s"), $this->_user, $result['errtext']['_']));
        }
        if (!isset($result['status']['_']) ||
            $result['status']['_'] != 'ok' ||
            empty($result)) {
            return PEAR::raiseError(sprintf(_("Cannot remove vacation notice for mail user %s."), $this->_user));
        }

        $this->_details = false;
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @param string $password  The password of the user.
     *
     * @return mixed  Returns 'Y' if forwarding is enabled for the user, 'N' if
     *                forwarding is currently disabled, false if the status
     *                cannot be determined, and PEAR_Error on error.
     */
    function isEnabledForwarding($password)
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig($password), 'PEAR_Error')) {
            return $checked;
        }
        if (is_a($details = $this->_getUserDetails($password), 'PEAR_Error')) {
            return $details;
        }

        return $details === false ? 'N' : 'Y';
    }

    /**
     * Checks if user is keeping a local copy of forwarded mail.
     *
     * @param string $password  The password of the user.
     *
     * @return boolean  True if user is keeping a local copy of mail,
     *                  otherwise false.
     */
    function isKeepLocal($password)
    {
        return true;
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @param string $password  The password of the user.
     *
     * @return string  The current forwarding mail address, false if no
     *                 forwarding is set, or PEAR_Error on error.
     */
    function currentTarget($password)
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig($password), 'PEAR_Error')) {
            return $checked;
        }
        return $this->_getUserDetails($password);
    }

    /**
     * Retrieves the current forwards details for the user.
     *
     * @param string $password  The password for user.
     *
     * @return mixed  Target address if enabled, false if disabled, PEAR_Error
     *                on failure.
     */
    function _getUserDetails($password)
    {
        if (!is_null($this->_details)) {
            return $this->_details;
        }

        if (is_a($checked = $this->_checkConfig($password),
                 'PEAR_Error')) {
            return $checked;
        }
        @list($user,) = explode('@', $this->_user);

        // Query the server.
        $request = '<mail><get_info><filter><domain_id>'
            . $this->_domain_id . '</domain_id><name>'
            . htmlspecialchars($user)
            . '</name></filter><redirect/></get_info></mail>';
        $result = $this->_request($password, $request);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        if (isset($result['mail']['get_info']['result']['status']['_']) &&
            $result['mail']['get_info']['result']['status']['_'] == 'error') {
            return PEAR::raiseError(sprintf(_("Cannot retrieve information about mail user %s: %s"), $this->_user, $result['mail']['get_info']['result']['errtext']['_']));
        } elseif (!isset($result['mail']['get_info']['result']['status']['_']) ||
                  $result['mail']['get_info']['result']['status']['_'] != 'ok' ||
                  !isset($result['mail']['get_info']['result']['mailname'])) {
            return PEAR::raiseError(sprintf(_("Cannot retrieve information about mail user %s."), $this->_user));
        }

        if (!isset($result['mail']['get_info']['result']['mailname']['redirect']) ||
            $result['mail']['get_info']['result']['mailname']['redirect']['enabled']['_'] == 'false') {
            $this->_details = false;
        } else {
            $this->_details = $result['mail']['get_info']['result']['mailname']['redirect']['address']['_'];
        }

        return $this->_details;
    }

    /**
     * Checks if the realm has a specific configuration. If not, tries to fall
     * back on the default configuration. If still not a valid configuration
     * then returns an error.
     *
     * @param string $password  The password for the user.
     */
    function _checkConfig($password)
    {
        if (!(@include_once 'Horde/DOM.php')) {
            return PEAR::raiseError(_("The Plesk driver requires the Horde_DOM package from Horde 3.2 or later. See http://pear.horde.org/index.php?package=Horde_DOM"));
        }

        // If no realm passed in, or no host config for the realm passed in,
        // then we fall back to the default realm
        if (empty($this->_params[$this->_realm]['host'])) {
            $this->_realm = 'default';
        }

        if (!isset($this->_domain_id)) {
            @list(, $domain) = explode('@', $this->_user, 2);
            $request = '<domain><get><filter><domain_name>'
                . htmlspecialchars($domain)
                . '</domain_name></filter><dataset><gen_info/></dataset></get></domain>';
            $result = $this->_request($password, $request);
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }
            if (isset($result['domain']['get']['result']['status']['_']) &&
                $result['domain']['get']['result']['status']['_'] == 'ok' &&
                isset($result['domain']['get']['result']['id']['_'])) {
                $this->_domain_id = $result['domain']['get']['result']['id']['_'];
            } elseif (isset($result['domain']['get']['result']['status']['_']) &&
                      $result['domain']['get']['result']['status']['_'] == 'error') {
                return PEAR::raiseError(sprintf(_("Cannot retrieve domain ID of domain %s: %s"), $domain, $result['domain']['get']['result']['errtext']['_']));
            } else {
                return PEAR::raiseError(sprintf(_("Cannot retrieve domain ID of domain %s."), $domain));
            }
        }

        // If still no host/port, then we have a misconfigured module.
        if (empty($this->_params[$this->_realm]['host']) ||
            empty($this->_params[$this->_realm]['user']) ||
            empty($this->_params[$this->_realm]['pass'])) {
            return PEAR::raiseError(_("The forwards application is not properly configured."));
        }
    }

    /**
     * Connects to the Plesk RPC API server and sends a request.
     *
     * @param string $password  The password to connect with.
     * @param string $packet    The XML fragment for the request.
     *
     * @return boolean  True on success, PEAR_Error otherwise.
     */
    function _request($password, $packet)
    {
        if (!$this->_curl) {
            $url = 'https://' . $this->_params[$this->_realm]['host']
                . ':8443/enterprise/control/agent.php';
            $headers = array(
                'HTTP_AUTH_LOGIN: ' . $this->_params[$this->_realm]['user'],
                'HTTP_AUTH_PASSWD: ' . $this->_params[$this->_realm]['pass'],
                'Content-Type: text/xml');
            $this->_curl = curl_init();
            curl_setopt($this->_curl, CURLOPT_SSL_VERIFYHOST, 0);
            curl_setopt($this->_curl, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($this->_curl, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($this->_curl, CURLOPT_URL, $url);
            curl_setopt($this->_curl, CURLOPT_RETURNTRANSFER, true);
        }

        $content = '<?xml version="1.0" encoding="' . NLS::getCharset()
            . '"?><packet version="1.4.2.0">' . $packet . '</packet>';
        curl_setopt($this->_curl, CURLOPT_POSTFIELDS, $content);
        $retval = curl_exec($this->_curl);
        if ($retval === false) {
            return PEAR::raiseError(curl_error($this->_curl));
        }

        $doc = Horde_DOM_Document::factory(
            array('xml' => $retval,
                  'options' => HORDE_DOM_LOAD_REMOVE_BLANKS));
        $result = array();
        $this->_parseResponse($doc->root(), $result);
        if (isset($result['packet']['system']['status']['_']) &&
            $result['packet']['system']['status']['_'] == 'error') {
            return PEAR::raiseError($result['packet']['system']['errtext']['_']);
        }

        return $result['packet'];
    }

    /**
     * Parses the XML response body of the Plesk API call into a hash.
     *
     * @param Horde_DOM_Node $node  A DOM node object.
     * @param array $array          The result hash.
     */
    function _parseResponse($node, &$array)
    {
        $name = $node->node_name();
        $element = array();
        if (isset($array[$name])) {
            $array[$name] = array($array[$name]);
            $array[$name][] = &$element;
        } else {
            $array[$name] = &$element;
        }
        $array[$name] = array();
        if ($node->has_child_nodes()) {
            for ($child = $node->first_child();
                 $child;
                 $child = $child->next_sibling())  {
                if ($child->node_type() == XML_TEXT_NODE) {
                    $array[$name]['_'] = $child->node_value();
                } else {
                    $this->_parseResponse($child, $array[$name]);
                }
            }
        }
    }

}
